<?php

require_once 'SOAP/Client.php';

/**
 * Forwards_Driver_soap:: implements the Forwards_Driver API for SOAP
 * services.
 *
 * $Horde: forwards/lib/Driver/soap.php,v 1.8.2.2 2007/01/02 13:54:06 jan Exp $
 *
 * Copyright 2004-2007 Marc Jauvin <marc@r4l.com>
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Marc Jauvin <marc@r4l.com>
 * @since   Forwards 3.0
 * @package Forwards
 */
class Forwards_Driver_soap extends Forwards_Driver {

    /**
     * The information that was retrieved from the SOAP server.
     *
     * @var array
     */
    var $_current_details;

    /**
     * Is the SOAP client initialized?
     *
     * @var boolean
     */
    var $_initialised = false;

    /**
     * Initialise the SOAP connection object.
     *
     * @return boolean  True or False based on success of connect.
     */
    function _initialise()
    {
        if (!$this->_initialised) {
            Horde::assertDriverConfig($this->_params, 'server',
                array(
                    'soap_proxy',
                    'soap_uri',
                    'fwd_fetch_func',
                    'fwd_enable_func',
                    'fwd_disable_func',
                    'username_varname',
                    'target_varname',
                    'keeplocal_varname',
                ),
                'SOAP initialisation'
            );

            $this->_client = &new SOAP_Client($this->_params['soap_proxy']);
            $this->_client->setOpt('curl', CURLOPT_FAILONERROR, false);
            $this->_client->setOpt('curl', CURLOPT_SSL_VERIFYPEER, false);

            $this->_initialised = true;
        }

        return true;
    }

    /**
     * Retrieves user details from the backend.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  SOAP result resource or (boolean) False.
     */
    function _getUserDetails($user, $realm, $password)
    {
        // _initialise() will die with Horde::fatal() upon failure.
        $this->_initialise();

        if ($this->_current_details) {
            return $this->_current_details;
        }

        // Build username.
        $account = $this->_buildUsername($user, $realm);

        $data[$this->_params['username_varname']] = $account;
        $args = &new SOAP_Value("args", "array", $data);

        $result = $this->_client->call($this->_params['fwd_fetch_func'], $args, $this->_params['soap_uri']);

        if (is_a($result, 'PEAR_Error')) {
            $this->_current_details = array();
            return false;
        } else {
            $this->_current_details = get_object_vars($result);
            return $this->_current_details;
        }
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if forwarding is enabled, or false otherwise.
     */
    function isEnabledForwarding($user, $realm, $password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails($user, $realm, $password);
        if ($current_details === false) {
            return false;
        }

        // Check forwarding flag.
        if ($current_details[$this->_params['target_varname']]) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($user, $realm, $password)
    {
        // Get current details.
        $current_details = $this->_getUserDetails($user, $realm, $password);
        if ($current_details === false) {
            return false;
        }

        if ($current_details[$this->_params['keeplocal_varname']]) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or false on error.
     */
    function currentTarget($user, $realm, $password)
    {
        $current_details = $this->_getUserDetails($user, $realm, $password);

        if ($current_details[$this->_params['target_varname']]) {
            return $current_details[$this->_params['target_varname']];
        } else {
            return false;
        }
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm, $password, $target, $keeplocal)
    {
        // _initialise() will die with Horde::fatal() upon failure.
        $this->_initialise();

        // Build username.
        $account = $this->_buildUsername($user, $realm);

        $data[$this->_params['username_varname']] = $account;
        if ($password && $this->_params['password_varname']) {
            $data[$this->_params['password_varname']] = $password;
        }
        $data[$this->_params['target_varname']] = $target;
        $data[$this->_params['keeplocal_varname']] = ($keeplocal === 'on') ? 1 : 0;

        $args = &new SOAP_Value('args', 'array', $data);

        $result = $this->_client->call($this->_params['fwd_enable_func'], $args, $this->_params['soap_uri']);
        if (is_a($result, 'PEAR_Error')) {
            $this->_current_details = array();
            return false;
        }

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm, $password)
    {
        // _initialise() will die with Horde::fatal() upon failure.
        $this->_initialise();

        // Build username.
        $account = $this->_buildUsername($user, $realm);

        $data[$this->_params['username_varname']] = $account;
        if ($password && $this->_params['password_varname']) {
            $data[$this->_params['password_varname']] = $password;
        }

        $args = &new SOAP_Value("args", "array", $data);

        $result = $this->_client->call($this->_params['fwd_disable_func'], $args, $this->_params['soap_uri']);
        if (is_a($result, 'PEAR_Error')) {
            $this->_current_details = array();
            return false;
        }

        return true;
    }

    /**
     * Builds a username based on presence of realm.
     *
     * @param string $user   Username.
     * @param string $realm  Realm name.
     *
     * @return string  Fully qualified username.
     */
    function _buildUsername($user, $realm)
    {
        if ($realm === 'default' ||
            $realm == '') {
            return $user;
        } else {
            return $user . '@' . $realm;
        }
    }

}
