/***************************************************************************
    smb4ksharewidget.cpp  -  The widget, that displays the mounted shares.
                             -------------------
    begin                : Sam M� 1 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlabel.h>
#include <qlayout.h>
#include <qdir.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kpopupmenu.h>
#include <kiconloader.h>

// application specific includes
#include "smb4ksharewidget.h"
#include "smb4ksharewidgetitem.h"
// #include "smb4kmountdialog.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kglobal.h"

using namespace Smb4KGlobal;



Smb4KShareWidget::Smb4KShareWidget( QWidget *parent, const char *name ) : KIconView( parent, name )
{
  setFrameShape( KIconView::PopupPanel );
  setFrameShadow( KIconView::Sunken );
  setSelectionMode( KIconView::Single );
  setResizeMode( KIconView::Adjust );
  setShowToolTips( false );
  setAutoArrange( true );
  setSorting( true, true );

  m_collection = new KActionCollection( this, "SharesView_ActionCollection", KGlobal::instance() );

  m_menu = new Smb4KShareActionMenu( Smb4KShareActionMenu::Full, m_collection, QString::null, QIconSet(), this, "SharesView_ActionMenu" );
  m_menu->popupMenu()->insertTitle( SmallIcon( "hdd_mount" ), i18n( "Shares" ), 0, 0 );

  m_tooltip = NULL;

  initActions();

  // Internal connections
  connect( this, SIGNAL( mouseButtonPressed( int, QIconViewItem *, const QPoint & ) ),
           this, SLOT( slotMouseButtonPressed( int, QIconViewItem *, const QPoint & ) ) );
  connect( this, SIGNAL( executed( QIconViewItem * ) ),
           this, SLOT( slotOpenFilemanager() ) );
  connect( this, SIGNAL( selectionChanged( QIconViewItem * ) ),
           this, SLOT( slotSelectionChanged( QIconViewItem * ) ) );

  connect( smb4k_core->mounter(), SIGNAL( updated() ),
           this,                  SLOT( slotMountedShares() ) );
}


Smb4KShareWidget::~Smb4KShareWidget()
{
  delete m_menu;
  delete m_tooltip;

  uint index = 0;

  while ( index < m_collection->count() )
  {
    delete m_collection->action( index++ );
  }

  m_collection->clear();
}


/****************************************************************************
   Initialize the actions needed by this class.
****************************************************************************/

void Smb4KShareWidget::initActions()
{
  m_unmount_action = m_menu->unmountAction();
  m_unmount_action->setGroup( "ShareWidget" );
  m_unmount_action->setEnabled( false );
  connect( m_unmount_action, SIGNAL( activated() ), this, SLOT( slotUnmountShare() ) );

  m_force_action = m_menu->forcedUnmountAction();
  m_force_action->setGroup( "ShareWidget" );
  m_force_action->setEnabled( false );
  connect( m_force_action, SIGNAL( activated() ), this, SLOT( slotForceUnmountShare() ) );

  m_unmount_all_action = m_menu->unmountAllAction();
  m_unmount_all_action->setGroup( "ShareWidget" );
  m_unmount_all_action->setEnabled( false );
  connect( m_unmount_all_action, SIGNAL( activated() ), this, SLOT( slotUnmountAllShares() ) );

  m_filemanager_action = m_menu->filemanagerAction();
  m_filemanager_action->setGroup( "ShareWidget" );
  m_filemanager_action->setEnabled( false );
  connect( m_filemanager_action, SIGNAL( activated() ), this, SLOT( slotOpenFilemanager() ) );

  m_sync_action = m_menu->syncAction();
  m_sync_action->setGroup( "ShareWidget" );
  m_sync_action->setEnabled( false );
  connect( m_sync_action, SIGNAL( activated() ), this, SLOT( slotSynchronize() ) );
}


/****************************************************************************
   Reads the options for the share view widget.
****************************************************************************/

void Smb4KShareWidget::readOptions()
{
  // Which appearance does the user want?
  config()->setGroup( "Appearance" );
  m_display = config()->readEntry( "Show Shares", "icons" );

  if ( QString::compare( m_display, "icons" ) == 0 )
  {
    setItemTextPos( KIconView::Bottom );
    setMaxItemWidth( 150 );
    setArrangement( KIconView::LeftToRight );
    setWordWrapIconText( true );
  }
  else if ( QString::compare( m_display, "list" ) == 0 )
  {
    setItemTextPos( KIconView::Right );
    setMaxItemWidth( 500 );
    setArrangement( KIconView::TopToBottom );
    setWordWrapIconText( false );
  }

  m_mountpoint = config()->readBoolEntry( "Show Mount Point", false );

  // Should all share be shown, that were mounted on the system?
  m_showExternal = config()->readBoolEntry( "Show All Shares", false );

  if ( count() != 0 )
  {
    for ( QIconViewItem *item = this->firstItem(); item; item = item->nextItem() )
    {
      int alignment = Smb4KShareWidgetItem::Icons;

      if ( QString::compare( m_display, "list" ) == 0 )
      {
        alignment = Smb4KShareWidgetItem::List;
      }

      ((Smb4KShareWidgetItem *)item)->setupItem( alignment, m_mountpoint );
    }
  }

  slotMountedShares();

  adjustItems();

  // Mount options.
  config()->setGroup( "Programs" );
  QString super = config()->readPathEntry( "super" );
  QString sudo = config()->readPathEntry( "sudo" );
  m_rsync = (config()->readPathEntry( "rsync" )).isEmpty() ? false : true;

  config()->setGroup( "Super User Privileges" );

  if ( !super.isEmpty() || !sudo.isEmpty() )
  {
    m_forcedUnmount = config()->readBoolEntry( "Force Unmount", false );
  }
  else
  {
    m_forcedUnmount = false;
  }
}


void Smb4KShareWidget::contentsMouseMoveEvent( QMouseEvent *e )
{
  Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)findItem( e->pos() );

  if ( item )
  {
    config()->setGroup( "Appearance" );

    if ( !m_tooltip && hasMouse() && config()->readBoolEntry( "Show Share Tooltip", true ) )
    {
      m_tooltip = new Smb4KShareTooltip( item, e->globalPos() );

      QTimer::singleShot( 2000, this, SLOT( slotShowTooltip() ) );
    }
  }
  else
  {
    if ( m_tooltip )
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
    else
    {
      m_tooltip = NULL;
    }
  }

  QIconView::contentsMouseMoveEvent( e );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KShareWidget::slotMouseButtonPressed( int button, QIconViewItem *item, const QPoint &pos )
{
  if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    // Enable the actions:
    if ( m_forcedUnmount )
    {
      m_force_action->setEnabled( true );
    }

    if ( !i->isBroken() )
    {
      m_filemanager_action->setEnabled( true );
      m_sync_action->setEnabled( m_rsync ? true : false );
    }
    else
    {
      m_filemanager_action->setEnabled( false );
      m_sync_action->setEnabled( false );
    }

    m_unmount_action->setEnabled( true );
  }
  else
  {
    m_unmount_action->setEnabled( false );
    m_filemanager_action->setEnabled( false );
    m_sync_action->setEnabled( false );
    m_force_action->setEnabled( false );

    clearSelection();
  }

  if ( button == KIconView::RightButton )
  {
    if ( item )
    {
      m_menu->popupMenu()->changeTitle( 0, SmallIcon( "hdd_mount" ), ((Smb4KShareWidgetItem *)item)->name() );
    }
    else
    {
      m_menu->popupMenu()->changeTitle( 0, SmallIcon( "hdd_mount" ), i18n( "Shares" ) );
    }

    m_menu->popupMenu()->exec( pos, 0 );
  }
}


void Smb4KShareWidget::slotSelectionChanged( QIconViewItem *item )
{
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    if ( !i->isBroken() )
    {
      m_filemanager_action->setEnabled( true );
      m_sync_action->setEnabled( m_rsync ? true : false );
    }
    else
    {
      m_filemanager_action->setEnabled( false );
      m_sync_action->setEnabled( false );
    }

    if ( m_forcedUnmount )
    {
      m_force_action->setEnabled( true );
    }

    m_unmount_action->setEnabled( true );
  }
}


void Smb4KShareWidget::slotShowTooltip()
{
  if ( m_tooltip )
  {
    if ( hasMouse() )
    {
      m_tooltip->showTip();
    }
    else
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }
}


/****************************************************************************
   External slots.
****************************************************************************/

void Smb4KShareWidget::slotMountedShares()
{
  QValueList<Smb4KShare *> shares = smb4k_core->mounter()->getShares();
  Smb4KShareWidgetItem *current = (Smb4KShareWidgetItem *)currentItem();

  if ( !shares.isEmpty() )
  {
    Smb4KShareWidgetItem *test_item = (Smb4KShareWidgetItem *)firstItem();
    Smb4KShareWidgetItem *next_item = NULL;

    // First remove those, that are not mounted anymore:
    while ( test_item )
    {
      Smb4KShare *share = smb4k_core->mounter()->findShareByPath( test_item->mountpoint() );
      next_item = (Smb4KShareWidgetItem *)test_item->nextItem();

      if( !share || (test_item->shareObject()->isForeign() && !m_showExternal) )
      {
        delete test_item;
        test_item = NULL;

        sort( sortDirection() );
      }

      test_item = next_item;
    }

    // Now process the entries in the list:
    for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
    {
      // Check, whether the share is already in the list. Look for the
      // mount point.
      bool alreadyInView = false;

      for ( Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)this->firstItem(); item; item = (Smb4KShareWidgetItem *)item->nextItem() )
      {
        if ( QString::compare( item->mountpoint(), (*it)->path() ) == 0 ||
             QString::compare( item->mountpoint(), (*it)->canonicalPath() ) == 0 )
        {
          if ( *it != item->shareObject() )
          {
            item->replaceShareObject( *it );
          }

          alreadyInView = true;
          break;
        }
        else
        {
          continue;
        }
      }

      if ( !alreadyInView )
      {
        if ( QString::compare( m_display, "icons" ) == 0 )
        {
          if ( !m_showExternal && (*it)->isForeign() )
          {
            continue;
          }
          else
          {
            new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::Icons, m_mountpoint );

            continue;
          }
        }
        else if ( QString::compare( m_display, "list" ) == 0 )
        {
          if ( !m_showExternal && (*it)->isForeign() )
          {
            continue;
          }
          else
          {
            new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::List, m_mountpoint );

            continue;
          }
        }

        // Do the sorting:
        this->sort( this->sortDirection() );
      }
    }

    if ( current != (Smb4KShareWidgetItem *)currentItem() )
    {
      m_unmount_action->setEnabled( false );
      m_force_action->setEnabled( false );
      m_filemanager_action->setEnabled( false );
      m_sync_action->setEnabled( false );

      clearSelection();
    }

    m_unmount_all_action->setEnabled( true );
  }
  else
  {
    delete current;
    clear();

    m_unmount_action->setEnabled( false );
    m_force_action->setEnabled( false );
    m_unmount_all_action->setEnabled( false );
    m_filemanager_action->setEnabled( false );
    m_sync_action->setEnabled( false );
  }
}


/***************************************************************************
   Slots for the menu.
***************************************************************************/


void Smb4KShareWidget::slotOpenFilemanager()
{
  if ( currentItem() && !((Smb4KShareWidgetItem *)currentItem())->isBroken() )
  {
    smb4k_core->open( ((Smb4KShareWidgetItem *)currentItem())->shareObject() );
  }
}


void Smb4KShareWidget::slotUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    smb4k_core->mounter()->unmountShare( i->shareObject() );
  }
}


void Smb4KShareWidget::slotUnmountAllShares()
{
  smb4k_core->mounter()->unmountAllShares();
}


void Smb4KShareWidget::slotForceUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    smb4k_core->mounter()->forceUnmountShare( i->shareObject() );
  }
}


void Smb4KShareWidget::slotSynchronize()
{
  smb4k_core->synchronizer()->synchronize( ((Smb4KShareWidgetItem *)currentItem())->shareObject() );
}

#include "smb4ksharewidget.moc"
