
% imdisplay.sl: Render a stack of images or display an animation. {{{
%
%		The image specifications passed in may be names of files
%		(of various types), GDK pixbufs, or raw S-Lang arrays
%		(2D/greyscale, 3D/RGB, or 3D/RGBA).  If any input image
%		contains an alpha channel then the rendered result will
%		as well.  Input image files may be of any type readable
%		by a GdkPixbuf loader module (including FITS), while
%		output may be saved to any writable GdkPixbuf format
%		(again, including FITS).
%
%		Hint: don't rely upon XV (which is otherwise great!) to
%		judge whether transparency has been correctly preserved
%		in output files.  Most "standard" installations of XV do
%		not really support layering, and my observation,
%		corroborated by others on the WWW, is that it will assign
%		colors to transparent pixels in an inconsistent manner
%		(sometimes white, sometimes black, ...).  Instead, try
%		using a more actively developed tool, and with better RGBA
%		support, such as the GIMP or ImageMagick.
%		
%		This file is part of SLgtk, the S-Lang bindings to Gtk.
%		Copyright (C) 2003-2005 Massachusetts Institute of Technology
%		Copyright (C) 2002 Michael S. Noble (mnoble@space.mit.edu)
% }}}

require("gtk");

static define load(image, ctx)	% {{{
{
   variable pb = NULL;

   if (ctx.animation) return;

   switch( typeof(image))
   { case String_Type:
      
	pb = gdk_pixbuf_animation_new_from_file(image);
  	if (pb != NULL) {
	   if (gdk_pixbuf_animation_is_static_image(pb)) {
		pb = gdk_pixbuf_animation_get_static_image(pb);
		image = path_basename(image);
	   }
	   else {
		ctx.title = image;
		ctx.animation = 1;
		ctx.w = gdk_pixbuf_animation_get_width(pb);
		ctx.h = gdk_pixbuf_animation_get_height(pb);
		ctx.pixbuf = pb;
		return;
	   }
	}
   }
   { case Array_Type:

	variable dims; (dims, ,) = array_info(image);
	if (length(dims) == 2) {
	   variable rgbbuf = UChar_Type[dims[0], dims[1], 3];
	   image = norm_array(image);
	   rgbbuf[*,*,0] = image;
	   rgbbuf[*,*,1] = image;
	   rgbbuf[*,*,2] = image;
	   image = rgbbuf;
	}
      	pb = gdk_pixbuf_new_from_data(image);
   }
   { case GdkPixbuf:  pb = image; }

   if (pb == NULL) {
	() = fprintf(stderr,"Could not find or create pixbuf from: %S\n",image);
	return;
   }

   variable w = gdk_pixbuf_get_width(pb), h = gdk_pixbuf_get_height(pb);
   if (ctx.pixbuf == NULL) {
	ctx.pixbuf = pb;
	ctx.w = w; ctx.h = h;
	ctx.has_alpha = gdk_pixbuf_get_has_alpha(ctx.pixbuf);
	ctx.title = sprintf("Imdisplay: %S",image);
	return;
   }

   if (gdk_pixbuf_get_has_alpha(pb) and not(ctx.has_alpha)) {
	ctx.pixbuf = gdk_pixbuf_add_alpha(ctx.pixbuf, FALSE, 0, 0, 0);
	ctx.has_alpha = 1;
   }

   if (w != ctx.w or h != ctx.h) {

	variable w2 = max([ctx.w, w]), h2 = max ([ctx.h, h]);
	variable bigger = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, w2, h2);
	gdk_pixbuf_fill(bigger, 0xFFFFFF00);	% 100% transparent, white

	% Copy existing image into new pixbuf, then composite new image
	% Pixels not covered by these two operations remain transparent
	gdk_pixbuf_copy_area(ctx.pixbuf, 0, 0, ctx.w, ctx.h, bigger, 0, 0);
   	gdk_pixbuf_composite(pb, bigger, 0, 0, w, h, 0, 0, 1, 1,
						GDK_INTERP_BILINEAR, 255);
	ctx.pixbuf = bigger;
	ctx.w = w2; ctx.h = h2;
	ctx.has_alpha = 1;
   }
   else
	gdk_pixbuf_composite(pb, ctx.pixbuf, 0, 0, w, h, 0, 0, 1, 1,
						GDK_INTERP_BILINEAR, 255);

} % }}}

% Expose handling {{{
static variable RESIZE_NONE   = 0;
static variable RESIZE_TILED  = 1;
static variable RESIZE_SCALED = 2;

static define image_size(ctx)
{
   variable w, h;

   switch(ctx.resize_style)
   { case RESIZE_NONE: w = ctx.w; h = ctx.h; }
   { gdk_drawable_get_size(ctx.drawable, &w, &h); }

   gtk_label_set_text(ctx.size_label, sprintf("%S x %S", w, h));

   return w, h;
}

static define expose_cb(event, ctx)
{
   variable x, y, w, h, pb, iw, ih;

   (iw, ih) = image_size(ctx);

   switch(ctx.resize_style)
   { case RESIZE_TILED : return TRUE; }
   { case RESIZE_NONE : x = 0; y = 0; w = ctx.w; h = ctx.h; pb = ctx.pixbuf; }
   { case RESIZE_SCALED:

	variable a = event.area;
	if (iw <= ctx.w and ih <= ctx.h)
	   pb = ctx.pixbuf;
	else {
	   if (ctx.resize_w != iw or ctx.resize_h != ih) {

		ctx.resize_pixbuf = gdk_pixbuf_scale_simple (ctx.pixbuf,
						iw, ih, GDK_INTERP_BILINEAR);
		ctx.resize_w = iw; ctx.resize_h = ih;
	   }
	   pb = ctx.resize_pixbuf;
	}
	x = a.x; y = a.y; w = a.width; h = a.height;
   }

   gdk_draw_pixbuf (ctx.drawable, ctx.gc, pb, x, y, x, y, w, h,
						GDK_RGB_DITHER_NORMAL, 0, 0);
   return TRUE;
}
% }}}

static define realize_cb(ctx)  % {{{
{
   ctx.drawable = gtk_widget_get_window(ctx.widget);
   ctx.gc = gdk_gc_new(ctx.drawable);
   gtk_widget_modify_bg(ctx.widget, GTK_STATE_NORMAL, @gdk_white);
   gdk_gc_set_foreground(ctx.gc, @gdk_white);
   () = g_signal_connect_swapped(ctx.widget,"expose_event", &expose_cb, ctx);
} % }}}

static define save_file(file, kind, ctx) % {{{
{
   variable pb, w, h;

   if (ctx.resize_style == RESIZE_TILED) {
	gdk_drawable_get_size(ctx.drawable, &w, &h);
	pb = gdk_pixbuf_get_from_drawable(NULL, ctx.drawable, NULL,0,0,0,0,w,h);
	if (ctx.has_alpha)
	   pb = gdk_pixbuf_add_alpha(pb, FALSE, 0, 0, 0);
   }
   else if (ctx.resize_style == RESIZE_SCALED)
	pb = ctx.resize_pixbuf;
   else
	pb = ctx.pixbuf;

   _pixbuf_save (pb, file, kind);
   return FALSE;
} % }}}

static define save_cb(ctx) % {{{
{  
   variable file, kind;
   (file, kind) = _image_save_dialog(["png", "jpeg", "fits"]);

   if (file != NULL) {
	% Save the drawn window, rather than pixmap, in order to get the
	% tiling for free.  So, raise it to ensure that it's fully exposed
	gtk_window_present(ctx.win);
	() = gtk_timeout_add(750, &save_file, file, kind, ctx);
   }
} % }}}

static define resize_cb(ctx, resize_style) % {{{
{
   if (ctx.resize_style == resize_style) return;

   ctx.resize_style = resize_style;
   switch (resize_style)
   { case RESIZE_TILED:

	if (ctx.pixmap == NULL) {
	   ctx.pixmap = gdk_pixmap_new(ctx.drawable, ctx.w, ctx.h, -1);
	   gdk_draw_rectangle(ctx.pixmap, ctx.gc, 1, 0, 0, ctx.w, ctx.h);
	   gdk_draw_pixbuf(ctx.pixmap, ctx.gc, ctx.pixbuf, 0,0,0,0,
				ctx.w, ctx.h, GDK_RGB_DITHER_NORMAL, 0, 0);
	}

      	gdk_window_set_back_pixmap(ctx.drawable, ctx.pixmap,0);
   }
   { gdk_window_set_background(ctx.drawable, gdk_white); }

   gtk_widget_queue_draw(ctx.widget);
} % }}}

static define revert_cb(revert_button, default_radio_button, ctx) % {{{
{  
   gtk_window_resize(ctx.win, ctx.viewport_w, ctx.viewport_h);
   gtk_toggle_button_set_active(default_radio_button, TRUE);
} % }}}

define imdisplay() % {{{
{
   if (_NARGS == 0)
	usage("imdisplay( FileName_or_ImageArray [, ...])");

   % Imaging context
   variable ctx = struct { win, title, widget, drawable, pixmap, pixbuf,
      				resize_pixbuf, resize_style, gc, has_alpha,
				w, h, size_label, resize_w, resize_h,
				viewport_w, viewport_h, 
				animation };

   ctx.animation = 0;
   foreach(__pop_args(_NARGS)) { variable arg = (); load(arg.value, ctx); }
   if (ctx.pixbuf == NULL ) {
	() = fprintf(stderr,"No drawable images were found!\n");
	return;
   }

   ctx.win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   if (_NARGS > 1) ctx.title += " ...";
   gtk_window_set_title(ctx.win, ctx.title);
   gtk_container_set_border_width(ctx.win, 2);

   variable vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_add(ctx.win, vbox);

   variable hbox = gtk_hbox_new(FALSE, 5);
   gtk_box_pack_end(vbox,hbox,FALSE,FALSE,0);

   ctx.resize_style = RESIZE_NONE;
   if (ctx.animation) {
	ctx.widget = gtk_image_new_from_animation(ctx.pixbuf);
	gtk_window_set_resizable(ctx.win, FALSE);
   }
   else {
	ctx.widget = gtk_drawing_area_new();
	() = g_signal_connect_swapped(ctx.widget, "realize", &realize_cb, ctx);
	gtk_widget_set_size_request (ctx.widget, ctx.w, ctx.h);

	variable revert_button = gtk_button_new_with_label("Revert");
	gtk_widget_unset_flags (revert_button, GTK_CAN_FOCUS);
	gtk_box_pack_start(hbox,revert_button,FALSE,FALSE,0);
   }

   ctx.viewport_w = min ([ ctx.w, 7 * gdk_screen_width()  / 10]);
   ctx.viewport_h = min ([ ctx.h, 7 * gdk_screen_height() / 10]);

   if (ctx.viewport_w < ctx.w or ctx.viewport_h < ctx.h) {
	gtk_window_resize(ctx.win, ctx.viewport_w, ctx.viewport_h);
	variable sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (sw, 0);
	gtk_scrolled_window_set_policy(sw, GTK_POLICY_AUTOMATIC,
		 					GTK_POLICY_AUTOMATIC);
	gtk_container_add(vbox, sw);
	gtk_scrolled_window_add_with_viewport(sw, ctx.widget);
   }
   else
	gtk_container_add(vbox, ctx.widget);

   variable button = gtk_button_new_with_label("Done");
   gtk_box_pack_end(hbox,button,FALSE,FALSE,0);
   () = g_signal_connect_swapped(button,"clicked",&gtk_widget_destroy,ctx.win);
   gtk_widget_grab_focus(button);

   !if (ctx.animation) {

	ctx.size_label = gtk_label_new(sprintf("%S x %S",ctx.w, ctx.h));
	gtk_box_pack_start(hbox,ctx.size_label,TRUE,FALSE,0);

	button = gtk_button_new_with_label("Save");
	gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
	gtk_box_pack_end(hbox,button,FALSE,FALSE,0);
	() = g_signal_connect_swapped(button,"clicked", &save_cb, ctx);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_end(vbox,hbox,FALSE,FALSE,0);

	variable frame = gtk_frame_new("Expansion Rule");
	gtk_container_add(hbox,frame);

	variable resize_hbox = gtk_hbox_new(TRUE, 0);
	gtk_container_add(frame,resize_hbox);

	button = gtk_radio_button_new_with_label(NULL, "Neither ");
	() = g_signal_connect_swapped(button, "clicked", &resize_cb, ctx, 0);
	gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
	gtk_box_pack_end(resize_hbox,button,FALSE,FALSE,0);
	() = g_signal_connect(revert_button,"clicked", &revert_cb, button, ctx);

	button = gtk_radio_button_new_with_label_from_widget(button, "Scale");
	() = g_signal_connect_swapped(button, "clicked", &resize_cb, ctx, 2);
	gtk_box_pack_end(resize_hbox,button,FALSE,FALSE,0);

	button = gtk_radio_button_new_with_label_from_widget(button, "Tile");
	() = g_signal_connect_swapped(button, "clicked", &resize_cb, ctx, 1);
	gtk_box_pack_end(resize_hbox,button,FALSE,FALSE,0);

	frame = gtk_frame_new("Transparent");
	gtk_container_add(hbox,frame);
	if (ctx.has_alpha)
	   gtk_container_add(frame, gtk_label_new("Yes"));
	else
	   gtk_container_add(frame, gtk_label_new("No"));
   }

   gtk_widget_show_all (ctx.win);

   if (gtk_main_level() <= 0) {
	() = g_signal_connect(ctx.win, "destroy", &gtk_main_quit);
	gtk_main();
   }
} % }}}

provide("imdisplay");
