#include "sjog.h"
#include "sjog_config.h"
#include "sjog_list.h"
#include "sjog_brightness.h"
#include "sjog_volume.h"
#include "sjog_scroll.h"
#include "sjog_spic.h"
#include "sjog_sonypi.h"

/* -------------------------------------------------------------------------- */

gboolean reverseBrightness = TRUE;
gboolean useSpic = FALSE;
gboolean mousewheelFakeKeys = FALSE;
gboolean centerWindow = TRUE;

/* -------------------------------------------------------------------------- */

/* 
 * Sets the current mode.
 * Mode can be one of list, brightness, volume or scroll.
 * See sjog_mode enum in sjog.h
 *
 * @param mode the new current mode.
 */
void
sjog_set_mode(guint mode)
{
   currentMode = mode;
}

/* -------------------------------------------------------------------------- */

/* 
 * Returns the current mode.
 */
guint
sjog_get_mode()
{
   return currentMode;
}

/* -------------------------------------------------------------------------- */

void
sjog_window_set_position(GtkWindow * window)
{
   if (centerWindow)
      gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
}

/* -------------------------------------------------------------------------- */

/* 
 * Gets the current jog position and call a function according to the
 * current mode.
 */
void
sjog_jog_position_changed(gint jog)
{

#ifdef DEBUG
   fprintf(stdout, "sjog_jog_position_changed MODE=%d\n", sjog_get_mode());
#endif

   switch (sjog_get_mode())
   {
     case SJOG_MODE_VOLUME:
        sjog_volume_check_jog_position(jog);
        break;
     case SJOG_MODE_BRIGHTNESS:
        sjog_brightness_check_jog_position(jog);
        break;
     case SJOG_MODE_LIST:
        sjog_list_check_jog_position(jog);
        break;
     case SJOG_MODE_SCROLL:
        sjog_scroll_check_jog_position(jog);
   }

}

/* -------------------------------------------------------------------------- */

/*
 * Parse the command line and set the options.
 */
static void
sjog_check_options(gint argc, gchar * argv[])
{
   int i;

   for (i = 1; i < argc; i++)
   {
      /* some vaio models (i.e. 505) reverse the brightness value
         so this option allows to reverse the brightness
         setting behavior.
       */
      if (!strcmp(argv[i], "-rb") || !strcmp(argv[i], "--reverse-brightness"))
         reverseBrightness = FALSE;

      /* keep compatibility with legacy spicdriver */
      else if (!strcmp(argv[i], "-s") || !strcmp(argv[i], "--spic"))
         useSpic = TRUE;

      else if (!strcmp(argv[i], "-fk") || !strcmp(argv[i], "--fake-keys"))
         mousewheelFakeKeys = TRUE;

      else if (!strcmp(argv[i], "-nc") || !strcmp(argv[i], "--no-center"))
         centerWindow = FALSE;

      /* usage information */
      else
      {
         printf("S-Jog -- A tool to make Sony's Jog Dial less useless.\n"

                "Usage: %s [OPTIONS]\n"
                "\n"
                
                "  -rb,  --reverse-brightness      "
                "reverse brightness setting behaviour\n"

                "  -fk,  --fake-keys               "
                "send page up/down instead of buttons 4/5\n"

                "  -nc,  --no-center               "
                "do not center windows on screen\n"

                "  -s ,  --spic                    "
                "use the legacy spic driver\n"

                "  -h ,  --help                    "
                "display this message\n" "\n",
                argv[0]);
         exit(0);
      }

   }

}

/* -------------------------------------------------------------------------- */

gint
main(gint argc, gchar * argv[])
{

   /* read command line options */
   sjog_check_options(argc, argv);

   /* gtk initialization */
   gtk_init(&argc, &argv);

   /* choose between legacy spic driver polling or new sonypi
    * driver IRQ handling depending on command line option.
    * Default is to use the new sonypi driver.
    */
   if (useSpic == TRUE)
   {
      /* prepare to read the device */
      sjog_spic_init();
      /* the legacy driver requires us to poll the device instead */
      gtk_timeout_add(100, (GtkFunction) sjog_spic_read, NULL);
   }
   else
   {
      /* prepare to read the device */
      sjog_sonypi_init();
   }

   /* show initial window */
   sjog_list_show();

   /* main event loop */
   gtk_main();

   return 0;

}
