/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  Author : Richard GAYRAUD - 04 Nov 2003
 *           Marc LAMBERTON
 *           Olivier JACQUES
 *           From Hewlett Packard Company.
 *           F. Tarek Rogers
 *           Peter Higginson
 *           Vincent Luba
 */

#define GLOBALS_FULL_DEFINITION

#include "sipp.hpp"

#ifdef _USE_OPENSSL
SSL_CTX  *sip_trp_ssl_ctx = NULL; /* For SSL cserver context */
SSL_CTX  *sip_trp_ssl_ctx_client = NULL; /* For SSL cserver context */
SSL_CTX  *twinSipp_sip_trp_ssl_ctx_client = NULL; /* For SSL cserver context */

#define CALL_BACK_USER_DATA "ksgr"

/*
** passwd_call_back_routine implementation
**
*/

int
passwd_call_back_routine(char  *buf , int size , int flag, void *passwd)
{
  strncpy(buf, (char *)(passwd), size);
  buf[size - 1] = '\0';
  return(strlen(buf));
}
#endif

/***************** System Portability Features *****************/

unsigned int getmilliseconds()
{
  struct timeval LS_system_time;
  unsigned long long int VI_milli;
  static unsigned long long int VI_milli_base = 0;
  
  gettimeofday(&LS_system_time, NULL);
  VI_milli = ((unsigned long long) LS_system_time.tv_sec) 
    * 1000LL + (LS_system_time.tv_usec / 1000LL);
  if (!VI_milli_base) VI_milli_base = VI_milli - 1;
  VI_milli = VI_milli - VI_milli_base;
  
  return (unsigned int) VI_milli;
}

#ifdef _USE_OPENSSL
int send_nowait_tls(SSL *ssl, const void *msg, int len, int flags)
{
  int initial_fd_flags;
  int rc;
  int fd;
  int fd_flags;
  if ( (fd = SSL_get_fd(ssl)) == -1 ) {
    return (-1);
  }
  fd_flags = fcntl(fd, F_GETFL , NULL);
  initial_fd_flags = fd_flags;
  fd_flags |= O_NONBLOCK;
  fcntl(fd, F_SETFL , fd_flags);
  rc = SSL_write(ssl,msg,len);
  if ( rc <= 0 ) {
    return(rc);
  }
  fcntl(fd, F_SETFL , initial_fd_flags);
  return rc;
}
#endif 

int send_nowait(int s, const void *msg, int len, int flags)
{
#ifdef MSG_DONTWAIT
  return send(s, msg, len, flags | MSG_DONTWAIT);
#else
  int fd_flags = fcntl(s, F_GETFL , NULL);
  int initial_fd_flags;
  int rc;

  initial_fd_flags = fd_flags;
  //  fd_flags &= ~O_ACCMODE; // Remove the access mode from the value
  fd_flags |= O_NONBLOCK;
  fcntl(s, F_SETFL , fd_flags);
  
  rc = send(s, msg, len, flags);

  fcntl(s, F_SETFL , initial_fd_flags);

  return rc;
#endif 
}

char * get_inet_address(struct sockaddr_storage * addr)
{
  static char * ip_addr = NULL;
  int           res;

  if (!ip_addr) {
    ip_addr = (char *)malloc(1024*sizeof(char));
  }

  if (getnameinfo(_RCAST(struct sockaddr *, addr),
                  sizeof(struct sockaddr),
                  ip_addr,
                  1024,
                  NULL,
                  0,
                  NI_NUMERICHOST) != 0) {
    strcpy(ip_addr, "addr not supported");
  }

  return ip_addr;
}

void get_host_and_port(char * addr, char * host, int * port)
{
  /* Separate the port number (if any) from the host name.
   * Thing is, the separator is a colon (':').  The colon may also exist
   * in the host portion if the host is specified as an IPv6 address (see
   * RFC 2732).  If that's the case, then we need to skip past the IPv6
   * address, which should be contained within square brackets ('[',']').
   */
  char *p;
  p = strchr( addr, '[' );                      /* Look for '['.            */
  if( p != NULL ) {                             /* If found, look for ']'.  */
    p = strchr( p, ']' );
  }
  if( p == NULL ) {                             /* If '['..']' not found,   */
    p = addr;                                   /* scan the whole string.   */
  } else {                                      /* If '['..']' found,       */
    char *p1;                                   /* extract the remote_host  */
    char *p2;
    p1 = strchr( addr, '[' );
    p2 = strchr( addr, ']' );
    *p2 = '\0';
    strcpy(host, p1 + 1);
  }
  /* Starting at <p>, which is either the start of the host substring
   * or the end of the IPv6 address, find the last colon character.
   */
  p = strchr( p, ':' );
  if( NULL != p ) {
    *p = '\0';
    *port = atol(p + 1);
  } else {
    *port = 0;
  }
}

char * strcasestr2(char *s, char *find) {
  char c, sc;
  size_t len;

  if ((c = *find++) != 0) {
    c = tolower((unsigned char)c);
    len = strlen(find);
    do {
      do {
        if ((sc = *s++) == 0)
        return (NULL);
      } while ((char)tolower((unsigned char)sc) != c);
    } while (strncasecmp(s, find, len) != 0);
    s--;
  }
  return ((char *)s);
}


/******************** Recv Poll Processing *********************/

int                  pollnfds;
struct pollfd        pollfiles[SIPP_MAXFDS];
call               * pollcalls[SIPP_MAXFDS];
#ifdef _USE_OPENSSL
SSL                * ssl_list[SIPP_MAXFDS];
#endif


/***************** Check of the message received ***************/

bool sipMsgCheck (char *P_msg, int P_msgSize
#ifdef __3PCC__
		  ,int P_pollSetIdx
#endif
		  ) {
  
  const char C_sipHeader[] = "SIP/2.0" ;

#ifdef __3PCC__
  if (pollfiles[P_pollSetIdx].fd == twinSippSocket) {
    return true ;
  } else {
#endif // __3PCC__

    if (strstr(P_msg, C_sipHeader) !=  NULL) {
      return true ;
    }

    return false ;

#ifdef __3PCC__
  }
#endif // __3PCC__
}

void pollset_reset()
{
  pollnfds = 0;

  memset((void *)pollfiles,0,SIPP_MAXFDS*sizeof(struct pollfd));
  pollfiles[pollnfds].fd      = main_socket;
  pollfiles[pollnfds].events  = POLLIN | POLLERR;
  pollfiles[pollnfds].revents = 0;
  pollcalls[pollnfds]         = NULL;
  pollnfds++;
  
  if(tcp_multiplex) {
    /* Adds the TCP multiplex in the file descriptor array */
    pollfiles[pollnfds].fd      = tcp_multiplex;
    pollfiles[pollnfds].events  = POLLIN | POLLERR;
    pollfiles[pollnfds].revents = 0;
    pollcalls[pollnfds]         = NULL;
    pollnfds++;
  } 

#ifdef __3PCC__
  if(twinSippSocket) {
    /* Adds the twinSippSocket */
    pollfiles[pollnfds].fd      = twinSippSocket;
    pollfiles[pollnfds].events  = POLLIN | POLLERR;
    pollfiles[pollnfds].revents = 0;
    pollcalls[pollnfds]         = NULL;
    pollnfds++;
  } 

  if(localTwinSippSocket) {
    /* Adds the twinSippSocket */
    pollfiles[pollnfds].fd      = localTwinSippSocket;
    pollfiles[pollnfds].events  = POLLIN | POLLERR;
    pollfiles[pollnfds].revents = 0;
    pollcalls[pollnfds]         = NULL;
    pollnfds++;
  } 
#endif
}

int pollset_add(call * p_call, int sock)
{  
  // WARNING_P2("Adding socket %d at idx = %d", sock, pollnfds);

  pollfiles[pollnfds].fd      = sock;
  pollfiles[pollnfds].events  = POLLIN | POLLERR;
  pollfiles[pollnfds].revents = 0;
  pollcalls[pollnfds]         = p_call;
  pollnfds++;
  
  return pollnfds - 1;
}

void pollset_remove(int idx)
{  
  if(idx >= pollnfds) {
    ERROR("Pollset error");
  }

  /* Adds call sockets in the array */
  if(pollnfds) {
    // WARNING_P2("Removing socket %d at idx = %d", pollfiles[idx].fd, idx);
    pollnfds--;
    pollfiles[idx] = pollfiles[pollnfds];
    pollcalls[idx] = pollcalls[pollnfds];

    if((pollcalls[idx]) && (pollcalls[idx] -> pollset_index)) {
      pollcalls[idx] -> pollset_index = idx;
    }
  } else {
    ERROR("Pollset underflow");
  }
}

/************** Statistics display & User control *************/

void print_stats_in_file(FILE * f, int last)
{
  int index;
  static char temp_str[256];
  int divisor;

#define SIPP_ENDL "\r\n"

  /* Optional timestamp line for files only */
  if(f != stdout) {
    time_t tim;
    time(&tim);
    fprintf(f, "  Timestamp: %s" SIPP_ENDL, ctime(&tim));
  }
  
  /* Header line with global parameters */
  sprintf(temp_str, "%3.1f(%d ms)/%5.3fs", rate, duration, rate_period_s);
  if( toolMode == MODE_SERVER) {
    fprintf
      (f,
       "  Port   Total-time  Total-calls  Transport" 
       SIPP_ENDL
       "  %-5d %6d.%02d s     %8d  %s" 
       SIPP_ENDL SIPP_ENDL,
       local_port,
       clock_tick / 1000, (clock_tick % 1000) / 10,
       total_calls,
       TRANSPORT_TO_STRING(transport));
  } else {
    fprintf
      (f,
       "  Call-rate(length)     Port   Total-time  Total-calls  Remote-host" 
       SIPP_ENDL
       "%19s   %-5d %6d.%02d s     %8d  %s:%d(%s)" 
       SIPP_ENDL SIPP_ENDL,
       temp_str,
       local_port,
       clock_tick / 1000, (clock_tick % 1000) / 10,
       total_calls,
       remote_ip, 
       remote_port,
       TRANSPORT_TO_STRING(transport));
  }
  
  /* 1st line */
  if(total_calls < stop_after) {
    sprintf(temp_str, "%d new calls during %d.%03d s period ",
            total_calls - last_report_calls,
            (clock_tick-last_report_time) / 1000, 
            ((clock_tick-last_report_time) % 1000));
  } else {
    sprintf(temp_str, "Call limit reached (-m %d), %d.%03d s period ",
            stop_after,
            (clock_tick-last_report_time) / 1000, 
            ((clock_tick-last_report_time) % 1000));
  }
  divisor = scheduling_loops; if(!divisor) { divisor = 1; }
  fprintf(f,"  %-38s %d ms scheduler resolution" 
         SIPP_ENDL,
         temp_str,
         (clock_tick-last_report_time) / divisor);

  /* 2nd line */
  if( toolMode == MODE_SERVER) { 
    sprintf(temp_str, "%d concurrent calls", open_calls);
  } else {
    sprintf(temp_str, 
            "%d concurrent calls (limit %d)",
            open_calls,
            open_calls_allowed);
  }
  fprintf(f,"  %-38s Peak was %d calls, after %d s" SIPP_ENDL, 
         temp_str, 
         open_calls_peak, 
         open_calls_peak_time);
  
  /* 3rd line (optional) */
  if( toolMode != MODE_SERVER) { 
    sprintf(temp_str,"%d out-of-call msg (discarded)", 
            nb_out_of_the_blue);
    fprintf(f,"  %-37s", temp_str);
  }
  if(compression) {
    fprintf(f,"  Comp resync: %d sent, %d recv" , 
           resynch_send, resynch_recv);
  }
  if(compression || (toolMode != MODE_SERVER)) {
    fprintf(f,SIPP_ENDL);
  }
  
  /* 4th line , sockets and optional errors */ 
  sprintf(temp_str,"%d open sockets", 
          pollnfds);
  fprintf(f,"  %-38s", temp_str);
  if(nb_net_recv_errors || nb_net_send_errors || nb_net_cong) {
    fprintf(f,"  %d/%d/%d %s errors (send/recv/cong)" SIPP_ENDL,
           nb_net_send_errors, 
           nb_net_recv_errors,
           nb_net_cong,
           TRANSPORT_TO_STRING(transport));
  } else {
    fprintf(f,SIPP_ENDL);
  }

  /* 5th line, RTP echo statistics */
  if (media_socket > 0) {
    sprintf(temp_str, "%d Total echo RTP pckts",
            rtp_pckts);

    // AComment: Fix for random coredump when using RTP echo
    if (clock_tick-last_report_time) {
       fprintf(f,"  %-38s %d.%03d last period RTP rate (kB/s)" SIPP_ENDL,
              temp_str,
              (rtp_bytes)/(clock_tick-last_report_time),
              (rtp_bytes)%(clock_tick-last_report_time));
    }
    
    rtp_bytes = 0;
  }

  /* Scenario counters */
  fprintf(f,SIPP_ENDL);
  if(!lose_packets) {
    fprintf(f,"                                 "
           "Messages  Retrans   Timeout   Unexpected-Msg" 
           SIPP_ENDL);
  } else {
    fprintf(f,"                                 "
           "Messages  Retrans   Timeout   Unexp.    Lost" 
           SIPP_ENDL);
  }
  for(index = 0;
      index < scenario_len;
      index ++) {
    
    if(scenario[index] -> send_scheme) {
      char *dest, *src;
      int len;
      dest = temp_str;
      src  = scenario[index] -> send_scheme;

      if( strncmp(src, "SIP/2.0", 7) == 0) {
        src += 8;
      }
      while((*src) && (*src != ' ') && (*src != '\t') && (*src != '\n')) {
        *dest++ = *src ++;
      }
      *dest = 0;
      if(toolMode == MODE_SERVER) {
        fprintf(f,"  <---------- %-10s %-8s", 
               temp_str, 
               scenario[index] -> start_rtd ? 
               "B-RTD" :
               scenario[index] -> stop_rtd ? "E-RTD" : "");
      } else { 
        fprintf(f,"  %10s ----------> %-8s", 
               temp_str, 
               scenario[index] -> start_rtd ? 
               "B-RTD" :
               scenario[index] -> stop_rtd ? "E-RTD" : "");
      }

      if(scenario[index] -> retrans_delay) {
        fprintf(f,"%-9d %-9d %-9d" ,
               scenario[index] -> nb_sent,
               scenario[index] -> nb_sent_retrans,
               scenario[index] -> nb_timeout);
      } else {
        fprintf(f,"%-9d %-9d                    " ,
               scenario[index] -> nb_sent,
               scenario[index] -> nb_sent_retrans);
      }
    } else if(scenario[index] -> recv_response) {
      if(toolMode == MODE_SERVER) {
        fprintf(f,"  ----------> %-10d %-8s", 
               scenario[index] -> recv_response, 
               scenario[index] -> start_rtd ?
               "B-RTD" :
               scenario[index] -> stop_rtd ? "E-RTD" : "");
      } else { 
        fprintf(f,"  %10d <---------- %-8s", 
               scenario[index] -> recv_response, 
               scenario[index] -> start_rtd ?
               "B-RTD" :
               scenario[index] -> stop_rtd ? "E-RTD" : "");
      }

      if(toolMode == MODE_SERVER) {
        fprintf(f,"%-9d %-9d           %-9d" ,
               scenario[index] -> nb_recv,
               scenario[index] -> nb_recv_retrans,
               scenario[index] -> nb_unexp);
      } else {
        fprintf(f,"%-9d %-9d           %-9d" ,
               scenario[index] -> nb_recv,
               scenario[index] -> nb_recv_retrans,
               scenario[index] -> nb_unexp);
      }
    } else if (scenario[index] -> pause) {
      int pause;
      if(scenario[index] -> pause < 0) {
        pause = duration;
      } else {
        pause = scenario[index] -> pause;
      }
      if(toolMode == MODE_SERVER) {
        fprintf(f,"  [%6d ms]",  pause);
      } else { 
        fprintf(f,"             [%6d ms]", pause);
      }
    } else if(scenario[index] -> recv_request) {
      if(toolMode == MODE_SERVER) {
        fprintf(f,"  ----------> %-10s %-8s", 
               scenario[index] -> recv_request, 
               scenario[index] -> start_rtd ? 
               "B-RTD" :
               scenario[index] -> stop_rtd ? "E-RTD" : "");
      } else { 
        fprintf(f,"  %10s <---------- %-8s", 
               scenario[index] -> recv_request, 
               scenario[index] -> start_rtd ? 
               "B-RTD" :
               scenario[index] -> stop_rtd ? "E-RTD" : "");
      }
      fprintf(f,"%-9d %-9d           %-9d" ,
             scenario[index] -> nb_recv,
             scenario[index] -> nb_recv_retrans,
             scenario[index] -> nb_unexp);
    }
#ifdef __3PCC__
    else if(scenario[index] -> M_type == MSG_TYPE_RECVCMD) {
      fprintf(f,"       [ Received Command ]        ");
      fprintf(f,"%-9d %-9s           %-9s" ,
             scenario[index] -> M_nbCmdRecv,
             "",
             "");
      
    } else if(scenario[index] -> M_type == MSG_TYPE_SENDCMD) {
      fprintf(f,"         [ Sent Command ]          ");
      fprintf(f,"%-9d %-9s           %-9s" ,
             scenario[index] -> M_nbCmdSent,
             "",
             "");
    }
#endif
    else {
      ERROR("Not Implemented\n");
    }
    
    if(lose_packets && (scenario[index] -> nb_lost)) {
      fprintf(f," %-9d" SIPP_ENDL,
             scenario[index] -> nb_lost);
    } else {
      fprintf(f,SIPP_ENDL);
    }
    
    if(scenario[index] -> crlf) {
      fprintf(f,SIPP_ENDL);
    }
  }
}

void print_header_line(FILE *f, int last)
{  
  switch(currentScreenToDisplay)
    {
    case DISPLAY_STAT_SCREEN :
      fprintf(f,"----------------------------- Statistics Screen ------- [1-4]: Change Screen --" SIPP_ENDL);
      break;
    case DISPLAY_REPARTITION_SCREEN :
      fprintf(f,"---------------------------- Repartition Screen ------- [1-4]: Change Screen --" SIPP_ENDL);
      break;
    case DISPLAY_VARIABLE_SCREEN  :
      fprintf(f,"----------------------------- Variables Screen -------- [1-4]: Change Screen --" SIPP_ENDL);
      break;
    case DISPLAY_SCENARIO_SCREEN :
    default:
      fprintf(f,"------------------------------ Scenario Screen -------- [1-4]: Change Screen --" SIPP_ENDL);
      break;
    }
}

void print_bottom_line(FILE *f, int last)
{
  if(last) {
    fprintf(f,"------------------------------ Test Terminated --------------------------------" SIPP_ENDL);
  } else if(quitting) {
    fprintf(f,"------- Waiting for active calls to end. Press [Ctrl-c] to force exit. --------" SIPP_ENDL );
  } else if(paused) {
    fprintf(f,"----------------- Traffic Paused - Press [p] again to resume ------------------" SIPP_ENDL );
  } else if(cpu_max) {
    fprintf(f,"-------------------------------- CPU CONGESTED ---------------------------------" SIPP_ENDL);
  } else {
    switch(toolMode)
      {
      case MODE_SERVER :
        fprintf(f,"------------------------------ Sipp Server Mode -------------------------------" SIPP_ENDL);
        break;
#ifdef __3PCC__
      case MODE_3PCC_CONTROLLER_B :
        fprintf(f,"----------------------- 3PCC Mode - Controller B side -------------------------" SIPP_ENDL);
        break;
      case MODE_3PCC_A_PASSIVE :
        fprintf(f,"------------------ 3PCC Mode - Controller A side (passive) --------------------" SIPP_ENDL);
        break;
      case MODE_3PCC_CONTROLLER_A :
        fprintf(f,"----------------------- 3PCC Mode - Controller A side -------------------------" SIPP_ENDL);
        break;
#endif
      case MODE_CLIENT :
      default:
        fprintf(f,"------ [+|-|*|/]: Adjust rate ---- [q]: Soft exit ---- [p]: Pause traffic -----" SIPP_ENDL);
        break;
      }
  }
  fprintf(f,SIPP_ENDL);
  fflush(stdout);
}

void print_variable_list()
{
  CActions  * actions;
  CAction   * action;
  CVariable * variable;
  int i;
  bool found;

  printf("Action defined Per Message :" SIPP_ENDL);
  found = false;
  for(i=0; i<scenario_len; i++)
    {
      actions = scenario[i]->M_actions;
      if(actions != NULL)
        {
          switch(scenario[i]->M_type)
            {
            case MSG_TYPE_RECV:
              printf("=> Message[%d] (Receive Message) - "
                     "[%d] action(s) defined :" SIPP_ENDL,
                     i,             
                     actions->getUsedAction());
              break;
#ifdef __3PCC__
            case MSG_TYPE_RECVCMD:
              printf("=> Message[%d] (Receive Command Message) - "
                     "[%d] action(s) defined :" SIPP_ENDL,
                     i,             
                     actions->getUsedAction());
              break;
#endif
            default:
              printf("=> Message[%d] - [%d] action(s) defined :" SIPP_ENDL,
                     i,             
                     actions->getUsedAction());
              break;
            }
      
          for(int j=0; j<actions->getUsedAction(); j++)
            {
              action = actions->getAction(j);
              if(action != NULL)
                {
                  printf("   --> action[%d] = ", j);
                  action->afficheInfo();
                  printf(SIPP_ENDL);
                  found = true;
                }
            }
        }
    }
  if(!found) printf("=> No action found on any messages"SIPP_ENDL);
  
  printf(SIPP_ENDL);
  printf("Setted Variable List:" SIPP_ENDL);
  found = false;
  for(i=0; i<SCEN_VARIABLE_SIZE; i++)
    {
      variable = scenVariableTable[i];
      if(variable != NULL)
        {
          printf("=> Variable[%d] : setted regExp[%s]" SIPP_ENDL,
                 i,
                 variable->getRegularExpression());
          found = true;
        }
    }
  if(!found) printf("=> No variable found for this scenario"SIPP_ENDL);
  
}

/* Function to dump all available screens in a file */
void print_screens(void)
{
  int oldScreen = currentScreenToDisplay;
  
  currentScreenToDisplay = DISPLAY_SCENARIO_SCREEN;  
  print_header_line(   screenf, 0);
  print_stats_in_file( screenf, 0);
  print_bottom_line(   screenf, 0);

  currentScreenToDisplay = DISPLAY_STAT_SCREEN;  
  print_header_line(   screenf, 0);
  CStat::instance()->displayStat(screenf);
  print_bottom_line(   screenf, 0);
  
  currentScreenToDisplay = DISPLAY_REPARTITION_SCREEN;  
  print_header_line(   screenf, 0);
  CStat::instance()->displayRepartition(screenf);
  print_bottom_line(   screenf, 0);

  currentScreenToDisplay = oldScreen;
}

void print_statistics(int last)
{
  static int first = 1;

  if(backgroundMode == false) {
    if(!last) {
      screen_clear();
    }

    if(first) {
      first = 0;
      printf("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n"
             "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");
    }
    print_header_line(stdout,last);
    switch(currentScreenToDisplay) {
      case DISPLAY_STAT_SCREEN :
        CStat::instance()->displayStat(stdout);
        break;
      case DISPLAY_REPARTITION_SCREEN :
        CStat::instance()->displayRepartition(stdout);
        break;
      case DISPLAY_VARIABLE_SCREEN  :
        print_variable_list();
        break;
      case DISPLAY_SCENARIO_SCREEN :
      default:
        print_stats_in_file(stdout, last);
        break;
    }
    print_bottom_line(stdout,last);

    if(last) { fprintf(stdout,"\n"); }
  }
}

void set_rate(double new_rate)
{

  double L_temp ;
  
  if(toolMode == MODE_SERVER) {
    rate = 0;
    open_calls_allowed = 0;
  }

  rate = new_rate;
  if(rate < 0) {
    rate = 0;
  }

  last_rate_change_time = clock_tick;
  calls_since_last_rate_change = 0;
  
  if(!open_calls_user_setting) {
    
    int call_duration_min =  scenario_duration;

    if(duration > call_duration_min) call_duration_min = duration;

    if(call_duration_min < 1000) call_duration_min = 1000;
    
    L_temp = (3 * rate * call_duration_min) / rate_period_s / 1000 ;
    open_calls_allowed = (unsigned int) L_temp ;

  }
}

void sipp_sigusr1(int /* not used */)
{
  /* Smooth exit: do not place any new calls and exit */
  quitting = 1;
}

void sipp_sigusr2(int /* not used */)
{
  if (!signalDump) {
     signalDump = true ;
  }
}

/* User interface thread */

void keyb_thread (void * param)
{
  int c;

  while(!feof(stdin)){

    c = screen_readkey();
    
    switch (c) {
    case '1':
      currentScreenToDisplay = DISPLAY_SCENARIO_SCREEN;
      print_statistics(0);
      break;

    case '2':
      currentScreenToDisplay = DISPLAY_STAT_SCREEN;
      print_statistics(0);
      break;

    case '3':
      currentScreenToDisplay = DISPLAY_REPARTITION_SCREEN;
      print_statistics(0);
      break;

    case '4':
      currentScreenToDisplay = DISPLAY_VARIABLE_SCREEN;
      print_statistics(0);
      break;

    case '+':
      set_rate(rate + 1);
      print_statistics(0);
      break;
      
    case '-':
      set_rate(rate - 1);
      print_statistics(0);
      break;

    case '*':
      set_rate(rate + 10);
      print_statistics(0);
      break;
      
    case '/':
      set_rate(rate - 10);
      print_statistics(0);
      break;

    case 'p':
      if(paused) { 
        paused = 0;
        set_rate(rate);
      } else {
        paused = 1;
      }
      print_statistics(0);
      break;

    case 'q':
      quitting = 1;
      print_statistics(0);
      return;
    }
  }
}

/*************************** Mini SIP parser ***************************/

char * get_peer_tag(char *msg)
{
  char        * to_hdr;
  char        * ptr; 
  char        * end_ptr;
  static char   tag[MAX_HEADER_LEN];
  int           tag_i = 0;
  
  to_hdr = strstr(msg, "To:");
  if(!to_hdr) to_hdr = strstr(msg, "to:");
  if(!to_hdr) to_hdr = strstr(msg, "TO:");
  if(!to_hdr) {
    ERROR("No valid To: header in reply");
  }

  end_ptr = strchr(to_hdr,'\n');

  ptr = strchr(to_hdr, '>');
  if (!ptr) {
    return NULL;
  }
  
  ptr = strchr(to_hdr, ';'); 
  
  if(!ptr) {
    return NULL;
  }
  
  to_hdr = ptr;

  ptr = strstr(to_hdr, "tag");
  if(!ptr) { ptr = strstr(to_hdr, "TAG"); }
  if(!ptr) { ptr = strstr(to_hdr, "Tag"); }

  if(!ptr) {
    return NULL;
  }

  if (ptr>end_ptr) {
    return NULL ;
  }
  
  ptr = strchr(ptr, '='); 
  
  if(!ptr) {
    ERROR("Invalid tag param in To: header");
  }

  ptr ++;

  while((*ptr)         && 
        (*ptr != ' ')  && 
        (*ptr != ';')  && 
        (*ptr != '\t') && 
        (*ptr != '\t') && 
        (*ptr != '\r') &&  
        (*ptr != '\n') && 
        (*ptr)) {
    tag[tag_i++] = *(ptr++);
  }
  tag[tag_i] = 0;
  
  return tag;
}

char * get_call_id(char *msg)
{
  static char call_id[MAX_HEADER_LEN];
  char * ptr1, * ptr2, backup;

  ptr1 = strstr(msg, "Call-ID:");
  if(!ptr1) { ptr1 = strstr(msg, "Call-Id:"); }
  if(!ptr1) { ptr1 = strstr(msg, "Call-id:"); }
  if(!ptr1) { ptr1 = strstr(msg, "call-Id:"); }
  if(!ptr1) { ptr1 = strstr(msg, "call-id:"); }
  if(!ptr1) { ptr1 = strstr(msg, "CALL-ID:"); }
  if(!ptr1) { ERROR_P1("(1) No valid Call-ID: header in reply '%s'", msg); }
  
  ptr1 += 8;
  
  while((*ptr1 == ' ') || (*ptr1 == '\t')) { ptr1++; }
  
  if(!(*ptr1)) { ERROR("(2) No valid Call-ID: header in reply"); }
  
  ptr2 = ptr1;

  while((*ptr2) && 
        (*ptr2 != ' ') && 
        (*ptr2 != '\t') && 
        (*ptr2 != '\r') && 
        (*ptr2 != '\n')) { 
    ptr2 ++;
  } 

  if(!*ptr2) { ERROR("(3) No valid Call-ID: header in reply"); }

  backup = *ptr2;
  *ptr2 = 0;
  strcpy(call_id, ptr1);
  *ptr2 = backup;
  return (char *) call_id;
}

unsigned long get_reply_code(char *msg)
{
  while((*msg != ' ') && (*msg != '\t')) msg ++;
  while((*msg == ' ') || (*msg == '\t')) msg ++;

  return atol(msg);
}

/*************************** I/O functions ***************************/

#ifdef _USE_OPENSSL
int recv_all_tls(SSL *ssl, char *buffer, int size, int trace_id)
{
  int    recv_size = 0;
  char * start_buffer = buffer;
  int    to_be_recvd = size;
  int    part_size ;

  recv_size = SSL_read(ssl,buffer, size);
  
  if(recv_size <= 0) {
    if(recv_size != 0) {
      nb_net_recv_errors++;
      WARNING_P3("TLS %d Recv error : size = %d,Dummy : %d ",
                 trace_id, recv_size, trace_id);
    } else {
      /* This is normal for a server to have its client close
       * the connection */
      if(toolMode != MODE_SERVER) {
        WARNING_P3("TLS %d Recv error : size = %d, dummy : %d  "
                   "remote host closed connection",
                   trace_id, recv_size,trace_id);
        nb_net_recv_errors++;
      }
    }
  }
  
  return recv_size;
}
#endif

int recv_all_tcp(int sock, char *buffer, int size, int trace_id)
{
  int    recv_size = 0;
  char * start_buffer = buffer;
  int    to_be_recvd = size;
  int    part_size ;
  do {
    part_size = recv(sock, start_buffer, to_be_recvd, 0);
    
    if(part_size > 0) {
      to_be_recvd -= part_size;
      start_buffer += part_size;
      recv_size += part_size;
    } else {
      recv_size = part_size;
    }
    
  } while((part_size > 0) && to_be_recvd);
  
  if(recv_size <= 0) {
    if(recv_size != 0) {
      nb_net_recv_errors++;
      WARNING_P3("TCP %d Recv error : size = %d, sock = %d",
                 trace_id, recv_size, sock);
      WARNING_NO("TCP Recv error");
      // ERROR_NO("TCP recv error");
    } else {
#ifdef __3PCC__
      if (toolMode == MODE_3PCC_CONTROLLER_B) {
        /* In 3PCC controller B mode, twin socket is closed at peer closing.
         * This is a normal case: 3PCC controller B should end now */
        ERROR("3PCC controller A has ended -> exiting");
      } else
#endif
        /* This is normal for a server to have its client close
         * the connection */
        if(toolMode != MODE_SERVER) {
          WARNING_P3("TCP %d Recv error : size = %d, sock = %d, "
                     "remote host closed connection",
                     trace_id, recv_size, sock);
#ifdef __3PCC__
	  if(sock == twinSippSocket || sock == localTwinSippSocket) {
            int L_poll_idx = 0 ;
	    quitting = 1;
	    for((L_poll_idx) = 0;
	        (L_poll_idx) < pollnfds;
	        (L_poll_idx)++) {
	         if(pollfiles[L_poll_idx].fd == twinSippSocket) {
		   pollset_remove(L_poll_idx);
                  }
		 if(pollfiles[L_poll_idx].fd == localTwinSippSocket) {
		    pollset_remove(L_poll_idx);
                  }
              }
	      if(twinSippSocket) {
		       shutdown(twinSippSocket, SHUT_RDWR);
		       close(twinSippSocket);
		       twinSippSocket = 0 ;
              }
	      if(localTwinSippSocket) {
		       shutdown(localTwinSippSocket, SHUT_RDWR);
		       close(localTwinSippSocket);
		       localTwinSippSocket = 0 ;
              }
          }
#endif


          nb_net_recv_errors++;
        }
    }
  }
  
  return recv_size;
}


#ifdef _USE_OPENSSL
int recv_tls_message(SSL * ssl,
                     char *buffer,
                     int buffer_size,
                     E_Alter_YesNo alter_msg)
{
  int len = 0;
  int recv_size;
  char * ctl_hdr;
  int content_length;

  len = recv_size = recv_all_tls(ssl, buffer, buffer_size, 1);

  if(recv_size <= 0) {
    return recv_size;
  }

  if(len >= buffer_size) {
    ERROR("TLS msg too big");
  }
  buffer[len] = 0;
  return len;
}
#endif
    
int recv_tcp_message(int sock,
                     char *buffer,
                     int buffer_size,
                     E_Alter_YesNo alter_msg,
                     E_Alter_YesNo isControlMsg = E_ALTER_NO)
{
  int len = 0;
  int recv_size;
  char * ctl_hdr;
  int content_length;


  // Try to read SIP Header Message only
  // or CMD Message
  while(1) {
    
    // Read one char on tcp socket
    recv_size = recv_all_tcp(sock, &(buffer[len]), 1, 1);
    
    // Check read problem return
    if(recv_size <= 0) {
      return recv_size;
    }

    len++;

    // Search the end Message condition 
    if (isControlMsg == E_ALTER_NO) {
      // In case of SIP Message \r\n follow by 
      // \r\n is header end
      if((buffer[len-1] == '\n') && 
         (buffer[len-2] == '\r') && 
         (buffer[len-3] == '\n') && 
         (buffer[len-4] == '\r')) {
        /* CRLF CRLF Detected */
        buffer[len] = 0;
        break;
      }
    }
    else
    {
      // In case of CMD Message
      // escape char is the end of message
      if((alter_msg==E_ALTER_NO) &&
         (buffer[len-1] == 27)) {
        /* End delimitor detected, stop receiving */
        buffer[len-1] = 0;
        return (len - 1);
      }
    }
  }

  if(len >= buffer_size) {
    ERROR("TCP msg too big");
  }
  
  // Now search the content length of the body
  // part of SIP or CMD Message
  ctl_hdr = strstr(buffer, "Content-Length:");
  if(!ctl_hdr) ctl_hdr = strstr(buffer, "Content-length:");
  if(!ctl_hdr) ctl_hdr = strstr(buffer, "content-Length:");
  if(!ctl_hdr) ctl_hdr = strstr(buffer, "content-length:");
  if(!ctl_hdr) ctl_hdr = strstr(buffer, "CONTENT-LENGTH:");

  // Content Length was found
  // Read its value
  if((ctl_hdr) && (alter_msg==E_ALTER_YES)) {
    ctl_hdr += 15;
    content_length = atoi(ctl_hdr);
  } else {
    content_length = 0;
  }
  
  // If a body exist read it
  if(content_length) {

    /* Ensure remaining content will fit in remaining buffer size */
    if(content_length > (buffer_size - len)) {
      ERROR("TCP msg too big");
    }
    
    // Read Body part 
    do {
      recv_size = recv_all_tcp(sock, &(buffer[len]), content_length, 2);
      
      if(recv_size <= 0) {
        return recv_size;
      }
      
      len += recv_size;
      content_length -= recv_size;
    } while(content_length);
  }
  
  // Add the final '\0'
  buffer[len] = 0;
  
  return len;
}

int decompress_if_needed(int sock, char *buff,  int len, void **st)
{
  if(compression && len) {
    struct timeval currentTime;
    GET_TIME (&currentTime);
    TRACE_MSG((s,
               "----------------------------------------------- %s\n"
               "Compressed message received, header :\n"
               "0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x "
               "0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\n",
               CStat::instance()->formatTime(&currentTime),
               buff[0] , buff[1] , buff[2] , buff[3],
               buff[4] , buff[5] , buff[6] , buff[7],
               buff[8] , buff[9] , buff[10], buff[11],
               buff[12], buff[13], buff[14], buff[15]));
    
    int rc = comp_uncompress(st,
                             buff, 
                             (unsigned int *)&len);
    
    switch(rc) {
    case COMP_OK:
      TRACE_MSG((s,"Compressed message decompressed properly.\n"));
      break;

    case COMP_REPLY:
      TRACE_MSG((s, 
                 "Compressed message KO, sending a reply (resynch).\n"));
      sendto(sock,
             buff, 
             len, 
             0,
             (sockaddr *)(void *)&remote_sockaddr,
             SOCK_ADDR_SIZE(&remote_sockaddr));
      resynch_send++;
      return 0;

    case COMP_DISCARD:
      TRACE_MSG((s, "Compressed message discarded by pluggin.\n"));
      resynch_recv++;
      return 0;

    default:
    case COMP_KO:
      ERROR("Compression pluggin error");
      return 0;
    }
  }
  return len;
}

void sipp_customize_socket(int s)
{
  unsigned int buffsize = 65535;

  /* Allows fast TCP reuse of the socket */
#ifdef _USE_OPENSSL
  if (transport == T_TCP || transport == T_TLS ) { 
#else
  if (transport == T_TCP) { 
#endif
    int sock_opt = 1;
    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *)&sock_opt,
                   sizeof (sock_opt)) == -1) {
      ERROR_NO("setsockopt(SO_REUSEADDR) failed");
    }
#ifndef SOL_TCP
#define SOL_TCP 6
#endif
    if (setsockopt (s, SOL_TCP, TCP_NODELAY, (void *)&sock_opt,
                    sizeof (sock_opt)) == -1) {
      {
        ERROR_NO("setsockopt(TCP_NODELAY) failed");
      }
    }

    {
      struct linger linger;
      
      linger.l_onoff = 1;
      linger.l_linger = 1;
      if (setsockopt (s, SOL_SOCKET, SO_LINGER, 
                      &linger, sizeof (linger)) < 0) {
        ERROR_NO("Unable to set SO_LINGER option");
      }
    }
  }
    
    /* Increase buffer sizes for this sockets */
  if(setsockopt(s,
                SOL_SOCKET,
                SO_SNDBUF,
                &buffsize,
                sizeof(buffsize))) {
    ERROR_NO("Unable to set socket sndbuf");
  }
  
  buffsize = 65535;
  if(setsockopt(s,
                SOL_SOCKET,
                SO_RCVBUF,
                &buffsize,
                sizeof(buffsize))) {
    ERROR_NO("Unable to set socket rcvbuf");
  }
  
}
  
#ifdef _USE_OPENSSL
int send_message_tls(SSL *ssl, void ** comp_state, char * msg)
{
	int rc;
	
	 rc = send_nowait_tls(ssl, msg, strlen(msg), 0);

  if(rc == 0) {
    nb_net_send_errors++;
    ERROR_NO("Unable to send TLS message");
    return -2;
  }

  return rc;
}
#endif

int send_message(int s, void ** comp_state, char * msg)
{

  struct sockaddr_storage *L_dest = &remote_sockaddr;

  if(transport == T_TCP) {

    int rc;

    rc = send_nowait(s, 
                     msg, 
                     strlen(msg), 
                     0);
    
#ifdef EAGAIN
    if(errno == EAGAIN) {
      nb_net_cong++;
      return -1;
    }
#endif

    if(errno == EWOULDBLOCK) {
      nb_net_cong++;
      return -1;
    }
    
    if(errno == EPIPE) {
      nb_net_send_errors++;
      ERROR("Broken pipe on TCP connection, remote peer "
            "probably closed the socket");
    }

    if(rc <= 0) {
      nb_net_send_errors++;
      ERROR_NO("Unable to send TCP message");
      return -2;
    }
  } else { /* UDP */

    unsigned int len = strlen(msg);
    
    if(compression) {
      static char comp_msg[SIPP_MAX_MSG_SIZE];
      strcpy(comp_msg, msg);
      if(comp_compress(comp_state,
                       comp_msg, 
                       &len) != COMP_OK) {
        ERROR("Compression pluggin error");
      }
      msg = (char *)comp_msg;

      TRACE_MSG((s, "---\nCompressed message len: %d\n",
                 len));
    }

    // different remote sending address from received
    if (use_remote_sending_addr) {
      L_dest = &remote_sending_sockaddr ;
    }

    if(sendto(s, 
              msg, 
              len, 
              0,
              (struct sockaddr *)(void *)L_dest,
              SOCK_ADDR_SIZE(L_dest)) == -1) {
      nb_net_send_errors++;
      ERROR_NO("Unable to send UDP message");
      return -2;
    }
  }
  return 0;
}

/****************************** Network Interface *******************/

int recv_message(char * buffer, int buffer_size, int * poll_idx)
{
  int size = 0;

#ifdef _USE_OPENSSL 
  BIO *bio;
  SSL *ssl;
#endif
  int err;
  
  for((*poll_idx) = 0;
      (*poll_idx) < pollnfds;
      (*poll_idx)++) {

    if(pollfiles[(*poll_idx)].revents) {
      
      call * recv_call = pollcalls[(*poll_idx)];
      int s = pollfiles[(*poll_idx)].fd;
      int ss = s;

      pollfiles[(*poll_idx)].revents = 0;

#ifdef __3PCC__
      if(s == localTwinSippSocket)
        {
          sipp_socklen_t len = sizeof(twinSipp_sockaddr);
          twinSippSocket = accept(s,
                                  (sockaddr *)(void *)&twinSipp_sockaddr,
                                  &len);
          
          pollset_add(0, twinSippSocket);
          return(-2);
        } 
      else if (s == twinSippSocket)
        {
          size = recv_tcp_message(s,
                                  buffer,
                                  buffer_size,
                                  E_ALTER_NO,
                                  E_ALTER_YES);
          if(size >= 0) {
            buffer[size] = 0;
          }
          else
            buffer[0] = 0;
          return size;
        }
        else
        {
#endif
      
#ifdef _USE_OPENSSL
      if(transport == T_TCP ||  transport == T_TLS ) {
#else
      if(transport == T_TCP ) {
#endif
        
        if(s == main_socket) {
          /* New incoming connection */
          sipp_socklen_t len = SOCK_ADDR_SIZE(&remote_sockaddr);
          int new_sock = accept(s,
                                (sockaddr *)(void *)&remote_sockaddr,
                                &len);
          
#ifdef _USE_OPENSSL
          if (transport == T_TLS) {
            if ( (bio = BIO_new_socket(new_sock,BIO_NOCLOSE)) == NULL) {
              ERROR("Unable to create the BIO- New TLS connection - recv_message\n");
            }

            /* Create a SSL object */
            if (!(ssl = SSL_new(sip_trp_ssl_ctx))){
              ERROR("Unable to create new SSL context recv_message: Fail SSL_new\n");
            }

            SSL_set_bio(ssl,bio,bio);

            if ( (err = SSL_accept(ssl)) < 0 ) {
              ERROR("SSL_accept Fails - recv_message()\n");
            }
            ssl_list[new_sock] = ssl;
          }
#endif

          pollset_add(0, new_sock);
          return -2;
        }
#ifdef _USE_OPENSSL
        if ( transport == T_TLS ) {
          int ss = s;
          ssl = ssl_list[s];
          size = recv_tls_message(ssl,
                                  buffer,
                                  buffer_size,
                                  E_ALTER_YES);
        } else {
#endif
        size = recv_tcp_message(s,
                                buffer,
                                buffer_size,
                                E_ALTER_YES);
#ifdef _USE_OPENSSL
        }
#endif
        
        if(size <= 0) { /* Remote side closed TCP connection */
          
          nb_net_recv_errors++;
          
          /* Preventive cleaning */
          if(size < 0) {
            WARNING_P2("TCP/TLS recv error on socket %d, index = %d",
                       s, *poll_idx);
            ERROR_NO("TCP/TLS recv_error");
          }

          if(recv_call) {
            recv_call -> call_socket = 0;
            if(recv_call -> pollset_index) {
              recv_call -> pollset_index = 0;
            }
          }
          
          pollset_remove((*poll_idx));
          shutdown(s, SHUT_RDWR);
          close(s);
          return 0;
        }
        
      } else { /* T_UDP */
        
        if(toolMode == MODE_SERVER) {
          sipp_socklen_t len = SOCK_ADDR_SIZE(&remote_sockaddr);
          
          size  = recvfrom(s,
                           buffer,
                           buffer_size,
                           0,
                           (sockaddr *)(void *)&remote_sockaddr,
                           &len);
          
        } else {
          size  = recvfrom(s, 
                           buffer, 
                           buffer_size, 
                           0, NULL, NULL);
        }
        
        if(size < 0) {
          WARNING_P3("Unexpected UDP recv error, idx = %d, "
                     "socket = %d, recv_call = 0x%08x",
                     (*poll_idx), s, recv_call);
          ERROR_NO("Unexpected UDP recv error");
#if 0
          nb_net_recv_errors++;
          pollset_remove((*poll_idx));
          shutdown(s, SHUT_RDWR);
          close(s);
#endif
          return 0;
        }

        if (size > 0) {

          size = decompress_if_needed(s,
                                      buffer,
                                      size,
                                      ((recv_call) ? 
                                       (&(recv_call -> comp_state)) : 
                                       &monosocket_comp_state));
        }

      } /* else ... T_UDP */
      
      break;

#ifdef __3PCC__
      }
#endif
    } /* if(pollfiles[(*poll_idx)].revents) */
  } /* for((*poll_idx)) */
  
  buffer[size] = 0;

  struct timeval currentTime;
  GET_TIME (&currentTime);
  TRACE_MSG((s, "----------------------------------------------- %s\n"
             "%s message received [%d] bytes :\n\n%s\n",
             CStat::instance()->formatTime(&currentTime),
             TRANSPORT_TO_STRING(transport), size,
             buffer));

  return size;
}

void pollset_process(bool ipv6)
{
  int rs; /* Number of times to execute recv().
	     For TCP with 1 socket per call:
	         no. of events returned by poll
	     For UDP and TCP with 1 global socket:
	         recv_count is a flag that stays up as
	         long as there's data to read */

  int loops = MAX_RECV_LOOPS;
  
  while((loops-- > 0) && /* Ensure some minimal statistics display sometime */
        (rs = poll(pollfiles, pollnfds,  1)) > 0) {
    if((rs < 0) && (errno == EINTR)) {
      return;
    }
    
    if(rs < 0) {
      ERROR_NO("poll() error");
    }
    
    while(rs > 0) {
      char            msg[SIPP_MAX_MSG_SIZE];
      int             msg_size;
      char          * call_id;
      call          * call_ptr;
      int             pollset_index = 0;
      
      memset(msg,0,sizeof(msg));
      msg_size = recv_message(msg, 
                              SIPP_MAX_MSG_SIZE, 
                              &pollset_index);
      
      if(msg_size > 0) {
	
	if (sipMsgCheck(msg, 
			msg_size
#ifdef __3PCC__
			,pollset_index
#endif // __3PCC__
			) == true) {
	  
          call_id = get_call_id(msg);
          call_ptr = get_call(call_id);
        
          if(!call_ptr)
            {
              if(toolMode == MODE_SERVER)
                {
                  // Adding a new INCOMING call !
                  CStat::instance()->computeStat
                    (CStat::E_CREATE_INCOMING_CALL);
                  call_ptr = add_call(call_id, ipv6);
            
                  if((pollset_index) && 
                     (pollfiles[pollset_index].fd != main_socket) && 
                     (pollfiles[pollset_index].fd != tcp_multiplex) ) {
				  
                    call_ptr -> call_socket = pollfiles[pollset_index].fd;
                  }
                }
#ifdef __3PCC__
              else if(toolMode == MODE_3PCC_CONTROLLER_B || toolMode == MODE_3PCC_A_PASSIVE)
                {
                  // Adding a new OUTGOING call !
                  CStat::instance()->computeStat
                    (CStat::E_CREATE_OUTGOING_CALL);
                  call_ptr = add_call(call_id, ipv6);
            
                  if((pollset_index) && 
                     (pollfiles[pollset_index].fd != main_socket) && 
                     (pollfiles[pollset_index].fd != tcp_multiplex) &&
                     (pollfiles[pollset_index].fd != localTwinSippSocket) &&
                     (pollfiles[pollset_index].fd != twinSippSocket)) {
                    call_ptr -> call_socket = pollfiles[pollset_index].fd;
                  }
                }
#endif
              else // mode != from SERVER and 3PCC Controller B
                {
                  nb_out_of_the_blue++;
                  CStat::instance()->computeStat
                    (CStat::E_OUT_OF_CALL_MSGS);
                  // This is a message that is not relating to any call
                }
            }
		
          if(call_ptr)
            {
#ifdef __3PCC__
              if( (pollfiles[pollset_index].fd == localTwinSippSocket) ||
                  (pollfiles[pollset_index].fd == twinSippSocket))
                {
                  if(!call_ptr -> process_twinSippCom(msg))
                    {
                      return;
                    } 
                }
              else
#endif
                {
                  if(!call_ptr -> process_incomming(msg))
                    {
                      /* Needs to rebuild the pollset (socket removed, 
                       * call deleted, etc... Cause pollcalls is now 
                       * invalid and will alway lead poll() to return 
                       * an error.*/
                      return;
                    }
                }
            }
	} else { // sipMsgCheck == false
	  // unrecognized message => discard it
	  WARNING("non SIP message discarded");
	}
	if (pollnfds > 0) /* refer to note at the beginning of this function */
               rs--;
           
      } // end if msg >=0
      else 
      rs--;
    }
  }
  cpu_max = loops <= 0;
}

/* Send loop & trafic generation*/

void traffic_thread(bool ipv6)
{
  unsigned int calls_to_open = 0;
  unsigned int new_time;
  unsigned int last_time;
  int          timer_resolution = DEFAULT_TIMER_RESOLUTION;
  bool         firstPass;

  /* create the file */
  char         L_file_name [200];
  sprintf (L_file_name, "%s_%d_screen.log", scenario_file, getpid());

  firstPass = true;
  last_time = getmilliseconds();
 
  /* Prepare pollset with basic sockets */
  pollset_reset();

  while(1) {

    scheduling_loops ++;

    /* update local time, except if resetted*/
    new_time = getmilliseconds();

    clock_tick += (new_time - last_time);
    last_time = new_time;

    if (signalDump) {
       /* Screen dumping in a file */
       if (screenf) {
          print_screens();
       } else {
         /* If the -trace_screen option has not been set, */
         /* create the file at this occasion              */
         screenf = fopen(L_file_name, "a");
	 if (!screenf) {
            WARNING_P1("Unable to create '%s'", L_file_name); 
         }
	 print_screens();
	 fclose(screenf);
	 screenf = 0;
       }
       signalDump = false ;
    }

    if ((!quitting) && (!paused)) {

      calls_to_open = 
        (unsigned int) (((clock_tick - last_rate_change_time) * rate/rate_period_s) / 1000)
        - calls_since_last_rate_change;

      if( (toolMode == MODE_CLIENT)
#ifdef __3PCC__
          || (toolMode == MODE_3PCC_CONTROLLER_A)
#endif
          )
        {
          while((calls_to_open--) && 
                (open_calls < open_calls_allowed) &&
                (total_calls < stop_after)) 
            {
              // adding a new OUTGOING CALL
              CStat::instance()->computeStat(CStat::E_CREATE_OUTGOING_CALL);
              call * call_ptr = add_call(ipv6);
              call_ptr -> run();
            }
        
          if(open_calls >= open_calls_allowed) {
            set_rate(rate);
          }
        }

        // Quit after asked number of calls is reached
        if(total_calls >= stop_after) {
          quitting = 1;
        }
      
      
    } else if (quitting) {
      /* Quitting and no more openned calls, close all */
      if(!open_calls) {
        // Dump the latest statistics if necessary
        if(dumpInFile) {
          CStat::instance()->dumpData();
        }
        /* Screen dumping in a file if asked */
        if(screenf) {
          print_screens();
        }
        screen_exit(EXIT_TEST_RES_UNKNOWN);
      }
    }

    if(compression) {
      timer_resolution = 50;
    }

    /* Schedule all pending calls and process their timers */
    if((clock_tick - last_timer_cycle) > timer_resolution) {
      call_map * calls = get_calls();
      call_map::iterator iter;
      
      /* Workaround hpux problem with iterators. Deleting the
       * current object when iterating breaks the iterator and
       * leads to iterate again on the destroyed (deleted)
       * object. Thus, we have to wait ont step befere actual
       * deletion of the object*/
      call * last = NULL;

      for(iter = calls->begin(); iter != calls->end(); iter++) {
        if(last) { last -> run(); }
        last = iter -> second;
      }      
      if(last) { last -> run(); }

      last_timer_cycle = clock_tick;
    }
    
    /* Receive incomming messages */
    pollset_process(ipv6);
   
    if(firstPass)
      {
        // dumping (to create file on disk) and showing 
        // screen at the beginning even if the report
        // period is not reach
        firstPass = false;
        print_statistics(0);
        /* Dumping once to create the file on disk */
        if(dumpInFile)
          {
            CStat::instance()->dumpData();
          }
      }

    if((clock_tick - last_report_time) >= report_freq)
      {
        print_statistics(0);
        CStat::instance()->computeStat(CStat::E_RESET_PD_COUNTERS);
        last_report_time  = clock_tick;
        last_report_calls = total_calls;
        rtd_sum = 0;
        rtd_nb = 0;
        call_duration_sum = 0;
        call_duration_nb = 0;
        scheduling_loops = 0;
      }

    // FIXME - Should we recompute time ? print stat take 
    // a lot of time, so the clock_time is no more 
    // the current time !
    if(dumpInFile) {
      if((clock_tick - last_dump_time) >= report_freq_dumpLog)  {
        CStat::instance()->dumpData();
        CStat::instance()->computeStat(CStat::E_RESET_PL_COUNTERS);
        last_dump_time  = clock_tick;
      }
    }
  }
}

/*************** RTP ECHO THREAD ***********************/

void rtp_echo_thread (void * param)
{
  char msg[2048];
  size_t nr, ns;
  sipp_socklen_t len;

   int                   rc;
   sigset_t              mask;
   sigfillset(&mask); /* Mask all allowed signals */
   rc = pthread_sigmask(SIG_BLOCK, &mask, NULL);

  for (;;) {
    len = sizeof(remote_rtp_addr);
    nr = recvfrom(media_socket, 
                  msg, 
                  sizeof(msg), 0, 
                  (sockaddr *)(void *) &remote_rtp_addr, 
                  &len);

    if (((long)nr) < 0) {
      WARNING_P2("%s %i", 
                 "Error on RTP echo reception - stopping echo - errno=", 
                 errno);
      return;
    }
    ns = sendto(media_socket, msg, nr, 
                0, (sockaddr *)(void *) &remote_rtp_addr, 
                len);

    if (ns != nr) {
      WARNING_P2("%s %i", 
                 "Error on RTP echo transmission - stopping echo - errno=", 
                 errno);
      return;
    }
    
    rtp_pckts++;
    rtp_bytes += ns;
  }
}

/* Help screen */

void help() 
{
  printf
    ("\n"
     "Usage:\n"
     "\n"
     "  sipp remote_host[:remote_port] [options]\n"
     "\n"
     "  Available options:\n"
     "\n"
     "   -v               : Display version and copyright information.\n"
     "\n"
     "   -bg              : Launch the tool in background mode.\n"
     "\n"
     "   -p local_port    : Set the local port number. Default is a\n"
     "                      random free port chosen by the system.\n"
     "\n"
     "   -i local_ip      : Set the local IP address for 'Contact:',\n"
     "                      'Via:', and 'From:' headers. Default is\n"
     "                      primary host IP address.\n"
     "\n"
     "   -inf file_name   : Inject values from an external CSV file during calls\n"
     "                      into the scenarios.\n"
     "                      First line of this file say whether the data is \n"
     "                      to be read in sequence (SEQUENTIAL) or random \n"
     "                      (RANDOM) order.\n"
     "                      Each line corresponds to one call and has one or \n"
     "                      more ';' delimited data fields. Those fields can be \n"
     "                      referred as [field0], [field1], ... in the xml \n"
     "                      scenario file.\n"
     "\n"
     "   -d duration      : Controls the length (in milliseconds) of\n"
     "                      calls. More precisely, this controls\n"
     "                      the duration of 'pause' instructions in\n"
     "                      the scenario, if they do not have a\n"
     "                      'milliseconds' section. Default value is 0.\n"
     "\n"
     "   -r rate (cps)    : Set the call rate (in calls per seconds).\n"
     "                      This value can be changed during test by\n"
     "                      pressing '+','_','*' or '/'. Default is 10.\n"
     "                      pressing '+' key to increase call rate by 1,\n"
     "                      pressing '-' key to decrease call rate by 1,\n"
     "                      pressing '*' key to increase call rate by 10,\n"
     "                      pressing '/' key to decrease call rate by 10.\n"
     "                      If the -rp option is used, the call rate is\n"
     "                      calculated with the period in ms given \n"
     "                      by the user.\n"
     "\n"
     "   -rp period (ms)  : Specify the rate period in milliseconds for the call\n"
     "                      rate.\n"
     "                      Default is 1 second.\n"
     "                      This allows you to have n calls every m milliseconds \n"
     "                      (by using -r n -rp m).\n"
     "                      Example: -r 7 -rp 2000 ==> 7 calls every 2 seconds.\n"
     "\n"
     "   -sf filename     : Loads an alternate xml scenario file.\n"
     "                      To learn more about XML scenario syntax,\n"
     "                      use the -sd option to dump embedded \n"
     "                      scenarios. They contain all the necessary\n"
     "                      help.\n"
     "\n"
     "   -sn name         : Use a default scenario (embedded in\n"
     "                      the sipp executable). If this option is omitted,\n"
     "                      the Standard SipStone UAC scenario is loaded.\n"
     "                      Available values in this version:\n"
     "\n"
     "                        'uac'      : Standard SipStone UAC (default).\n"
     "                        'uas'      : Simple UAS responder.\n"
     "                        'regexp'   : Standard SipStone UAC - with\n"
     "                                     regexp and variables.\n"
     "                        'branchc'  : Branching and conditional\n"
     "                                     branching in scenarios - client.\n"
     "                        'branchs'  : Branching and conditional\n"
     "                                     branching in scenarios - server.\n"
#ifdef __3PCC__
     "\n"
     "                      Default 3pcc scanerios (see -3pcc option):\n"
     "\n"
     "                        '3pcc-C-A' : Controller A side (must be started\n"
     "                                     after all other 3pcc scenarios)\n"
     "                        '3pcc-C-B' : Controller B side.\n"
     "                        '3pcc-A'   : A side.\n"
     "                        '3pcc-B'   : B side.\n"
#endif
     "\n"
     "   -sd name         : Dumps a default scenario (embeded in\n"
     "                      the sipp executable)\n"
     "\n"
#ifdef _USE_OPENSSL
     "   -t [u1|un|t1|tn|l1|ln] : Set the transport mode:\n"
#else
     "   -t [u1|un|t1|tn] : Set the transport mode:\n"
#endif
     "\n"
     "                        u1: UDP with one socket (default),\n"
     "                        un: UDP with one socket per call,\n"
     "                        t1: TCP with one socket,\n"
     "                        tn: TCP with one socket per call,\n"
#ifdef _USE_OPENSSL
     "                        l1: TLS with one socket,\n"
     "                        ln: TLS with one socket per call.\n"
#endif
     "\n");
  if(!strlen(comp_error)) {
    printf
      ("                      It appears that you installed the\n"
       "                      " COMP_PLUGGIN " plugin. 2 additionnal\n"
       "                      transport modes are available:\n"
       "\n"
       "                        c1: u1 + compression,\n"
       "                        cn: un + compression.\n"
       "\n");
  }
  printf
    ("   -trace_msg       : Displays sent and received SIP messages in\n"
     "                      <scenario file name>_<ppid>_messages.log\n"
     "\n"
     "   -trace_screen    : Dump statistic screens in the \n"
     "                      <scenario_name>_<ppid>_screens.log file when\n"
     "                      quitting SIPp. Useful to get a final status report\n"
     "                      in background mode (-bg option).\n"
     "\n"
     "   -trace_timeout   : Displays call ids for calls with timeouts in\n"
     "                      <scenario file name>_<ppid>_timeout.log\n"
     "\n"
     "   -trace_stat      : Dumps all statistics in <scenario_name>_<ppid>.csv\n"
     "                      file. Use the '-h stat' option for a detailed\n"
     "                      description of the statistics file content.\n"
     "\n"
     "   -stf file_name   : Set the file name to use to dump statistics\n"
     "\n"
     "   -trace_err       : Trace all unexpected messages in\n"
     "                      <scenario file name>_<ppid>_errors.log.\n"
     "\n"
     "   -trace_logs      : Allow tracing of <log> actions in\n"
     "                      <scenario file name>_<ppid>_logs.log.\n"
     "\n"
     "   -s service_name  : Set the username part of the resquest URI.\n"
     "                      Default is 'service'.\n"
     "\n"
#ifdef _USE_OPENSSL
     "   -ap password     : Set the password for authentication challenges.\n"
     "                      Default is 'password'\n"
     "\n"
#endif
     "   -f frequency     : Set the statistics report frequency on screen\n"
     "                      (in seconds). Default is 1.\n"
     "\n"
     "   -fd frequency    : Set the statistics dump log report frequency\n"
     "                      (in seconds). Default is 60.\n"
     "\n"
     "   -l calls_limit   : Set the maximum number of simultaneous\n"
     "                      calls. Once this limit is reached, traffic\n"
     "                      is decreased until the number of open calls\n"
     "                      goes down. Default:\n"
     "\n"
     "                        (3 * call_duration (s) * rate).\n"
     "\n"
     "   -m calls         : Stop the test and exit when 'calls' calls are\n"
     "                      processed.\n"
     "\n"
     "   -mp local_port   : Set the local RTP echo port number. Default\n"
     "                      is none. RTP/UDP packets received on that\n"
     "                      port are echoed to their sender.\n"
     "\n"
     "   -mi local_rtp_ip : Set the local IP address for RTP echo.\n"
     "\n"
#ifdef __3PCC__
     "   -3pcc ip:port    : Launch the tool in 3pcc mode (\"Third Party\n"
     "                      call control\"). The passed ip address\n"
     "                      is depending on the 3PCC role.\n"
     "                      - When the first twin command is 'sendCmd' then\n"
     "                      this is the address of the remote twin socket.\n"
     "                      Example: 3PCC-C-A scenario.\n"
     "                      - When the first twin command is 'recvCmd' then\n"
     "                      this is the address of the local twin socket.\n"
     "                      Example: 3PCC-C-B scenario.\n"
     "\n"
#endif
     "   -nr              : Disable retransmission in UDP mode.\n"
     "\n"
     "   -nd              : No Default. Disable all default behavior of SIPp\n"
     "                      which are the following:\n"
     "                      - On UDP retransmission timeout, abort the call by\n"
     "                      sending a BYE or a CANCEL\n"
     "                      - On unexpected BYE send a 200 OK and close the call\n"
     "                      - On unexpected CANCEL send a 200 OK and close the call\n"
     "                      - On unexpected PING send a 200 OK and continue the call\n"
     "                      - On any other unexpected message, abort the call by\n"
     "                      sending a BYE or a CANCEL\n"
     "\n"
     "   -rsa host:port   : Set the remote sending address to host:port.\n"
     "                      for sending the messages.\n"
     "\n"
     "Signal handling:\n"
     "\n"
     "   SIPp can be controlled using posix signals. The following signals\n"
     "   are handled:\n"
     "   USR1: Similar to press 'q' keyboard key. It triggers a soft exit\n"
     "         of SIPp. No more new calls are placed and all ongoing calls\n"
     "         are finished before SIPp exits.\n"
     "         Example: kill -SIGUSR1 732\n"
     "   USR2: Triggers a dump of all statistics screens in\n"
     "         <scenario_name>_<ppid>_screens.log file. Especially useful \n"
     "         in background mode to know what the current status is.\n"
     "         Example: kill -SIGUSR2 732\n"
     "\n"
     "Exit code:\n"
     "\n"
     "   Upon exit (on fatal error or when the number of asked calls (-m\n"
     "   option) is reached, sipp exits with one of the following exit\n"
     "   code:\n"
     "    0: All calls were successful\n"
     "    1: At least one call failed\n"
     "   97: exit on internal command. Calls may have been processed\n"
     "   99: Normal exit without calls processed\n"
     "   -1: Fatal error\n"
     "\n"
     "\n"
     "Example:\n"
     "\n"
     "   Run sipp with embedded server (uas) scenario:\n"
     "     ./sipp -sn uas\n"
     "   On the same host, run sipp with embedded client (uac) scenario\n"
     "     ./sipp -sn uac 127.0.0.1\n"
     "\n");
}


void help_stats() 
{
  printf(
"\n"
"  The  -trace_stat option dumps all statistics in the\n"
"  <scenario_name.csv> file. The dump starts with one header\n" 
"  line with all counters. All following lines are 'snapshots' of \n"
"  statistics counter given the statistics report frequency\n"
"  (-fd option). This file can be easily imported in any\n"
"  spreadsheet application, like Excel.\n"
"\n"
"  In counter names, (P) means 'Periodic' - since last\n"
"  statistic row and (C) means 'Cumulated' - since sipp was\n"
"  started.\n"
"\n"
"  Available statistics are:\n"
"\n"
"  - StartTime: \n"
"    Date and time when the test has started.\n"
"\n"
"  - LastResetTime:\n"
"    Date and time when periodic counters where last reseted.\n"
"\n"
"  - CurrentTime:\n"
"    Date and time of the statistic row.\n"
"\n"
"  - ElapsedTime:\n"
"    Elapsed time.\n"
"\n"
"  - CallRate:\n"
"    Call rate (calls per seconds).\n"
"\n"
"  - IncomingCall:\n"
"    Number of incoming calls.\n"
"\n"
"  - OutgoingCall:\n"
"    Number of outgoing calls.\n"
"\n"
"  - TotalCallCreated:\n"
"    Number of calls created.\n"
"\n"
"  - CurrentCall:\n"
"    Number of calls currently ongoing.\n"
"\n"
"  - SuccessfulCall:\n"
"    Number of successful calls.\n"
"\n"
"  - FailedCall:\n"
"    Number of failed calls (all reasons).\n"
"\n"
"  - FailedCannotSendMessage:\n"
"    Number of failed calls because Sipp cannot send the\n"
"    message (transport issue).\n"
"\n"
"  - FailedMaxUDPRetrans:\n"
"    Number of failed calls because the maximum number of\n"
"    UDP retransmission attempts has been reached.\n"
"\n"
"  - FailedUnexpectedMessage:\n"
"    Number of failed calls because the SIP message received\n"
"    is not expected in the scenario.\n"
"\n"
"  - FailedCallRejected:\n"
"    Number of failed calls because of Sipp internal error.\n"
"    (a scenario sync command is not recognized or a scenario\n"
"    action failed or a scenario variable assignment failed).\n"
"\n"
"  - FailedCmdNotSent:\n"
"    Number of failed calls because of inter-Sipp\n"
"    communication error (a scenario sync command failed to\n"
"    be sent).\n"
"\n"
"  - FailedRegexpDoesntMatch:\n"
"    Number of failed calls because of regexp that doesn't\n"
"    match (there might be several regexp that don't match\n"
"    during the call but the counter is increased only by\n"
"    one).\n"
"\n"
"  - FailedRegexpHdrNotFound:\n"
"    Number of failed calls because of regexp with hdr    \n"
"    option but no matching header found.\n"
"\n"
"  - OutOfCallMsgs:\n"
"    Number of SIP messages received that cannot be associated\n"
"    to an existing call.\n"
"\n"
"  - AutoAnswered:\n"
"    Number of unexpected specific messages received for new Call-ID.\n"
"    The message has been automatically answered by a 200 OK\n"
"    Currently, implemented for 'PING' message only.\n"
"\n");
}

/************* exit handler *****************/

void print_last_stats()
{
  interrupt = 1;
  // print last current screen
  print_statistics(1);
  // and print statistics screen
  currentScreenToDisplay = DISPLAY_STAT_SCREEN;
  print_statistics(1);
}

void releaseGlobalAllocations()
{
  message * L_ptMsg = NULL;

  CStat::instance()->close();

  for(int i=0; i<SCEN_VARIABLE_SIZE; i++)
    {
      if (scenVariableTable[i] != NULL)
        delete(scenVariableTable[i]);
      scenVariableTable[i] = NULL;
    }

  for(int i=0; i<scenario_len; i++)
  {
    L_ptMsg = scenario[i];
    if (L_ptMsg != NULL)
    {
      delete(L_ptMsg);
      scenario[i] = NULL;
    }
  }
}

char* remove_pattern(char* P_buffer, char* P_extensionPattern) {

  char *L_ptr = P_buffer;

  if (P_extensionPattern == NULL) {
    return P_buffer ;
  }

  if (P_buffer == NULL) {
    return P_buffer ;
  }

  L_ptr = strstr(P_buffer, P_extensionPattern) ;
  if (L_ptr != NULL) {
    *L_ptr = '\0' ;
  }

  return P_buffer ;
  
}

/* Main */
int main(int argc, char *argv[])
{
  int                  argi = 0;
  int                  index = 0;
  struct sockaddr_storage   local_sockaddr;
  struct sockaddr_storage   localTwin_sockaddr;
  struct sockaddr_storage   media_sockaddr;
  int                  user_port = 0;
  pthread_t            pthread_id, pthread2_id;
  int                  argiFileName = 0;
  int                  argiInputFile = 0;
  char                *scenario_name = NULL;
  char                 hostname[80];
  bool                 is_ipv6 = false;
  int                  err;

  /* At least one argument is needed */
  if(argc < 2) {
    help();
    exit(EXIT_OTHER);
  }

  /* Ignore the SIGPIPE signal */
  {
    struct sigaction action_pipe;
    memset(&action_pipe, 0, sizeof(action_pipe));
    action_pipe.sa_handler=SIG_IGN;
    sigaction(SIGPIPE, &action_pipe, NULL);
	 
    /* sig usr1 management */
    struct sigaction action_usr1;
    memset(&action_usr1, 0, sizeof(action_usr1));
    action_usr1.sa_handler = sipp_sigusr1;
    sigaction(SIGUSR1, &action_usr1, NULL);

    /* sig usr2 management */
    struct sigaction action_usr2;
    memset(&action_usr2, 0, sizeof(action_usr2));
    action_usr2.sa_handler = sipp_sigusr2;
    sigaction(SIGUSR2, &action_usr2, NULL);
  }

  screen_set_exename((char *)"sipp");
  
  pid = getpid();
  memset(local_ip, 0, 40);
  memset(media_ip,0, 40);
  
  /* Load compression pluggin if available */
  comp_load();
  
  /* Command line parsing */
  
  for(argi = 1; argi < argc; argi++) {
	 
    int processed = 0;
	 
    if((!strcmp(argv[argi], "-h"    )) ||
       (!strcmp(argv[argi], "--h"   )) || 
       (!strcmp(argv[argi], "--help")) || 
       (!strcmp(argv[argi], "-help" ))    ) {
      if(((argi+1) < argc) && (!strcmp(argv[argi+1], "stat"))) {
        help_stats();
      } else {
        help();
      }
      exit(EXIT_OTHER);
    }
	 
    if(!strcmp(argv[argi], "-p")) {
      if((++argi) < argc) {
        user_port = atol(argv[argi]);
        processed = 1;
      } else {
        ERROR_P1("Missing argument for param '%s'.\nUse 'sipp -h' for details",
                 argv[argi-1]);
      }
    }
	 
    if(!strcmp(argv[argi], "-mp")) {
      if((++argi) < argc) {
        media_port = atol(argv[argi]);
	processed = 1;
      } else {
        ERROR_P1("Missing argument for param '%s'.\nUse 'sipp -h' for details",
                 argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-mi")) {
      if((++argi) < argc) {
        processed = 1;
        strcpy(media_ip, argv[argi]);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-t")) {
      if((++argi) < argc) {
        processed = 1;
        if(!strcmp(argv[argi], "u1")) {
          transport = T_UDP;
          multisocket = 0;
        } else if(!strcmp(argv[argi], "un")) {
          transport = T_UDP;
          multisocket = 1;
        } else if(!strcmp(argv[argi], "t1")) {
          transport = T_TCP;
          multisocket = 0;
        } else if(!strcmp(argv[argi], "tn")) {
          transport = T_TCP;
          multisocket = 1;
#ifdef _USE_OPENSSL
        } else if(!strcmp(argv[argi], "l1")) {
          transport = T_TLS;
          multisocket = 0;
          if ( init_OpenSSL() != 1) {
             printf("OpenSSL Initialization problem\n");
             exit ( -1);
          } 
        } else if (!strcmp(argv[argi], "ln")) {       
          transport = T_TLS;
          multisocket = 1;
          if ( init_OpenSSL() != 1) {
             printf("OpenSSL Initialization problem\n");
             exit ( -1);
          }
#endif
        } else if(!strcmp(argv[argi], "c1")) {
          if(strlen(comp_error)) {
            ERROR_P1("No " COMP_PLUGGIN " pluggin available:\n%s", comp_error);
          }
          transport = T_UDP;
          multisocket = 0;
          compression = 1;
        } else if(!strcmp(argv[argi], "cn")) {
          if(strlen(comp_error)) {
            ERROR_P1("No " COMP_PLUGGIN " pluggin available:\n%s", comp_error);
          }
          transport = T_UDP;
          multisocket = 1;
          compression = 1;
        } else {
          ERROR_P1("Invalid argument for -t param : '%s'.\n"
                   "Use 'sipp -h' for details",  argv[argi]);
        }          
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-nr")) {
      processed = 1;
      retrans_enabled = 0;
    }

    if(!strcmp(argv[argi], "-nd")) {
      processed = 1;
      default_behavior = 0;
    }

    if(!strcmp(argv[argi], "-trace_msg")) {
      useMessagef = 1 ;
      processed = 1;
    }

    if(!strcmp(argv[argi], "-trace_screen")) {
      useScreenf = 1 ;
      processed = 1;
    }

    if(!strcmp(argv[argi], "-trace_err")) {
      processed = 1;
      print_all_responses = 1;
    }

    if(!strcmp(argv[argi], "-trace_timeout")) {
      useTimeoutf = 1 ;
      processed = 1;
    }

    if(!strcmp(argv[argi], "-trace_stat")) {
      processed  = 1;
      dumpInFile = 1;
    }

    if(!strcmp(argv[argi], "-trace_logs")) {
      processed  = 1;
      useLogf = 1;
    }

    if(!strcmp(argv[argi], "-stf")) {
      if((++argi) < argc) {
        processed = 1;
        argiFileName = argi;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-inf")) {
      if((++argi) < argc) {
        processed = 1;
        argiInputFile = argi;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-d")) {
      if((++argi) < argc) {
        processed = 1;
        duration = atol(argv[argi]);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-i")) {
      if((++argi) < argc) {
        processed = 1;
        int dummy_port;
        strcpy(local_ip, argv[argi]);
        get_host_and_port(local_ip, local_ip, &dummy_port);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }
    
    if(!strcmp(argv[argi], "-m")) {
      if((++argi) < argc) {
        processed = 1;
        stop_after  = atol(argv[argi]);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-f")) {
      if((++argi) < argc) {
        processed = 1;
        report_freq  = atol(argv[argi]) * 1000;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-fd")) {
      if((++argi) < argc) {
        processed = 1;
        report_freq_dumpLog = atol(argv[argi]) * 1000;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-l")) {
      if((++argi) < argc) {
        processed = 1;
        open_calls_allowed = atol(argv[argi]);
        open_calls_user_setting = 1;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-bg")) {
      processed = 1;
      backgroundMode = true;
    }

    if(!strcmp(argv[argi], "-v")) {
#ifdef _USE_OPENSSL
      printf("\n Sipp v1.1pre-with-TLS, built %s, %s.\n\n", __DATE__, __TIME__); 
#else
      printf("\n Sipp v1.1pre-without-TLS, built %s, %s.\n\n", __DATE__, __TIME__); 
#endif
      printf
        (" This program is free software; you can redistribute it and/or\n"
         " modify it under the terms of the GNU General Public License as\n"
         " published by the Free Software Foundation; either version 2 of\n"
         " the License, or (at your option) any later version.\n"
         "\n"
         " This program is distributed in the hope that it will be useful,\n"
         " but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
         " MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
         " GNU General Public License for more details.\n"
         "\n"
         " You should have received a copy of the GNU General Public\n"
         " License along with this program; if not, write to the\n"
         " Free Software Foundation, Inc.,\n"
         " 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA\n"
         "\n"
         " Author: Richard GAYRAUD - 04 Nov 2003.\n\n");
      
      exit(EXIT_OTHER);
    }
    
    if(!strcmp(argv[argi], "-r")) {
      if((++argi) < argc) {
        processed = 1;
        rate = atof(argv[argi]);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-rp")) {
      if((++argi) < argc) {
        processed = 1;
        rate_period_s = atof(argv[argi]) / 1000 ;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    
    if(!strcmp(argv[argi], "-s")) {
      if((++argi) < argc) {
        processed = 1;
        service = argv[argi];
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }
    
#ifdef _USE_OPENSSL
    if(!strcmp(argv[argi], "-ap")) {
      if((++argi) < argc) {
        auth_password = argv[argi];
	    processed = 1;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }
#endif

    if(!strcmp(argv[argi], "-sf")) {
      if((++argi) < argc) {
        processed = 1;
        load_scenario(argv[argi], 0);
        scenario_file = new char [strlen(argv[argi])+1] ;
        sprintf(scenario_file,"%s", argv[argi]);
        CStat::instance()->setFileName(argv[argi], (char*)".csv");
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    // Remote sending address different from the received messages
    if(!strcmp(argv[argi], "-rsa")) {
      if((++argi) < argc) {
	char *remote_s_address ;
        int   remote_s_p;
        int   temp_remote_s_p;

        processed = 1;
        temp_remote_s_p = 0;
	remote_s_address = argv[argi] ;
        get_host_and_port(remote_s_address, remote_s_address, &temp_remote_s_p);
        if (temp_remote_s_p != 0) {
          remote_s_p = temp_remote_s_p;
        }
        struct addrinfo   hints;
        struct addrinfo * local_addr;

        printf("Resolving remote sending address %s...\n", remote_s_address);
        
        memset((char*)&hints, 0, sizeof(hints));
        hints.ai_flags  = AI_PASSIVE;
        hints.ai_family = PF_UNSPEC;

        if (getaddrinfo(remote_s_address,
                        NULL,
                        &hints,
                        &local_addr) != 0) {
	    ERROR_P1("Unknown remote host '%s'.\n"
		     "Use 'sipp -h' for details", remote_s_address);
	  }

        memcpy(&remote_sending_sockaddr,
               local_addr->ai_addr,
               SOCK_ADDR_SIZE(
                 _RCAST(struct sockaddr_storage *, local_addr->ai_addr)));

        if (remote_sending_sockaddr.ss_family == AF_INET) {
          (_RCAST(struct sockaddr_in *, &remote_sending_sockaddr))->sin_port =
            htons((short)remote_s_p);
        } else {
          (_RCAST(struct sockaddr_in6 *, &remote_sending_sockaddr))->sin6_port =
            htons((short)remote_s_p);
	}
        use_remote_sending_addr = 1 ;
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    
    if(!strcmp(argv[argi], "-sn")) {
      if((++argi) < argc) {
        processed = 1;
        if(!strcmp(argv[argi], "uac")) {
          CStat::instance()->setFileName((char*)"uac", (char*)".csv");
          load_scenario(0, 0);
        } else if(!strcmp(argv[argi], "uas")) {
          CStat::instance()->setFileName((char*)"uas", (char*)".csv");
          load_scenario(0, 1);
        } else if(!strcmp(argv[argi], "regexp")) {
          CStat::instance()->setFileName((char*)"regexp", (char*)".csv");
          load_scenario(0, 2);
        } else if(!strcmp(argv[argi], "3pcc-C-A")) {
          CStat::instance()->setFileName((char*)"3pcc-C-A", (char*)".csv");
          load_scenario(0, 3);
        } else if(!strcmp(argv[argi], "3pcc-C-B")) {
          CStat::instance()->setFileName((char*)"3pcc-C-B", (char*)".csv");
          load_scenario(0, 4);
        } else if(!strcmp(argv[argi], "3pcc-A")) {
          CStat::instance()->setFileName((char*)"3pcc-A", (char*)".csv");
          load_scenario(0, 5);
        } else if(!strcmp(argv[argi], "3pcc-B")) {
          CStat::instance()->setFileName((char*)"3pcc-B", (char*)".csv");
          load_scenario(0, 6);
        } else if(!strcmp(argv[argi], "branchc")) {
          CStat::instance()->setFileName((char*)"branchc", (char*)".csv");
          load_scenario(0, 7);
        } else if(!strcmp(argv[argi], "branchs")) {
          CStat::instance()->setFileName((char*)"branchs", (char*)".csv");
          load_scenario(0, 8);
        } else {
          ERROR_P1("Invalid default scenario name '%s'.\n", argv[argi]);
        }
        scenario_file = new char [strlen(argv[argi])+1] ;
        sprintf(scenario_file,"%s", argv[argi]);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

    if(!strcmp(argv[argi], "-sd")) {
      if((++argi) < argc) {
        processed = 1;
        if(!strcmp(argv[argi], "uac")) {
          fprintf(stdout, "%s", default_scenario[0]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "uas")) {
          fprintf(stdout, "%s", default_scenario[1]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "regexp")) {
          fprintf(stdout, "%s", default_scenario[2]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "3pcc-C-A")) {
          fprintf(stdout, "%s", default_scenario[3]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "3pcc-C-B")) {
          fprintf(stdout, "%s", default_scenario[4]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "3pcc-A")) {
          fprintf(stdout, "%s", default_scenario[5]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "3pcc-B")) {
          fprintf(stdout, "%s", default_scenario[6]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "branchc")) {
          fprintf(stdout, "%s", default_scenario[7]);
          exit(EXIT_OTHER);
        } else if(!strcmp(argv[argi], "branchs")) {
          fprintf(stdout, "%s", default_scenario[8]);
          exit(EXIT_OTHER);
        } else {
          ERROR_P1("Invalid default scenario name '%s'.\n", argv[argi]);
        }
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }

#ifdef __3PCC__
    if(!strcmp(argv[argi], "-3pcc")) {
      if((++argi) < argc) {
        processed = 1;
        twinSippMode = true;
        strcpy(twinSippHost, argv[argi]);
      } else {
        ERROR_P1("Missing argument for param '%s'.\n"
                 "Use 'sipp -h' for details",  argv[argi-1]);
      }
    }
#endif

    if(!processed) {
      if((argv[argi])[0] != '-') {
        strcpy(remote_host, argv[argi]);
      } else {
        help();
        ERROR_P1("Invalid argument: '%s'.\n"
                 "Use 'sipp -h' for details", argv[argi]);
      }
    }
  }

  /* trace file setting */
  if (scenario_file == NULL) {
    scenario_file = new char [ 5 ] ;
    sprintf(scenario_file, "%s", "sipp");
  } else {
    scenario_file = remove_pattern (scenario_file, (char*)".xml");
  }

  if (useMessagef == 1) {
    char *L_file_name ;
    /* 14 for '_messages.log' and 6 for pid */
    L_file_name = new char [ strlen(scenario_file)+14+6 ] ;
    sprintf (L_file_name, "%s_%d_messages.log", scenario_file, getpid());
    messagef = fopen(L_file_name, "w");
    if(!messagef) {
      ERROR_P1("Unable to create '%s'", L_file_name);
    }
    delete [] L_file_name ;
    L_file_name = NULL ;
  }
  
  if (useScreenf == 1) {
    char *L_file_name ;
    /* 11 for '_screen.log' and 6 for pid */
    L_file_name = new char [ strlen(scenario_file)+11+6 ] ;
    sprintf (L_file_name, "%s_%d_screen.log", scenario_file, getpid());
    screenf = fopen(L_file_name, "w");
    if(!screenf) {
      ERROR_P1("Unable to create '%s'", L_file_name);
    }
    delete [] L_file_name ;
    L_file_name = NULL ;
  }

  if (useTimeoutf == 1) {
    char *L_file_name ;
    /* 13 for '_timeout.log' and 6 for pid */
    L_file_name = new char [ strlen(scenario_file)+13+6 ] ;
    sprintf (L_file_name, "%s_%d_timeout.log", scenario_file, getpid());
    timeoutf = fopen(L_file_name, "w");
    if(!timeoutf) {
      ERROR_P1("Unable to create '%s'", L_file_name);
    }
    delete [] L_file_name ;
    L_file_name = NULL ;
  }
  
  if (useLogf == 1) {
    char *L_file_name ;
    /* 9 for '_logs.log' and 6 for pid */
    L_file_name = new char [ strlen(scenario_file)+9+6 ] ;
    sprintf (L_file_name, "%s_%d_logs.log", scenario_file, getpid());
    logfile = fopen(L_file_name, "w");
    if(!logfile) {
      ERROR_P1("Unable to create '%s'", L_file_name);
    }
    delete L_file_name ;
    L_file_name = NULL ;
  }

  /* Initialization:  boost open file limit to the max (AgM)*/
  {
    struct rlimit rlimit;
    
    if (getrlimit (RLIMIT_NOFILE, &rlimit) < 0) {
      ERROR_NO("getrlimit error");
    }
    if (rlimit.rlim_max > FD_SETSIZE) {
      fprintf (stderr, "Warning: open file limit > FD_SETSIZE; "
               "limiting max. # of open files to FD_SETSIZE = %d\n",
               FD_SETSIZE);
      rlimit.rlim_max = FD_SETSIZE;
    }
    
    rlimit.rlim_cur = rlimit.rlim_max;
    if (setrlimit (RLIMIT_NOFILE, &rlimit) < 0) {
      ERROR_P1("Unable to increase the open file limit to FD_SETSIZE = %d",
               FD_SETSIZE);
    }
  }
  
  /* Load default scenario in case nothing was loaded */
  if(!scenario_len) {
    load_scenario(0, 0);
    CStat::instance()->setFileName((char*)"uac", (char*)".csv");
    sprintf(scenario_file,"uac");
  }
  
  if(argiFileName) {
    CStat::instance()->setFileName(argv[argiFileName]);
  }

  if(argiInputFile) {
    call::readInputFileContents(argv[argiInputFile]);
  }
  
  /* In which mode the tool is launched ? */
  computeSippMode();

  /* checking if we need to launch the tool in background mode */ 
  if(backgroundMode == true)
    {
      pid_t l_pid;
      switch(l_pid = fork())
        {
        case -1:
          // error when forking !
          ERROR_NO("Forking error");
          exit(EXIT_FATAL_ERROR);
        case 0:
          // child process - poursuing the execution
          break;
        default:
          // parent process - killing the parent - the child get the parent pid
          printf("Background mode - PID=[%d]\n", l_pid);
          exit(EXIT_OTHER);
        }
    }
	 
  /* Setting the rate and its dependant params (open_calls_allowed) */
  set_rate(rate);
	 
  if(!strlen(remote_host)) {
    if(toolMode != MODE_SERVER) {
      ERROR("Missing remote host parameter. This scenario requires it");
    }
  } else {
    int temp_remote_port;
    get_host_and_port(remote_host, remote_host, &temp_remote_port);
    if (temp_remote_port != 0) {
      remote_port = temp_remote_port;
    }
    
    /* Resolving the remote IP */
    {
      struct addrinfo   hints;
      struct addrinfo * local_addr;

      printf("Resolving remote host '%s'...\n", remote_host);

      memset((char*)&hints, 0, sizeof(hints));
      hints.ai_flags  = AI_PASSIVE;
      hints.ai_family = PF_UNSPEC;

      if (getaddrinfo(remote_host,
                      NULL,
                      &hints,
                      &local_addr) != 0) {
        ERROR_P1("Unknown remote host '%s'.\n"
                 "Use 'sipp -h' for details", remote_host);
      }

      memset(&remote_sockaddr, 0, sizeof( remote_sockaddr ));
      memcpy(&remote_sockaddr,
             local_addr->ai_addr,
             SOCK_ADDR_SIZE(
               _RCAST(struct sockaddr_storage *,local_addr->ai_addr)));

      strcpy(remote_ip, get_inet_address(&remote_sockaddr));
      if (remote_sockaddr.ss_family == AF_INET) {
        (_RCAST(struct sockaddr_in *, &remote_sockaddr))->sin_port =
          htons((short)remote_port);
        strcpy(remote_ip_escaped, remote_ip); 
      } else {
        (_RCAST(struct sockaddr_in6 *, &remote_sockaddr))->sin6_port =
          htons((short)remote_port);
        sprintf(remote_ip_escaped, "[%s]", remote_ip); 
      }
    }
    }

  if(gethostname(hostname,64) != 0) {
    ERROR_NO("Can't get local hostname in 'gethostname(hostname,64)'");
  }
  
  {
    char            * local_host = NULL;
    struct addrinfo   hints;
    struct addrinfo * local_addr;

    if (!strlen(local_ip)) {
      local_host = (char *)hostname;
    } else {
      local_host = (char *)local_ip;
    }

    memset((char*)&hints, 0, sizeof(hints));
    hints.ai_flags  = AI_PASSIVE;
    hints.ai_family = PF_UNSPEC;

  /* Resolving local IP */
    if (getaddrinfo(local_host,
                    NULL,
                    &hints,
                    &local_addr) != 0) {
      ERROR_P2("Can't get local IP address in getaddrinfo, local_host='%s', local_ip='%s'", 
        local_host, 
        local_ip);
    }
    
    memset(&local_sockaddr,0,sizeof(struct sockaddr_storage));
    local_sockaddr.ss_family = local_addr->ai_addr->sa_family;

    if (!strlen(local_ip)) {
    strcpy(local_ip,
             get_inet_address(
               _RCAST(struct sockaddr_storage *, local_addr->ai_addr)));
    }
    if (local_sockaddr.ss_family == AF_INET6) {
      is_ipv6 = true;
      sprintf(local_ip_escaped, "[%s]", local_ip); 
    } else {
      strcpy(local_ip_escaped, local_ip); 
    }
  }
  
  /* Creating and binding the local socket */
  if((main_socket = socket(is_ipv6 ? AF_INET6 : AF_INET,
                           (transport == T_UDP) ? SOCK_DGRAM : SOCK_STREAM,
                           0)) == -1) {
    ERROR_NO("Unable to get the local socket");
  }
  
  /* Trying to bind local port */
  if(!user_port) {
    unsigned short l_port;
    for(l_port = DEFAULT_PORT;
        l_port < (DEFAULT_PORT + 60);
        l_port++) {
      if (is_ipv6) {
        (_RCAST(struct sockaddr_in6 *, &local_sockaddr))->sin6_port
          = htons((short)l_port);
      } else {
        (_RCAST(struct sockaddr_in *, &local_sockaddr))->sin_port
          = htons((short)l_port);
      }
      if(!bind(main_socket,
               (sockaddr *)(void *)&local_sockaddr,
               SOCK_ADDR_SIZE(&local_sockaddr))) {
        local_port = l_port;
        break;
      }
    }
  }

  if(!local_port) {
    /* Not already binded, use user_port of 0 to leave
     * the system choose a port. */
    if (is_ipv6) {
      (_RCAST(struct sockaddr_in6 *, &local_sockaddr))->sin6_port
        = htons((short)user_port);
    } else {
      (_RCAST(struct sockaddr_in *, &local_sockaddr))->sin_port
        = htons((short)user_port);
    }
    if(bind(main_socket, 
            (sockaddr *)(void *)&local_sockaddr,
            SOCK_ADDR_SIZE(&local_sockaddr))) {
      ERROR_NO("Unable to bind main socket");
    }
  }
  
  /* Recover system port */
  {
    sipp_socklen_t len = SOCK_ADDR_SIZE(&local_sockaddr);
    getsockname(main_socket,
                (sockaddr *)(void *)&local_sockaddr,
                &len);
    if (is_ipv6) {
      local_port =
        ntohs((short)
          (_RCAST(struct sockaddr_in6 *,&local_sockaddr))->sin6_port);
    } else {
      local_port =
        ntohs((short)
          (_RCAST(struct sockaddr_in *,&local_sockaddr))->sin_port);
    }

  }
  sipp_customize_socket(main_socket);

#ifdef _USE_OPENSSL
  if((!multisocket) && (transport == T_TCP || transport == T_TLS) &&
#else
  if((!multisocket) && (transport == T_TCP) &&
#endif
   (toolMode != MODE_SERVER)) {
    if((tcp_multiplex = socket(is_ipv6 ? AF_INET6 : AF_INET,
                               SOCK_STREAM,
                               0))== -1) {
      ERROR_NO("Unable to get a TCP socket");
    }
    
    /*
    struct sockaddr_storage *L_dest = &remote_sockaddr;

    if (use_remote_sending_addr) {
        L_dest = &remote_sending_sockaddr ;
    }
               (struct sockaddr *)(void *)L_dest,
    */

    if(connect(tcp_multiplex,
               (struct sockaddr *)(void *)&remote_sockaddr,
               SOCK_ADDR_SIZE(&remote_sockaddr))) {
      
      if(errno == EINVAL){
        /* This occurs sometime on HPUX but is not a true INVAL */
        ERROR_NO("Unable to connect a TCP socket, remote peer error.\n"
              "Use 'sipp -h' for details");
      } else {
        ERROR_NO("Unable to connect a TCP socket.\n"
                 "Use 'sipp -h' for details");
      }
    }
#ifdef _USE_OPENSSL
    if ( transport == T_TLS ) {
      if ((sip_trp_ssl_ctx_client = SSL_CTX_new(SSLv23_method())) == NULL){
        ERROR("Unable to init the SSL_CTX - SSL_CTX_new Fail\n");
      }
    
      if ( (bio = BIO_new_socket(tcp_multiplex,BIO_NOCLOSE)) == NULL) {
        ERROR("Unable to create BIO object:Problem with BIO_new_socket()\n");
      }
    
      if (!(ssl_tcp_multiplex = SSL_new(sip_trp_ssl_ctx_client))){
        ERROR("Unable to create SSL object : Problem with SSL_new() \n");
      }
    
      SSL_set_bio(ssl_tcp_multiplex,bio,bio);
      if ( (err = SSL_connect(ssl_tcp_multiplex)) < 0 ) {
        ERROR("Error in SSL connection \n");
      }
      ssl_list[tcp_multiplex] = ssl_tcp_multiplex;
    }
#endif
    
    sipp_customize_socket(tcp_multiplex);
  }

#ifdef _USE_OPENSSL
  if (transport == T_TLS ) {
    if ((sip_trp_ssl_ctx = SSL_CTX_new(SSLv23_method())) == NULL) {
      ERROR("SSL_CTX_new() - main() - Fails\n");
    }
    /* Load the Certificate into the SSL context */
  
    if ( SSL_CTX_use_certificate_file(sip_trp_ssl_ctx,CERTFILE,SSL_FILETYPE_PEM) != 1 ) {
       ERROR_P1("Unable to load Certificate SSL_CTX_use_certificate_file '%s'\n",
                CERTFILE);
    }
    if ( SSL_CTX_use_PrivateKey_file(sip_trp_ssl_ctx,PRIVATE_KEY_FILE,SSL_FILETYPE_PEM) != 1) {
       ERROR_P1("SSL_CTX_use_PrivateKey_file Fails with file '%s'\n",
                PRIVATE_KEY_FILE);
    }
  }
  if(transport == T_TCP || transport == T_TLS) {
#else
  if(transport == T_TCP) {
#endif
    if(listen(main_socket, 100)) {
      ERROR_NO("Unable to listen main socket");
    }
  }

#ifdef __3PCC__
  /* Trying to connect to Twin Sipp in 3PCC mode */
  if(twinSippMode) {
    if(toolMode == MODE_3PCC_CONTROLLER_A || toolMode == MODE_3PCC_A_PASSIVE) {
      if(strstr(twinSippHost, ":")) {
              twinSippPort = atol(strstr(twinSippHost, ":")+1);
              *(strstr(twinSippHost, ":")) = 0;
            }

          /* Resolving the twin IP */
            printf("Resolving twin address : %s...\n", twinSippHost);
      struct addrinfo   hints;
      struct addrinfo * local_addr;
      memset((char*)&hints, 0, sizeof(hints));
      hints.ai_flags  = AI_PASSIVE;
      hints.ai_family = PF_UNSPEC;
      is_ipv6 = false;
      /* Resolving local IP */
      if (getaddrinfo(twinSippHost,
                      NULL,
                      &hints,
                      &local_addr) != 0) {
              ERROR_P1("Unknown twin host '%s'.\n"
                       "Use 'sipp -h' for details", twinSippHost);
            }

      memcpy(&twinSipp_sockaddr,
             local_addr->ai_addr,
             SOCK_ADDR_SIZE(
               _RCAST(struct sockaddr_storage *,local_addr->ai_addr)));

      if (twinSipp_sockaddr.ss_family == AF_INET) {
       (_RCAST(struct sockaddr_in *,&twinSipp_sockaddr))->sin_port =
         htons((short)twinSippPort);
      } else {
        (_RCAST(struct sockaddr_in6 *,&twinSipp_sockaddr))->sin6_port =
          htons((short)twinSippPort);
        is_ipv6 = true;
      }
      strcpy(twinSippIp, get_inet_address(&twinSipp_sockaddr));
      if((twinSippSocket = socket(is_ipv6 ? AF_INET6 : AF_INET,
          SOCK_STREAM, 0))== -1) {
              ERROR_NO("Unable to get a TCP socket in 3PCC controller A mode");
            }
    
          if(connect(twinSippSocket,
                     (struct sockaddr *)(void *)&twinSipp_sockaddr,
                 SOCK_ADDR_SIZE(&twinSipp_sockaddr))) {
        if(errno == EINVAL) {
                  /* This occurs sometime on HPUX but is not a true INVAL */
                  ERROR_NO("Unable to connect a TCP socket in 3PCC controller "
                        "A mode, remote peer error.\n"
                        "Use 'sipp -h' for details");
        } else {
                  ERROR_NO("Unable to connect a TCP socket in 3PCC controller "
                           "A mode.\n"
                           "Use 'sipp -h' for details");
                }
            }

          sipp_customize_socket(twinSippSocket);
    } else if(toolMode == MODE_3PCC_CONTROLLER_B) {
      if(strstr(twinSippHost, ":")) {
              twinSippPort = atol(strstr(twinSippHost, ":")+1);
              *(strstr(twinSippHost, ":")) = 0;
            }

          /* Resolving the listener IP */
            printf("Resolving listener address : %s...\n", twinSippHost);
            struct addrinfo   hints;
            struct addrinfo * local_addr;
            memset((char*)&hints, 0, sizeof(hints));
            hints.ai_flags  = AI_PASSIVE;
            hints.ai_family = PF_UNSPEC;
            is_ipv6 = false;
            
            /* Resolving local IP */
            if (getaddrinfo(twinSippHost,
                           NULL,
                           &hints,
                           &local_addr) != 0) {
               ERROR_P1("Unknown twin host '%s'.\n"
                        "Use 'sipp -h' for details", twinSippHost);
             }
             memcpy(&twinSipp_sockaddr,
                    local_addr->ai_addr,
                    SOCK_ADDR_SIZE(
                      _RCAST(struct sockaddr_storage *,local_addr->ai_addr)));
      
             if (twinSipp_sockaddr.ss_family == AF_INET) {
              (_RCAST(struct sockaddr_in *,&twinSipp_sockaddr))->sin_port =
                htons((short)twinSippPort);
             } else {
               (_RCAST(struct sockaddr_in6 *,&twinSipp_sockaddr))->sin6_port =
                 htons((short)twinSippPort);
               is_ipv6 = true;
             }
             strcpy(twinSippIp, get_inet_address(&twinSipp_sockaddr));
      
             if((localTwinSippSocket = socket(is_ipv6 ? AF_INET6 : AF_INET,
                 SOCK_STREAM, 0))== -1) {
                ERROR_NO("Unable to get a TCP socket in 3PCC controller B mode");
              }

           memset(&localTwin_sockaddr, 0, sizeof(struct sockaddr_storage));
           if (!is_ipv6) {
            localTwin_sockaddr.ss_family = AF_INET;
            (_RCAST(struct sockaddr_in6 *,&localTwin_sockaddr))->sin6_port =
              htons((short)twinSippPort);
           } else {
             localTwin_sockaddr.ss_family = AF_INET6;
             (_RCAST(struct sockaddr_in *,&localTwin_sockaddr))->sin_port =
               htons((short)twinSippPort);
           }
          if(bind(localTwinSippSocket, 
                  (sockaddr *)(void *)&localTwin_sockaddr,
                   SOCK_ADDR_SIZE(&localTwin_sockaddr))) {
              ERROR_NO("Unable to bind twin sipp socket in "
                    "3PCC_CONTROLLER_B mode");
            }

          if(listen(localTwinSippSocket, 100))
            ERROR_NO("Unable to listen twin sipp socket in "
                     "3PCC_CONTROLLER_B mode");
          sipp_customize_socket(localTwinSippSocket);
    } else {
        ERROR("TwinSipp Mode enabled but toolMode is different "
              "from 3PCC_CONTROLLER_B and 3PCC_CONTROLLER_A\n");
    }
  } /* end if(twinSippMode) */
#endif
   
  /* MaL : Create and Bind RTP socket */
  if (media_port > 0) {
    if((media_socket = socket(is_ipv6 ? AF_INET6 : AF_INET,
                              SOCK_DGRAM, 0)) == -1) {
      ERROR_NO("Unable to get the RTP echo socket");
    }

    /* retrieve RTP local addr */
    if (media_ip[0] == 0) {
      strcpy(media_ip, local_ip);
    }

    struct addrinfo   hints;
    struct addrinfo * local_addr;

    memset((char*)&hints, 0, sizeof(hints));
    hints.ai_flags  = AI_PASSIVE;
    hints.ai_family = PF_UNSPEC;

    is_ipv6 = false;

    /* Resolving local IP */
    if (getaddrinfo(media_ip,
                    NULL,
                    &hints,
                    &local_addr) != 0) {
      ERROR_P1("Unknown RTP address '%s'.\n"
               "Use 'sipp -h' for details", media_ip);
    }

    memcpy(&media_sockaddr,
           local_addr->ai_addr,
           SOCK_ADDR_SIZE(
             _RCAST(struct sockaddr_storage *,local_addr->ai_addr)));

    if (media_sockaddr.ss_family == AF_INET) {
     (_RCAST(struct sockaddr_in *,&media_sockaddr))->sin_port =
       htons((short)media_port);
     strcpy(media_ip_escaped, media_ip);
    } else {
      (_RCAST(struct sockaddr_in6 *,&media_sockaddr))->sin6_port =
        htons((short)media_port);
      is_ipv6 = true;
      sprintf(media_ip_escaped, "[%s]", media_ip);
    }

    if(bind(media_socket, 
            (sockaddr *)(void *)&media_sockaddr,
            SOCK_ADDR_SIZE(&media_sockaddr))) {
      ERROR_NO("Unable to bind RTP socket");
    }
    
  } else {
    /* If media_port is not defined, need to put a default 
     * value for XML script variables */
    media_port = 10000;
    if (media_ip[0] == 0) {
      strcpy(media_ip, local_ip);
    }
  }

  if( backgroundMode == false ) {

    if (print_all_responses) {
    char *L_file_name ;
    /* 12 for '_errors.log' and 6 for pid */
    L_file_name = new char [ strlen(scenario_file)+12+6 ] ;
      sprintf (L_file_name, "%s_%d_errors.log", scenario_file, getpid());
    screen_init(L_file_name, print_last_stats);
      delete [] L_file_name ;
      L_file_name = NULL ;
    } else {
      screen_init(NULL, print_last_stats);
    }

    /* Creating the keyb thread */
    if (pthread_create
        (&pthread_id,
         NULL,
         (void *(*)(void *)) keyb_thread,
         (void*)NULL) 
        == -1) {
      ERROR_NO("Unable to create recv thread");
    }
  }

  /* Creating the RTP echo thread */
  if (media_socket > 0) {
    if (pthread_create
        (&pthread2_id,
         NULL,
         (void *(*)(void *)) rtp_echo_thread,
         (void*)NULL) 
        == -1) {
      ERROR_NO("Unable to create RTP echo thread");
    }
  }

  traffic_thread(is_ipv6);

  if (scenario_file != NULL) {
    delete [] scenario_file ;
    scenario_file = NULL ;
  }
}
