/*
**  Sinek (Video Player)
**  Copyright (c) 2001 - 2002 the Sinek Team, see the AUTHORS file.
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  playlist
*/

#include "common.h"

static GtkWidget *pl_win;
static GtkWidget *clist_wid;
static GtkWidget *add_win;
static gint pl_row = -1, pl_cur = -1;

typedef struct pl_item_struct
{
	gchar *buf[3];
	char *mrl;
	int time;
	int format;
} pl_item;

static void cb_load(GtkWidget *w, gpointer data);
static void cb_save(GtkWidget *w, int to_default);
static void cb_play(GtkWidget *w, gpointer data);
static void cb_selection(GtkWidget *w, gint row, gint col, GdkEventButton *ev, gpointer data);
static gint cb_double_selection(GtkWidget *w, GdkEventButton *ev, gpointer data);
static void cb_add(GtkWidget *w, gpointer data);
static void cb_shuffle(GtkWidget *w, gpointer data);
static void cb_remove(GtkWidget *w, gpointer data);
void pl_load(char *fname);

#define BUTTON(title,sigfunc,data) \
	but = gtk_button_new_with_label(title); \
	gtk_widget_show(but); \
	gtk_box_pack_start(GTK_BOX(hb), but, TRUE, TRUE, 4); \
	gtk_signal_connect(GTK_OBJECT(but), "clicked", GTK_SIGNAL_FUNC(sigfunc), (gpointer)data)


int pl_init(void)
{
	gchar *titles[] =  { _("Media"), NULL };
	gchar *fname = NULL, *t;
	GtkWidget *vb, *info, *sw, *hb, *but;

	pl_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(pl_win), _("Sinek Playlist"));
	gtk_window_set_wmclass(GTK_WINDOW(pl_win), "Sinek Playlist", "Sinek");
	gtk_window_set_default_size(GTK_WINDOW(pl_win), 450, 200);

	vb = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(vb);
	gtk_container_add(GTK_CONTAINER(pl_win), vb);
	gtk_container_set_border_width(GTK_CONTAINER(vb), 4);

	info = gtk_entry_new();
	gtk_widget_show(info);
	gtk_box_pack_start(GTK_BOX(vb), info, FALSE, FALSE, 4);
	gtk_entry_set_editable(GTK_ENTRY(info), FALSE);

	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(vb), sw, TRUE, TRUE, 0);

	clist_wid = gtk_clist_new_with_titles(1, titles);
	gtk_widget_show(clist_wid);
	gtk_container_set_border_width(GTK_CONTAINER(clist_wid), 2);
	gtk_clist_set_reorderable(GTK_CLIST(clist_wid), TRUE);
	gtk_signal_connect(GTK_OBJECT(clist_wid), "select_row", GTK_SIGNAL_FUNC(cb_selection), NULL);
	gtk_signal_connect(GTK_OBJECT(clist_wid), "button_press_event", GTK_SIGNAL_FUNC(cb_double_selection), NULL);
	gtk_container_add(GTK_CONTAINER(sw), clist_wid);

	hb = gtk_hbox_new(TRUE, 0);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, TRUE, 4);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 5);
	BUTTON(_("Play"), cb_play, 0);
	BUTTON(_("Shuffle"), cb_shuffle, 0);
	BUTTON(_("Add..."), cb_add, 0);
	BUTTON(_("Remove"), cb_remove, 0);
	BUTTON(_("Remove All"), pl_remove_all, 0);

	but = gtk_hseparator_new();
	gtk_widget_show(but);
	gtk_box_pack_start(GTK_BOX(vb), but, FALSE, TRUE, 0);

	hb = gtk_hbox_new(TRUE, 0);
	gtk_widget_show(hb);
	gtk_box_pack_start(GTK_BOX(vb), hb, FALSE, TRUE, 4);
	gtk_container_set_border_width(GTK_CONTAINER(hb), 5);
	BUTTON(_("Load..."), cb_load, 0);
	BUTTON(_("Save As..."), cb_save, 0);
	BUTTON(_("Save As Default"), cb_save, 1);

	wm_manage(pl_win, WM_NORMAL);

	if(sinek.start_mrl)
	{
		t = strrchr(sinek.start_mrl, '.');
		if(t && strcasecmp(t, ".m3u") == 0)
		{
			fname = g_strdup(sinek.start_mrl);
			sinek.start_mrl = NULL;
		}
	}
	if(fname == NULL)
	{
		fname = g_strconcat(g_get_home_dir(), "/.xine/sinek.plist", NULL);
	}
	pl_load(fname);
	g_free(fname);

	sinek.playlist_win = pl_win;

	return 1;
}


void pl_show(void)
{
	gtk_widget_show(pl_win);
}


void pl_prev(void)
{
	pl_item *item = NULL;

	if(pl_cur > 0)
	{
		pl_cur--;
		item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), pl_cur);
	}
	if(item) video_play(item->mrl); else pl_cur = -1;
}


void pl_next(void)
{
	pl_item *item;

	pl_cur++;
	while(1)
	{
		item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), pl_cur);
		if(item)
		{
			if(video_play(item->mrl)) return;
			pl_cur++;
		}
		else
		{
			if(sinek.repeat_all)
			{
				pl_cur = 0;
			}
			else
			{
				pl_cur = -1;
				return;
			}
		}
	}
}


char *pl_get_mrl(int row)
{
	pl_item *item;

	item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), row);
	if(item) return(item->mrl);
	return NULL;
}


void pl_append(char *mrl)
{
	pl_item *item;
	int row;

	if(!mrl || mrl[0] == 0) return;
	item = malloc(sizeof(pl_item));
	if(!item) return;

	item->mrl = strdup(mrl);
	item->time = 0;
	item->buf[0] = g_basename(item->mrl);
	item->buf[1] = "";
	item->buf[2] = "";

	row = gtk_clist_append(GTK_CLIST(clist_wid), item->buf);
	gtk_clist_set_row_data(GTK_CLIST(clist_wid), row, (gpointer)item);
	gtk_clist_columns_autosize(GTK_CLIST(clist_wid));
}


void pl_remove(pl_item *item)
{
	if(item->mrl) free(item->mrl);
	free(item);
}


void pl_remove_all(void)
{
	pl_item *item;
	int row;

	for(row = 0; (item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), row)); row++)
	{
		pl_remove(item);
	}
	gtk_clist_clear(GTK_CLIST(clist_wid));
	pl_row = -1;
	pl_cur = -1;
}


void pl_load(char *fname)
{
	gchar buf[2048], *t, *path;
	FILE *f;

	f = fopen(fname, "r");
	if(f)
	{
		while(fgets(buf, 2040, f))
		{
			strtok(buf, "\r\n");
			if(buf[0] != '#')
			{
				while((t = strchr(buf, '\\')) != NULL)
					*t = '/';
				if(buf[0] != '/' && !strstr(buf, "://"))
				{
					path = g_strdup(fname);
					t = strrchr(path, '/');
					if(t)
					{
						*t = '\0';
						t = g_strdup_printf("%s/%s", path, buf);
						pl_append(t);
						free(t);
					}
					else
					{
						pl_append(buf);
					}
				}
				else
				{
					pl_append(buf);
				}
			}
		}
		fclose(f);
	}
}


void pl_save(char *filename)
{
	pl_item *item;
	int row;
	FILE *f;

	f = fopen(filename, "w");
	if(f)
	{
		for(row = 0; (item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), row)); row++)
		{
			fprintf(f, "%s\n", item->mrl);
		}
		fclose(f);
	}
}


static void cb_load(GtkWidget *w, gpointer data)
{
	select_file(pl_load, _("Load playlist..."));
}


static void cb_save(GtkWidget *w, int to_default)
{
	gchar *fname;

	if(to_default)
	{
		fname = g_strconcat(g_get_home_dir(), "/.xine/sinek.plist", NULL);
		pl_save(fname);
		g_free(fname);
	}
	else
		select_file(pl_save, _("Save playlist as..."));
}


static void cb_play(GtkWidget *w, gpointer data)
{
	pl_item *item;

	if(pl_row == -1) return;
	item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), pl_row);
	if(item)
	{
		pl_cur = pl_row;
		video_play(item->mrl);
	}
}


static void cb_selection(GtkWidget *w, gint row, gint col, GdkEventButton *ev, gpointer data)
{
	pl_row = row;
}


static gint cb_double_selection(GtkWidget *w, GdkEventButton *ev, gpointer data)
{
	if(GTK_IS_CLIST(w) && ev->type == GDK_2BUTTON_PRESS)
	{
		cb_play(clist_wid, NULL);
	}
	return FALSE;
}


static void cb_add(GtkWidget *w, gpointer data)
{
	if(NULL == add_win)
	{
		add_win = file_request(1);
		wm_manage(add_win, 0);
	}
	gtk_widget_show(add_win);
}


static void cb_remove(GtkWidget *w, gpointer data)
{
	pl_item *item;

	if(pl_row == -1) return;
	item = gtk_clist_get_row_data(GTK_CLIST(clist_wid), pl_row);
	gtk_clist_remove(GTK_CLIST(clist_wid), pl_row);
	gtk_clist_columns_autosize(GTK_CLIST(clist_wid));
	pl_remove(item);
	pl_row = -1;
}


static void cb_shuffle(GtkWidget *w, gpointer data)
{
	int *deal, *newdeal;
	int len = GTK_CLIST(clist_wid)->rows;
	int i, j;

	if(len < 2) return;

	deal = g_malloc0(sizeof(int) * len);
	newdeal = g_malloc0(sizeof(int) * len);
	for(i = 0; i < len; i++)
		deal[i] = i;

	for(i = 0; i < len; i++)
	{
		j = (((double)len - i) * rand()) / RAND_MAX;
		newdeal[i] = deal[j];
		if(j != (len - i - 1)) deal[j] = deal[len - i - 1];
	}

	gtk_clist_freeze(GTK_CLIST(clist_wid));
	for(i = 0; i < len; i++)
	{
		j = newdeal[i];
		if(newdeal[j] != -1 && i != j)
			gtk_clist_swap_rows(GTK_CLIST(clist_wid), i, j);
		newdeal[i] = -1;
	}
	gtk_clist_unselect_all(GTK_CLIST(clist_wid));
	gtk_clist_thaw(GTK_CLIST(clist_wid));

	pl_row = -1;
	pl_cur = -1;

	g_free(newdeal);
	g_free(deal);
}
