/* files.c  --  The file holding all i/o functions required by Sigit..
   
   Copyright (C) Kenneth 'Redhead' Nielsen <kn@redhead.dk>
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****
 * $Id: files.c,v 1.7 2003/07/22 22:03:42 redhead Exp $ 
 * $Log: files.c,v $
 * Revision 1.7  2003/07/22 22:03:42  redhead
 * Cleaned up the code, which included rewriting every structure in the program.
 * Fixed some error checking, to make sure theres no undocumentet parts that might
 * produce an error.
 *
 * Revision 1.6  2003/07/18 10:58:10  redhead
 * This is going to be the 0.2.4 release, just fixed some spelling errors in teh documentation
 * fixed the bug with zombie processes of boxes, when using the boxing feature
 * fixed the long time waited sigit.rc(5) error reported by moffe
 * plus some minor changes here and there.
 *
 * Revision 1.5  2003/07/16 21:02:13  redhead
 * Fixed some more error checking, and worked on killing the zombie boxes
 * processes that seems to emerge.. At this point still no solution for it.
 *
 * Revision 1.4  2003/07/16 12:53:37  redhead
 * Added a conversion char %L to the usual conversions, this one will
 * display the load avarage on the machine.
 * Fixed some error checkign to ensure the right info will be displayed.
 *
 * Revision 1.3  2003/03/07 20:36:48  redhead
 * Fixed a flaw in the call to boxes, so it now takes several arguments in the
 * boxes_flags setting.
 * Added the functionality of having it giving random boxes, when the boxes_config
 * setting is pointing to the location of your boxes designs.
 *
 * Revision 1.2  2002/10/18 18:25:23  redhead
 * Added the use of boxes to create a surrounding box around the signature
 * This introduced the --with-boxes flag to configure.
 * If that flag isn't used it will default be assumed boxes isn't installed
 * Else the configure script will search for boxes, if none should be found
 * in the search path, you can use --with-boxes=/where to specify it.
 *
 ****/

#include "files.h" /* #include "config.h" */

#ifdef sigit

/* just a small function used to create random numbers  
 *
 * It takes a MAX value, and then returns a random number 
 * between 0 - MAX, since its using srandom() with a seed of
 * clock(), it will favor the small numbers. 
 * But for practical purpus, we can live with that.
 */

unsigned long int get_rand (unsigned long int max) 
{
  size_t res;
  time_t tim, c_start;
  if(max <= 0)
    return 0;
  c_start = clock();
  srandom(c_start);
  time(&tim);
  srandom(tim);
  res = random() % max;
  return res;
}


/* Basicaly, this function will only be available, when the 
 * program is compiled with boxes support.
 *  
 * What it does, is to look through the boxs_config file, 
 * searching for a random box design, it will then send the 
 * -d <design> flag along to boxes, hoping that it will 
 * overwrite a posible set one from the boxes_flags with
 * the assumption, that the last '-d' setting will be chosen.
 */

#ifdef BOXING_SUPPORT
int get_rand_box(FIFO* fifo, char box[])
{
  unsigned long int box_count = 0;
  int i, n = 0, space = 0;
  char buffer[BUFF_SIZE +1];
  FILE* in_file;

  pthread_mutex_lock(&mutex);

  /* open boxes config file */
  if(fifo->verbose > 0)
    logging(fifo, "[get_rand_box]\t\t: Opening boxes config file {%s}\n", fifo->boxes_config);  
  if(!(in_file = fopen(fifo->boxes_config, "r")))
    {
      if(fifo->verbose > 0)
	logging(fifo, "[get_rand_box]\t\t: ERROR: Couldn't open boxes config file {%s}\n", fifo->boxes_config);  
      pthread_mutex_unlock(&mutex);
      return CONF_MI;
    }
  /* read through it counting the BOX there */
  while(fgets(buffer, BUFF_SIZE, in_file))
    {
      if(fifo->verbose > 3)
	logging(fifo, "[get_rand_box]\t\t: Read line {%s} from boxes config file.\n", buffer);      
      if(!strncmp(buffer, "BOX", 3))
	{
	  if(fifo->verbose > 3)
	    logging(fifo, "[get_rand_box]\t\t: Found the line to be holding 'BOX'\n", fifo->boxes_config);  
	  box_count++;
	}
  }
  if(box_count <= 0)
    {
      if(fifo->verbose > 0)
	logging(fifo, "[get_rand_box]\t\t: ERROR: Couldn't find any'BOX' line in boxes config file {%s}\n", fifo->boxes_config);  
      fclose(in_file);
      pthread_mutex_unlock(&mutex);
      return COUNT_MISS;
    }
  if(fseek(in_file, 0L, SEEK_SET) == -1)
    {
      if(fifo->verbose > 0)
	logging(fifo, "[get_rand_box]\t\t: ERROR: Couldn't seek to beginning of file.\n", fifo->boxes_config);  
      fclose(in_file);
      pthread_mutex_unlock(&mutex);
      return SEEK_ERR;
    }
  /* find the random box there */
  if(fifo->verbose > 2)
    logging(fifo, "[get_rand_box]\t\t: Calling get_rand with {%d}\n", box_count);  
  box_count = get_rand(box_count);
  if(fifo->verbose > 2)
    logging(fifo, "[get_rand_box]\t\t: returning from get_rand with {%d}\n", box_count);  
  if(fifo->verbose > 1)
    logging(fifo, "[get_rand_box]\t\t: Searching for box design {%d}\n", box_count);  
  while(fgets(buffer, BUFF_SIZE, in_file))
    {
      if(!strncmp(buffer, "BOX", 3))
	{
	  if(box_count <= 1)
	    {
	    /* make sure, we start after the 'BOX' in the beginning */
	      if(fifo->verbose > 2)
		logging(fifo, "[get_rand_box]\t\t: Found the line {%s} to be holding 'BOX'\n", buffer);
	      for(i=3; i < strlen(buffer) && n < LINE_SIZE; i++)
		{ 
		  /* start copying */
		  if(fifo->verbose > 3)
		    logging(fifo, "[get_rand_box]\t\t: From the line {%s} we are testing {%c}\n", buffer, buffer[i]);
		  if(space > 0 && buffer[i] != ' ' && buffer[i] != '\t' && 
		     buffer[i] != '\n' && buffer[i] != '\r' && buffer[i] != '\0')
		    box[n++] = buffer[i];
		  /* first char that isn't space or tab, must be the name */
		  if(!space && buffer[i] != ' ' && buffer[i] != '\t')
		    {
		      box[n++] = buffer[i];
		      space++;
		    }
		  /* should the next char be a space or some form of legible ending char,
		   * we should stop the copying 
		  */
		  if(space > 0 && (buffer[i] == ' ' || buffer[i] == '\n' || 
				   buffer[i] == '\t' || buffer[i] == '\r' || buffer[i] == '\0'))	      
		    break;	      
	      }
	      fclose(in_file);
	      box[n] = '\0';
	      if(fifo->verbose > 0)
		logging(fifo, "[get_rand_box]\t\t: Found the box design to be {%s}\n", box);  
	      pthread_mutex_unlock(&mutex);
	      return SUCCESS;
	    }
	  else
	    box_count--;
	}
    }
  if(fifo->verbose > 0)
    logging(fifo, "[get_rand_box]\t\t: ERROR: Couldn't find any useful 'BOX' setting in the boxes config file {%s}\n", fifo->boxes_config);  
  fclose(in_file);
  pthread_mutex_unlock(&mutex);
  return FAILURE;
}
#endif

/* a function to change order of two integer, asuming first one has to be the smallest
 */
int change_order(int* first, int* second)
{
        int temp = 0;
        temp = *first;
        *first = (*first < *second) ? *first : *second;
        *second = (temp > *second) ? temp : *second;
        return 0;
}

/* This function gets a random numbered signature form the data file
 * It simply read the amount of signatures given in the file, then fetches
 * the signature placed as the random number given with a call to get_rand()
 * with MAX set to number of signatures in data file.
 */

int get_sig(FIFO* fifo)
{
  gzFile in_file;
  COUNTER counter;
  OLD_COUNTER old_counter;
  int failed_try = 0, first_sig = 0, second_sig = 0, seek_indent = 0;
  char *temp_buff[3*HEXCOUNT], *delimit = "-: ,;.";

  /* make sure any test later on will fail 
   * should we have been called earlier, there
   * might be some garbage in the sig.ident
   */
  fifo->sig.ident[0] = '\0';

  /* if they are using the same database file on multiple fifos, and at some point
   * one thread wants to access it while another thread is getting a sig, then we better 
   * lock the threads. 
   */
  
  pthread_mutex_lock(&mutex);

  if(fifo->verbose > 0)
    logging(fifo, " [get_sig]\t\t: Getting sig from file: {%s}\n", fifo->data_name);
  if(!(in_file = gzopen(fifo->data_name, "rb")))
    { /* at this point we should be at the top of the file */
      /* some error happened */
      if(fifo->verbose > 0)
	logging(fifo, "[get_sig]\t\t: Error: Can not open file.\n");
      /* we better unlock the threads no matter what happens */
      pthread_mutex_unlock(&mutex);
      return CONF_MI;
    }
  if(0 >= gzread(in_file, (void*) &counter, sizeof(COUNTER)))
    {
      /* some error happened */
      if(fifo->verbose > 0)
	logging(fifo, "[get_sig]\t\t: Error: The file is empty.\n");
      gzclose(in_file);
      /* we better unlock the threads no matter what happens */
      pthread_mutex_unlock(&mutex);
      return COUNT_MISS;
    }
  if(0 < strcmp("0.1.4", counter.ver))
    {
      /* lets mangle with our counter */
      gzclose(in_file);
      if(!(in_file = gzopen(fifo->data_name, "rb")))
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[get_sig]\t\t: Error: Couldn't open {%s} file for reading.\n", fifo->data_name);
	  return CONF_ER;
	} 
      if(0 >= gzread(in_file, (void*) &old_counter, sizeof(OLD_COUNTER)))
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[get_sig]\t\t: Error: Couldn't read from file {%s}\n", fifo->data_name);
	  return NO_READ;
	} 
      strcpy(counter.ver, old_counter.ver);
      counter.count = old_counter.count;
      counter.remain = 0;
    }
  counter.count = (int) ntohl(counter.count);

  if(fifo->verbose > 0)
    logging(fifo, "[get_sig]\t\t: Counter 1: {%d}\n", counter.count);
  /* now we know the max count, so lets get the random number for that */
 fetch_sig:
  if(failed_try > 3)
    {
      if(fifo->verbose > 0)
	{
	  logging(fifo, "[get_sig]\t\t: Error: Failed to fetch a signature from {%s} more than 3 times.\n", fifo->data_name);
	  logging(fifo, "[get_sig]\t\t: Error: Changing signature to default values.\n");
	}
#ifdef I__HAVE__STRNCPY
      strncpy(fifo->sig.sig, DEFAULT_SIG, LINE_SIZE*NUM_LINES);
      strncpy(fifo->sig.ident, "DEFAULT", HEXCOUNT);
#else
      strcpy(fifo->sig.sig, DEFAULT_SIG);
      strcpy(fifo->sig.ident, "DEFAULT");
#endif
      goto fail_safe;
    }

#warning Please fix this speed hog in searching for the signatures to show

  if(fifo->limit_sig[0] != '\0' && !fifo->limit_start)
    {
      if(fifo->verbose > 1)
	{
	  logging(fifo, "[get_sig]\t\t: Splitting limit_sig flag [%s]\n", fifo->limit_sig);
	}
      /* see how many, and which signatures to limit the display by */
      fifo->limit_start = strtok_r(fifo->limit_sig, delimit, temp_buff);
      if(fifo->verbose > 1)
	{
	  logging(fifo, "[get_sig]\t\t: Start signature ID [%s]\n", fifo->limit_start);
	}
      if(!fifo->limit_start)
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[get_sig]\t\t: Error: Can't get a valid sig ID from limit_sig.\n");
	  return COUNT_MISS;
	}
      else
	{
	  if(fifo->verbose > 1)
	    {
	      logging(fifo, "[get_sig]\t\t: Splitting rest of limit_sig flag [%s]\n", fifo->limit_sig);
	    }
	  /* search for end signature */
	  fifo->limit_end = strtok_r(NULL, delimit, temp_buff);
	find_sig:
	  if(fifo->verbose > 1)
	    {
	      logging(fifo, "[get_sig]\t\t: End signature ID [%s]\n", fifo->limit_end);
	    }

	  /* search for the first signature there is */
	  while(!gzeof(in_file) && 0 != strncmp(fifo->sig.ident, fifo->limit_start, HEXCOUNT))
	    {
	      /* lets read that signature we have now */
	      if(0 >= gzread(in_file, (void*) &(fifo->sig), sizeof(SIG)))
		{
		  if(fifo->verbose > 0)
		    logging(fifo, "[get_sig]\t\t: Error: The file is empty.\n");
		  gzclose(in_file);
		  /* we better unlock the threads no matter what happens */
		  pthread_mutex_unlock(&mutex);
		  return NO_READ;
		}
	      first_sig++;
	    }
	  if(0 < strcmp("0.1.4", counter.ver))
	    seek_indent = sizeof(OLD_COUNTER);
	  else
	    seek_indent = sizeof(COUNTER);
	  if(!fifo->limit_end)
	    {
	      /* if only one signature wanted, no need to be here any more
	       * Just go to the display part, since it's allready in fifo->sig */
	      goto show_me;
	    }

	  /* reset to the place where first sig was found, 
	   * to make sure, no one befor that is available.
	   */
	  if( 0 > gzseek(in_file, seek_indent, SEEK_SET))
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[get_sig]\t\t: Error: Can't seek in file.\n");
	      gzclose(in_file);
	      /* we better unlock the threads no matter what happens */
	      pthread_mutex_unlock(&mutex);
	      return SEEK_ERR;
	    }
	  
	  /* search for that second one, to limit this show and tell */
	  while(!gzeof(in_file) && 0 != strncmp(fifo->sig.ident, fifo->limit_end, HEXCOUNT))
	    {
	      /* lets read that signature we have now */
	      if(0 >= gzread(in_file, (void*) &(fifo->sig), sizeof(SIG)))
		{
		  if(fifo->verbose > 0)
		    logging(fifo, "[get_sig]\t\t: Error: The file is empty.\n");
		  gzclose(in_file);
		  /* we better unlock the threads no matter what happens */
		  pthread_mutex_unlock(&mutex);
		  return NO_READ;
		}
	      second_sig++;
	    }

	  if(fifo->verbose > 1)
	    {
	      logging(fifo, "[get_sig]\t\t: Signatur ID: %s is located as number: %d\n", fifo->limit_start, first_sig);
	      if(second_sig)
		logging(fifo, "[get_sig]\t\t: Signatur ID: %s is located as number: %d\n", fifo->limit_end, second_sig);
	    }
	  /* make sure first_sig allways comes befor second_sig */
	  change_order(&first_sig, &second_sig);
	  if(fifo->verbose > 1)
	    {	  
	      logging(fifo, "[get_sig]\t\t: Signatures is to be fetched between signature {%d} and {%d}\n", 
		      first_sig, second_sig);
	    }
	  /* find the random one between them */
	  fifo->seek_indent = get_rand(second_sig - first_sig);
	  if(fifo->verbose > 1)
	    {	  
	      logging(fifo, "[get_sig]\t\t: Searching for signature number: %d\n", fifo->seek_indent);
	    }
	  goto seek_me;
	}
    }
  else
    {
      /*
       * If it is the second time we're here, fifo->limit_sig is empty but
       * fifo->limit_start isn't, so it's time to look for the limiting sigs
       */
      if(fifo->limit_start != NULL)
	{
	  if(fifo->verbose > 1)
	    {	  
	      logging(fifo, "[get_sig]\t\t: Start signature ID [%s]\n", fifo->limit_start);
	    }
	  goto find_sig;
	}
      /* no limited signature, so fetch a random one */
      fifo->seek_indent = get_rand(counter.count);
      if(fifo->verbose > 0)
	logging(fifo, "[get_sig]\t\t: Counter 2: {%d}\n", fifo->seek_indent);
    seek_me:
      if(0 < strcmp("0.1.4", counter.ver))
	seek_indent = sizeof(OLD_COUNTER);
      else
	seek_indent = sizeof(COUNTER);

      /* reset to the place where first sig was found, 
       * to make sure, no one befor that is available.
       */      
      if( 0 > gzseek(in_file, seek_indent, SEEK_SET))
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[get_sig]\t\t: Error: Can't seek in file.\n");
	  gzclose(in_file);
	  /* we better unlock the threads no matter what happens */
	  pthread_mutex_unlock(&mutex);
	  return SEEK_ERR;
	}
      if(!fifo->limit_start)
	seek_indent = fifo->seek_indent;
      else
	seek_indent = (first_sig - 1 + fifo->seek_indent);
      /* then lets search for that signature */
      if( 0 > gzseek(in_file, seek_indent*sizeof(SIG), SEEK_CUR))
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[get_sig]\t\t: Error: Can't seek in file.\n");
	  gzclose(in_file);
	  /* we better unlock the threads no matter what happens */
	  pthread_mutex_unlock(&mutex);
	  return SEEK_ERR;
	}
    }
  /* lets read that signature we have now */
  if(0 >= gzread(in_file, (void*) &(fifo->sig), sizeof(SIG)))
    {
      if(fifo->verbose > 0)
	logging(fifo, "[get_sig]\t\t: Error: The file is empty.\n");
      gzclose(in_file);
      /* we better unlock the threads no matter what happens */
      pthread_mutex_unlock(&mutex);
      return NO_READ;
    }
 show_me:
  if(strlen(fifo->sig.sig) < 1)
    {
      if(fifo->verbose > 0)
	logging(fifo, "[get_sig]\t\t: Error: The signature fetched from {%s} was empty.\n", fifo->data_name);
      failed_try++;
      sleep(1); /* in order to make sure the random will return a different number */
      goto fetch_sig;
    }
 fail_safe:
  if(fifo->verbose > 0)
    logging(fifo, "[get_sig]\t\t:   DONE\n");
  gzclose(in_file);
  /* we better unlock the threads no matter what happens */
  pthread_mutex_unlock(&mutex);
  return SUCCESS;
}


/* spred_box_flags() a function, which given roughly a number of char* arguments
 * will return with a char** instead.
 *
 * It will check for the currently useful flags thats provided by boxes version 1.0
 * And only parse those in the returned char**
 *
 */

#ifdef BOXING_SUPPORT
/* 
 * #warning please fix the speed hog in this spred_box_flags() 
 * At this moment, after 5 month of testing, there is nothing which will
 * enhance the speed of spred_box_flags(), so for now we will leave it as is.
 */

char** spred_box_flags(char** flags, FIFO* fifo, const char* box_design)
{
  int i, j = 0, n, max_members = 0;
  /* run through the flags parsed to the fifo and select the different. */
  for(i = 0; i < strlen(fifo->boxes_flags); i++)
    {
      switch (fifo->boxes_flags[i++])
	{
	case '-':
	  /* it's a setting, we better see if it's an accepted one,
	   * and copy what it tell us into the flags array 
	   */

	  /* 
	   * looking to see, if j > max_members, and then add extra space.
	   */
	  if (j >= max_members -1) {
	    max_members += 5; /* we only allocate 5 flags at a time, since theres rarely need for more */
	    flags = (char**)realloc (flags, max_members * sizeof (char*));
	    if (!flags)
	      {
		if(fifo->verbose > 0)
		  logging(fifo, "[spred_box_flags]\t: Error: There wasn't enough heap space to reallocate the argument list.\n");
		return NULL;
	      }
	  }
	  
	  switch(fifo->boxes_flags[i++])
	    {
	    case ' ':
	    case '\t':
	    case '\0':
	      break; /* is just a stupid seperator, or we're at the end */
	    case 'a': /* to align the text inside the box */
	    case 'd': /* design setting, we can't leave that out, eventho we might end up using our own */
	    case 'f': /* if the configuration file has a different location, it would be bad if thats missing */
	    case 'i': /* If our user should deside, to use an indentation mode */
	    case 'k': /* If we like to kill leading/trailing blank lines */
	    case 'p': /* Should the user decide to use a padding setting */
	    case 's': /* For specific settings of the box size. */
	    case 't': /* What, you dont like 8 char tabbing distance ? */
	      /* 
	       * We better make some room for the flags, thats about to get copied.
	       */
	      flags[j] = malloc(LINE_SIZE +1);
	      if(!flags[j])
		{
		  if(fifo->verbose > 0)
		    logging(fifo, "[spred_box_flags]\t: Error: There wasn't enough heap space to allocate for the flag: {-%c}\n", fifo->boxes_flags[i -1]);
		  return NULL;
		}
	      /* first add the flag intended for the setting */
	      flags[j][0] = '-';
	      flags[j][1] = fifo->boxes_flags[i -1];
	      /* then copy the setting for that flag */
	      for(n = 2; n < LINE_SIZE && i < strlen(fifo->boxes_flags) && fifo->boxes_flags[i] != '-'; n++)
		{
		  if(fifo->boxes_flags[i] == ' ')
		    {
		      i++; /* do nothing it must be a space in the setting, like "-d peek" */
		      n--;
		    }
		  else
		    flags[j][n] = fifo->boxes_flags[i++];
		}
	      i--; /* to correct the extra i++ for the '-' test in the for loop */
	      flags[j++][n] = '\0';
	      if(fifo->verbose > 1)
		logging(fifo, "[spred_box_flags]\t: The found flag for boxes was {%s}\n", flags[j -1]);
	      break;
	    default:
	      if(fifo->verbose > 0)
		logging(fifo, "[spred_box_flags]\t: Error: The '-%c' flag to boxes isn't supported, and will be omited.\n", fifo->boxes_flags[i -1]);
	      break;
	    }
	default:
	  if(fifo->verbose > 1)
	    logging(fifo, "[spred_box_flags]\t: Error: Skipping the {%c} in boxes_flags, since it isn't a '-'.\n", fifo->boxes_flags[i -1]);
	  break;
	}
    }
  /* 
   * for the final deal, to overwrite any previus -d flags we place our 
   * own -d as the last one, guessing boxes will chose it over any other
   */
  if(box_design[0] != '\0')
    {
      flags[j] = malloc(LINE_SIZE +1);
      if(!flags[j])
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[spred_box_flags]\t: Error: There wasn't enough heap space to allocate for the flag: -d\n");
	  return NULL;
	}
#ifdef I__HAVE__SNPRINTF
      snprintf(flags[j++], LINE_SIZE, "-d%s", box_design);
#else
      sprintf(flags[j++], "-d%s", box_design);
#endif
    }
  flags[j] = NULL;
  return flags;
}
#endif

/*
 * SSHHH, this is magic, no one even breath.
 * 
 * Nah seriusly, it is just a simple function, gets a conversion char ch.
 * from that one decides, what should be added to buffer.
 * 
 * The follwoing conversion chars are supported:
 *
 * Char  :  What they display
 *  
 *  C    :  Callsign
 *  H    :  Homepage
 *  L    :  Load level
 *  M    :  Machine architecture
 *  N    :  Node name (machine name)
 *  P    :  Package
 *  R    :  Kernel version
 *  S    :  System name (Linux/BSD/..)
 *  T    :  Current time
 *  U    :  Uptime
 *  V    :  Version
 *  %    :  %
 */

int buffer_magic(char ch, char buffer[], 
		 int sig_write, 
		 int *l_count, 
		 int *pretty_sig_count, 
		 int *line_count, 
		 int *buffer_count)
{

#ifdef I__HAVE__PROC
  FILE  *fp;
  int  days = 0, hours = 0, min = 0, read_number = 0;
  float upt = 0.0, temp = 0.0, load1 = 0.0, load2 = 0.0, load3 = 0.0;
#endif
  int counter = 0;
  char uname_buffer[LINE_SIZE +1], tmp[10];
#ifdef I__HAVE__UNAME
  struct utsname unamebuf;
#endif
  struct tm *tm_set;
  time_t tim, t;
  /* to make sure, the garbage in memory
     isnt causing any problems.*/
#ifdef I__HAVE__UNAME
  uname_buffer[0] = '\0';
#endif
  tmp[0] = '\0';
#ifdef I__HAVE__UNAME
  uname(&unamebuf);  
#endif
  switch(ch)
    {
    case('t'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
      break;
    case ('C'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__SNPRINTF
      snprintf(uname_buffer, LINE_SIZE, "%s", CALLSIGN);
#else
      sprintf(uname_buffer, "%s", CALLSIGN);
#endif
      uname_buffer[strlen(CALLSIGN)] = '\0';
      *line_count = *line_count + strlen(CALLSIGN);
      break;
    case ('H'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__STRNCPY
      strncpy(uname_buffer, HOMEPAGE, LINE_SIZE);
#else
      strcpy(uname_buffer, HOMEPAGE);
#endif
      uname_buffer[strlen(HOMEPAGE)] = '\0';
      *line_count = *line_count + strlen(HOMEPAGE);
      break; 
    case ('L'):
      /* I know the HP_UX havn't got the same /proc structure as Linux
	 There might be other architechtures suffering of the same*/
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__PROC
      fp = fopen("/proc/loadavg", "r");
      if(fp != NULL)
	{
	  read_number = fscanf(fp, "%f %f %f", &load1, &load2, &load3);
	  fclose(fp);
	}
      if(read_number == 3)
	{
#ifdef I__HAVE__SNPRINTF
	  snprintf(uname_buffer, LINE_SIZE, "%.2f %.2f %.2f", load1, load2, load3);
#else
	  sprintf(uname_buffer, "%.2f %.2f %.2f", load1, load2, load3);
#endif
	}
      else
	{
#ifdef I__HAVE__SNPRINTF
	  snprintf(uname_buffer, LINE_SIZE, "Unavailable /proc/loadavg");
#else
	  sprintf(uname_buffer,  "Unavailable /proc/loadavg");
#endif
	}
	  *line_count = *line_count + strlen(uname_buffer);
#endif   /* I__HAVE__PROC */
      break;
    case ('M'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__SNPRINTF
  #ifdef I__HAVE__UNAME
      snprintf(uname_buffer, LINE_SIZE, "%s", unamebuf.machine);
  #else
      snprintf(uname_buffer, LINE_SIZE, "%s", "Undefined machine");
  #endif
#else
  #ifdef I__HAVE__UNAME
      sprintf(uname_buffer, "%s", unamebuf.machine);
  #else
      sprintf(uname_buffer, "%s", "Undefined machine");
  #endif
#endif
  #ifdef I__HAVE__UNAME
      uname_buffer[strlen(unamebuf.machine)] = '\0';
      *line_count = *line_count + strlen(unamebuf.machine);
  #else
      uname_buffer[17] = '\0';
      *line_count = *line_count + 17;
  #endif
      break;
    case('N'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__SNPRINTF
  #ifdef I__HAVE__UNAME
      snprintf(uname_buffer, LINE_SIZE, "%s", unamebuf.nodename);
  #else
      snprintf(uname_buffer, LINE_SIZE, "%s", "Undefined machine");
  #endif
#else
  #ifdef I__HAVE__UNAME
      sprintf(uname_buffer, "%s", unamebuf.nodename);
  #else
      sprintf(uname_buffer, "%s", "Undefined machine");
  #endif
#endif
  #ifdef I__HAVE__UNAME
      uname_buffer[strlen(unamebuf.nodename)] = '\0';
      *line_count = *line_count + strlen(unamebuf.nodename);
  #else
      uname_buffer[17] = '\0';
      *line_count = *line_count + 17;
  #endif
      break;
    case ('P'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__STRNCPY
      strncpy(uname_buffer, PACKAGE, LINE_SIZE);
#else
      strcpy(uname_buffer, PACKAGE);
#endif
      uname_buffer[strlen(PACKAGE)] = '\0';
      *line_count = *line_count + strlen(PACKAGE);
      break; 
    case ('R'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__SNPRINTF
  #ifdef I__HAVE__UNAME
      snprintf(uname_buffer, LINE_SIZE, "%s", unamebuf.release);
  #else
      snprintf(uname_buffer, LINE_SIZE, "%s", "Undefined release");
  #endif
#else
  #ifdef I__HAVE__UNAME
      sprintf(uname_buffer, "%s", unamebuf.nodename);
  #else
      sprintf(uname_buffer, "%s", "Undefined release");
  #endif
#endif
  #ifdef I__HAVE__UNAME
      uname_buffer[strlen(unamebuf.release)] = '\0';
      *line_count = *line_count + strlen(unamebuf.release);
  #else
      uname_buffer[17] = '\0';
      *line_count = *line_count + 17;
  #endif
      break; 
    case ('S'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__SNPRINTF
  #ifdef I__HAVE__UNAME
      snprintf(uname_buffer, LINE_SIZE, "%s", unamebuf.sysname);
  #else
      snprintf(uname_buffer, LINE_SIZE, "%s", "Undefined sysname");
  #endif
#else
  #ifdef I__HAVE__UNAME
      snprintf(uname_buffer, "%s", unamebuf.sysname);
  #else
      snprintf(uname_buffer, "%s", "Undefined sysname");
  #endif
#endif
  #ifdef I__HAVE__UNAME
      uname_buffer[strlen(unamebuf.sysname)] = '\0';
      *line_count = *line_count + strlen(unamebuf.sysname);
  #else
      uname_buffer[17] = '\0';
      *line_count = *line_count + 17;
  #endif
      break;
    case ('T'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
      tim = time(&t);
      tm_set = localtime(&tim);
#ifdef I__HAVE__SNPRINTF
      snprintf(uname_buffer, LINE_SIZE, "%.2d:%.2d", tm_set->tm_hour, tm_set->tm_min);
#else
      sprintf(uname_buffer, "%.2d:%.2d", tm_set->tm_hour, tm_set->tm_min);
#endif
      *line_count = *line_count + strlen(uname_buffer);
      break;
    case ('U'):
      /* I know the HP_UX havn't got the same /proc structure as Linux
       * There might be other architechtures suffering of the same
       */
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;

#ifdef I__HAVE__PROC
      fp = fopen("/proc/uptime", "r");
      if(fp != NULL)
	{
	  read_number = fscanf(fp, "%f %f", &upt, &temp);
	  fclose(fp);
	}
      if(read_number == 2)
	{
	  days = ((int)upt)/86400;
	  upt = ((int)upt)%86400;
	  hours = ((int)upt)/3600;
	  upt = ((int)upt)%3600;
	  min = ((int)upt)/60;
	  tim = time(&t);
	  tm_set = localtime(&tim);
	  if(days >0)
	    {
#ifdef I__HAVE__SNPRINTF
	      snprintf(uname_buffer, LINE_SIZE, "%.2d:%.2d up %.2d days, ", tm_set->tm_hour, tm_set->tm_min, days);
#else
	      sprintf(uname_buffer, "%.2d:%.2d up %.2d days, ", tm_set->tm_hour, tm_set->tm_min, days);
#endif
	    }
	  else
	    {
#ifdef I__HAVE__SNPRINTF
	      snprintf(uname_buffer, LINE_SIZE, "%.2d:%.2d up ", tm_set->tm_hour, tm_set->tm_min);
#else
	      sprintf(uname_buffer, "%.2d:%.2d up ", tm_set->tm_hour, tm_set->tm_min);
#endif
	    }
	  sprintf(tmp, "%.2d:%.2d", hours, min);
	  strcat(uname_buffer, tmp);
	}
      else
	{
#ifdef I__HAVE__SNPRINTF
	  snprintf(uname_buffer, LINE_SIZE, "Unavailable /proc/uptime");
#else
	  sprintf(uname_buffer, "Unavailable /proc/uptime");
#endif
	}
      *line_count = *line_count + strlen(uname_buffer);
#endif   /* I__HAVE__PROC */
      break;
    case ('V'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
#ifdef I__HAVE__STRNCPY
      strncpy(uname_buffer, VERSION, LINE_SIZE);
#else
      strcpy(uname_buffer, VERSION);
#endif
      uname_buffer[strlen(VERSION)] = '\0';
      *line_count = *line_count + strlen(VERSION);
      break; 
    case ('%'):
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
      strcpy(uname_buffer, "%");
      uname_buffer[1] = '\0';
      *line_count = *line_count +1;
      break;
    default:
      if(sig_write == 0)
	*l_count = *l_count +1;
      else
	*pretty_sig_count = *pretty_sig_count +1;
      break;
    }
  for(counter = 0; 
      *buffer_count < *line_count && 
	*buffer_count < LINE_SIZE && 
	uname_buffer[counter] != '\0'; 
      *buffer_count = *buffer_count +1, counter++)
      buffer[*buffer_count] = uname_buffer[counter];

  return SUCCESS;
}


/* 
 * This is the pretty printer.. 
 * And it's pretty big... And complex...
 * 
 * It will print the signature to the fifo, taking care of all the conversions and placements of 
 * every desired display, the pretty{} setting will be made, that if it isn't consisting of more
 * lines than the actual signature, the remaining signature lines, will be placed with 
 * strlen(last_pretty_setting_line) -3 spaces distortion, meaning you can have a smooth resignation 
 * from the pretty filling ie:
 * 
 * The pretty setting ends |  First line of signature
 *                        second line of signature
 *                    third line of signature...
 *
 *
 * It also handles, if you have a %r placed first in the line of your signature, it will 
 * right align that line, so if you have a signature like this:
 *
 * The signature can be customized in 
 * %rmany ways
 * like with many other things.
 *
 * This will be displayed in the fifo as:
 *
 * The signatures can be cusomized in
 *                          many ways
 * like with many other things.
 *
 * And do I have to mention it handles the printing of "name", "email", "top_line", 
 * "bottom_line", "spam_catcher" etc
 * 
 * Plus, if you chose to have boxes support (and boxes is pressent on your system) 
 * then it will ceate a pipe to boxes stdin and a pipe to boxes stdout, routing
 * the signature through boxes, befor sending it to the fifo. This is acomplished 
 * in a way, kinda like this:
 * 
 *        .-----.                             .-----.
 *  Sigit-      |                             |      -Sigit
 *        '--.  |                             |  .--'
 *           |  '------.               .------'  |  
 *           |   PIPE  |               |  PIPE   |
 *           '-------. |               | .-------' 
 *                   | '---.       .---' |
 *                   |      -boxes-      |
 *                   '-----'       '-----'
 *
 *  
 *  
 *
 *
 *
 *
 */                     

int pretty_print(FIFO* fifo)
{
  FILE *out_file;
  int line_count = 0, last_line = 0, sig_write = 0, buffer_count = 0, longest_line = 0, info = 0, statement = 0;  
  int pretty_count = 0, sig_count = 0, l_count = 0, one_two = 0, started = 0, len_out_buffer = 0, len_buffer_count = 0;
  int spaces, i, status;
  pid_t pid;
#ifdef BOXING_SUPPORT
  int nread, nwrite, sig2box[2], box2sig[2], flags_here = 0;
  char box_design[LINE_SIZE +1], **boxes_flags = NULL;
#endif
  char ch, line_buff[LINE_SIZE +3], info_buff1[LINE_SIZE +3], info_buff2[LINE_SIZE +3], buffer[LINE_SIZE +3], *out_buffer;
  char state_line[LINE_SIZE], opinion_line[3*LINE_SIZE +1];

  len_out_buffer = (strlen(fifo->pretty.pretty_setting) + strlen(fifo->pretty.top_line) +
		    strlen(fifo->pretty.bottom_line) + strlen(fifo->pretty.email_address) +
		    strlen(fifo->pretty.name_setting) + strlen(fifo->sig.sig) + strlen(fifo->sig.ident) +
		    strlen(fifo->show_sig_ident) + 6*LINE_SIZE);
  if(fifo->verbose > 3)
    logging(fifo, "[pretty_print]\t\t: Making space in out_buffer for {%d} chars.\n", len_out_buffer);
  out_buffer = (char*) USE__MALLOC(len_out_buffer);
  strcpy(out_buffer, ""); /* initialize it, so we can use strncat() without fear */

  if(fifo->verbose > 2)
    logging(fifo, "[pretty_print]\t\t: Preparing to copy signature: {%s} to out_buffer.\n", fifo->sig.ident);
  if(strlen(fifo->pretty.top_line) > 2)
    {
      if(fifo->verbose > 2)
	logging(fifo, "[pretty_print]\t\t: Preparing to copy t_line {%s} if it will fit.\n", fifo->pretty.top_line);
#ifdef I__HAVE__STRNCPY
      strncpy(line_buff, fifo->pretty.top_line, LINE_SIZE);
#else
      strcpy(line_buff, fifo->pretty.top_line);
#endif
      /* #### very dangerous, counting to -1, but if we should allready #### */
      /* #### print one line, then the counter will be 0 for the first  #### */
      /* ####               run through the pretty buffer.               #### */
      pretty_count--;
    }
  else
    { /* we need to avoid the sig_count to be -1, so we have the started flag */
      if(fifo->verbose > 2)
	logging(fifo, "[pretty_print]\t\t: No top_line to copy, altering started flag.\n");
      started = 1;
      sig_write = 1;
      one_two++;
      statement = 1;
    }
  for( ; sig_write != 4 ; )
    { 
      if(fifo->verbose > 2)
	logging(fifo, "[pretty_print]\t\t: Currently using sig_write setting {%d} in the process.\n", sig_write);
      
      /*
       * sig_write == 0  writing either top line or bottom line
       * sig_write == 1  writing presetting
       * sig_write == 2  writing signature
       * sig_write == 3  done writing signature
       * sig_write == 4  We've finished bottom line and its time to get out.
       */
      
      /* if its either the top line or the pre setting
       * we have the conversion chars to check 
       */
      if(sig_write <= 1)
	{
	  if(fifo->verbose > 2)
	    logging(fifo, "[pretty_print]\t\t: Checking conversion chars.\n");
	  if(sig_write == 0)
	    ch = line_buff[l_count];
	  else
	    ch = fifo->pretty.pretty_setting[pretty_count];
 	  if(ch == '%')
 	    {
	      if(sig_write == 0)
		l_count++;
	      else
		pretty_count++;
	      if(sig_write == 0)
		ch = line_buff[l_count];
	      else
		ch = fifo->pretty.pretty_setting[pretty_count];
	      if(fifo->verbose > 2)
		logging(fifo, "[pretty_print]\t\t: Calling buffer_magick() with {ch: %c, sig_write: %d, l_count: %d, pretty_count: %d, line_count: %d, buffer_count: %d}\n",ch, sig_write, l_count, pretty_count, line_count, buffer_count);
	      
	      buffer_magic(ch, buffer, 
			   sig_write, 
			   &l_count, 
			   &pretty_count, 
			   &line_count, 
			   &buffer_count);
	      if(fifo->verbose > 2)
		logging(fifo, "[pretty_print]\t\t: Returning from buffer_magick() with {ch: %c, sig_write: %d, l_count: %d, pretty_count: %d, line_count: %d, buffer_count: %d}\n",ch, sig_write, l_count, pretty_count, line_count, buffer_count);
	      
	    }
	  if(sig_write == 0)
	    {
	      if(line_buff[l_count] != '\0' && 
		 line_count < LINE_SIZE && 
		 buffer_count < LINE_SIZE)
		{
		  if(line_buff[l_count] == '\n')
		    {
		      l_count++;
		      sig_write = 4;
		    }	      
		  else
		    {
		      buffer[buffer_count++] = line_buff[l_count++];
		      line_count++;
		      if(line_count >= LINE_SIZE)
			sig_write = 4;
		    }
		}
	      else 
		sig_write = 4;
	    }
	  else
	    {
	      if(strlen(fifo->pretty.pretty_setting) > 1)
		{ /* as long as theres something in the pretty, we just copy it to our buffer */
		  if(fifo->pretty.pretty_setting[pretty_count] != '\n' && 
		     fifo->pretty.pretty_setting[pretty_count] != '\0' && 
		     line_count < LINE_SIZE) 
		    {
		      buffer[buffer_count++] = fifo->pretty.pretty_setting[pretty_count++];
		      line_count++;
		    } /* if its a \n we've reached its time to write the signature */
		  if(fifo->pretty.pretty_setting[pretty_count] == '\n')
		    {
		      buffer[buffer_count++] = ' ';
		      buffer[buffer_count++] = ' ';
		      pretty_count++;
		      last_line = line_count;
		      sig_write = 2;
		    } 
		  /* if we've reached this part, and yet there was more signature to write
		   * then we just fill blanks for the remaining preset, so there won't be
		   * some odd mengling of the signature.. 
		   */
		  if(fifo->verbose > 2)
		    logging(fifo, "[pretty_print]\t\t: Filling the preset space with ' ' to accomplish smooth transformation.\n");
		  
		  if(fifo->pretty.pretty_setting[pretty_count] == '\0')
		    {
		      line_count = 0;
		      for(i = 0; i <= last_line; i++)
			{ 
			  buffer[buffer_count++] = ' ';
			  line_count++;
			}
		      sig_write = 2;
		      /* making a slow ajustment to the normal line_size */
                      if(line_count <= 2)
                        last_line = 0;
                      else
			last_line = (line_count -2);
		    }
		} 
	      else
		sig_write = 2;
	    }  
	} /* now its time to write the signature part */
      if(sig_write == 2)
	{

	  if(fifo->sig.sig[sig_count] != '\0' && 
	     line_count < LINE_SIZE)
	    { /* the '\n' will be added later */
	      if(fifo->sig.sig[sig_count] == '\n')
		{
		  sig_count++;
		  sig_write = 3;
		}	      
	      else
		if(fifo->sig.sig[sig_count] == '%' && fifo->sig.sig[sig_count +1] == 'r')
		  {
		    if(!sig_count)
		      longest_line = LINE_SIZE -2;
		    else
		      if(fifo->sig.sig[sig_count -1] != '\n')
			goto out; /* Real programmers arent afraid of using gotos */
		    sig_count += 2;
		    
		    /* removing any leading spaces in the signature */
		    while(fifo->sig.sig[sig_count] == ' ' || fifo->sig.sig[sig_count] == '\t')
		      sig_count++;
		    
		    for(spaces = 0; 
			(fifo->sig.sig[sig_count + spaces] != '\n' &&
			 fifo->sig.sig[sig_count + spaces] != '\0') &&
			  spaces < LINE_SIZE; spaces++)
		      /* just count what we need.. */ ;
		    
		    /* setting the spaces needed, to right place it */
		    if(longest_line - spaces > 0)
		      for(i = 0; i < longest_line - spaces - last_line && 
			    line_count + spaces < LINE_SIZE; i++)
			{
			  buffer[buffer_count++] = ' ';
			  line_count++;
			}
		    
		    
		  }
		else
		  { /* we just copy it into our buffer. The easiest way */
		  out:
		    buffer[buffer_count++] = fifo->sig.sig[sig_count++];
		    line_count++;
		  }
	    }
	  else 
	    sig_write = 3;
	}/* this is reached, if we have a full buffer to write.. */
      if(sig_write >= 3 || line_count >= LINE_SIZE)
	{
	  /* #### In order to correct the -1 in the pretty_count #### */
	  if(pretty_count == -1)
	    pretty_count++;
	  else
	    {
	      if(fifo->pretty.pretty_setting[pretty_count] != '\0')
		pretty_count++;
	    } /* was it the bottom line we were writing ? */
	  if(sig_write == 4)
	    {
	      started = 1;
	      one_two++;
	      line_buff[0] = '\0';
	    }
	  else
	    {
	      if(fifo->verbose > 1)
		logging(fifo, "[pretty_print]\t\t: Finding the last white space, where we can split the line.\n");

	      /* it must have been part of the signature and preset, 
	       * so if were in the middle of a word, we better go back to a white space 
	       */
	      if(fifo->sig.sig[sig_count] != '\0' && 
		 fifo->sig.sig[sig_count] != '\n' && 
		 fifo->sig.sig[sig_count] != ' ' && 
		 fifo->sig.sig[sig_count] != '\t' && 
		 started &&
		 fifo->sig.sig[sig_count -1] != '\n')
		for(; fifo->sig.sig[sig_count] != ' ' && 
		      fifo->sig.sig[sig_count] !='\n' && 
		      fifo->sig.sig[sig_count] !='\t' && 
		      fifo->sig.sig[sig_count] != '\0'; sig_count--, buffer_count--)
		  /* lets just count.. */ line_count--;
	      else /* it was the first line, so its OK to start searching for white spaces */
		started = 1;
	    }/* if theres nothing in the buffer, then why bother filling more */	
	  if(!buffer_count &&
	      fifo->sig.sig[sig_count] == '\0' && 
	      fifo->pretty.pretty_setting[pretty_count] == '\0')
	    sig_write = 4;
	  else
	    {/* write what we got so far */
	      if(started && fifo->sig.sig[sig_count] == '\n')
		{
		  sig_count++;
		  /* this might be where the buffer[buffer_count] = '\n'; should be */
		}
	      buffer[buffer_count++] = '\n';
	      buffer[buffer_count] = '\0';
	      buffer_count = 0;
	      if(fifo->verbose > 2)
		logging(fifo, "[pretty_print]\t\t: Copying signature part {%s} to the out_buffer.\n", buffer);

	      if(statement > 0 && statement < 2 && fifo->show_sig_ident[0] != '\0')
		{
		  /* sig_write is 0, we must assume this is in the start of writing,
		   * where top_line has been written, so we're ready to write ident 
		   */
		  if(fifo->verbose > 2)
		      logging(fifo, "[pretty_print]\t\t: Copying ident {%s} to out_buffer.\n", fifo->sig.ident);
#ifdef I__HAVE__SNPRINTF
		  snprintf(state_line, LINE_SIZE, "%s %.3dx%s:\n", 
			   fifo->show_sig_ident, ntohl(fifo->sig.count), fifo->sig.ident);
#else
		  sprintf(state_line, "%s %.3dx%s:\n", 
			   fifo->show_sig_ident, ntohl(fifo->sig.count), fifo->sig.ident);
#endif

		  if(!strncat(out_buffer, state_line, strlen(state_line)))
		    {
		      
		      if(fifo->verbose > 0)
			logging(fifo, "[pretty_print]\t\t: Error: Can't copy ident to out_buffer.\n");
		      return WRITE_ERR;
		    }
		  statement = 2;
		}
	      else
		if(statement < 2)
		  statement = 1;
	      if(fifo->verbose > 3)
		logging(fifo, "[pretty_print]\t\t: out_buffer is currently holding [%s].\n", out_buffer);
	      if(!strncat(out_buffer, buffer, strlen(buffer)))
		{
		  if(fifo->verbose > 0)
		    logging(fifo, "[pretty_print]\t\t: Error: Can't copy signature to out_buffer.\n");		 
		  return WRITE_ERR;
		}/* prepare for the writing of presetting next */

	      sig_write = 1;
	      if(sig_count > 1 && longest_line < line_count)
		longest_line = line_count;
	      line_count = 0;  
	    }
	  
	  /* hmmm are we done writing the signature and should we select bottom line now ? */
	  if(fifo->pretty.pretty_setting[pretty_count] == '\0' && fifo->sig.sig[sig_count] == '\0')
	    {
	      /* if atleast we have some user info to write, we better express that */
	      if(!info && (strlen(fifo->pretty.name_setting) > 1 || strlen(fifo->pretty.email_address) > 1))
		{
		  info_buff1[0] = '\0';
		  info_buff2[0] = '\0';
		  line_buff[0] = '\0';
		  
		  if(fifo->verbose > 0)
		    logging(fifo, "[pretty_print]\t\t: Starting to copy name and address info to the out_buffer.\n");
		  
#ifdef I__HAVE__SNPRINTF
		  if(strlen(fifo->pretty.name_setting) > 1)	  
		    snprintf(line_buff, LINE_SIZE, "%s", fifo->pretty.name_setting);
		  if(fifo->verbose > 2)
		    logging(fifo, "[pretty_print]\t\t: %s and %s\n", fifo->pretty.name_setting, line_buff);
		  if(strlen(fifo->pretty.email_address) > 1)
		    snprintf(info_buff1, LINE_SIZE, " <%s>",fifo->pretty.email_address);
		  snprintf(info_buff2, LINE_SIZE, "%s%s\n", line_buff, info_buff1);
#else
		  if(strlen(fifo->pretty.name_setting) > 1)		  
		    sprintf(line_buff, "%s", fifo->pretty.name_setting);
		  if(fifo->verbose > 2)
		    logging(fifo, "[pretty_print]\t\t: %s and %s\n", fifo->pretty.name_setting, line_buff);
		  if(strlen(fifo->pretty.email_address) > 1)		  
		    sprintf(info_buff1, " <%s>", fifo->pretty.email_address);
		  sprintf(info_buff2, "%s%s\n", line_buff, info_buff1);
#endif
		  if(fifo->verbose > 1)   
		    logging(fifo, "[pretty_print]\t\t: {%s} and {%s} produce: {%s}\n", line_buff, info_buff1, info_buff2);
		  i = 0;
		  buffer[0] = '\0';
		  longest_line = LINE_SIZE -2;
		  /* see how many spaces are needed..  */
		  for(spaces = 0; 
		      (info_buff2[spaces] != '\n' &&
		       info_buff2[spaces] != '\0') &&
			spaces < LINE_SIZE; spaces++)
		    /* just count what we need.. */ ;
		  
		  /* setting the spaces needed, to right place it */
		  if(longest_line - spaces > 0)
		    for(; i < longest_line - spaces && 
			  i + spaces < LINE_SIZE; i++)
		      {
			buffer[i] = ' ';
		      }
		  buffer[i] = '\0';
#ifdef I__HAVE__SNPRINTF
		  snprintf(opinion_line, 3*LINE_SIZE, "\nThe opinion expressed above is mine, not at all related to my employer.\n%s%s", buffer, info_buff2);
#else
		  sprintf(opinion_line, "\nThe opinion expressed above is mine, not at all related to my employer.\n%s%s", buffer, info_buff2);
#endif		    
		  if(!strncat(out_buffer, opinion_line, strlen(opinion_line)))
		    {
		      if(fifo->verbose > 0)
			logging(fifo, "[pretty_print]\t\t: Error: Can't copy name and address info to out_buffer.\n");
		      return WRITE_ERR;
		    }
		  if(fifo->spam_catcher)
		    if(!strncat(out_buffer, "   <rchong@fcc.gov><rhundt@fcc.gov><uce@ftc.gov><sness@fcc.gov>    Spambot\n<president@whitehouse.gov><haesslich@loyalty.org><jquello@fcc.gov> harvest\n", 150))
		      {
			if(fifo->verbose > 0)
			  logging(fifo, "[pretty_print]\t\t: Error: Can't copy spambot seed to out_buffer.\n");
			return WRITE_ERR;
		      }
		  info = 1;
		}
	      
	      if(line_buff[0] == '\0' && one_two > 1)
		sig_write = 4; /* if we allready are done with the bottom line we exit */
	      if(one_two == 1)
		{ /* if there has been just one (indicating top line ) go on */
		  if(strlen(fifo->pretty.bottom_line) > 1) 
		    {/* if theres something in the bottom line we can write it */
		      sig_write = 0;
#ifdef I__HAVE__STRNCPY
		      strncpy(line_buff, fifo->pretty.bottom_line, LINE_SIZE);
#else
		      strcpy(line_buff, fifo->pretty.bottom_line);
#endif
		      l_count = 0;
		    }
		  else /* else we might aswell exit.. */
		    sig_write = 4;
		}
	    }
	}
    }
  /* 
   * Now we have all our output in out_buffer, all we need is to see if it's a boxing desire, 
   * and then create the pipes, if not, just print what is in out_buffer to our fifo 
   */
#ifdef BOXING_SUPPORT
  if(fifo->boxes_support)
    {
      /* well we need to feed boxes, thus we need to have a valid termination of out_buffer */
      len_buffer_count = strlen(out_buffer);
      if(len_buffer_count > len_out_buffer)
	{
	  /* oh poop, we're over the limit, so we have to rewind */
	  out_buffer[len_out_buffer] = '\0';
	}
      if(len_buffer_count <= 0)
	{
	  /* the length is 0, what the hell something must have gone wrong.. */
	  if(fifo->verbose > 0)
	    logging(fifo, "[pretty_print]\t\t: Error: the length of expected signature is 0, usign DEFAULT_SIG instead.");
#ifdef I__HAVE__STRNCPY
	  strncpy(out_buffer, DEFAULT_SIG, LINE_SIZE*NUM_LINES);
#else
	  strcpy(out_buffer, DEFAULT_SIG);
#endif
	}      
      box_design[0] = '\0';
      /* call get_rand_box() if we have fifo->boxes_config set to anything */
      if(fifo->boxes_config[0] != '\0')
	if(0 != get_rand_box(fifo, box_design))
	  {	   
	    if(fifo->verbose > 0)
	      logging(fifo, "[pretty_print]\t\t: Error: get_rand_box() failed to find a design, using C comment as default.\n");
	    box_design[0] = 'c'; /* the C comment should be pressent in the box design */
	    box_design[1] = '\0';
	  }
      if (pipe(sig2box) == -1 || pipe(box2sig) == -1)
	{
 	  if(fifo->verbose > 0)
	    logging(fifo, "[pretty_print]\t\t: Error: piping sig2box and box2sig.\n");
	  exit(1);
	}
      /* prepare the boxes_flags that are needed in this run. */
      if(fifo->verbose > 2)
	logging(fifo, "[pretty_print]\t\t: calling spred_box_flags with {%s}.\n", fifo->boxes_flags);
      if(!(boxes_flags = spred_box_flags(boxes_flags, fifo, box_design)))
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[pretty_print]\t\t: Error: spred_box_flags() returned with an error.\n");
	  return EXCEEDED;
	}
      if(fifo->verbose > 2)
	{
	  while(boxes_flags[flags_here] != NULL)
	    logging(fifo, "[pretty_print]\t\t: returning from spred_box_flags with {%s}.\n", boxes_flags[flags_here++]);
	}      

      /* Fork child */
      switch (fork())
	{
	case -1:
	  perror("fork");
	  exit(1);
	  
	case 0:
	  /* Child section */
	  /* 
	   * Duplicate child's reading end of sig2box into stdin,
	   * and child's writing end of box2sig into stdout
	   */
	  if (dup2(sig2box[0], STDIN_FILENO) ==-1 ||
	      dup2(box2sig[1], STDOUT_FILENO) ==-1)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Child duping sig2box and box2sig.\n");
	      exit(1);
	    }
	  
	  /* Close unused filedescriptors */
	  if(close(sig2box[0]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Child closing sig2box[0].\n");
	      exit(1);
	    }
	  if(close(sig2box[1]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Child closing sig2box[1].\n");
	      exit(1);
	    }
	  if(close(box2sig[0]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Child closing box2sig[0].\n");
	      exit(1);
	    }
	  if(close(box2sig[1]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Child closing box2sig[1].\n");
	      exit(1);
	    }
	  
	  /* 
	   * All set: exec boxes with the given settings.
	   */
	  
	  execvp(BOXES_PROGRAM, boxes_flags);
	  if(fifo->verbose > 0)
	    logging(fifo, "[pretty_print]\t\t: Error: Child executing boxes {%s}.\n", BOXES_PROGRAM);
	  exit(1);
	  
	default:
	  /* Parent section */
	  /* Close unused fildescriptor */
	  if(close(sig2box[0]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Parent closing sig2box[0].\n");
	      exit(1);
	    }
	  if(close(box2sig[1]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Parent closing box2sig[1].\n");
	      exit(1);
	    }
	  /* Write found lines to pipe */
	  if((nwrite = write(sig2box[1], (void*)out_buffer, strlen(out_buffer)+1)) < 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Parent writing to sig2box[1] pipe.\n");
	      exit(1);
	    }
	  /* 
	   * Close feeding pipe to tell boxes it can start boxing it,
	   * then read all its output, and print the lines to our fifo
	   */
	  if(close(sig2box[1]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Parent closing sig2box[1].\n");
	      exit(1);
	    }
	  /* open the fifo file, and prepare to write it */
	  if(fifo->verbose > 0)
	    logging(fifo, "[pretty_print]\t\t: Parent opening the fifo: {%s} for writing.\n", fifo->fifo_name);
	  if(!(out_file = fopen(fifo->fifo_name, "w")))
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Parent can't open file {%s} for writing.\n", fifo->fifo_name);
	      /* some error happened */
	      return CONF_MI;
	    }

	  /* make the normal required "-- \n" as the first line of a sig. */
	  if(fifo->pretty.slash)
	    { /* if the user told, that the system dosnt put it in */
	      if(!fwrite((void*)"\n-- \n", 5, 1, out_file))
		{
		  if(fifo->verbose > 0)
		    logging(fifo, "[pretty_print]\t\t: Error: Parent can't write {-- \\n} to file {%s}.\n", fifo->fifo_name);
		  fclose(out_file);
		  return WRITE_ERR;
		}
	      if(fifo->verbose > 2)
		logging(fifo, "[pretty_print]\t\t: Parent wrote {-- \\n} to file {%s}.\n", fifo->fifo_name);
	    }
	  if(fifo->verbose > 1)
	    logging(fifo, "[pretty_print]\t\t: Parent writing boxed out_buffer to file {%s}.\n", fifo->fifo_name);
	  while((nread = read(box2sig[0], line_buff, LINE_SIZE)) > 0)
	    {
	      line_buff[nread] = '\0';
	      if ((nwrite = fwrite((void*)line_buff, strlen(line_buff), 1, out_file)) < 0)
		{
		  if(fifo->verbose > 0)
		    logging(fifo, "[pretty_print]\t\t: Parent writing boxed out_buffer to file {%s} failed.\n", fifo->fifo_name);
		  fclose(out_file);
		  exit(1);
		}
	    }
	  /* close the last pipe that reads from boxes */
	  if(close(box2sig[0]) != 0)
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Parent closing box2sig[0].\n");
	      exit(1);
	    }
	  if(fifo->verbose > 1)
	    logging(fifo, "[pretty_print]\t\t: Parent wrote boxed out_buffer to file {%s}.\n", fifo->fifo_name);
	  fclose(out_file);
	  pid = waitpid(0,&status,0);
	  if(fifo->verbose > 1 && pid <= 1)
	    logging(fifo, "[pretty_print]\t\t: Child exit failed with code %d.\n", status);
	}
      BFREE(boxes_flags);
      /* All done */
      
    }
  else
    {
#endif
      /* just print what is needed, since theres no required boxes */
      if(fifo->verbose > 0)
	logging(fifo, "[pretty_print]\t\t: Opening the fifo: {%s} for writing.\n", fifo->fifo_name);
      if(!(out_file = fopen(fifo->fifo_name, "w")))
	{
	  if(fifo->verbose > 0)
	    logging(fifo, "[pretty_print]\t\t: Error: Can't open file {%s} for writing.\n", fifo->fifo_name);
	  /* some error happened */
	  return CONF_MI;
	}
      /* make the normal required "-- \n" as the first line of a sig. */
      if(fifo->pretty.slash)
	{ /* if the user told, that the system dosnt put it in */
	  if(!fwrite((void*)"\n-- \n", 5, 1, out_file))
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Can't write {-- \\n} to file {%s}.\n", fifo->fifo_name);
	      fclose(out_file);
	      return WRITE_ERR;
	    }
	  if(fifo->verbose > 2)
	    logging(fifo, "[pretty_print]\t\t: Wrote {-- \\n} to file {%s}.\n", fifo->fifo_name);
	}
      if(fifo->verbose > 2)
	logging(fifo, "[pretty_print]\t\t: out_buffer contains: {%s}.\n", out_buffer);
      if(fifo->verbose > 1)
	logging(fifo, "[pretty_print]\t\t: Writing out_buffer to file {%s}.\n", fifo->fifo_name);
      if(!fwrite((void*)out_buffer, strlen(out_buffer), 1, out_file))
	    {
	      if(fifo->verbose > 0)
		logging(fifo, "[pretty_print]\t\t: Error: Can't write out_buffer to file {%s}.\n", fifo->fifo_name);
	      fclose(out_file);
	      return WRITE_ERR;
	    }
	  if(fifo->verbose > 2)
	    logging(fifo, "[pretty_print]\t\t: Wrote out_buffer to file {%s}.\n", fifo->fifo_name);
	  fclose(out_file);

#ifdef BOXING_SUPPORT
    }
#endif
  if(fifo->verbose > 0)
    logging(fifo, "[pretty_print]\t\t:    DONE\n");
  if(fifo->verbose > 1)
    logging(fifo, "\n** So Long and Thanks for all the Fish. **\n\n");
  return SUCCESS;
}

#endif

#ifdef sigitdb


/* 
 * hmmm lets hope people won't read this far in the file.. 
 * Else get your calculator within reach, and start punching.
 */ 

int _shine_on_you_crazy_diamond_(void){return(printf("%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c",10,34,84,104,101,32,98,97,110,100,32,105,115,32,106,117,115,116,32,102,97,110,116,97,115,116,105,99,44,10,32,116,104,97,116,32,105,115,32,114,101,97,108,121,32,119,104,97,116,32,73,32,116,104,105,110,107,46,10,32,45,79,104,32,98,121,32,116,104,101,32,119,97,121,44,32,119,104,105,99,104,32,111,110,101,32,105,115,32,80,105,110,107,32,63,34,10,9,45,32,45,32,80,105,110,107,32,70,108,111,121,100,32,45,32,72,97,118,101,32,65,32,67,105,103,97,114,46,10,10));}

#endif
