/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "imagefileitem.h"

#include <qfileinfo.h>
#include <qdir.h>
#include <qapplication.h>
#include <qdragobject.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qiconview.h>
#include <qpixmapcache.h>
#include <qurloperator.h>

#include <kiconloader.h>
#include <kapp.h>
#include <kpixmap.h>
#include <kmessagebox.h>
#include <kglobalsettings.h>
#include <konq_operations.h>
#include <kio/job.h>

ImageFileItem::ImageFileItem (ImageListeView * parent, Directory * parentDir,
			      const char *filename, const char *path,
			      ImageViewer * imv):
   FileItem (parent, path, filename, "image"),
   f(filename)
{
	iv = imv;
	imageList=parent;

	readable = TRUE;
	selected = false;
	this->parentDir = parentDir;
	full.append (path);
	full.append (filename);
	setRenameEnabled(false);
	info = new QFileInfo (full);
	size = new QString ();
	size->sprintf ("%10d", info->size ());
	QDateTime epoch( QDate( 1980, 1, 1 ) );
	date = new QString ();
	date->sprintf("%08d", epoch.secsTo(info->lastModified())) ;
	extension = new QString (info->extension ().upper ());
	type = new QString ("file");
	p_open = NULL;
	p_none = NULL;
	pixLoad = false;
	doPreload = true;
	setName("ImageFileItem");
}


ImageFileItem::~ImageFileItem ()
{
	delete (size);
	delete (extension);
	delete (type);
	delete (p_open);
	delete (p_none);
}


void
ImageFileItem::setName(QString n)
{
	FileItem::setName(n);
}


void
ImageFileItem::setText( const QString & text )
{
	if(text==this->text(0))
		return;

	QFileInfo *itemFileInfo = new QFileInfo( full );
	QDir dir( itemFileInfo->dir() );

	if(QFileInfo(itemFileInfo->dirPath(TRUE)+"/"+text).exists())
	{
		KMessageBox::error(iv, QString().sprintf(i18n("The file '<b>%s</b>' already exists"),(const char*)text));
		return;
	}

	if ( dir.rename( itemFileInfo->fileName(), text ) )
	{
		QString itemFileName = itemFileInfo->dirPath( TRUE ) + "/" + text;
		full = QCString(itemFileInfo->dirPath( TRUE ) + "/" + text);

		delete itemFileInfo;
		itemFileInfo = new QFileInfo( itemFileName );
	
		f.setName(text);
		QIconViewItem::setText( this->text() );
	}
	else
	{
		KMessageBox::error(iv, QString().sprintf(i18n("The file <b>%s</b> can't be renamed"), (const char*)text));
	}
}


bool
ImageFileItem::suppression (bool suprFile)
{
	if (suprFile)
	{
		return suppression ();
	}
	else
	{
		parentDir->removeImage (this);
		return true;
	}
}


bool
ImageFileItem::suppression ()
{
	if (QDir ().remove (fullName ()))
	{
		parentDir->removeImage (this);
		return true;
	}
	else
	{
		return false;
	}
}


bool 
ImageFileItem::moveToTrash()
{
	KURL org("file:/"+fullName()), dest("file:/"+KGlobalSettings::trashPath()+getFullName(new QString(fullName())));
	KIO::file_move(org, dest, -1, true, false, false);
	return suppression(false);
};


bool
ImageFileItem::shred()
{
	KIO::del(KURL("file:/"+fullName ()), true, false);
	suppression(false);
	return true;
}


QString
ImageFileItem::fullName ()
{
	return QString(full);
}

QString
ImageFileItem::name()
{
	return FileItem::name();
}


QString
ImageFileItem::shrink(QString str, int len) const
{
	if(str.length()<=len)
		return str;
	else
	{
		return QString(str.left(len/2) + "..." + str.right(len/2));
	}
}

QString
ImageFileItem::text () const
{
		return  f.name()+
				i18n("\nLocation: ")+
				shrink(QString(parentDir->fullName()),20)+
				i18n("\nSize: ")+
				size->stripWhiteSpace()+
				i18n(" bytes\nDate: ")+
				info->lastModified().toString();
}



QString
ImageFileItem::text (int column) const
{
	
	if (column == 0)
		return f.name ();
	else if (column == 1)
		return (*extension);
	else if (column == 2)
		return (*size);
	else if (column == 3)
		return (*type);
	else
		return QString ();

}


void
ImageFileItem::setSelected (bool s)
{
	QIconViewItem::setSelected (s);
	if (s)
	{
		QApplication::setOverrideCursor (waitCursor);	  // this might take time
		iv->loadImage ((const char *) full);
		qApp->processEvents ();
		if (nextItem () && imageList->doPreload())
		{
			iv->preloadImage (((ImageFileItem *) nextItem ())->fullName ());
		}
		QApplication::restoreOverrideCursor ();   // restore original cursor
	}
	else
	{
	}
}


int
ImageFileItem::width () const
{
	return QIconViewItem::width();
}

