/* 
 * Sherman's aquarium v2.1.2
 * (Formely known as aquarium applet.)
 *
 * Copyright (C) 2002 Jonas Aaberg <cja@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You did not receive a copy of the GNU General Public License along with
 * this program; chances are, you already have at least 10 copies of this
 * license somewhere on your system.  If you still want it, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, 
 * MA 02111-1307, USA.
 *
 * Based upon timecop's <timecop@japan.co.jp> fishmon.
 *
 */


/* general includes */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <math.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#ifdef DARWIN
#include "getopt.h"
#else
#ifdef SUNOS
#include "getopt.h"
#else
#include <getopt.h>
#endif
#endif

/* Same include files for both 1.2.x and 2.0.x */
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk-pixbuf/gdk-pixbuf.h>


#include "draw.h"
#include "aquarium.h"
#include "defines.h"
#include "sherman.h"
#include "stamp.h"

#include "wmapplet.h"
#include "detectbrokenwm.h"
#include "preferences.h"


#ifdef GNOME

#ifndef GNOME2
#include <gnome.h>
#include <applet-widget.h>
#include "gnome-applet.h"
#else
#include <libgnomeui/libgnomeui.h>
#include <libgnome/libgnome.h>
#include <panel-applet.h>
#include "gnome2-applet.h"
#endif

#endif

/*
#define DEBUG
*/

#ifdef THERMOMETER
#include "sys_include.h"
#endif

#ifdef LEDS
#include "kleds.h"
#endif

/* Yeah, this shouldn't be here really. ;-) */
#ifdef MATRIX
#include "matrix.h"
#endif

#ifdef FULLSCREEN
#include "fullscreen.h"
#endif

/* Global variables */



Settings mys;
Fish_buffer fish_buffer[NUMOFFISHTYPES];

Display *display=NULL;


int ymax, xmax, ymin, xmin;
int virtual_aquarium_x, virtual_aquarium_y;

int windowsize_x, windowsize_y;




int goad_server;
int force_es;
int brokenwm;		/* Is a broken window mananger such as sawfish 
				   and enlightenment running or not? */
int posx, posy;




/* Full screen stuff */
#ifdef FULLSCREEN
int full_screen;
int keep_resolution;
int escape_only;
int xscreensaver;
int window_id;
static int have_sea_floor;
static int start_sea_floor;

static Fish_buffer sea_floor;
static Fish_buffer bottom_stuff;
#endif

/* Fullscreen needs to know about those two. */
Fish_buffer blowup_data;
Fish_buffer prey_hunter_data;

/* keeps track of distance between bubble state changes (depends on ymax) */
int bubble_state_change;	



/*
   Static variables
   ================
*/
 
static int start_wm;

#ifdef MATRIX
static int fade;
#endif


/* current bubble count */
static int nr_bubbles;

/* keeps track of mouse focus */
static int proximity;


static AquariumData ad;
static Bubble *bubbles;


/* Images and data for them */
static Fish_buffer numlock, scrollock, capslock;
static Fish_buffer dig, thermometer;
static Fish_buffer bubbles_image_data;


/* Where the graphics are stored */
static char datadir[] = DATADIRPREFIX;
static char *altdatadir;
static int use_altdatadir;


/* data for special actions */
static int special_action;






/* Functions */



#ifdef GTK2

/* This function is called if Sherman's aquarium can't open a graphic file */
void display_file_error(char *msg, char *func, GError *errmsg)
{
  char tmpmsg[1024];
  g_snprintf(tmpmsg,1024, "Can't load %s - %s()\n%s",msg,func,errmsg->message);
#ifdef GNOME
  if(!start_wm)
    gnome_error_dialog(tmpmsg);
  else
#endif
    printf("%s",tmpmsg);
  exit(-1);
}

#else
/* This function is called if Sherman's aquarium can't open a graphic file */
void display_file_error(char *msg, char *func)
{
  char tmpmsg[256];
  g_snprintf(tmpmsg,256, "Can't load %s - %s()\n",msg,func);
#ifdef GNOME
  if(!start_wm)
    gnome_error_dialog(tmpmsg);
  else
#endif
    printf("%s",tmpmsg);
  exit(-1);
}

#endif



/* Remember to g_free() every filename!! */
char *filename(char *name)
{
    char *tmp;

    if (!use_altdatadir) {
	tmp = g_malloc0(strlen(datadir) + strlen(name) + 10);
	if (datadir[strlen(datadir) - 1] == '/') {
	    g_snprintf(tmp,strlen(datadir)+strlen(name)+5, "%s%s", datadir, name);
	} else {
	    g_snprintf(tmp,strlen(datadir)+strlen(name)+5, "%s/%s", datadir, name);
	}
    } else {
	tmp = g_malloc0(strlen(altdatadir) + strlen(name) + 10);
	if (altdatadir[strlen(altdatadir) - 1] == '/') {
	    g_snprintf(tmp,strlen(altdatadir)+strlen(name)+5, "%s%s", altdatadir, name);
	} else {
	    g_snprintf(tmp,strlen(altdatadir)+strlen(name)+5, "%s/%s", altdatadir, name);
	}
    }
    return tmp;
}

int entering_aquarium(void)
{
    int i;
    proximity = 1;
    for (i = 0; i < mys.fishes; i++)
	ad.fishes[i].speed_mul = 2.5;

    return 1;
}

int leaving_aquarium(void)
{
    int i;
    proximity = 0;

    /* get fish moving again, but slowly - checking out */
    for (i = 0; i < mys.fishes; i++)
	ad.fishes[i].speed_mul = 0.5;

    return 1;
}




void update_screen()
{
#ifdef GNOME
    if (start_wm) {
#ifndef GTK2
	if (!brokenwm)
	    gdk_draw_rgb_image(ad.iconwin, ad.gc, xmin, ymin, xmax, ymax,
			       GDK_RGB_DITHER_NONE, ad.rgb, xmax * 3);
	else
#endif
	    gdk_draw_rgb_image(ad.win, ad.gc, xmin, ymin, xmax, ymax,
			       GDK_RGB_DITHER_NONE, ad.rgb, xmax * 3);
    } else
	gdk_draw_rgb_image(ad.drawingarea->window, ad.gc, xmin, ymin, xmax,
			   ymax, GDK_RGB_DITHER_NONE, ad.rgb, (xmax) * 3);
#else

#ifndef GTK2
    if (!brokenwm)
	gdk_draw_rgb_image(ad.iconwin, ad.gc, xmin, ymin, xmax, ymax,
			   GDK_RGB_DITHER_NONE, ad.rgb, xmax * 3);
    else
#endif
	gdk_draw_rgb_image(ad.win, ad.gc, xmin, ymin, xmax, ymax,
			   GDK_RGB_DITHER_NONE, ad.rgb, xmax * 3);



#endif
    gdk_flush();



}



void thermometer_update()
{
#ifdef THERMOMETER
    /* Draw thermometer */
    if (mys.thermo) {
	if (ymax > thermometer.height) {
	    draw_pic_alpha(thermometer.image, thermometer.width,
			   thermometer.height, xmax - thermometer.width,
			   ymax - thermometer.height, 0, 80);

	    /* The temperature line is very much depending of the thermometer graphics. */

	    anti_line(xmax - 5, ymax - 5, xmax - 5,
		      ymax - 5 - (30 * system_cpu() / 100), 2, 0xe00000);


	    
	}
    }
#endif
}

void leds_update()
{
#ifdef LEDS

    int status;

    if (mys.leds) {
	if (xmax > (8 + 3 * capslock.width + 2)
	    && ymax > (2 + capslock.height)) {
	    status = check_kleds();
	    if (status == -1)
		return;

	    if ((status & 1) == 1)
		draw_pic_alpha(capslock.image, capslock.width,
			       capslock.height, 8 + numlock.width + 1,
			       ymax - capslock.height, 0, 80);

	    if ((status & 2) == 2)
		draw_pic_alpha(numlock.image, numlock.width,
			       numlock.height, 8, ymax - numlock.height, 0,
			       80);

	    if ((status & 4) == 4)
		draw_pic_alpha(scrollock.image, scrollock.width,
			       scrollock.height,
			       8 + (capslock.width + numlock.width) + 2,
			       ymax - scrollock.height, 0, 80);
	}
    }
#endif
}



/* Does this function leak memory?! */

void clock_update()
{

#ifdef CLOCK

    time_t now;

    struct tm *mt;

    if (!mys.clock)
	return;
    if (xmax < (4 + (dig.width + 1) * 4 + 4) || ymax < (4 + dig.height))
	return;

    now = time(NULL);
    mt = localtime(&now);

    draw_pic_alpha(dig.image, dig.width, dig.height, 4, 4,
		   mt->tm_hour / 10, 80);

    draw_pic_alpha(dig.image, dig.width, dig.height, 4 + dig.width + 1, 4,
		   mt->tm_hour % 10, 80);

    draw_pic_alpha(dig.image, dig.width, dig.height,
		   4 + (dig.width + 1) * 2 + 4, 4, mt->tm_min / 10, 80);

    draw_pic_alpha(dig.image, dig.width, dig.height,
		   4 + (dig.width + 1) * 3 + 4, 4, mt->tm_min % 10, 80);

#endif

}

void bubble_update()
{
    int i, x, y;
    int seq;

#ifdef FULLSCREEN
    if(full_screen){
      for(i=0;i<nr_bubbles;i++)
	fullscreen_clean(bubbles[i].x - VIEWPOINT_START_X ,((int)bubbles[i].y) - VIEWPOINT_START_Y,
			 bubbles_image_data.width, bubbles_image_data.height);
    }
#endif


    /* make a new bubble, if needed */
    if ((nr_bubbles < mys.bubbles) && ((rand() % 101) <= 32)) {
	bubbles[nr_bubbles].x = VIEWPOINT_START_X + (rand() % xmax);
	bubbles[nr_bubbles].y = VIEWPOINT_START_Y + ymax;

#ifdef VARIABLE_BUBBLE_SPEED
	bubbles[nr_bubbles].speed = ((rand() % 10) + 1);
#else
	bubbles[nr_bubbles].speed = 0.015625;
#endif
	nr_bubbles++;
    }

    /* Update and draw the bubbles */
    for (i = 0; i < nr_bubbles; i++) {

	/* Move the bubble vertically */
	bubbles[i].y -= (bubbles[i].speed / 13);

	/* Did we lose it? */
	if (bubbles[i].y < VIEWPOINT_START_Y) {
	    /* Yes; nuke it */
	    bubbles[i].x = bubbles[nr_bubbles - 1].x;
	    bubbles[i].y = bubbles[nr_bubbles - 1].y;
	    bubbles[i].speed = bubbles[nr_bubbles - 1].speed;
	    nr_bubbles--;

	    /* We must check the previously last bubble, which is
	     * now the current bubble, also. */
	    i--;
	    continue;
	}

	/* Draw the bubble */
	x = bubbles[i].x - VIEWPOINT_START_X;
	y = ((int) bubbles[i].y) - VIEWPOINT_START_Y;

	/* calculate bubble sequence - 0 to 4 (determine bubble sprite idx) */
	seq = y / bubble_state_change;
#ifdef DEBUG
	/*	printf("drawing bubble");*/
	fflush(NULL);
#endif
	/* draw the bubble, using offset-to-center calculated from current
	 * sequence, and make the bubble bigger as we go up. 120 - alpha */

#ifdef FULLSCREEN
	if(!full_screen){
#endif
	  draw_pic_alpha(bubbles_image_data.image, bubbles_image_data.width, 
			 bubbles_image_data.height, x, y,
			 seq, 120);
#ifdef FULLSCREEN
	}
	else {
	    fullscreen_draw_fish(x,y,seq, bubbles_image_data.width, 
				 bubbles_image_data.height,bubbles_image_data.image);
	}
#endif


#ifdef DEBUG
	/*	printf("done.\n");*/
#endif
    }
}

int get_fish(int start, int wantedfish)
{
  int i;
  for(i=start+1;i<mys.fishes;i++)
    if(ad.fishes[i].type==wantedfish) return i;
  
  return -1;
}

/* This rouitine handles both the eating fish happening and
   the swordish hits a blowfish happening. */

/* 
   This rountine is the most messy you've seen. Sorry! 
   But it works ok. :-)
*/

void prey_hunter_hit()
{
  int i,j,k;
  int victim_val=-1, doer_val=-1;


  for(i=0;i<mys.fishes;i++){
    doer_val=get_fish(doer_val, HUNTER);
    if(doer_val==-1) break;

    /* Check if this one is eating right now. */
    if(ad.fishes[doer_val].num_animation==7) continue;

    victim_val=-1;

    for(j=0;j<mys.fishes;j++){
      victim_val=get_fish(victim_val, PREY);
      if(victim_val==-1) break;
      if(ad.fishes[victim_val].rev==ad.fishes[doer_val].rev) continue;


      /* If parts of the prey is beyond the right edge, let it live */
      if((ad.fishes[victim_val].tx+ad.fishes[victim_val].width) 
	 >(xmax+VIEWPOINT_START_X)) continue;
      /* If the fish is below the edge, let it live */
      if((ad.fishes[victim_val].y) >(ymax+VIEWPOINT_START_Y) ) continue;

      if(ad.fishes[victim_val].tx < (xmin-ad.fishes[victim_val].width)) continue;

      /* If the prey is too high, let it live */
      if((ad.fishes[victim_val].y) < VIEWPOINT_START_Y) continue;



      
      if(abs((ad.fishes[victim_val].y+ad.fishes[victim_val].height)-
	     (ad.fishes[doer_val].y+ad.fishes[doer_val].height))> (ad.fishes[doer_val].height/2)) continue;


      
      /* rev=1 prey moves to towards right.*/
      if(ad.fishes[victim_val].rev){

	if((ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   >(10+ad.fishes[victim_val].width) &&
	   (ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   <(30+ad.fishes[victim_val].width)){
	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height)
	     >(ad.fishes[doer_val].y+ad.fishes[doer_val].height)) 
	    ad.fishes[doer_val].y+=2;
	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height)
	     <(ad.fishes[doer_val].y+ad.fishes[doer_val].height)) 
	    ad.fishes[doer_val].y-=2;
 
	  continue;
	}

	if((ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   >(10+ad.fishes[victim_val].width)) continue;
	if((ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   <ad.fishes[victim_val].width) continue;

      }
      else{
	if((ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   >(10+ad.fishes[victim_val].width) &&
	   (ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   <(30+ad.fishes[victim_val].width)){

	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height)
	     >(ad.fishes[doer_val].y+ad.fishes[doer_val].height)) 
	    ad.fishes[doer_val].y+=2;
	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height)
	     <(ad.fishes[doer_val].y+ad.fishes[doer_val].height)) 
	    ad.fishes[doer_val].y-=2;

	  continue;
	}


	if((ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   >(10+ad.fishes[victim_val].width)) continue;
	if((ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   <ad.fishes[victim_val].width) continue;
      }
      /*
      printf("hunter %d eats prey %d!\n",doer_val, victim_val);
      */

	
      /* Removing eaten prey */


#ifdef FULLSCREEN
      /* Removing the poor sucker that gets eaten from screen */
      if(full_screen){
	fullscreen_clean((int)ad.fishes[victim_val].tx, ad.fishes[victim_val].y - VIEWPOINT_START_Y, 
			 ad.fishes[victim_val].width, ad.fishes[victim_val].height);
      }
#endif


      if(doer_val>victim_val) doer_val--;
      mys.fishes--;
      
      for(k=victim_val;k<mys.fishes;k++){
	ad.fishes[k]=ad.fishes[k+1];
      }
      
      /* Towards left */
      
      if(!ad.fishes[doer_val].rev)
	ad.fishes[doer_val].tx-=(prey_hunter_data.width-ad.fishes[doer_val].width);

      ad.fishes[doer_val].pic=prey_hunter_data.image;
      ad.fishes[doer_val].frame=0;
      ad.fishes[doer_val].width=prey_hunter_data.width;
      ad.fishes[doer_val].height=prey_hunter_data.height;
      ad.fishes[doer_val].animation=prey_hunter_animation;
      ad.fishes[doer_val].num_animation=7;
      ad.fishes[doer_val].speed=prey_hunter_speed;
      ad.fishes[doer_val].fast_frame_change=5.0;
      break;
      
    }

  }
}

/* Not quite ok yet...*/
void blowup_hit()
{

  int i,j;
  int victim_val=-1, doer_val=-1;


  for(i=0;i<mys.fishes;i++){
    doer_val=get_fish(doer_val, SWORDFISH);
    if(doer_val==-1) break;

    victim_val=-1;

    for(j=0;j<mys.fishes;j++){
      victim_val=get_fish(victim_val, BLOWFISH);
      if(victim_val==-1) break;

      /* Is this blowfish blowing up? - Then take next.*/
      if(ad.fishes[victim_val].num_animation==2) continue;

       
      if(ad.fishes[victim_val].rev==ad.fishes[doer_val].rev) continue;


      /* If parts of the prey is beyond the right edge, let it live */
      if((ad.fishes[victim_val].tx+ad.fishes[victim_val].width) 
	 >(xmax+VIEWPOINT_START_X)) continue;
      /* If the fish is below the edge, let it live */
      if((ad.fishes[victim_val].y) >(ymax+VIEWPOINT_START_Y) ) continue;

      if(ad.fishes[victim_val].tx < (xmin-ad.fishes[victim_val].width)) continue;

      /* If the prey is too high, let it live */
      if((ad.fishes[victim_val].y) < VIEWPOINT_START_Y) continue;

      
      if(abs(ad.fishes[victim_val].y - ad.fishes[doer_val].y)
	 > (ad.fishes[victim_val].height/2)) continue;


      
      /* rev=1 prey moves to towards right.*/
      if(ad.fishes[victim_val].rev){

	if((ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   >0 &&
	   (ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   <20){
	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height/2)
	     >(ad.fishes[doer_val].y+ad.fishes[doer_val].height/2)) 
	    ad.fishes[doer_val].y+=2;
	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height/2)
	     <(ad.fishes[doer_val].y+ad.fishes[doer_val].height/2))
	    ad.fishes[doer_val].y-=2;
 
	  continue;
	}

	if((ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   >0) continue;
	if((ad.fishes[doer_val].tx-ad.fishes[victim_val].tx-ad.fishes[victim_val].width)
	   <-10) continue;
	/*
	printf("killing from right!\n vx:%d - vy:%d - dx:%d - dy:%d", 
	       (int)ad.fishes[victim_val].tx, ad.fishes[victim_val].y,
	       (int)ad.fishes[doer_val].tx, ad.fishes[doer_val].y);
	*/
      }
      else{
	if((ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   >0 &&
	   (ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   <20){

	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height/2) 
	     >(ad.fishes[doer_val].y+ad.fishes[doer_val].height/2)) 
	    ad.fishes[doer_val].y+=2;

	  if((ad.fishes[victim_val].y+ad.fishes[victim_val].height/2)
	     <(ad.fishes[doer_val].y+ad.fishes[doer_val].height/2)) 
	    ad.fishes[doer_val].y-=2;

	  continue;
	}


	if((ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   >0) continue;
	if((ad.fishes[victim_val].tx-ad.fishes[doer_val].width-ad.fishes[doer_val].tx)
	   <-10) continue;
	/*
	printf("killing from left!\n vx:%d - vy:%d - dx:%d - dy:%d", 
	       (int)ad.fishes[victim_val].tx, ad.fishes[victim_val].y,
	       (int)ad.fishes[doer_val].tx, ad.fishes[doer_val].y);

	*/
      }
      /*
      printf("swordfish hit %d blowfish %d!\n",doer_val, victim_val);
      */


#ifdef FULLSCREEN
      /* Removing the blowfish from screen */
      if(full_screen){
	fullscreen_clean((int)ad.fishes[victim_val].tx, ad.fishes[victim_val].y - VIEWPOINT_START_Y, 
			 ad.fishes[victim_val].width, ad.fishes[victim_val].height);
      }
#endif


      ad.fishes[victim_val].pic=blowup_data.image;
      ad.fishes[victim_val].frame=0;
      ad.fishes[victim_val].width=blowup_data.width;
      ad.fishes[victim_val].height=blowup_data.height;
      ad.fishes[victim_val].animation=normal_animation;
      ad.fishes[victim_val].num_animation=2;
      ad.fishes[victim_val].speed=blowup_speed;
      ad.fishes[victim_val].fast_frame_change=1.0;
      break;
      
    }

  }

}



void fish_update(void)
{
    int i,j,k;

    if(special_action==1 || mys.eat!=1)
      prey_hunter_hit();

    if(special_action==2 || mys.explode!=1)
      blowup_hit();

#ifdef DEBUG
    printf("\t\t - Done special hits.\n");
#endif


#ifdef FULLSCREEN    
    if(full_screen){
      for(i=0;i<mys.fishes;i++)
	fullscreen_clean((int)ad.fishes[i].tx, ad.fishes[i].y - VIEWPOINT_START_Y, 
			 ad.fishes[i].width, ad.fishes[i].height);
    }
#endif

    /* We should check after fullscreen_clean if any fishes are to be removed.
       explode and hunter eats prey.*/

    for(i=0;i<mys.fishes;i++){
      if (ad.fishes[i].frame >= (ad.fishes[i].num_animation-1)){
	if(mys.eat){
	  
	  if(ad.fishes[i].pic==prey_hunter_data.image){
	    /*
	      printf("Done eating.Continue.\n");
	    */
	    
	    if(ad.fishes[i].rev) ad.fishes[i].tx+=ad.fishes[i].width-fish_buffer[HUNTER].width;
	    /* Left movement is ok. */
	    /*		  else ad.fishes[i].tx-=ad.fishes[i].width-fish_buffer[HUNTER].width;*/
	    
	    ad.fishes[i].width=fish_buffer[HUNTER].width;
	    ad.fishes[i].height=fish_buffer[HUNTER].height;
	    ad.fishes[i].pic=fish_buffer[HUNTER].image;
	    ad.fishes[i].animation=normal_animation;
	    ad.fishes[i].num_animation=4;
	    ad.fishes[i].speed=hunter_speed;
	    ad.fishes[i].fast_frame_change=1.0;
	    ad.fishes[i].frame = 0;
	    
	    
	  }
	}
	
	if(mys.explode) {
	  
	  if(ad.fishes[i].pic==blowup_data.image){
	    /*		  printf("Done blowing up.\n");*/
	    mys.fishes--;
	    for(k=i;k<mys.fishes;k++){
	      ad.fishes[k]=ad.fishes[k+1];
	    }
	    
	  }
	}
      }
    }
    




    for (i = 0; i < mys.fishes; i++) {

	/* frozen fish doesn't need to be handled, or drawn */
	if (ad.fishes[i].speed_mul == 0)
	    continue;


	/* move fish in horizontal direction, left or right */
	/* Large aquarium, the fishes are shown more seldom */

	if (!ad.fishes[i].rev) {

	    ad.fishes[i].tx -=
		(ad.fishes[i].speed[ad.fishes[i].frame]) * (ad.fishes[i].speed_mul);

	    if (ad.fishes[i].tx < (-ad.fishes[i].width - ad.fishes[i].travel)) {
		/* we moved out of bounds. change direction,
		 * position, speed. */


		ad.fishes[i].travel =
		    (rand() % ((virtual_aquarium_x - xmax) / 2)) +
		    ad.fishes[i].width;
		ad.fishes[i].tx = -ad.fishes[i].width - ad.fishes[i].travel;
		ad.fishes[i].rev = 1;
		ad.fishes[i].y =
		    rand() % (virtual_aquarium_y - ad.fishes[i].height);
		if (proximity)
		    ad.fishes[i].speed_mul = 0;
		else
		    ad.fishes[i].speed_mul = 1;

	    }
	} else {


	    ad.fishes[i].tx +=
		(ad.fishes[i].speed[ad.fishes[i].frame]) * (ad.fishes[i].speed_mul);


	    if (ad.fishes[i].tx > xmax + ad.fishes[i].travel) {
		/* we moved out of bounds. change direction,
		 * position, speed. */

		ad.fishes[i].travel =
		    (rand() % ((virtual_aquarium_x - xmax) / 2)) +
		    ad.fishes[i].width;
		ad.fishes[i].tx = xmax + ad.fishes[i].travel;
		ad.fishes[i].rev = 0;
		ad.fishes[i].y =
		    rand() % (virtual_aquarium_y - ad.fishes[i].height);

		if (proximity)
		    ad.fishes[i].speed_mul = 0;
		else
		    ad.fishes[i].speed_mul = 1;

	    }
	}

	/* move fish in vertical position randomly by one pixel up or down */
	/* If the fish last time moved up or down, larger chance that it does that again. */

	j = rand() % 16;

	/* The bottondweller & hacktorn shall be still ;-) */
	if(ad.fishes[i].type == BDWELLER) j = 0;
	if(ad.fishes[i].type == HAWTHORNE) j = 0;

	if (((ad.fishes[i].updown == 1) && (j == 6 || j == 7 || j == 8))
	    || j == 8) {
	    ad.fishes[i].y++;
	    ad.fishes[i].updown = 1;
	} else if (((ad.fishes[i].updown == -1)
		    && (j == 12 || j == 13 || j == 14)) || j == 12) {
	    ad.fishes[i].y--;
	    ad.fishes[i].updown = -1;
	} else
	    ad.fishes[i].updown = 0;
#ifdef DEBUG
	/*	printf("\t\t - Calling draw_fish()\n");*/
#endif
#ifdef FULLSCREEN
	if(full_screen){

	  if(!ad.fishes[i].rev){
	  /* animate fishes using fish_animation array */
	    fullscreen_draw_fish((int) ad.fishes[i].tx,
				 ad.fishes[i].y - VIEWPOINT_START_Y,
				 ad.fishes[i].animation[ad.fishes[i].frame],
				 ad.fishes[i].width,
				 ad.fishes[i].height, ad.fishes[i].pic);
	  }
	  else{
	    if(ad.fishes[i].pic==blowup_data.image){
	      fullscreen_draw_fish((int) ad.fishes[i].tx,
				   ad.fishes[i].y - VIEWPOINT_START_Y,
				   ad.fishes[i].animation[ad.fishes[i].frame],
				   ad.fishes[i].width,
				   ad.fishes[i].height, blowup_data.rev);


	    }
	    else if(ad.fishes[i].pic==prey_hunter_data.image){
	      fullscreen_draw_fish((int) ad.fishes[i].tx,
				   ad.fishes[i].y - VIEWPOINT_START_Y,
				   ad.fishes[i].animation[ad.fishes[i].frame],
				   ad.fishes[i].width,
				   ad.fishes[i].height, prey_hunter_data.rev);

	    }
	    else{
	      fullscreen_draw_fish((int) ad.fishes[i].tx,
				   ad.fishes[i].y - VIEWPOINT_START_Y,
				   ad.fishes[i].animation[ad.fishes[i].frame],
				   ad.fishes[i].width,
				   ad.fishes[i].height, fish_buffer[ad.fishes[i].type].rev);
	    }
	  }


	}
	else{
#endif
	  /* animate fishes using fish_animation array */
	  draw_fish((int) ad.fishes[i].tx,
		    ad.fishes[i].y - VIEWPOINT_START_Y,
		    ad.fishes[i].animation[ad.fishes[i].frame],
		    ad.fishes[i].rev, ad.fishes[i].width,
		    ad.fishes[i].height, ad.fishes[i].pic);
#ifdef FULLSCREEN
	}
#endif

	/* switch to next swimming frame */


	ad.fishes[i].delay += ad.fishes[i].speed[ad.fishes[i].frame]*ad.fishes[i].fast_frame_change;
#ifdef DEGUG
	printf("delay:%f speed:%f frame:%d num_ani:%d anipic:%d\n",ad.fishes[i].delay, ad.fishes[i].speed[ad.fishes[i].frame],ad.fishes[i].frame, ad.fishes[i].num_animation, ad.fishes[i].animation[ad.fishes[i].frame]);
#endif
	if (ad.fishes[i].delay >= (7*ad.fishes[i].speed[ad.fishes[i].frame])) {
	  if (ad.fishes[i].frame >= (ad.fishes[i].num_animation-1)){
		ad.fishes[i].frame = 0;
	  }	  
	  else ad.fishes[i].frame ++;
	  ad.fishes[i].delay = 0;
	}

    }
}



/* Loads a given image file into a 24bit buffer, and gives the width and height */


unsigned char *load_image_n_scale(char *fname, int *width, int *height, int *image,
				  int frames, int scale)
{
    unsigned char *buffer;
    int w, h, newh, neww;
    GdkPixbuf *tmpbuff, *tmpbuff2;
    char *fname2;
#ifdef GTK2
    GError *imerr=NULL;
#endif

    fname2 = filename(fname);

#ifdef GTK2
    
    tmpbuff = gdk_pixbuf_new_from_file(fname2, &imerr);
    if (!tmpbuff) {
      display_file_error(fname2,"load_image_n_scale", imerr);
    }
#else
    tmpbuff = gdk_pixbuf_new_from_file(fname2);
    if (!tmpbuff) {
      display_file_error(fname2,"load_image_n_scale");
    }
#endif

    g_free(fname2);

    w = gdk_pixbuf_get_width(tmpbuff);
    h = gdk_pixbuf_get_height(tmpbuff);

    if (scale != 100) {
	(*width) = neww = (w * scale) / 100;
	newh = h * scale / 100;
	(*height) = ((h / frames) * scale) / 100;


	tmpbuff2 =
	    gdk_pixbuf_scale_simple(tmpbuff, neww, newh, GDK_INTERP_BILINEAR);
	buffer = gdk_pixbuf_get_pixels(tmpbuff2);

	gdk_pixbuf_unref(tmpbuff);

	/* Have to save the gdkpixbuf data, since if we resize, we need to free it*/
	(*image) =(int) tmpbuff2;

    } else {
	(*width) = w;
	(*height) = h / frames;
	buffer = gdk_pixbuf_get_pixels(tmpbuff);

	/* Have to save the gdkpixbuf data, since if we resize, we need to free it*/
	(*image) =(int) tmpbuff;
    }

    printf(".");
    fflush(NULL);


    return buffer;
}

unsigned char *load_image(char *fname, int *width, int *height, 
			  int *image, int frames)
{
    return load_image_n_scale(fname, width, height, image, frames, 100);
}

#ifdef FULLSCREEN
void make_bottom(void)
{
  int i,j,k,l, step_max, current_loc,x;
  int old_height=0,old_x=0, old_current_loc=0, bx[10], bw[10], nfa;


  for(i=(start_sea_floor-sea_floor.width);i<xmax;i+=sea_floor.width){
    
    draw_fish(i,
	      ymax-sea_floor.height,
	      0,
	      0, sea_floor.width,
	      sea_floor.height, sea_floor.image);
  }
  
  /* Number of plants and stones Max 15*/


  j=rand()%15+1;
  
  step_max=sea_floor.height/j;

  /* The smallest stone/plant is 32 high */
  current_loc=ymax - sea_floor.height + rand()%step_max+30;
      

  /* See if we have a bottom dweller and or a hacktorn*/
  nfa=0;

  for(i=0;i<mys.fishes;i++){
    if(ad.fishes[i].type == BDWELLER || ad.fishes[i].type == HAWTHORNE){

      for(;;){
	x=(int)rand()%(xmax-ad.fishes[i].width);
	for(l=0;l<nfa;l++)
	  if(x > (bx[l]-bottom_stuff.width) && x < (bx[l]+bw[l])) break;
	if(l==nfa) break;
      }
      
      bx[nfa]=x;
      bw[nfa]=ad.fishes[i].width;
      ad.fishes[i].tx=bx[nfa];
      ad.fishes[i].y=current_loc + sea_floor.height/5+20 +rand()%sea_floor.height;
      if(ad.fishes[i].type == BDWELLER)
	ad.fishes[i].fast_frame_change=0.05;
      else
	ad.fishes[i].fast_frame_change=0.15;

      ad.fishes[i].speed_mul=0.0001;
      nfa++;
    }
  }
      

  /* Freeing sea floor. Not needed any longer */
  g_free(sea_floor.image);
  /*gdk_pixbuf_unref(sea_floor.pixbuf);*/


      
  for(i=0;i<j;i++){
    k=rand()%NUMOFBOTTOMITEMS;


    bottom_stuff.image = load_image_n_scale(bottom_items[k].image,
					    &bottom_stuff.width,
					    &bottom_stuff.height, 
					    (int *)&bottom_stuff.pixbuf,1,85+rand()%30);
    
	
	
    /* from -50 to xmax */
    if(nfa!=0){
      for(;;){
	x=rand()%(xmax+50)-50;
	for(l=0;l<nfa;l++)
	  if(x > (bx[l]-bottom_stuff.width) && x < (bx[l]+bw[l])) break;
	if(l==nfa) break;
      }
    }
    else
      x=rand()%(xmax+50)-50;


    /* Are they too close? */
    if(abs(x-old_x)<bottom_stuff.width)
      if((old_current_loc+old_height)>(bottom_stuff.height+current_loc)){
	current_loc=old_current_loc+old_height-bottom_stuff.height+5;
      }
	
    
    draw_fish(x, current_loc-bottom_stuff.height,
	      0,rand()%2,
	      bottom_stuff.width,bottom_stuff.height,
	      bottom_stuff.image);
    
    old_current_loc=current_loc;
    current_loc+=rand()%step_max;
    
    g_free(bottom_stuff.image);
    /*	gdk_pixbuf_unref(bottom_stuff.pixbuf);
	bottom_stuff.pixbuf=NULL;*/
    bottom_stuff.image=NULL;

    old_height=bottom_stuff.height;
    old_x=x;
  }
}
#endif


void prepare_backbuffer(void)
{
    int i, j, k, y, rowstride, alpha;

    float d;
    GdkPixbuf *water, *water2 = NULL;
    char *water_file;
#ifdef GTK2
    GError *imerr=NULL;
#endif
    unsigned char *buff;

    if (mys.wateralike || mys.yourown) {
      if(mys.wateralike)
	water_file = filename("aquarium/water.png");
      else{
	water_file = mys.imagename;
      }
#ifdef GTK2
      water = gdk_pixbuf_new_from_file(water_file, &imerr);
      if (!water) {
	display_file_error(water_file,"prepare_backbuffer",imerr);
      }
#else
      water = gdk_pixbuf_new_from_file(water_file);
      if (!water) {
	display_file_error(water_file,"prepare_backbuffer");
      }
#endif


      if(mys.wateralike)
	g_free(water_file);
      

	if (xmax != XMAX || ymax != YMAX) {

	    water2 =
		gdk_pixbuf_scale_simple(water, xmax, ymax,
					GDK_INTERP_BILINEAR);
	    buff = gdk_pixbuf_get_pixels(water2);

	    rowstride = gdk_pixbuf_get_rowstride(water2);
	    alpha=(int)gdk_pixbuf_get_has_alpha(water2);
	} else {
	    buff = gdk_pixbuf_get_pixels(water);
	    rowstride = gdk_pixbuf_get_rowstride(water);
	    alpha=(int)gdk_pixbuf_get_has_alpha(water);

	}
       
	k = 0;
	for (i = 0; i < ymax; i++) {
	    y = i * rowstride;
	    for (j = 0; j < ((xmax * (3+alpha))); j += (3+alpha)) {
		ad.rgb[k + 0] = buff[y + j + 0];
		ad.rgb[k + 1] = buff[y + j + 1];
		ad.rgb[k + 2] = buff[y + j + 2];
		k += 3;
	    }
	}

	gdk_pixbuf_unref(water);
	if (xmax != XMAX || ymax != YMAX)
	    gdk_pixbuf_unref(water2);

    }

    else {
	/* draw the water . deep water gradient */

	d = 0x55 / (float) ymax;
	for (i = 0; i < ymax; i++) {
	    for (j = 0; j < xmax; j++) {
		int t = (i * xmax * 3) + j * 3;
		if (mys.solidcolour) {
		    ad.rgb[t + 0] = (unsigned char)mys.r;
		    ad.rgb[t + 1] = (unsigned char)mys.g;
		    ad.rgb[t + 2] = (unsigned char)mys.b;

		}
		if (mys.shadedblue) {
		    ad.rgb[t] = 0x00;
		    ad.rgb[t + 1] = 0x55 - (int) ((float) i * d);
		    ad.rgb[t + 2] = 0xff - (int) (2 * (float) i * d);
		}
	    }
	}
    }

#ifdef FULLSCREEN
    if(have_sea_floor!=0 && full_screen!=0)
      make_bottom();
#endif


    /* copy it to the "frequent use" buffer */
    memcpy(ad.bgr, ad.rgb, xmax * ymax * 3);



}





int sel_fish(int j)
{
  /* This is ugly! */

  if(j<mys.fish1)
    return FISH1;
  else
    j-=mys.fish1;
  
  if(j<mys.fish2)
    return FISH2;
  else
    j-=mys.fish2;
  
  if(j<mys.fish3)
    return FISH3;
  else
    j-=mys.fish3;

  if(j<mys.fish4)
    return FISH4;
  else
    j-=mys.fish4;

  if(j<mys.fish5)
    return FISH5;
  else
    j-=mys.fish5;

  if(j<mys.fish6)
    return FISH6;
  else
    j-=mys.fish6;


  if(j<mys.swordfish)
    return SWORDFISH;
  else
    j-=mys.swordfish;


  if(j<mys.blowfish)
    return BLOWFISH;
  else
    j-=mys.blowfish;


  if(j<mys.fillmore)
    return FILLMORE;
  else
    j-=mys.fillmore;


  if(j<mys.sherman)
    return SHERMAN;
  else
    j-=mys.sherman;


  if(j<mys.prey)
    return PREY;
  else
    j-=mys.prey;


  if(j<mys.hunter)
    return HUNTER;
  else
    j-=mys.hunter;


  if(j<mys.lori)
    return LORI;
  else
    j-=mys.lori;


  if(j<mys.ernest)
    return ERNEST;
  else
    j-=mys.ernest;


  if(j<mys.squid)
    return SQUID;
  else
    j-=mys.squid;

  /* This is still ok */
  if(j==0) return 0;
  
  /* This should never happend */
  printf("Error choose fish! - %d\n",j);
  return 0; /* Returning -1 would make it even worser */

}

void prepare_graphics(void)
{
    int i = 0;
    int type, special_fish=0;

#ifdef LEDS
    /* Prepare Leds */
    if (mys.leds) {
      if(numlock.image==NULL)
	numlock.image = load_image("aquarium/numlock.png",
				    &numlock.width, &numlock.height,
				   (int *)&numlock.pixbuf, 1);
      if(scrollock.image==NULL)
	scrollock.image = load_image("aquarium/scrollock.png",
				     &scrollock.width, &scrollock.height,
				     (int *)&scrollock.pixbuf,1);
      if(capslock.image==NULL)
	capslock.image =
	    load_image("aquarium/capslock.png", &capslock.width,
		       &capslock.height, (int *)&capslock.pixbuf, 1);
    }
#endif

#ifdef CLOCK
    if (mys.clock)
      if(dig.image==NULL)
	dig.image = load_image("aquarium/dig.png", &dig.width, &dig.height, 
			       (int *)&dig.pixbuf,10);
#endif


#ifdef THERMOMETER
    if (mys.thermo)
      if(thermometer.image==NULL)
	thermometer.image = load_image("aquarium/thermometer.png",
					&thermometer.width,
					&thermometer.height, 
				        (int *)&thermometer.pixbuf,1);
#endif


#ifdef FULLSCREEN
    if(have_sea_floor){
      if(sea_floor.image==NULL)
	sea_floor.image= load_image_n_scale("aquarium/sherman/bottom.png",
					    &sea_floor.width,
					    &sea_floor.height, 
					    (int *)&sea_floor.pixbuf,1,mys.scale);
      start_sea_floor=rand() % sea_floor.width;
    }
#endif
    


    /* Load bubbles */
    if(bubbles_image_data.image==NULL)
      bubbles_image_data.image = load_image("aquarium/bubbles.png",
					    &bubbles_image_data.width, 
					    &bubbles_image_data.height,
					    (int *)&bubbles_image_data.pixbuf, 
					    BUBBLES_FRAMES);


    memset(fish_buffer, 0, sizeof(Fish_buffer) * NUMOFFISHTYPES);


    for (i = 0; i < mys.fishes; i++) {

      if(!mys.choose) {
#ifdef FULLSCREEN
	if(i<2 && (rand()%10)<6 && have_sea_floor!=0 && full_screen!=0) {
	  if(rand()%2) type = BDWELLER;
	  else type = HAWTHORNE;
	}
	else
#endif
	  do{
	    type = (rand() % NUMOFFISHTYPES); 
	  } while(type==BDWELLER || type==HAWTHORNE);
      }
      else{
	type=sel_fish(i);
      }


      /*      if(i==0)
       	type = SWORDFISH; 
      else
	type = BLOWFISH;
      */

	/*
	type = HUNTER;
      else
	type = PREY;
	*/

      if(type == PREY) special_fish|=1;
      if(type == HUNTER) special_fish|=2;
      if(type == SWORDFISH) special_fish|=4;
      if(type == BLOWFISH) special_fish|=8;





	if (fish_buffer[type].image == NULL) {
	    fish_buffer[type].image =
		load_image_n_scale(fish_animation[type].file, &ad.fishes[i].width,
				   &ad.fishes[i].height, (int *)&fish_buffer[type].pixbuf,
				   fish_animation[type].pics,
				   mys.scale);
	    fish_buffer[type].width = ad.fishes[i].width;
	    fish_buffer[type].height = ad.fishes[i].height;
	} else {
	    ad.fishes[i].width = fish_buffer[type].width;
	    ad.fishes[i].height = fish_buffer[type].height;
	}

	ad.fishes[i].pic = fish_buffer[type].image;


	ad.fishes[i].num_animation = fish_animation[type].frames;
	ad.fishes[i].animation = fish_animation[type].animation;

	ad.fishes[i].speed = fish_animation[type].speed;

	ad.fishes[i].type = type;
	ad.fishes[i].frame = rand() % ad.fishes[i].num_animation;

	ad.fishes[i].rev = (i % 2);
	ad.fishes[i].tx = rand() % (virtual_aquarium_x - ad.fishes[i].width);

	ad.fishes[i].speed_mul = 1;
	ad.fishes[i].fast_frame_change=1.0;

	ad.fishes[i].updown = 0;
	ad.fishes[i].travel =
	    (rand() % ((virtual_aquarium_x - xmax) / 2)) + ad.fishes[i].width;
	ad.fishes[i].y = rand() % (virtual_aquarium_y - ad.fishes[i].height);

    }

    special_action=0;

    
    if((special_fish&3)==3 && mys.eat==1){
      prey_hunter_data.image =
		load_image_n_scale("aquarium/sherman/eating.png", 
				   &prey_hunter_data.width,
				   &prey_hunter_data.height, 
				   (int *)&prey_hunter_data.pixbuf,
				   7,mys.scale);

      special_action=1;
    }

    if((special_fish&12)==12 && mys.explode==1){
      blowup_data.image =
		load_image_n_scale("aquarium/sherman/blowup.png", 
				   &blowup_data.width,
				   &blowup_data.height, 
				   (int *)&blowup_data.pixbuf,
				   7, mys.scale);

      special_action=2;
    }



    /*    
    printf("\nI've selected those ad.fishes for your aquvarium: ");

    for (i = 0; i < mys.fishes; i++) {
	printf("%d", ad.fishes[i].type + 1);

	if ((i + 1) != mys.fishes)
	    printf(", ");
	else
	    printf("\n");
    }
    */

}


/*

-h,--help		Display help
-v,--version		Display version
-c,--noclock		Turn off clock
-t,--nothermometer	Turn off thermometer
-b X,			1,2,3 backgrounds
-n X, --numad.fishes X	X ad.fishes on the screen.
-u X, --numbubbles X	X bubbles

*/

/* Used letters:


abcdefhiklmnoprstuvwy
K

*/

/* Lgg till: quiet */

void do_help(void)
{
    printf("\nSherman's aquarium v%s\nUsage: sherman [options]\n\n"
	   " NOTICE!! All those options, except help,info and version, are only for use with\n"
	   " the Windowmaker applet variant! With Gnome, use the Preferences menu instead.\n\n"
	   " --help\t\t\tShow this message and exits.\n"
	   " -v\t--version\tShow version and exits.\n"
	   " -i\t--info\t\tShow compile info and exits.\n"
	   " -t\t--nothermometer\tRemoves the thermometer.\n"
	   " -c\t--noclock\tRemoves the digital clock.\n"
	   " -k\t--nokeyleds\tRemoves the keyboard leds indicator.\n"
	   " -d\t--datadir\tAlternativ datadir. Overrides default directory\n"
	   "\t\t\tand the enviroment variable SHERMANS_AQUARIUM_DATA_DIR.\n"
	   " -b #\t--background #\tSelect background:\n"
	   "\t\t\t1 - Solid blue\n"
	   "\t\t\t2 - Shaded blue\n"
	   "\t\t\t3 - Waterlike (default)\n"
	   " -bf <file>\t\t--backgroundfile <file>\tSelect your own background.\n"
	   " -n #\t--numfish\tChoose the number of fish you want in your aquarium.\n"
	   "\t\t\tFive is default.\n"
#ifdef MATRIX
	   " -m\t--matrix\tShow matrix scroller when mouse pointer is over. Default off.\n"
#endif
	   " -u #\t--numbubbles\tChoose the number of bubbles. Default is ten.\n"
	   " -x #\t\t\tX position on the screen.\n"
	   " -y #\t\t\tY position on the screen.\n\t\t\t-x and -y must be given together.\n"
#ifdef GNOME
	   " -wm\t--startwm\tStart WindowMaker applet instead of Gnome applet.\n\t\t\t(Gnome applet is default if the panel is running.)\n"
#endif
	   " -s\t--scale\t\tScale the fish. (100 is scale 1:1. 50 is half size.)\n"
	   " -h #\t--height #\tChoose the window height. Default is 64 for wm.\n"
	   " -w #\t--width #\tChoose the window width. Default is 64 for wm.\n"
	   " -fe\t\t\tForce Enlightenment settings.\n\t\t\t(Act like a normal program)\n"
	   
#ifdef FULLSCREEN
	   " -window-id\t\tRun Sherman's aquarium in a special window\n"
	   " -l\t--fullscreen\tStart in fullscreen instead of as an applet.\n"
	   " -K\t--keep-resolution\n\t\t\tKeeps the current resolution instead of switching.\n"
	   " -e\t--escape-only\tFullscreen quits only when the escape key is pressed.\n"
	   " --xscreensaver\t\tFor xscreensaver only.DO NOT USE THIS SWITCH OTHERWISE!!\n"
	   " -f\t--seafloor\tAdd a sea floor on the background in fullscreen mode.\n"
#endif
#ifdef GNOME
	   " -p\t--preferences\tStarts the preferences dialog. The preferences will be\n"
	   "\t\t\tsaved and used next time Sherman's aquarium is started.\n"
	   " -r\t\t\tReset settings. If Sherman's aquarium\n\t\t\tcrashes at startup, "
	   "this might help.\n"
#endif
	   "\nThis is a applet that creates an aquarium with randomly selected fish.\n",
	   VERSION);
}


void do_version(void)
{
    printf("%s\n", VERSION);
}

void do_info(void)
{
    int i;

    printf("\nSherman's aquarium version %s\n\n%s%s%s", VERSION, cominfo,
	   libinfo, gccinfo);
#ifdef GNOME
    printf(" - with gnome support.\n");
#else
    printf(" - without Gnome support.\n");
#endif

#ifdef LEDS
    printf(" - with keyboard led support.\n");
#else
    printf(" - without keyboard led support.\n");
#endif

#ifdef THERMOMETER
    printf(" - with thermometer.\n");
#else
    printf(" - without thermometer.\n");
#endif

#ifdef CLOCK
    printf(" - with clock.\n");
#else
    printf(" - without clock.\n");
#endif
#ifdef FULLSCREEN
    printf(" - with fullscreen support.\n");
#else
    printf(" - without fullscreen support.\n");
#endif
#ifdef MATRIX
    printf(" - with matrix code.\n");
#endif
    

    i = detect_brokenwm();
    if (i > 0)
	printf("It seems like you are running ");
    if (i == 1)
	printf("enlightenment\n");
    if (i == 2)
	printf("sawfish\n");
    if (i == 3)
	printf("twm\n");
    if (i == 4)
      printf("fvwm\n");
    if(i ==5)
      printf("fvwm2\n");
    if (i == 0)
	printf
	    ("You are running some wm that should handle wmapplets ok.\n");
#ifdef GNOME
#ifndef GNOME2
    if (!detect_panel())
	printf("You are using the gnome v1.x panel.\n");
    else
#endif
	printf("You are not using the gnome v1.x panel.\n");
#endif
    printf("\n");
    printf("Datadir:%s\n", datadir);
    altdatadir = getenv("SHERMANS_AQUARIUM_DATADIR");
    if (altdatadir != NULL)
	printf("Alternative directory is given and will be used:%s\n",
	       altdatadir);


}

#ifdef FULLSCREEN
/* Converts a hex string to an interger. libc function exists? */
int htoi(char *str)
{
  int i,sum=0,d;

  if(str[0]!='0' || str[1]!='x') return -1;
  for(i=2;i<strlen(str);i++){
    d=0;
    if(str[i]>='0' && str[i]<='9') d=(int)(str[i]-'0');
    if(str[i]>='A' && str[i]<='F') d=(int)(str[i]-'A'+10);
    if(str[i]>='a' && str[i]<='f') d=(int)(str[i]-'a'+10);

    sum+=d;
    sum=sum<<4;
  }

  return(sum>>4);
}
#endif

void parse_options(int argc, char *argv[])
{
    int c = 0,i;
    FILE *btest;

    struct option long_options[] = {
	{"help", no_argument, NULL, 1},
	{"version", no_argument, NULL, 2},
	{"v", no_argument, NULL, 2},
#ifdef CLOCK
	{"noclock", no_argument, NULL, 3},
	{"c", no_argument, NULL, 3},
#endif
#ifdef THERMOMETER
	{"nothermometer", no_argument, NULL, 4},
	{"t", no_argument, NULL, 4},
#endif
	{"background", required_argument, NULL, 5},
	{"b", required_argument, NULL, 5},
	{"numfish", required_argument, NULL, 6},
	{"n", required_argument, NULL, 6},
	{"numbubbles", required_argument, NULL, 7},
	{"u", required_argument, NULL, 7},
#ifdef LEDS
	{"nokeyleds", no_argument, NULL, 8},
	{"k", no_argument, NULL, 8},
#endif
#ifdef GNOME
	{"startwm", no_argument, &start_wm, 9},
	{"wm", no_argument, &start_wm, 9},
#endif
	{"scale", required_argument, NULL, 10},
	{"s", required_argument, NULL, 10},
	{"bf", required_argument, NULL, 11},
	{"backgroundfile", required_argument, NULL, 11},
	{"x", required_argument, NULL, 12},
	{"y", required_argument, NULL, 13},
	{"w", required_argument, NULL, 14},
	{"h", required_argument, NULL, 15},
	{"i", no_argument, NULL, 16},
	{"info", no_argument, NULL, 16},
	{"d", required_argument, NULL, 17},
	{"datadir", required_argument, NULL, 17},
	{"fe", no_argument, &force_es, 18},
	{"p", no_argument, NULL, 19},
	{"preferences", no_argument,NULL, 19},
#ifdef GNOME
	/* This is gnome commands - No idea what they do */
	{"activate-goad-server=shermans_aquarium", no_argument, &goad_server,20},
	{"goad-fd", no_argument, NULL, 21},
	{"r",no_argument,NULL,22},
#endif
#ifdef FULLSCREEN
	{"l",no_argument, &full_screen,23},
	{"fullscreen",no_argument, &full_screen,23},
	{"K",no_argument, &keep_resolution,24},
	{"keep-resolution",no_argument, &keep_resolution,24},
	{"e",no_argument, &escape_only,25},
	{"escape-only",no_argument, &escape_only,25},
	{"window-id",required_argument,NULL,26},
	{"xscreensaver",no_argument,&xscreensaver,27},
	{"seafloor",no_argument,&have_sea_floor,28},
	{"f",no_argument,&have_sea_floor,28},
#endif
#ifdef MATRIX
	{"m",no_argument, &mys.matrix,29},
	{"matrix",no_argument, &mys.matrix, 29},
#endif

	{0, 0, 0, 0}
    };


    while ((c =
	    getopt_long_only(argc, argv, "", long_options, NULL)) != -1) {
	switch (c) {
	case 1:
	    do_help();
	    exit(0);
	    break;
	case 2:
	    do_version();
	    exit(0);
	    break;
	case 3:
	  mys.clock=0;
	  break;
	case 4:
	  mys.thermo=0;
	  break;

	case 5:
	    i = atoi(optarg);
	    if (i > 3 || i<1) {
		printf("Valid range for background selection is 1-3\n");
		exit(1);
	    }
	    mys.solidcolour=mys.shadedblue=mys.wateralike=mys.wateralike=0;

	    if(i==1) mys.solidcolour=1;
	    if(i==2) mys.shadedblue=1;
	    if(i==3) mys.wateralike=1;

	    break;
	case 6:
	    mys.fishes = atoi(optarg);
	    if(mys.fishes>MAX_FISHES){
		printf("Maximum %d fish are allowed. Increase the defintion \nMAX_FISHES and recompile if you want more.\n",MAX_FISHES);
		exit(1);
	    }
	    break;

	case 7:
	    mys.bubbles = atoi(optarg);
	    break;
	case 8:
	  mys.leds=0;
	  break;
	case 10:
	    mys.scale = atoi(optarg);
	    break;
	case 11:
	  btest = fopen(optarg,"r");
	  if(!btest) printf("\nCan't load background file: %s\n",optarg);
	  else{
	    fclose(btest);
	    mys.imagename=g_malloc0(strlen(optarg)+10);
	    g_snprintf(mys.imagename,strlen(optarg)+8,"%s",optarg);
	    mys.solidcolour=mys.shadedblue=mys.wateralike=0;
	    mys.yourown=1;
	  }

	  break;
	case 12:
	    posx = atoi(optarg);
	    break;

	case 13:
	    posy = atoi(optarg);
	    break;


	case 14:
	    windowsize_x = atoi(optarg);
	    break;

	case 15:
	    windowsize_y = atoi(optarg);
	    break;
	case 16:
	    do_info();
	    exit(0);
	    break;
	case 17:
	  altdatadir = g_malloc0(strlen(optarg) + 10);
	  strcpy(altdatadir, optarg);
	  use_altdatadir = 1;
	  break;
#ifdef GNOME
	case 19:
	  /* Send only with the name of the program file, nothing more */
	  gnome_init ("Sherman's aquarium - Preferences", VERSION, 1, argv);
	  /* Tell the preference dialog, it's not from the gnome applet */
	  create_preferences(0);
	  gtk_main();
	  gtk_exit(0);
	  break;
#else
	case 19:
	  printf("You have not compiled Sherman's aquarium with gnome support. The preferences window\n"
		 "requires that. You can still use Sherman's aquarium as a windowmaker applet if you compile\n"
		 "it with gnome support.\n");
	  exit(-1);
	  break;
#endif
#ifdef GNOME
	case 22:
	  printf("Resetting all settings..");
	  fflush(NULL);
	  reset_settings("shermans_aquarium/");
	  printf("done\n");
	  exit(0);
	  break;
#endif
#ifndef FULLSCREEN
	case 23:
	  printf("You have not compiled Sherman's aquarium with fullscreen support.Make sure that\n"
		 "you have libSDL >=1.2.0 installed(www.libsdl.org) and then reconfigure, remake and\n"
		 "reinstall Sherman's aquarium, if you want to have fullscreensupport.\n");

#endif

#ifdef FULLSCREEN
	case 26:
	  window_id=htoi(optarg);
	  break;
#endif
	}

    }

#ifdef FULLSCREEN
    if(xscreensaver!=0){
      keep_resolution=1;
      full_screen=1;
    }
#endif
    if ((posy == -1 && posx != -1) || (posy != -1 && posy == -1)) {
	printf("\nYou must give both -x and -y. Not one alone!\n");
	/*0do_help();*/
	exit(-1);
    }
    if (mys.scale != 100) 
	printf("\nLoading and scaling images");
    else 
	printf("\nLoading images");
    fflush(NULL);

}


int aquarium_update(gpointer data)
{

#ifdef DEBUG
  printf(" - Entering aquarium_update()\n");
#endif

#ifdef FULLSCREEN
  if(!full_screen)
#endif
    memcpy(ad.rgb, ad.bgr, ymax * xmax * 3);


#ifdef MATRIX
#ifdef FULLSCREEN
  if(mys.matrix && !full_screen){
#else
    if(mys.matrix){
#endif
      if(proximity){
	if(fade==255){
	  init_matrix_2();
	  fade-=5;
	}
	else{
	  if(fade<255 && fade>0)
	    fade-=5;
	} 
      }
      else{
	
	if(fade==255)
	  free_matrix();
	else
	  fade+=5;
      }
    }

#endif

#ifdef DEBUG
  printf("\t - Calling fish_update()\n");
#endif

    fish_update();

#ifdef DEBUG
  printf("\t - Calling bubble_update()\n");
#endif
 
  bubble_update();
 

#ifdef DEBUG
  printf("\t - Calling thermometer_update\n");
#endif

#ifdef FULLSCREEN
  if(!full_screen)
#endif
    thermometer_update();

#ifdef DEBUG
  printf("\t - Calling leds_update()\n");
#endif
#ifdef FULLSCREEN
  if(!full_screen)
#endif
    leds_update();

#ifdef DEBUG
  printf("\t - Calling clock_update()\n");
#endif
#ifdef FULLSCREEN
  if(!full_screen)
#endif
    clock_update();

#ifdef DEBUG
  printf("\t - Calling update_screen()\n");
#endif


#ifdef MATRIX
#ifdef FULLSCREEN
  if(mys.matrix && !full_screen) 
#else
  if(mys.matrix) 
#endif
    if(fade<255){
      update_matrix();
      draw_pic_alpha(matrix_screen, xmax, ymax, 0, 0, 0, fade);
    }
#endif

#ifdef FULLSCREEN
  if(!full_screen)
#endif
    update_screen();

#ifdef DEBUG
    printf("\t - Leaving aquarium_update()\n");
#endif

    return 1;

}
  

#ifdef GNOME

void free_all()
{
  int i;

  /* Freeing all fish images */
  for(i=0;i<NUMOFFISHTYPES;i++){
    if(fish_buffer[i].pixbuf!=NULL)
      gdk_pixbuf_unref(fish_buffer[i].pixbuf);
  }
  if(prey_hunter_data.image!=NULL)
    gdk_pixbuf_unref(prey_hunter_data.pixbuf);
  

  if(blowup_data.image!=NULL)
    gdk_pixbuf_unref(blowup_data.pixbuf);

  memset(&blowup_data,0,sizeof(Fish_buffer));
  memset(&prey_hunter_data,0,sizeof(Fish_buffer));
    
  g_free(ad.fishes);
  g_free(bubbles);


}

void reload_all()
{
  int size;

  /* This is not always alloced by us. releasing it here could be wrong! */

  /*  if(mys.imagename!=NULL)
    g_free(mys.imagename);
  */

  gnome_timer_off();

  free_all();


  mys = load_settings("shermans_aquarium/");

  /* Re initialize variables and load the new settings */
  

  ad.fishes=g_malloc0(sizeof(Fish)*mys.fishes);
  bubbles=g_malloc0(sizeof(Fish)*mys.bubbles);

#ifndef GNOME2
  size = applet_widget_get_panel_pixel_size(APPLET_WIDGET(ad.applet));
#else
  size = panel_applet_get_size(PANEL_APPLET(ad.applet));
#endif
  gnome_background(size, 0);

  prepare_graphics();



#ifdef MATRIX
    if(mys.matrix)
      init_matrix_1();
#endif


  gnome_timer_on();


}
#endif

/* This function (re)sets all the global variables to their default */
/* Some globals, don't have a default, those are not set ofcourse */

void init(int argc, char **argv)
{

  memset(&numlock,0,sizeof(Fish_buffer));
  memset(&scrollock,0,sizeof(Fish_buffer));
  memset(&capslock,0,sizeof(Fish_buffer));
  memset(&dig,0,sizeof(Fish_buffer));
  memset(&thermometer,0,sizeof(Fish_buffer));
  memset(&bubbles_image_data,0,sizeof(Fish_buffer));
  memset(&blowup_data,0,sizeof(Fish_buffer));
  memset(&prey_hunter_data,0,sizeof(Fish_buffer));
  /*  memset(&sea_texture,0,sizeof(Fish_buffer));*/


#ifdef FULLSCREEN
  keep_resolution=0;
  full_screen=0;
  escape_only=0;
  xscreensaver=0;
  window_id=0;
  have_sea_floor=0;
  start_sea_floor=0;
  memset(&sea_floor,0,sizeof(Fish_buffer));

#endif


#ifdef GNOME
  start_wm = 0;
#endif


#ifdef MATRIX
  fade = 255;
#endif




  ymax = YMAX;
  ymin = YMIN;
  xmin = XMIN;
  xmax = XMAX;
  virtual_aquarium_x = XMAX + 2 * VIRTUAL_AQUARIUM_DX;
  virtual_aquarium_y = YMAX + 2 * VIRTUAL_AQUARIUM_DY;
  windowsize_x = WINDOWSIZE_X;
  windowsize_y = WINDOWSIZE_Y;
  nr_bubbles = 0;
  goad_server = 0;
  force_es = 0;
  brokenwm = 0;
  bubble_state_change = 0;
  proximity = 0;
  posx=posy=-1;

  use_altdatadir = 0;
  special_action = 0;

#ifdef GNOME
#ifdef GNOME2
  gnome_init("Sherman's aquarium",VERSION,1,argv);
#endif
  mys = load_settings("shermans_aquarium/");

#else

  mys.eat=1;
  mys.explode=1;

#ifdef THERMOMETER
  mys.thermo=1;
#else
  mys.thermo=0;
#endif

#ifdef CLOCK
  mys.clock=1;
#else
  mys.clock=0;
#endif

  mys.leds=0;

  mys.doublewh=0;
  mys.matrix=0;
  mys.scale=DEFAULT_SCALE;
  mys.fishes=NRFISH;
  mys.bubbles=MAXBUBBLES;

  mys.choose=0;
  mys.fish1=1;
  mys.fish2=1;
  mys.fish3=1;
  mys.fish4=1;
  mys.fish5=1;
  mys.fish6=1;
  mys.fillmore=1;
  mys.sherman=1;
  mys.ernest=1;
  mys.lori=1;
  mys.hunter=1;
  mys.prey=1;
  mys.swordfish=1;
  mys.blowfish=1;
  mys.squid=1;
  mys.solidcolour=0;

    /* Default solid color */
  mys.r=0x00;
  mys.g=0x55;
  mys.b=0xff;

  mys.shadedblue=0;
  mys.wateralike=1;
  mys.yourown=0;
  mys.imagename=NULL;


#endif
}



int main(int argc, char **argv)
{

  /* Load saved settings and reset others */
  init(argc,argv);


  /* parse command line options */
  parse_options(argc, argv);

  srand(time(NULL)*getpid());

    /* zero main data structure */
  memset(&ad, 0, sizeof(ad));

    ad.fishes = g_malloc0(sizeof(Fish) * mys.fishes);
    bubbles = g_malloc0(sizeof(Bubble) * mys.bubbles);


    if (!use_altdatadir) {
	altdatadir = getenv("SHERMANS_AQUARIUM_DATA_DIR");
	if (altdatadir != NULL)
	    use_altdatadir = 1;
    }


#ifdef THERMOMETER
#ifdef FREEBSD
    init_cpu_load();
#endif
#endif


#ifdef GNOME2
    /* detects if gnome 2 panel is running us 1=yes, 0=no*/
    start_wm=!detect_gnome2_panel(argc,argv);
#endif


    /* create dockapp window. creates windows, allocates memory, etc */
#ifdef FULLSCREEN
    if(!full_screen){
#endif

#ifdef GNOME
      if (!start_wm) {
	/* Starts a gnome applet, if the panel is running. Else start a wm applet. */
	if (aquarium_make_new_applet_gnome(argc, argv)) {
	  start_wm = 1;
	  aquarium_make_new_applet_wm(argc, argv);
	}
      } else
	aquarium_make_new_applet_wm(argc, argv);
#else
    aquarium_make_new_applet_wm(argc, argv);
#endif
#ifdef FULLSCREEN
    }
    else{
      gtk_init(&argc, &argv);
      if(window_id==0){
	if(keep_resolution!=0)
	  fullscreen_init(gdk_screen_width(),gdk_screen_height());
	else
	  fullscreen_init(640,480);
      }
      else{

	full_screen=0;
	start_wm=1;
	mys.thermo=0;
	mys.clock=0;
	mys.leds=0;
	mys.matrix=0;
	aquarium_make_new_applet_wm_with_parent(window_id,argc,argv);
      }
    }
#endif

    /* Allocated memory for background and work area */
    ad.rgb = g_malloc0((xmax + 2) * (ymax + 2) * 3);
    ad.bgr = g_malloc0((xmax + 2) * (ymax + 2) * 3);

    bubble_state_change = (ymax / BUBBLES_FRAMES);

#ifdef MATRIX
    if(mys.matrix)
      init_matrix_1();
#endif



    prepare_graphics();


    prepare_backbuffer();


    /* All loading dots written, make new line. */
    printf("\n");



#ifdef FULLSCREEN
    if(!full_screen){
#endif

#ifdef GNOME
#ifndef GNOME2
      if (!start_wm) 
	aquarium_main_gnome();
      else
#endif
#endif
       aquarium_main_wm();

#ifdef FULLSCREEN
    } else{
      if(window_id==0)
	fullscreen_main();
      else
	aquarium_main_wm();
    }
#endif


    return 0;
}


AquariumData *get_ad(void)
{
  return &ad;
}

