(* $Id: shell_sys.mli,v 1.1 2000/12/10 15:12:54 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)

(* System abstractions for Shell package *)

(**********************************************************************)
(***                       environments                             ***)
(**********************************************************************)

type environment

val create_env  : unit -> environment
val copy_env    : environment -> environment
val set_env     : environment -> string array -> unit
val get_env     : environment -> string array
val iter_env    : f:(string -> unit) -> environment-> unit
val iter_env_vars : f:(string -> string -> unit) -> environment -> unit
val set_env_var : environment -> string -> string -> unit
val get_env_var : environment -> string -> string
val current_env : unit -> environment


(**********************************************************************)
(***                    commands and processes                      ***)
(**********************************************************************)

exception Fatal_error of exn
  (* An error is fatal if it is not possible to recover from it in a
   * predictable manner
   *)

type command
type process

(* A _command_ is the description how to start a new process. A
 * _process_ is the running instance of a command; the same command
 * may be started several times.
 *)

val command :
      ?cmdname:string ->                   (* default: derived from filename *)
      ?arguments:(string array) ->         (* default: empty *)
      ?environment:environment ->          (* default: current environment *)
      ?descriptors:(Unix.file_descr list) -> 
                                           (* default: stdin, stdout, stderr *)
      ?assignments:((Unix.file_descr * Unix.file_descr) list) ->    
                                           (* default: empty *)
      filename:string ->
      unit ->
	command

(* cmdname: argv[0].
 * descriptors: the file descriptors which are open when the subprocess starts
 *)

exception Executable_not_found of string;;

val lookup_executable :
      ?path:(string list) ->     (* default: use the PATH variable *)
      string ->
	string

val get_cmdname     : command -> string
val get_arguments   : command -> string array
val get_environment : command -> environment
val get_descriptors : command -> Unix.file_descr list
val get_assignments : command -> (Unix.file_descr * Unix.file_descr) list
val get_filename    : command -> string

val set_cmdname     : command -> string          -> unit
val set_arguments   : command -> string array    -> unit
val set_environment : command -> environment     -> unit
val set_descriptors : command -> Unix.file_descr list -> unit
val set_assignments : command -> (Unix.file_descr * Unix.file_descr) list -> unit
val set_filename    : command -> string          -> unit

val copy_command : command -> command
  (* Returns a duplicate of the command description *)

val is_executable : command -> bool
  (* Returns 'true' if there is an executable file for the command, and
   * it is permitted to run this file.
   * 'false' means that the command can definitely not be executed. However,
   * even if the function returns 'true' there may be still reasons that
   * execution will fail.
   *)

type group_action =
    New_bg_group
  | New_fg_group
  | Join_group of int
  | Current_group

val run :
      ?group:group_action ->       (* default: Current_group *)
      ?pipe_assignments:((Unix.file_descr * Unix.file_descr) list) ->
                                   (* default: [] *)
      command ->
	process
  (* Executes the command concurrently with the current process. The function
   * does not wait until the process terminates; it returns immediately after
   * the exec system call has been successfully performed (yes, I said "exec"
   * and not "fork").
   *
   * On error, an exception is raised. In this case the caller can assume that
   * the process state has been cleaned up: any new child process has terminated;
   * any modifications of the global process state has been restored. Note that
   * a special mechanism makes it possible to also get exceptions raised in the
   * subprocess until the exec is successful.
   *
   * File descriptor assignments: First, the assignments in ~pipe_assignments
   * are performed, then the assignments contained in the command. This means
   * that for every pair (fd, fd') the descriptor fd is duplicated to fd'
   * (dup2). The assignments contained in ~pipe_assignments are executed first,
   * and they are done in a parallel way; the assignments contained in the
   * command are appended and performed in a sequential way.
   * [Note: ~pipe_assignments are used internally by run_group to connect
   * the processes through pipelines.] 
   *
   * Note: For users without very special needs, it is recommended to run
   * jobs instead of processes. See below for the job API.
   *)

val process_id : process -> int

val status : process -> Unix.process_status
  (* If the process has terminated, the status of the
   * process is returned.
   * If the process is still running, Not_found is raised.
   *
   * Note: 'status' does not check whether the process status changes; it
   * only reports the change associated with the last event. (See "wait" 
   * below.)
   *)

val command_of_process : process -> command

type process_event =
    File_read of Unix.file_descr
  | File_write of Unix.file_descr
  | File_except of Unix.file_descr
  | Process_event of process
      (* means: "get_status process" has changed *)
  | Signal

val wait : 
      ?wuntraced:bool ->                   (* default: false *)
      ?restart:bool ->                     (* default: false *)
      ?check_interval:float ->             (* default: 0.1 *)
      ?read:(Unix.file_descr list) ->      (* default: [] *)
      ?write:(Unix.file_descr list) ->     (* default: [] *)
      ?except:(Unix.file_descr list) ->    (* default: [] *)
      process list ->
        process_event list
 
(* wait: Watches the given list of processes and file descriptors and waits
 * until something happens. 
 * The function returns immediately with [] if it is no longer possible
 * that any event can happen.
 *
 * Precondition: The passed file descriptors must be open.
 *
 * Possible events:
 * - A process of the list terminates (regularly or because of a signal)
 * - ~wuntraced:true, and a process of the list is stopped
 * - A file descriptor of ~read is readable
 * - A file descriptor of ~write is writable
 * - A file descriptor of ~except has out-of-band data
 * 
 * Notes:
 * - The list of processes may contain terminated processes (that no longer
 *   exist) as long as the process status has already been recorded by a
 *   previous "wait" invocation.
 * - "wait" does not restart automatically if a signal happens. In this
 *   case, the exception Unix_error(EINTR,_,_) will be raised.
 * - "wait" does not use SIGCHLD itself.
 * - If a process causes both process and descriptor events at the same time, 
 *   it is not specified which events are reported first.
 * - File descriptor events have precedence over process events; the latter
 *   may be reported with some delay (see next note how to avoid this).
 *   The maximum delay is ~check_interval seconds.
 * - It is possible to install a SIGCHLD handler which does nothing 
 *   (Signal_handler (fun () -> ())). "wait" will then be interrupted on
 *   process events (EINTR error), and if you restart "wait" in the case
 *   EINTR happens, "wait" will report process events as soon as possible.
 *   (Caution: The SIGCHLD handler must have been installed before any of the
 *   watched processes are started!)
 * - By default, waiting is not restarted on EINTR. However, if you pass
 *   ~restart:true, "wait" will catch the EINTR condition and restart.
 *)

val call : command -> process
  (* Executes the command and waits until the process terminates
   * (synchronous execution a la "system", but no intermediate shell).
   * get_status is guaranteed to return WEXITED or WSIGNALED.
   *)

val kill :
      ?signal:int ->       (* default: SIGTERM *)
      process ->
        unit



(**********************************************************************)
(***                 system event handler type                      ***)
(**********************************************************************)

(* The types and functions in this section are only for very special
 * needs
 *)

type system_handler =
    { sys_register :
        ?wuntraced:bool ->                   (* default: false *)
	?check_interval:float ->             (* default: 0.1 *)
	?read:(Unix.file_descr list) ->      (* default: [] *)
	?write:(Unix.file_descr list) ->     (* default: [] *)
        ?except:(Unix.file_descr list) ->    (* default: [] *)
	process list ->
        (process_event list -> unit) ->   (* callback for events *)
	  unit;

      sys_wait :
	unit -> unit;
    }

(* sys_register: By calling this function a callback function for the specified
 *   events is registered. See [wait] above for an example.
 *
 * sys_wait: By calling this function the events are delivered to the
 *   registered callback. If exceptions are raised in the callback function
 *   these will not be caught, so the caller of sys_wait() will get them.
 *   It must be possible to restart sys_wait in this case.
 *
 *   The callback function can change the list of interesting events by
 *   calling sys_register again.
 *
 *   If effectively no events are interesting (sys_register is called without
 *   file descriptors and no running process) the callback function is called
 *   with an empty process_event list once. If it does not register a new
 *   callback, the event loop will stop, and sys_wait will return normally.
 *
 *)


(**********************************************************************)
(***                            jobs                                ***)
(**********************************************************************)

type job
type job_instance

  (* A job is the description of several commands which are
   * linked by pipelines (or which are just a logical unit). A job_instance
   * is the running instance of a job.
   *
   * Jobs are implemented on a higher layer than commands; the
   * following means of the operating system are used by job
   * invocations:
   * - Normally a job_instance corresponds to a Unix process group. In
   *   this case the last added command will result in the process group
   *   leader.
   * - Controlling the execution of jobs requires that signal
   *   handlers are set in many cases (see install_job_handlers)
   * - The processes of jobs are often interconnected by pipelines
   *   (see add_pipeline).
   * - It is possible to handle pipelines between the current process and
   *   processes of the job (see add_producer and add_consumer)
   *)

  (* IMPORTANT:
   * 
   * In order to run jobs efficiently (without busy waiting) and properly
   * it is strongly recommended to install the signal handlers using
   * install_job_handlers
   *)  


val new_job : unit -> job
  (* Creates a new job descriptor. Initially the job is empty, but you can
   * fill it with commands (add_command), pipelines (add_pipeline), consumers
   * (add_consumer) and producers (add_producer).
   * When the job is set up, you can start it (run_job/finish_job or
   * call_job).
   *)

val add_command : command -> job -> unit
  (* Adds a command to a job. 
   *
   * Note that you cannot add the same command twice; however you can
   * add a copy of a command already belonging to the job.
   *)


val add_pipeline :
      ?bidirectional:bool ->           (* default: false *)
      ?src_descr:Unix.file_descr ->    (* default: stdout *)
      ?dest_descr:Unix.file_descr ->   (* default: stdin *)
      src:command ->
      dest:command ->
      job ->
        unit
  (* Adds a pipeline which redirects the output of the command ~src to the
   * input of the command ~dest.
   * 
   * ~src_descr: determines the file descriptor of the source command which
   * is redirected.
   *
   * ~dest_descr: determines the file descriptor of the destination command
   * to which the data stream is sent
   *
   * ~bidirectional: if false, a classical pipe is created. If true, a
   * socketpair is created which is roughly a bidirectional pipe. (Note:
   * on some systems, there is essentially no difference as all pipes are
   * bidirectional.)
   *)


val add_producer :
      ?descr:Unix.file_descr ->     (* default: stdin *)
      producer:(Unix.file_descr -> bool) ->
      command ->
      job ->
        unit
  (* producer: this function is called if the passed descriptor is ready
   * for output. The function may write to the descriptor, and/or it may
   * close the descriptor. It must return true iff the descriptor is still
   * open.
   * The passed descriptor is in non-blocking mode.
   *)

val from_string :
      ?pos:int ->                  (* default: 0 *)
      ?len:int ->                  (* default: until end of string *)
      ?epipe:(unit -> unit) ->     (* default: empty function *)
      string ->
        (Unix.file_descr -> bool)
  (* Returns a function which can be used as ~producer argument and which
   * takes its material from a string (or, if ~pos or ~len are present,
   * from the specified substring).
   * If the pipeline crashes, the function ~epipe is called, and the 
   * descriptor is closed.
   *)

val from_stream :
      ?epipe:(unit -> unit) ->     (* default: empty function *)
      string Stream.t ->
        (Unix.file_descr -> bool)
  (* Returns a function which can be used as ~producer argument and which
   * takes its material from a stream of strings.
   * If the pipeline crashes, the function ~epipe is called, and the 
   * descriptor is closed.
   *)

val add_consumer : 
      ?descr:Unix.file_descr ->     (* default: stdout *)
      consumer:(Unix.file_descr -> bool) ->
      command ->
      job ->
        unit
  (* consumer: this function is called if data have arrived at the passed
   * descriptor, or if the end of file has just been reached. The function must
   * try to read from the descriptor. It must return true iff the eof has
   * not yet been reached.
   * The passed descriptor is in non-blocking mode.
   *)

val to_buffer :
      Buffer.t ->
        (Unix.file_descr -> bool)
  (* Returns a function which can be passed as ~consumer argument.
   * This function will collect the consumed material in the passed
   * buffer.
   *)


type group_mode = Same_as_caller | Foreground | Background
 
val run_job :
      ?mode:group_mode ->                (* default: Same_as_caller *)
      ?forward_signals:bool ->           (* default: true *)
      job ->
        job_instance

  (* Invokes the commands of the job such that they run concurrently
   * with the main process.
   *
   * The function returns a job_instance, i.e. a descriptor of the running
   * processes. Furthermore, the function has the side effect of adding the
   * job to the list of current jobs.
   *
   * ~mode:
   *   - Same_as_caller: the new processes belong to the same process group
   *     as the calling (current) process
   *   - Background: the new processes are started in a new background
   *     process group
   *   - Foreground: the new processes are started in a new foreground
   *     process group
   *
   * ~forward_signals: background process groups have a disadvantage:
   * terminal signals (e.g. CTRL-C) are not sent to such groups. However,
   * it is possible to catch such signals and forward them to the background
   * group. If ~forward_signals is 'true' AND ~mode is Background AND
   * the function install_job_handlers has been invoked, the signals
   * SIGINT and SIGQUIT are delivered to the background
   * group as well. See the function 
   * install_job_handlers for details and other effects.
   * 
   * The function returns normally if at least one process could be started.
   * If no process was startable (i.e. the first command was not startable), 
   * an exception is raised. If one or more processes could be started but
   * not all, job_status will return Job_partially_running. The caller 
   * should then discard the job and any intermediate result that might
   * already have been produced by the partial job.
   *
   * When all processes could be started and no other exceptional condition
   * happened, the function sets job_status to Job_running.
   *)


val register_job :
      system_handler ->
      job_instance -> unit
  (* Registers the job at the passed system_handler. This is not necessary
   * if you directly call [finish_job].
   *)


val finish_job :
      ?sys:system_handler ->
      job_instance -> unit
  (* Waits until all of the processes of the job have terminated.
   * The function handles all producer/consumer events and calls the
   * producer/consumer functions as necessary.
   *
   * Exceptions raised by the producer/consumer functions are not caught.
   * In this case, finish_job is restartable.
   *
   * The ~sys argument determines the system_handler (standard_system_handler
   * by default). The job instance is registered at the system handler,
   * and it is waited until the job finishes. Roughly, finish_job
   * is equivalent to
   *   register_job sys jobinst;
   *   sys.sys_wait()
   *)

val call_job :
      ?mode:group_mode ->                     (* default: Same_as_caller *)
      ?forward_signals:bool ->                (* default: true *)
      ?onerror:(job_instance -> unit) ->      (* default: abandon_job *)
      job ->
        job_instance
  (* Starts the job (see run_job) and waits until it finishes (see finish_job);
   * i.e. call_job = run_job + finish_job.
   * The function returns normally if all processes can be started; you can
   * examine job_status of the result to get the information whether all
   * processes returned the exit code 0.
   *
   * If not all of the processes can be started, the function passed by
   * ~onerror is invoked. By default, this function calls abandon_job
   * to stop the already running
   * processes. After the ~onerror function has returned, the original 
   * exception is raised again.
   *
   * Fatal error conditions are not caught.
   *)

val processes : job_instance -> process list
  (* Returns the processes that have actually been started for this job
   * by run_job; note that the corresponding Unix process group
   * may have additional processes (e.g. indirectly started processes).
   *)

exception No_Unix_process_group;;

val process_group_leader : job_instance -> process
  (* Returns the process group leader process.
   * This function is not available for jobs in the mode Same_as_caller.
   *)

val process_group_id : job_instance -> int
  (* Returns the Unix ID of the process group as number > 1.
   * This function is not available for jobs in the mode Same_as_caller.
   *)

val process_group_expects_signals : job_instance -> bool
  (* 'true' iff the group has ~mode=Background and ~forward_signals. *)

type job_status =
    Job_running            (* All commands could be started, and at least
			    * one process is still running
			    *)
  | Job_partially_running  (* Not all commands could be started, and at least
			    * one process is still running
			    *)
  | Job_ok                 (* all processes terminated with exit code 0 *)
  | Job_error              (* all processes terminated but some abnormally *)
  | Job_abandoned          (* the job has been abandoned (see abandon_job) *)

val job_status : job_instance -> job_status
  (* Returns the status. The status may only change after finish_job
   * has been called:
   *
   * run_job ...        ==> status is Job_running or Job_partially_running
   * finish_job ...     ==> if returning normally: status is Job_ok or 
   *                        Job_error. After an exception happened the other
   *                        states are possible, too
   *)

val kill_process_group : 
      ?signal:int ->              (* default: SIGTERM *)
      job_instance -> unit
  (* Kills the process group if it is still (at least partially) running.
   * This operation is not available if the mode is Same_as_caller
   * (exception No_Unix_process_group).
   *
   * Note 1: In the Unix terminology, "killing a job" only means to send
   * a signal to the job. So the job may continue running, or it may
   * terminate; in general we do not know this. Because of this, the job
   * will still have an entry in the job list.
   *
   * Note 2: Because sub-sub-processes are also killed, this function may send
   * the signal to more processes than kill_processes (below). On the other
   * hand, it is possible that sub-processes change their group ID such that
   * it is also possible that this function sends the signal to fewer processes
   * than kill_processes.
   *)

val kill_processes : 
      ?signal:int ->              (* default: SIGTERM *)
      job_instance -> unit
  (* Kills the individual processes of the job which are still running.
   *)

val abandon_job :
      ?signal:int ->              (* default: SIGTERM *)
      job_instance -> unit
  (* Tries to get rid of a running job. If the mode is Same_as_caller, the
   * signal is sent to the processes individually. If the mode is
   * Foreground or Background, the signal is sent to the process group 
   * corresponding to the job.
   *
   * This function removes the job from the job list; i.e. it is no longer
   * watched. Because of some magic spells it is guaranteed that the job dies
   * immediately without becoming a zombie (provided you have a SIGCHLD
   * handler).
   *)

val iter_job_instances :
      f:(job_instance -> unit) ->
        unit
  (* Iterates over the jobs in the job list and calls ~f for every
   * job_instance.
   *)


exception Already_installed;;

val configure_job_handlers :
      ?catch_sigint:bool ->      (* default: true *)
      ?catch_sigquit:bool ->     (* default: true *)
      ?catch_sigterm:bool ->     (* default: true *)
      ?catch_sighup:bool ->      (* default: true *)
      ?catch_sigchld:bool ->     (* default: true *)
      ?set_sigpipe:bool ->       (* default: true *)
      ?at_exit:bool ->           (* default: true *)
      unit ->
      unit
  (* Configures signal and at_exit handlers for jobs:
   * - The keyboard signals SIGINT and SIGQUIT are forwarded to all jobs
   *   which are running in the background (and thus are not
   *   automatically notified) and want to get such signals (~forward_signals).
   *   After the signals have been forwarded, the previous signal action
   *   is performed.
   * - The signals SIGTERM and SIGHUP are (if the handler is installed) 
   *   forwarded to all dependent processes (regardless whether they are
   *   running in their own Unix process group or not, and regardless of
   *   ~forward_signals).
   *   After the signals have been forwarded, the previous signal action
   *   is performed.
   * - The at_exit handler sends a SIGTERM to all dependent processes, too.
   * - the SIGCHLD handler does nothing; however the waiting routine will notice
   *   the signal because an EINTR condition will be raised.
   *   [Update: SIGCHLD checks whether abandoned jobs have terminated in the
   *   meantime]
   *   After the signal has been forwarded, the previous signal action
   *   is performed; however if the previous action was Signal_ignore 
   *   this is incorrectly interpreted as empty action (zombies are not
   *   avoided)
   * - The handler for SIGPIPE does nothing; note that a previous action
   *   is overwritten (the parameter is called ~set_sigpipe to stress this)
   *
   * Dependent processes are:
   * - For jobs with mode = Foreground or Background: the processes
   *   of the corresponding Unix process group
   * - For jobs with mode = Same_as_caller: the actually started
   *   children processes
   *
   * Note that if an uncaught exception leads to program termination,
   * this situation will not be detected; any running jobs will
   * not be terminated (sorry, this cannot be fixed).
   *
   * This function sets only which handlers will be installed when
   * install_job_handlers (below) is invoked.
   * The function fails if the handlers are already installed.
   *
   * KNOWN BUGS: At least for O'Caml 3.00, the handlers do not call the old
   * signal handlers after their own work has been done; this is due to an
   * error in Sys.signal.
   *)

val install_job_handlers : unit -> unit
  (* Installs handlers as configured before.
   * Raises Already_installed if the handlers are already installed.
   *)

(* General notes:
 * - The implementation is currently not thread-safe.
 *)


(* ======================================================================
 * History:
 * 
 * $Log: shell_sys.mli,v $
 * Revision 1.1  2000/12/10 15:12:54  gerd
 * 	Initial revision.
 *
 * 
 *)
