/*
 * Security Context Mapping Protocol Daemon
 *
 * Flnetlink interface to the kernel.
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>

#include "libflnetlink.h"
#include "libflutil.h"

#include "list.h"
#include "server.h"
#include "flnetlink.h"
#include "protocol.h"
#include "perimtab.h"


#define BUFSIZE		8192	/* Netlink dumps need page-sized buffer */

int flnetlink_fd = -1;
static struct fln_handle *h = NULL;
static unsigned char buf[BUFSIZE];

int flnetlink_init(void)
{
	h = fln_create_handle(FLN_F_ACKS, FLN_G_PERIM|FLN_G_CACHE);
	if (!h) {
		syslog(LOG_ERR, "fln_create_handle: %m: %s", fln_errstr());
		return -1;
	}
	
	flnetlink_fd = fln_fd(h);
	
	return 0;
}

int flnetlink_send_map_res(struct flmsg_map_res *res, size_t len)
{
	return fln_cache_map_res(h, res, len);
}

int flnetlink_perimeter_dump(void)
{
	return fln_perimeter_dump(h);
}

void flnetlink_pollin(void)
{
	int rc, type;
	
	memset(buf, 0, BUFSIZE);
	
	rc = fln_read(h, buf, BUFSIZE, 0);
	if (rc < 0)
		die(0, "fln_read: %m: %s", fln_errstr());
	
	type = fln_message_type(buf);
	
	switch (type) {
	
	case NLMSG_ERROR:
	{
		int errval = fln_get_msgerr(buf);
		
		/* netlink ack */
		if (errval == 0)
			break;
			
		syslog(LOG_ERR, "netlink error report: %s",
		       strerror(fln_get_msgerr(buf)));
		break;
	}
		
	case FLMSG_PERIM_ADD:
		perimtab_append_entry(fln_get_msg(buf));
		break;
	
	case FLMSG_PERIM_GET:
		fln_parse_perim_entry(h, buf, rc, perimtab_append_entry);
		break;
	                                                	
	case FLMSG_PERIM_DEL:
		perimtab_delete_entry(fln_get_msg(buf));
		break;
		
	case FLMSG_PERIM_FLUSH:
		perimtab_flush();
		break;
	
	case FLMSG_CACHE_MAP_REQ:
		scmp_send_map_request(fln_get_msg(buf));
		break;
	
	case FLMSG_CACHE_ADD:
	case FLMSG_CACHE_DEL:
	case FLMSG_CACHE_FLUSH:
	case NLMSG_DONE:
		break;
		
	default:
		syslog(LOG_ERR, "received unknown netlink message type: %d",
		       type);
		break;
		
	}
}

void flnetlink_exit(void)
{
	if (h) {
		fln_destroy_handle(h);
		h = NULL;
	}
}
