/*
 * Flask Networking Utility Library
 *
 * Copyright (c) 2001-2002 James Morris <jmorris@intercode.com.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Based mostly on Rusty's address parsing and general helper routines from
 * iptables.
 *
 */
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <libgen.h>
#include <netdb.h>
#include <stdio.h>

#include "libflutil.h"

int string_to_number(const char *s, int min, int max)
{
	long number;
	char *end;
	
	/* Handle hex, octal, etc. */
	errno = 0;
	number = strtol(s, &end, 0);
	if (*end == '\0' && end != s) {
		/* we parsed a number, let's see if we want this */
		if (errno != ERANGE && min <= number && number <= max)
			return number;
	}
	return -1;
}

struct in_addr *dotted_to_addr(const char *dotted)
{
	static struct in_addr addr;
	unsigned char *addrp;
	char *p, *q;
	int onebyte, i;
	char buf[20];

	/* copy dotted string, because we need to modify it */
	strncpy(buf, dotted, sizeof(buf) - 1);
	addrp = (unsigned char *) &(addr.s_addr);

	p = buf;
	for (i = 0; i < 3; i++) {
		if ((q = strchr(p, '.')) == NULL)
			return (struct in_addr *) NULL;

		*q = '\0';
		if ((onebyte = string_to_number(p, 0, 255)) == -1)
			return (struct in_addr *) NULL;

		addrp[i] = (unsigned char) onebyte;
		p = q + 1;
	}

	/* we've checked 3 bytes, now we check the last one */
	if ((onebyte = string_to_number(p, 0, 255)) == -1)
		return (struct in_addr *) NULL;

	addrp[3] = (unsigned char) onebyte;

	return &addr;
}

struct in_addr *network_to_addr(const char *name)
{
	struct netent *net;
	static struct in_addr addr;

	if ((net = getnetbyname(name)) != NULL) {
		if (net->n_addrtype != AF_INET)
			return (struct in_addr *) NULL;
		addr.s_addr = htonl((unsigned long) net->n_net);
		return &addr;
	}

	return (struct in_addr *)NULL;
}

void inaddrcpy(struct in_addr *dst, struct in_addr *src)
{
	dst->s_addr = src->s_addr;
}

struct in_addr *host_to_addr(const char *name, unsigned int *naddr)
{
	struct hostent *host;
	struct in_addr *addr;
	unsigned int i;
	*naddr = 0;
	
	if ((host = gethostbyname(name)) != NULL) {
		if (host->h_addrtype != AF_INET ||
		    host->h_length != sizeof(struct in_addr))
			return (struct in_addr *) NULL;
			
		while (host->h_addr_list[*naddr] != (char *) NULL)
			(*naddr)++;

		addr = calloc(*naddr, sizeof(struct in_addr));
		if (addr == NULL) {
			perror("calloc");
			exit(1);
		}
		
		for (i = 0; i < *naddr; i++)
			inaddrcpy(&(addr[i]),
			          (struct in_addr *) host->h_addr_list[i]);
	
		return addr;
	}
	return (struct in_addr *)NULL;
}

struct in_addr *parse_hostnetwork(const char *name, unsigned int *naddrs)
{
	struct in_addr *addrp, *addrptmp;

	if ((addrptmp = dotted_to_addr(name)) != NULL ||
	    (addrptmp = network_to_addr(name)) != NULL) {
		
		addrp = malloc(sizeof(struct in_addr));
		if (addrp == NULL) {
			perror("malloc");
			exit(1);
		}
		
		inaddrcpy(addrp, addrptmp);
		*naddrs = 1;
		return addrp;
	}
	
	if ((addrp = host_to_addr(name, naddrs)) != NULL)
		return addrp;

	return (struct in_addr *)NULL;
}

struct in_addr *parse_mask(char *mask)
{
	static struct in_addr maskaddr;
	struct in_addr *addrp;
	int bits;

	if (mask == NULL) {
		/* no mask at all defaults to 32 bits */
		maskaddr.s_addr = 0xFFFFFFFF;
		return &maskaddr;
	}

	if ((addrp = dotted_to_addr(mask)) != NULL)
		/* dotted_to_addr already returns a network byte order addr */
		return addrp;

	if ((bits = string_to_number(mask, 0, 32)) == -1)
		return NULL;	
			   
	if (bits != 0) {
		maskaddr.s_addr = htonl(0xFFFFFFFF << (32 - bits));
		return &maskaddr;
	}

	maskaddr.s_addr = 0L;
	return &maskaddr;
}

void parse_hostnetworkmask(const char *name, struct in_addr **addrpp,
                           struct in_addr *maskp, unsigned int *naddrs)
{
	struct in_addr *addrp;
	char buf[256];
	char *p;
	int i, j, k, n;

	strncpy(buf, name, sizeof(buf) - 1);
	if ((p = strrchr(buf, '/')) != NULL) {
		*p = '\0';
		addrp = parse_mask(p + 1);
	} else
		addrp = parse_mask(NULL);
		
	if (addrp == NULL) {
		*naddrs = 0;
		return;
	}	
		
	inaddrcpy(maskp, addrp);

	/* if a null mask is given, the name is ignored, like in "any/0" */
	if (maskp->s_addr == 0L)
		strcpy(buf, "0.0.0.0");

	addrp = *addrpp = parse_hostnetwork(buf, naddrs);
	n = *naddrs;
	for (i = 0, j = 0; i < n; i++) {
		addrp[j++].s_addr &= maskp->s_addr;
		for (k = 0; k < j - 1; k++) {
			if (addrp[k].s_addr == addrp[j - 1].s_addr) {
				(*naddrs)--;
				j--;
				break;
			}
		}
	}
}

/* This is from Zebra */
u_int8_t ip_masklen(u_int32_t netmask)
{
	u_int8_t len = 0;
	u_int8_t *pnt = (u_int8_t *) &netmask;
	u_int8_t *end = pnt + 4;
        u_int8_t val;
        
        while ((*pnt == 0xff) && pnt < end) {
        	len+= 8;
        	pnt++;
        } 
        
        if (pnt < end) {
        	val = *pnt;
        	while (val) {
        		len++;
        		val <<= 1;
        	}
        }
        return len;
}
