/*

	WM Icon Demo

	To compile:

	gcc wm_icon_demo.c -lX11 -lXext -lXpm -L/usr/X11R6/lib
	
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>

#include <X11/xpm.h>
#ifdef XPM_h
# define XPM_H
#endif

#ifndef XpmDefaultColorCloseness
# define XpmDefaultColorCloseness	40000
#endif


#define PROG_DESC	"Demostration of WM icon handling"


#define Pixel	unsigned long

static int runlevel;


static int SetWMIcon(
        Display *dpy, Window w,
        const char *filename, const char *iconname
);


/*
 *	Sets the icon specified by the data pointer to the toplevel
 *	window w. The icon will be given the name specified by name.
 */
static int SetWMIcon(
	Display *dpy, Window w,
	const char *filename, const char *iconname
)
{
#ifdef XPM_H
	int scr_num, depth;
	char *tmp_filename;
        Window icon_w = None;
	Window root;
        Pixmap icon_pm, icon_mask_pm;
        int xpm_status;
        XpmAttributes xpm_attr;
	XWMHints *wm_hints;
	XTextProperty text_prop;


        if((dpy == NULL) || (w == None) ||
           (filename == NULL) || (iconname == NULL)
	)
            return(-1);

	scr_num = DefaultScreen(dpy);
	depth = DefaultDepth(dpy, scr_num);
        root = DefaultRootWindow(dpy);

	/* Set up XPM attributes for XPM loading. */
	xpm_attr.valuemask = XpmSize | XpmCloseness | XpmDepth;
        xpm_attr.closeness = XpmDefaultColorCloseness;
        xpm_attr.depth = depth;
        xpm_attr.width = 0;		/* Reset size returns. */
        xpm_attr.height = 0;
	tmp_filename = strdup(filename);
        xpm_status = XpmReadFileToPixmap(
            dpy, root, tmp_filename,
            &icon_pm, &icon_mask_pm,
            &xpm_attr
        );
	free(tmp_filename);
        if(xpm_status == XpmSuccess)
        {
	    int count;
            int width = xpm_attr.width;
            int height = xpm_attr.height;
            Pixel black_pix = BlackPixel(dpy, scr_num);
            Pixel white_pix = WhitePixel(dpy, scr_num);
	    XIconSize *icon_sizes;

            /* Check if loaded icon is a valid size by confirming with
	     * X server.
	     */
	    if(XGetIconSizes(dpy, w, &icon_sizes, &count))
	    {
		int i;

		for(i = 0; i < count; i++)
		{
		    if((width >= icon_sizes[i].min_width) &&
                       (width <= icon_sizes[i].max_width) &&
                       (height >= icon_sizes[i].min_height) &&
                       (height <= icon_sizes[i].max_height)
                    )
                        break;
                }
                XFree(icon_sizes);

                /* Did not get valid icon size bounds? */
                if(i >= count)
                {
                    fprintf(
			stderr,
			"SetWMIcon(): Invalid icon size %ix%i.\n",
			width, height
		    );

		    if(icon_mask_pm != None)
			XFreePixmap(dpy, icon_mask_pm);
		    if(icon_pm != None)
			XFreePixmap(dpy, icon_pm);

                    return(-2);
                }
            }


	    /* Create icon window. */
	    icon_w = XCreateSimpleWindow(
                dpy, root,
                0, 0,		/* Coordinates. */
		width, height, 	/* Size. */
		0,		/* Border width. */
		white_pix,	/* Border color. */
		black_pix	/* Background color. */
	    );
	    if((icon_w != None) && (icon_mask_pm != None))
                XShapeCombineMask(
                    dpy, icon_w,
                    ShapeBounding,
                    0, 0,
                    icon_mask_pm,
                    ShapeSet
                );
	}

	/* Check if there is an XPM loading error by checking if icon_w
	 * is still None.
	 */
	if(icon_w == None)
	    return(-1);

	/* Allocate a WM hints structure. */
	wm_hints = XAllocWMHints();
	if(wm_hints != NULL)
	{
	    /* Set up WM hints structure to set icon on the window. */
	    wm_hints->flags = (IconPixmapHint | IconMaskHint |
	                       IconWindowHint
	    );
	    wm_hints->initial_state = NormalState;
	    wm_hints->icon_pixmap = icon_pm;
	    wm_hints->icon_mask = icon_mask_pm;
	    wm_hints->icon_window = icon_w;

	    XSetWMHints(dpy, w, wm_hints);
	    XFree(wm_hints);
	}

	/* Set up text property for setting of icon name. */
	text_prop.value = (unsigned char *)strdup(iconname);
	text_prop.encoding = XA_STRING;
	text_prop.format = 8;
	text_prop.nitems = strlen(iconname);

	/* Set icon name. */
	XSetWMIconName(dpy, w, &text_prop);

	free(text_prop.value);


	return(0);
#else
	return(0);
#endif	/* XPM_H */
}

int main(int argc, char *argv[])
{
	XEvent xev;
	Display *dpy;
	Pixel white_pix, black_pix;
	int scr_num, depth;
	GC gc;
	Colormap cmap;
	Window root, toplevel;
	int x = 64, y = 64;
	int width = 256, height = 128;
	Atom delete_window_atom;
	XSetWindowAttributes wa;


	runlevel = 1;

	/* Insufficient arguments? */
	if(argc < 3)
	{
	    /* Print usage. */
	    if(argc > 0)
		printf("Usage: %s <xpmfile> <iconname>\n", argv[0]);

	    return(1);
	}

	/* Connect to X server. */
	dpy = XOpenDisplay(NULL);
	if(dpy == NULL)
	    return(1);

	scr_num = DefaultScreen(dpy);
	cmap = DefaultColormap(dpy, scr_num);
	depth = DefaultDepth(dpy, scr_num);
	gc = DefaultGC(dpy, scr_num);
	black_pix = BlackPixel(dpy, scr_num);
	white_pix = WhitePixel(dpy, scr_num);
	root = DefaultRootWindow(dpy);
	delete_window_atom = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

	{
	    XFontStruct *xfont = XLoadQueryFont(dpy,
"-adobe-helvetica-medium-r-normal-*-16-*-*-*-p-*-iso8859-1"
	    );
	    if(xfont != NULL)
	    {
		XCharStruct *csp;

		printf("min %i  max %i\n",
xfont->min_char_or_byte2, xfont->max_char_or_byte2
		);

		if(xfont->per_char != NULL)
		{
		    if(!xfont->min_byte1 && !xfont->max_byte1)
		    {
/*
			int i = (int)xfont->min_char_or_byte2;
 			int m = (int)xfont->max_char_or_byte2;
 */
			int i = 0;
			int m = (int)xfont->max_char_or_byte2 -
			    (int)xfont->min_char_or_byte2;

			while(i <= m)
			{
			    csp = &xfont->per_char[i];
			    if(csp != NULL)
				printf(
"Char %i: L/W/R: %i - (%c)\n",
i, csp->width, i +
(int)xfont->min_char_or_byte2
				);
			    i++;
			}
		    }
		    else
		    {

		    }
		}
	    }
	    XFreeFont(dpy, xfont);
	}

	/* Set up window attributes. */
	wa.background_pixmap = None;
	wa.background_pixel = black_pix;
	wa.border_pixmap = None;
	wa.border_pixel = white_pix;
	wa.bit_gravity = NorthWestGravity;
	wa.win_gravity = UnmapGravity + NorthWestGravity;
	wa.backing_store = NotUseful;
	wa.backing_planes = 0;
	wa.backing_pixel = black_pix;
	wa.save_under = False;
	wa.event_mask = ExposureMask | StructureNotifyMask;
	wa.do_not_propagate_mask = 0;
	wa.override_redirect = False;
	wa.colormap = cmap;
	wa.cursor = None;

	toplevel = XCreateWindow(
	    dpy, root,
	    x, y,		/* Coordinates (not used on toplevels). */
	    width, height,	/* Size. */
	    0,			/* Border width. */
	    depth,		/* Depth. */
	    InputOutput,	/* Class. */
	    DefaultVisual(dpy, scr_num),	/* Visual. */
	    CWBackPixmap | CWBackPixel | CWBorderPixmap |
	    CWBorderPixel | CWBitGravity | CWWinGravity |
	    CWBackingStore | CWBackingPlanes | CWBackingPixel |
	    CWOverrideRedirect | CWSaveUnder | CWEventMask |
	    CWDontPropagate | CWColormap | CWCursor,
	    &wa
	);
	if(toplevel == None)
	{
	    XCloseDisplay(dpy);
	    return(1);
	}
	else
	{
	    Window rroot;
	    int rx, ry;
	    unsigned int rwidth, rheight, rdepth, rborder;
	    XSizeHints *sz_hints = XAllocSizeHints();
	    if(sz_hints != NULL)
	    {
		/* Set size hint values. */
		sz_hints->flags = USPosition | PSize | PBaseSize | PMinSize;
		sz_hints->x = x;
		sz_hints->y = y;
		sz_hints->width = width;
		sz_hints->height = height;
		sz_hints->base_width = 0;
		sz_hints->base_height = 0;
		sz_hints->min_width = 100;
		sz_hints->min_width = 75;
		XSetWMNormalHints(dpy, toplevel, sz_hints);
		XFree(sz_hints);
	    }
/*
XGetGeometry(
 dpy, toplevel,
 &rroot, &rx, &ry, &rwidth, &rheight, &rborder, &rdepth
);
printf("%i %i %i %i\n",
 rx, ry, rwidth, rheight
);
 */
            if(delete_window_atom != None)
                XSetWMProtocols(dpy, toplevel, &delete_window_atom, 1);
	    SetWMIcon(dpy, toplevel, argv[1], argv[2]);
	    XMapRaised(dpy, toplevel);
	}

	runlevel = 2;
	while(runlevel >= 2)
	{
	    XNextEvent(dpy, &xev);
	    switch(xev.type)
	    {
	      case Expose:
		if(xev.xany.window == toplevel)
                {
		    const char *strptr = PROG_DESC;

		    XClearWindow(dpy, toplevel);
		    XSetForeground(dpy, gc, white_pix);
		    XDrawString(
			dpy, toplevel, gc,
			32, 32, strptr, strlen(strptr)
		    );
                }
                break;

              case ClientMessage:
		if((xev.xclient.format == 32) &&
		   (xev.xclient.data.l[0] == delete_window_atom) &&
		   (xev.xany.window == toplevel)
		)
		{
		    runlevel = 1;
		}
		break;

	      case UnmapNotify:
		if(xev.xany.window == toplevel)
		{
		    /* Iconified. */
		}
		break;

	      case ConfigureNotify:
                if(xev.xany.window == toplevel)
                {
/*
		    printf("Move/resize: %i %i %i %i\n",
			xev.xconfigure.x, xev.xconfigure.y,
			xev.xconfigure.width, xev.xconfigure.height
		    );
 */
                }
                break;
	    }
	}



	if(toplevel != None)
	{
	    XWMHints *wm_hints = XGetWMHints(dpy, toplevel);
	    if(wm_hints != NULL)
	    {
                if(wm_hints->flags & IconWindowHint)
                    if(wm_hints->icon_window != None)
                        XDestroyWindow(dpy, wm_hints->icon_window);

		if(wm_hints->flags & IconPixmapHint)
		    if(wm_hints->icon_pixmap != None)
			XFreePixmap(dpy, wm_hints->icon_pixmap);

                if(wm_hints->flags & IconMaskHint)
                    if(wm_hints->icon_mask != None)
                        XFreePixmap(dpy, wm_hints->icon_mask);

		XFree(wm_hints);
	    }

	    XDestroyWindow(dpy, toplevel);
	    toplevel = None;
	}

	XCloseDisplay(dpy);
	dpy = NULL;

	return(0);
}
