#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "v3dtex.h"

#include "obj.h"
#include "smoke.h"


int SmokeCreate(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
	sar_position_struct *pos, sar_position_struct *respawn_offset,
	double radius_start,		/* In meters. */
	double radius_max,		/* In meters. */
	double radius_rate,		/* In meters per second. */
	int hide_at_max,
	int total_units,		/* Max smoke units in trail. */
        time_t respawn_int,             /* In ms. */
	const char *tex_name,
	int ref_object,			/* Can be -1 for none. */
	time_t life_span
);


/*
 *	Creates a smoke trail object, returning the new object's number
 *	or -1 on error.
 */
int SmokeCreate(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        sar_position_struct *pos, sar_position_struct *respawn_offset,
        double radius_start,            /* In meters. */
        double radius_max,              /* In meters. */
        double radius_rate,             /* In meters per second. */
        int hide_at_max,
        int total_units,                /* Max smoke units in trail. */
        time_t respawn_int,             /* In ms. */
        const char *tex_name,
	int ref_object,			/* Can be -1 for none. */
        time_t life_span
)
{
	int i, obj_num;
	sar_object_struct *obj_ptr;
	sar_object_smoke_struct *obj_smoke_ptr;


	if((scene == NULL) || (pos == NULL) || (tex_name == NULL))
	    return(-1);

	/* Create smoke trail object. */
	obj_num = SARObjCreate(
	    scene, ptr, total,
	    SAR_OBJ_TYPE_SMOKE
	);
	if(obj_num < 0)
	    return(-1);

	obj_ptr = (*ptr)[obj_num];
	if(obj_ptr == NULL)
	    return(-1);


	/* Need to auto calculate radius_rate? */
	if(radius_rate < 0.0)
	{
	    double dt = total_units * respawn_int / 1000.0;

	    if(dt > 0.0)
		radius_rate = (radius_max - radius_start) / dt;
	    else
		radius_rate = 0.0;
	}

	/* Set general values. */
	memcpy(
	    &obj_ptr->pos,
	    pos,
	    sizeof(sar_position_struct)
	);
	/* Set visible range based on maximum size. */
	if(radius_max >= 400.0)
	    obj_ptr->range = SFMMilesToMeters(18.0);
        else if(radius_max >= 200.0)
            obj_ptr->range = SFMMilesToMeters(16.0);
	else if(radius_max >= 100.0)
            obj_ptr->range = SFMMilesToMeters(13.0);
	else
	    obj_ptr->range = SFMMilesToMeters(10.0);

	SARObjAddContactBoundsSpherical(
	    obj_ptr,
	    0, 0,		/* Crash flags and crash type. */
	    radius_start	/* Use starting radius size as contact bounds. */
	);

	obj_ptr->life_span = life_span;


        /* Get pointer to smoke trail sub structure. */
        obj_smoke_ptr = (sar_object_smoke_struct *)obj_ptr->data;
        if(obj_smoke_ptr == NULL)
            return(-1);

	/* Set smoke trail specific values. */
	if(respawn_offset != NULL)
	    memcpy(
		&obj_smoke_ptr->respawn_offset,
		respawn_offset,
		sizeof(sar_position_struct)
	    );

	obj_smoke_ptr->radius_start = radius_start;
        obj_smoke_ptr->radius_max = radius_max;
	obj_smoke_ptr->radius_rate = radius_rate;

	obj_smoke_ptr->hide_at_max = hide_at_max;
	obj_smoke_ptr->delete_when_no_units = 0;

	obj_smoke_ptr->respawn_int = respawn_int;
	obj_smoke_ptr->respawn_next = 0;

	obj_smoke_ptr->tex_num = SARGetTextureRefNumberByName(
            scene, tex_name
        );

        obj_smoke_ptr->ref_object = ref_object;

	obj_smoke_ptr->total_units = total_units;
	if(obj_smoke_ptr->total_units > 0)
	{
	    obj_smoke_ptr->unit = (sar_object_smoke_unit_struct *)realloc(
		obj_smoke_ptr->unit,
		obj_smoke_ptr->total_units * sizeof(sar_object_smoke_unit_struct)
	    );
	    if(obj_smoke_ptr->unit == NULL)
	    {
		obj_smoke_ptr->total_units = total_units = 0;
	    }
	}
	for(i = 0; i < obj_smoke_ptr->total_units; i++)
	{
	    sar_object_smoke_unit_struct *smoke_unit_ptr = &obj_smoke_ptr->unit[i];

	    memset(&smoke_unit_ptr->pos, 0x00, sizeof(sar_position_struct));
	    smoke_unit_ptr->radius = 0.0;
	    smoke_unit_ptr->visibility = 0.0;
	}


	return(obj_num);
}
