#include <stdio.h> 
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "matrixmath.h"
#include "sfm.h"

#include "sarreality.h"
#include "obj.h"
#include "sarsound.h"
#include "smoke.h"
#include "explosion.h"
#include "messages.h" 
#include "simmanage.h"
#include "simutils.h"
#include "sar.h"
#include "config.h"


double SARSimGetFlatContactRadius(sar_object_struct *obj_ptr);
sar_object_struct *SARSimMatchObjectFromFDM(
        sar_object_struct **list, int total,
        SFMModelStruct *fdm, int *index
);
double SARSimThrottleOutputCoeff(
        int flight_model_type,
        double throttle,
        double collective, double collective_range
);

void SARSimWarpObjectRelative(
	sar_scene_struct *scene, sar_object_struct *obj_ptr,
	sar_object_struct **ptr, int total, int ref_obj_num,
	sar_position_struct *offset_pos,
	sar_direction_struct *offset_dir
);	
void SARSimWarpObject(
        sar_scene_struct *scene, sar_object_struct *obj_ptr,
        sar_position_struct *new_pos,
        sar_direction_struct *new_dir
);
int SARSimGetLandingGearState(sar_object_struct *obj_ptr);
void SARSimUpdateLandingGear(sar_obj_landing_gear_struct *lgear_ptr);
void SARSimOpLandingGear(  
        sar_scene_struct *scene, 
        sar_object_struct ***ptr,
        int *total,
        sar_object_struct *obj_ptr,
        int gear_state
);

void SARSimUpdateDoor(sar_obj_door_struct *door_ptr);
void SARSimOpDoor(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        sar_object_struct *obj_ptr,
        int state	/* 0 = close, 1 = open. */
);

void SARSimRotorUpdateControls(
        sar_obj_rotor_struct *rotor_ptr,
        double pitch, double bank
);

int SARSimGetEngineState(sar_object_struct *obj_ptr);
void SARSimOpEngine(
        sar_scene_struct *scene,
        sar_object_struct ***ptr,
        int *total,
        sar_object_struct *obj_ptr,
        int engine_state               
);

int SARSimGetLightsState(sar_object_struct *obj_ptr);
void SARSimOpLights(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	sar_object_struct *obj_ptr,
	int state
);
int SARSimGetStrobesState(sar_object_struct *obj_ptr);
void SARSimOpStrobes(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	sar_object_struct *obj_ptr,
	int state
);
int SARSimGetAttenuate(sar_object_struct *obj_ptr);
void SARSimOpAttenuate(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	sar_object_struct *obj_ptr,
	int state
);

int SARSimOpRepair(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);
int SARSimOpRefuel(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);

int SARSimOpPassengersSetLeave(
	sar_scene_struct *scene, sar_object_struct *obj_ptr,
	int passengers_leave_pending, int passengers_drop_pending
);
int SARSimOpPassengersUnloadAll(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);

double SARSimOpTransferFuelFromTanks(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);

int SARSimOpDropFuelTankNext(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num, sar_object_struct *obj_ptr
);

void SARSimSmokeSpawn(
        sar_core_struct *core_ptr, sar_object_struct *obj_ptr
);

int SARSimIsSlew(sar_object_struct *obj_ptr);
void SARSimSetSlew(
        sar_object_struct *obj_ptr, int enter_slew
);
int SARSimDoPickUpHuman(
	sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
	sar_object_human_struct *obj_human_ptr,
	int human_obj_num
);
int SARSimDoHoistIn(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr
);
int SARSimBoardObject(
	sar_core_struct *core_ptr,
	sar_object_struct *tar_obj_ptr, int src_obj_num
);
void SARSimSetFlyByPosition(
	sar_scene_struct *scene,
        sar_object_struct ***ptr,
        int *total,
        sar_object_struct *obj_ptr,
	sar_position_struct *pos_result
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Returns the flat (XY plane) contact bounds radius for
 *      the given object.
 *
 *      Can return 0.0 if contact bounds are not availbale for
 *      the object.
 */
double SARSimGetFlatContactRadius(sar_object_struct *obj_ptr)
{
        double contact_radius = 0.0;
        sar_contact_bounds_struct *cb;

        if(obj_ptr == NULL)
            return(0.0);

        cb = obj_ptr->contact_bounds;
        if(cb == NULL)
            return(0.0);

        switch(cb->contact_shape)
        {
          case SAR_CONTACT_SHAPE_SPHERICAL:
          case SAR_CONTACT_SHAPE_CYLENDRICAL:
            contact_radius = MAX(cb->contact_radius, 0.0);
            break;

          case SAR_CONTACT_SHAPE_RECTANGULAR:
            contact_radius = MAX(
                cb->contact_x_max - cb->contact_x_min,
                cb->contact_y_max - cb->contact_y_min
            );
            break;
        }

        return(MAX(contact_radius, 0.0));
}

/*
 *	Returns the pointer to the object in the give nobject list that
 *	contains an FDM that matches the given fdm or NULL on error.
 *
 *      If index is not NULL then the index to the object number on
 *      the objects list will be set or -1 on failed match.
 */
sar_object_struct *SARSimMatchObjectFromFDM(
        sar_object_struct **list, int total,
        SFMModelStruct *fdm, int *index
)
{
        int i;
        sar_object_struct *obj_ptr;
        sar_object_aircraft_struct *obj_aircraft_ptr;


        if((list == NULL) ||
           (fdm == NULL)
        )
        {
            if(index != NULL)
                (*index) = -1;
                
            return(NULL);
        }
  
        for(i = 0; i < total; i++)
        {
            obj_ptr = list[i]; 
            if(obj_ptr == NULL)
                continue;
        
            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)
                    obj_ptr->data;
                if(obj_aircraft_ptr == NULL)
                    break;

                if(obj_aircraft_ptr->fdm == fdm)
                {
                    if(index != NULL)
                        (*index) = i;
                    return(obj_ptr);
                }
                break;

/* Add checking for other object types which may have SFM structures
 * here when needed.
 */

            }
        }

        if(index != NULL)
            (*index) = -1;

        return(NULL);
}


/*
 *	Returns the actual throttle output based on the given
 *	values.
 */
double SARSimThrottleOutputCoeff(
        int flight_model_type,
        double throttle,
        double collective, double collective_range
)
{
	switch(flight_model_type)
	{
	  case SAR_FLIGHT_MODEL_AIRPLANE:
	    return(throttle);
	    break;

	  case SAR_FLIGHT_MODEL_HELICOPTER:
	    if(collective_range >= 0.0)
		return(CLIP(
		    (throttle + (throttle * collective * collective_range))
			/ (1.0 + collective_range),
		    0.0, 1.0
		));
	    else
		return(throttle);
	    break;

	  default:	/* Slew, return throttle as given. */
	    return(throttle);
	    break;
	}
}


/*
 *	Moves the given object obj_ptr relative to the object ref_obj_num
 *	if it is valid with respect to the given offset_pos and offset_dir
 *	(both relative to ref_obj_num).
 *
 *	If ref_obj_num is invalid then no operation will be performed.
 *
 *	This function automatically calls SARSimWarpObject() afterwards.
 */
void SARSimWarpObjectRelative(
        sar_scene_struct *scene, sar_object_struct *obj_ptr,
        sar_object_struct **ptr, int total, int ref_obj_num,
        sar_position_struct *offset_pos,
        sar_direction_struct *offset_dir
)
{
	double a[3 * 1], r[3 * 1];
	sar_position_struct *pos, *ref_pos;
	sar_direction_struct *dir, *ref_dir;
	sar_object_struct *ref_obj_ptr;


	if((scene == NULL) || (obj_ptr == NULL))
	    return;

	/* Check if referance object is valid. */
	if((ref_obj_num >= 0) && (ref_obj_num < total))
	    ref_obj_ptr = ptr[ref_obj_num];
	else
	    ref_obj_ptr = NULL;

	/* Got a referance object? */
	if(ref_obj_ptr == NULL)
	    return;

	/* Get pointers to position of referance object. */
	ref_pos = &ref_obj_ptr->pos;
        ref_dir = &ref_obj_ptr->dir;

	/* Set up and transform offset matrix a. */
	if(offset_pos != NULL)
	{
	    a[0] = offset_pos->x;
	    a[1] = offset_pos->y;
            a[2] = offset_pos->z;
	}
	else
	{
	    a[0] = 0.0;
	    a[1] = 0.0;
	    a[2] = 0.0;
	}

	/* Rotate the offset matrix a to get rotated offset matrix r. */
	MatrixRotateBank3(a, ref_dir->bank, r);
	MatrixRotatePitch3(r, ref_dir->pitch, a);
	MatrixRotateHeading3(a, ref_dir->heading, r);

	/* Move the given object to the new offset position relative
	 * to the referance object.
	 */
	pos = &obj_ptr->pos;
	pos->x = ref_pos->x + r[0];
        pos->y = ref_pos->y + r[1];
        pos->z = ref_pos->z + r[2];

	/* Rotate the given object by the relative object's direction
	 * added with the offset direction.
	 */
	dir = &obj_ptr->dir;
	if(offset_dir != NULL)
	{
	    dir->heading = SFMSanitizeRadians(
		ref_dir->heading + offset_dir->heading
	    );
	    dir->pitch = SFMSanitizeRadians( 
                ref_dir->pitch + offset_dir->pitch
            );
	    dir->bank = SFMSanitizeRadians(
                ref_dir->bank + offset_dir->bank
            );
	}
	else
	{
	    memcpy(dir, ref_dir, sizeof(sar_direction_struct));
	}


	/* Realize new position of the given object. */
	SARSimWarpObject(scene, obj_ptr, pos, dir);
}



/*
 *	Updates (realizes) all position and attitude values for the
 *	given object to the specified new position and direction.
 *
 *	If any input is NULL then the values will be left unchanged.
 *
 *	If the given input pointers match the pointers of the structures
 *	on the object then the structure in question will be left
 *	unchanged.
 *
 *	Rotatation values for the contact bounds and ground object
 *	structure will be updated as needed.
 */
void SARSimWarpObject(
        sar_scene_struct *scene, sar_object_struct *obj_ptr,  
        sar_position_struct *new_pos,
	sar_direction_struct *new_dir
)
{
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_contact_bounds_struct *cb;
	SFMModelStruct *fdm = NULL;


	if((scene == NULL) || (obj_ptr == NULL))
	    return;

	if((new_pos != NULL) && (new_pos != &obj_ptr->pos))
	{
	    obj_ptr->pos.x = new_pos->x;
            obj_ptr->pos.y = new_pos->y;
            obj_ptr->pos.z = new_pos->z;
	}
	if((new_dir != NULL) && (new_dir != &obj_ptr->dir))
	{
	    obj_ptr->dir.heading = new_dir->heading;
            obj_ptr->dir.pitch = new_dir->pitch;
            obj_ptr->dir.bank = new_dir->bank;
	}

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		/* Get pointer to SFM flight dynamics structure. */
		fdm = obj_aircraft_ptr->fdm;
	    }
	    break;

	  case SAR_OBJ_TYPE_GROUND:
	    obj_ground_ptr = (sar_object_ground_struct *)obj_ptr->data;
	    if(obj_ground_ptr != NULL)
	    {
		/* Update trig functions for ground's heightfield
		 * rotation.
		 */
		if(new_dir != NULL)
		{
		    obj_ground_ptr->cos_heading = cos(-new_dir->heading);
                    obj_ground_ptr->sin_heading = sin(-new_dir->heading);
		}
	    }
	    break;
	}

	/* Set values on FDM if exists. */
	if(fdm != NULL)
	{
	    if(new_pos != NULL)
	    {
		fdm->position.x = new_pos->x;
                fdm->position.y = new_pos->y;
                fdm->position.z = new_pos->z;
	    }
            if(new_dir != NULL)
            {
                fdm->direction.heading = new_dir->heading;
                fdm->direction.pitch = new_dir->pitch;
                fdm->direction.bank = new_dir->bank;
            }
	}

	/* Update values on contact bounds. */
	cb = obj_ptr->contact_bounds;
	if(cb != NULL)
	{
	    /* Update values based on contact shape. */
	    if(cb->contact_shape == SAR_CONTACT_SHAPE_RECTANGULAR)
	    {
		/* For rectangular contact bounds, the cos and sin
		 * values for the heading need to be updated based
		 * on the new rotation.
		 */
		if(new_dir != NULL)
		{
		    cb->cos_heading = cos(-new_dir->heading);
                    cb->sin_heading = sin(-new_dir->heading);
		}
	    }

	}
}

/*
 *	Gets the landing gear state of the first landing gear
 *	on the object. Returns true if gear is down, false if up.
 */
int SARSimGetLandingGearState(sar_object_struct *obj_ptr)
{
        sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_obj_landing_gear_struct *lgear_ptr = NULL;


        if(obj_ptr == NULL)
            return(0);

        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = obj_ptr->data;

	    if(obj_aircraft_ptr->total_landing_gears > 0)
		lgear_ptr = obj_aircraft_ptr->landing_gear[0];
            break;
        }

	if(lgear_ptr == NULL)
	    return(0);
	else
	    return(lgear_ptr->flags & SAR_LGEAR_FLAG_STATE);
}

/*
 *	Updates animated position of landing gear.
 */
void SARSimUpdateLandingGear(sar_obj_landing_gear_struct *lgear_ptr)
{
	sar_obj_flags_t flags;


	if(lgear_ptr == NULL)
	    return;

	flags = lgear_ptr->flags;

	if(flags & SAR_LGEAR_FLAG_STATE)
	{
	    /* Raising. */

            if(lgear_ptr->anim_pos == 0)
                return;

	    if((flags & SAR_LGEAR_FLAG_DAMAGED) ||
               (flags & SAR_LGEAR_FLAG_MISSING) ||
               (flags & SAR_LGEAR_FLAG_FIXED)
	    )
		return;

            lgear_ptr->anim_pos = (sar_grad_anim_t)MAX(
                (int)lgear_ptr->anim_pos -
                ((int)lgear_ptr->anim_rate * time_compensation *
                    time_compression),
                0
            );
	}
	else
	{
	    /* Lowering. */

            if(lgear_ptr->anim_pos == (sar_grad_anim_t)-1)
                return;

            if((flags & SAR_LGEAR_FLAG_DAMAGED) ||
               (flags & SAR_LGEAR_FLAG_MISSING)
            )
                return;

            lgear_ptr->anim_pos = (sar_grad_anim_t)MIN(
                (int)lgear_ptr->anim_pos +
                ((int)lgear_ptr->anim_rate * time_compensation *
                    time_compression),
                (sar_grad_anim_t)-1   
            );   
	}

	return;
}

/*
 *	Raise or lower landing gear on object.
 */
void SARSimOpLandingGear(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	sar_object_struct *obj_ptr,
	int gear_state
)
{
	int i;
        sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_obj_landing_gear_struct *lgear_ptr;


        if(obj_ptr == NULL)
            return;

        switch(obj_ptr->type)
        {
	  case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = obj_ptr->data;
            if(obj_aircraft_ptr == NULL)
                break;

	    /* No landing gears on object? */
	    if(obj_aircraft_ptr->total_landing_gears <= 0)
		break;

	    /* Cannot operate landing gears while landed. */
	    if(obj_aircraft_ptr->landed)
		break;

	    /* Get pointer to first landing gear. */
	    lgear_ptr = obj_aircraft_ptr->landing_gear[0];
	    if(lgear_ptr == NULL)
		break;

	    /* Gear is up, we want to lower it? */
	    if((lgear_ptr->flags & SAR_LGEAR_FLAG_STATE) &&
	       !gear_state
	    )
	    {
		/* Landing gear is currently up, we want to lower it. */

                for(i = 0; i < obj_aircraft_ptr->total_landing_gears; i++)
                {
                    lgear_ptr = obj_aircraft_ptr->landing_gear[i];
                    if(lgear_ptr == NULL)
                        continue;

                    if((lgear_ptr->flags & SAR_LGEAR_FLAG_DAMAGED) ||
		       (lgear_ptr->flags & SAR_LGEAR_FLAG_MISSING)
		    )
                        continue;

		    lgear_ptr->flags &= ~SAR_LGEAR_FLAG_STATE;
		}
	    }
            /* Gear is down, we want to raise it? */
	    else if(!(lgear_ptr->flags & SAR_LGEAR_FLAG_STATE) &&
                    gear_state
            )
            {
                /* Landing gear is currently down, we want to raise it. */

		for(i = 0; i < obj_aircraft_ptr->total_landing_gears; i++)
		{
		    lgear_ptr = obj_aircraft_ptr->landing_gear[i];
		    if(lgear_ptr == NULL)
			continue;

		    if((lgear_ptr->flags & SAR_LGEAR_FLAG_FIXED) ||
                       (lgear_ptr->flags & SAR_LGEAR_FLAG_MISSING) ||
                       (lgear_ptr->flags & SAR_LGEAR_FLAG_DAMAGED)
                    )
			continue;

                    lgear_ptr->flags |= SAR_LGEAR_FLAG_STATE;
		}
	    }
	    break;
	}

	return;	
}

/*
 *      Updates animated position of door.
 */
void SARSimUpdateDoor(
	sar_obj_door_struct *door_ptr
)
{
        sar_obj_flags_t flags;


        if(door_ptr == NULL)
            return;
            
        flags = door_ptr->flags;

        if(flags & SAR_DOOR_FLAG_STATE)
        {
            /* Opening. */

            if(door_ptr->anim_pos == (sar_grad_anim_t)-1)
                return;

            if(flags & SAR_DOOR_FLAG_FIXED)
                return;

            door_ptr->anim_pos = (sar_grad_anim_t)MIN(
		(int)door_ptr->anim_pos +
                ((int)door_ptr->anim_rate * time_compensation *
                    time_compression),
                (int)((sar_grad_anim_t)-1)
            );
        }
        else
        {
            /* Closing. */

            if(door_ptr->anim_pos == 0)
                return;

            if(flags & SAR_DOOR_FLAG_FIXED)
                return;

            door_ptr->anim_pos = (sar_grad_anim_t)MAX(
		(int)door_ptr->anim_pos -
                ((int)door_ptr->anim_rate * time_compensation *
                    time_compression),
                0
            );
        }

	return;
}

/*
 *	Open or close the first door on the given object.
 *
 *	This marks the door to either start opening or closing.
 *
 *	If the door is to be closed and has the SAR_DOOR_FLAG_STAY_OPEN
 *	flag set, then this flag will be unset.
 */
void SARSimOpDoor(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
	sar_object_struct *obj_ptr,
	int state		/* 0 = close, 1 = open. */
)
{
        sar_obj_door_struct *door_ptr;


        if(obj_ptr == NULL)
            return;

	/* Get the first door on the object if any. */
	door_ptr = SARObjGetDoorPtr(obj_ptr, 0, NULL);
	if(door_ptr != NULL)
        {
            /* Door is opened, we want to close it? */
            if((door_ptr->flags & SAR_DOOR_FLAG_STATE) &&
               !state
            )
            {
                if(!(door_ptr->flags & SAR_DOOR_FLAG_FIXED))
		{
                    door_ptr->flags &= ~SAR_DOOR_FLAG_STATE;
		    door_ptr->flags &= ~SAR_DOOR_FLAG_STAY_OPEN;
		}
            }
            /* Door is closed, we want to open it? */
            else if(!(door_ptr->flags & SAR_DOOR_FLAG_STATE) &&
                    state
            )
            {
                if(!(door_ptr->flags & SAR_DOOR_FLAG_FIXED) &&
                   !(door_ptr->flags & SAR_DOOR_FLAG_LOCKED)
		)
		    door_ptr->flags |= SAR_DOOR_FLAG_STATE;
            }
        }

	return;
}

/*
 *	Updates control positions on the rotor in the given list as
 *	needed in accordance with the given control positions pitch
 *	and bank which must be in the range of -1.0 to 1.0.
 *
 *	All inputs assumed valid.
 */
void SARSimRotorUpdateControls(
        sar_obj_rotor_struct *rotor_ptr,
        double pitch, double bank
)
{
	if(rotor_ptr->flags & SAR_ROTOR_FLAG_FOLLOW_PB)
	{
	    rotor_ptr->control_coeff_pitch = pitch;
            rotor_ptr->control_coeff_bank = bank;
	}

	return;
}

/*
 *	Returns the engine state of the object.
 *
 *	If object is not a type that has engines, then
 *	SAR_ENGINE_STATE_OFF is returned.
 */
int SARSimGetEngineState(sar_object_struct *obj_ptr)
{
	sar_object_aircraft_struct *obj_aircraft_ptr;


	if(obj_ptr == NULL)
	    return(SAR_ENGINE_STATE_OFF);

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = obj_ptr->data;
	    return(obj_aircraft_ptr->engine_state);
	    break;
	}

	return(SAR_ENGINE_STATE_OFF);
}

/*
 *	Turns engine off or starts engine, depending on 
 *	engine_state which can be either SAR_ENGINE_STATE_ON or
 *	SAR_ENGINE_STATE_OFF.
 *
 *	When passing engine_state as SAR_ENGINE_STATE_ON, it does not
 *	actually turn the engine on but rather sets engine_state on
 *	the object to initialize (start) and sets the initialization
 *	time depending on the current throttle position.
 */
void SARSimOpEngine(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	sar_object_struct *obj_ptr,
	int engine_state
)
{
	time_t engine_start_delay = 2200;	/* In ms. */
        sar_object_aircraft_struct *obj_aircraft_ptr;
  
 
        if(obj_ptr == NULL)
            return;

	/* Request to start engine as init is the same as on. */
	if(engine_state == SAR_ENGINE_STATE_INIT)
	    engine_state = SAR_ENGINE_STATE_ON;

        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
            if(obj_aircraft_ptr == NULL)
                break;

	    /* Cannot turn engine on if not flyable. */
	    if((obj_aircraft_ptr->air_worthy_state != SAR_AIR_WORTHY_FLYABLE) &&
               (engine_state == SAR_ENGINE_STATE_ON)
	    )
		break;

	    /* Turn engine on and engine is currently off? */
	    if((engine_state == SAR_ENGINE_STATE_ON) &&
               (obj_aircraft_ptr->engine_state == SAR_ENGINE_STATE_OFF)
	    )
	    {
		obj_aircraft_ptr->engine_state = SAR_ENGINE_STATE_INIT;

		/* Check if landed, if landed use normal engine start
	         * delay, otherwise start engine immediatly.
		 */
		if(obj_aircraft_ptr->landed)
		    obj_aircraft_ptr->next_engine_on = cur_millitime +
			(time_t)(engine_start_delay *
			    CLIP(1.0 - obj_aircraft_ptr->throttle, 0.0, 1.0)
			);
		else
		    obj_aircraft_ptr->next_engine_on = cur_millitime;
	    }
	    /* Turn engine off? */
	    else if(engine_state == SAR_ENGINE_STATE_OFF)
	    {
		obj_aircraft_ptr->engine_state = SAR_ENGINE_STATE_OFF;
		obj_aircraft_ptr->next_engine_on = 0;
	    }
	    break;

/* Add support for other object types that have engines. */

	}

	return;
}

/*
 *	Returns the state of the first light structure on the object.
 *
 *	Can return false if no lights exist or error.
 */
int SARSimGetLightsState(sar_object_struct *obj_ptr)
{
        int i;
        sar_light_struct *light_ptr;


        if(obj_ptr == NULL)
            return(0);

        for(i = 0; i < obj_ptr->total_lights; i++)
        {
            light_ptr = obj_ptr->light[i];
            if(light_ptr == NULL)
                continue;

            /* Is it a strobe light? */
            if(!(light_ptr->flags & SAR_LIGHT_FLAG_STROBE) &&
               !(light_ptr->flags & SAR_LIGHT_FLAG_ATTENUATE)
	    )
                return(light_ptr->flags & SAR_LIGHT_FLAG_ON);
        }

        return(0);
}

/*
 *	Turns all regular lights on object on or off.
 */
void SARSimOpLights(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
        int *total,
        sar_object_struct *obj_ptr,
        int state
)
{
	int i;
	sar_light_struct *light_ptr;

	if(obj_ptr == NULL)
            return;

	for(i = 0; i < obj_ptr->total_lights; i++)
	{
	    light_ptr = obj_ptr->light[i];
	    if(light_ptr == NULL)
		continue;

	    /* Skip non-standard lights. */
	    if((light_ptr->flags & SAR_LIGHT_FLAG_STROBE) ||
               (light_ptr->flags & SAR_LIGHT_FLAG_ATTENUATE)
	    )
		continue;

	    if(state)
	    {
		/* Turn on. */
		light_ptr->flags |= SAR_LIGHT_FLAG_ON;
	    }
	    else
	    {
		/* Turn off. */
		light_ptr->flags &= ~SAR_LIGHT_FLAG_ON;
	    }
	}

	return;
}

/*
 *      Returns the state of the first strobe light structure on the
 *	object.
 *
 *      Can return false if no lights exist or error.
 */
int SARSimGetStrobesState(sar_object_struct *obj_ptr)
{
	int i;
	sar_light_struct *light_ptr;


	if(obj_ptr == NULL)
	    return(0);

	for(i = 0; i < obj_ptr->total_lights; i++)
	{
	    light_ptr = obj_ptr->light[i];
	    if(light_ptr == NULL)
		continue;

            /* Is it a strobe light? */
            if(light_ptr->flags & SAR_LIGHT_FLAG_STROBE)
                return(light_ptr->flags & SAR_LIGHT_FLAG_ON);
	}

	return(0);
}

/*
 *	Turns all strobe lights on object on or off.
 */
void SARSimOpStrobes(
	sar_scene_struct *scene,
	sar_object_struct ***ptr,
	int *total,
	sar_object_struct *obj_ptr,
	int state
)
{
        int i;
        sar_light_struct *light_ptr;

        if(obj_ptr == NULL)
            return;

        for(i = 0; i < obj_ptr->total_lights; i++)
        {
            light_ptr = obj_ptr->light[i];
            if(light_ptr == NULL)
                continue;

            /* Skip non-strobe lights. */
            if(!(light_ptr->flags & SAR_LIGHT_FLAG_STROBE))
                continue;

            if(state)
            {
                /* Turn on. */
                light_ptr->flags |= SAR_LIGHT_FLAG_ON;
            }
            else
            {   
                /* Turn off. */
                light_ptr->flags &= ~SAR_LIGHT_FLAG_ON;
            }
        }

        return;
}

/*
 *      Returns the state of the first attenuation light structure on the
 *      object.
 *
 *      Can return false if no lights exist or error.
 */
int SARSimGetAttenuate(sar_object_struct *obj_ptr)
{
        int i;
        sar_light_struct *light_ptr;


        if(obj_ptr == NULL)
            return(0);

        for(i = 0; i < obj_ptr->total_lights; i++)
        {
            light_ptr = obj_ptr->light[i];
            if(light_ptr == NULL)  
                continue;

            /* Is it a attenuation light? */
            if(light_ptr->flags & SAR_LIGHT_FLAG_ATTENUATE)
                return(light_ptr->flags & SAR_LIGHT_FLAG_ON);
        }

        return(0);
}

/*
 *      Turns all attenuation lights on object on or off.
 */
void SARSimOpAttenuate(
        sar_scene_struct *scene,
        sar_object_struct ***ptr,
        int *total,
        sar_object_struct *obj_ptr,
        int state
)
{
        int i;
        sar_light_struct *light_ptr;

        if(obj_ptr == NULL)
            return;

        for(i = 0; i < obj_ptr->total_lights; i++)
        {
            light_ptr = obj_ptr->light[i];
            if(light_ptr == NULL)
                continue;

            /* Skip non-attenuation lights. */
            if(!(light_ptr->flags & SAR_LIGHT_FLAG_ATTENUATE))
                continue;

            if(state)
            {
                /* Turn on. */
                light_ptr->flags |= SAR_LIGHT_FLAG_ON;
            }
            else
            {
                /* Turn off. */
                light_ptr->flags &= ~SAR_LIGHT_FLAG_ON;
            }
        }

        return;
}

/*
 *	Repairs the object if possible.
 *
 *	Does not check if the object is near any repair faclity, the
 *	calling function must do that.
 *
 *      Returns -1 on error, -2 if the object cannot be repaired or
 *      0 on success.
 */
int SARSimOpRepair(
        sar_scene_struct *scene, sar_object_struct *obj_ptr
)
{
	sar_contact_bounds_struct *cb;
	sar_object_aircraft_struct *obj_aircraft_ptr;

	if((scene == NULL) || (obj_ptr == NULL))
	    return(-1);

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
/*
		obj_aircraft_ptr->engine_state = SAR_ENGINE_STATE_ON;
 */
		obj_aircraft_ptr->air_worthy_state = SAR_AIR_WORTHY_FLYABLE;
		obj_ptr->hit_points = obj_ptr->hit_points_max;

		/* Reset crash flags. */
		cb = obj_ptr->contact_bounds;
		if(cb != NULL)
		{
		    cb->crash_flags = SAR_CRASH_FLAG_CRASH_OTHER;
		}
		/* Return success. */
		return(0);
	    }
	    break;

/* Add support for other objects that can be repaired here. */
	  default:
	    break;
	}

	return(-2);
}

/*
 *	Refuels the object if possible.
 *
 *      Does not check if the object is near any refueling faclity, the
 *      calling function must do that.
 *
 *	Returns -1 on error, -2 if the object cannot be refueled or
 *	0 on success.
 */
int SARSimOpRefuel(
        sar_scene_struct *scene, sar_object_struct *obj_ptr
)
{
	int eft_num;
        sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_object_fueltank_struct *obj_fueltank_ptr;
	sar_external_fueltank_struct *eft_ptr;


        if((scene == NULL) || (obj_ptr == NULL))
            return(-1);

        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
            if(obj_aircraft_ptr != NULL)
            {
		/* Set current fuel to amount of maximum fuel. */
		obj_aircraft_ptr->fuel = obj_aircraft_ptr->fuel_max;
		/* Replace and refuel all external reserved tanks. */
		for(eft_num = 0; eft_num < obj_aircraft_ptr->total_external_fueltanks; eft_num++)
		{
		    eft_ptr = obj_aircraft_ptr->external_fueltank[eft_num];
		    if(eft_ptr == NULL)
			continue;

		    eft_ptr->flags |= SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD;
		    eft_ptr->fuel = eft_ptr->fuel_max;
		}
                return(0);
            }
            break;

	  case SAR_OBJ_TYPE_FUELTANK:
	    obj_fueltank_ptr = (sar_object_fueltank_struct *)obj_ptr->data;
            if(obj_fueltank_ptr != NULL)
            {
		/* Set current fuel to amount of maximum fuel. */
		obj_fueltank_ptr->fuel = obj_fueltank_ptr->fuel_max;
		return(0);
	    }
	    break;

/* Add support for other objects that can be refueled here. */
          default:
            break;
        }

        return(-2);
}


/*
 *	Marks the given number of passengers to leave or drop from the
 *	given object as pending. If either value is -1 then the value
 *	in question will be interprited as `all'.
 *
 *	Returns non-zero on error.
 */
int SARSimOpPassengersSetLeave(  
        sar_scene_struct *scene, sar_object_struct *obj_ptr,
        int passengers_leave_pending, int passengers_drop_pending
)
{
	int *crew, *passengers, *passengers_max;
	double *passengers_mass;
	int *cur_passengers_leave_pending, *cur_passengers_drop_pending;


	if((scene == NULL) || (obj_ptr == NULL))
            return(-1);

	if(SARObjGetOnBoardPtr(
	    obj_ptr,
	    &crew, &passengers, &passengers_max,
	    &passengers_mass,
	    &cur_passengers_leave_pending,
	    &cur_passengers_drop_pending
	))
	    return(-2);

	if(passengers_max == NULL)
	    return(-2);

	/* Set passengers leave pending. */
	if(cur_passengers_leave_pending != NULL)
	{
	  if(passengers_leave_pending < 0)
	  {
	    if((cur_passengers_leave_pending != NULL) &&
               (passengers_max != NULL)
	    )
		(*cur_passengers_leave_pending) = (*passengers_max);
	  }
	  else
	  {
	    if(passengers_max != NULL)
	    {
		if(passengers_leave_pending > (*passengers_max))
		    passengers_leave_pending = (*passengers_max);

		if(cur_passengers_leave_pending != NULL)
		    (*cur_passengers_leave_pending) = passengers_leave_pending;
	    }
	  }
	}

        /* Set passengers drop pending. */
	if(cur_passengers_drop_pending != NULL)
	{
          if(passengers_drop_pending < 0)
          {
            if((cur_passengers_drop_pending != NULL) &&
               (passengers_max != NULL)
            )
                (*cur_passengers_drop_pending) = (*passengers_max);
          }
          else
          {
            if(passengers_max != NULL)
            {
                if(passengers_drop_pending > (*passengers_max))
                    passengers_drop_pending = (*passengers_max);
             
                if(cur_passengers_drop_pending != NULL)
                    (*cur_passengers_drop_pending) = passengers_drop_pending;
            }
	  }
        }

	return(0);
}

/*
 *	Unloads all passengers from the given object by checking if the
 *	object can have passengers and setting the object's passengers
 *	count to 0 and passengers mass to 0.
 *
 *	Does not check if unloading passengers is permitted, the calling
 *	function must do that.
 *
 *	Returns -1 on error, -2 if the object does not have passengers.
 *	Returns 0 on success (even if no passengers were unloaded).
 */
int SARSimOpPassengersUnloadAll(
        sar_scene_struct *scene, sar_object_struct *obj_ptr
)
{
	int *crew, *passengers, *passengers_max;
	double *passengers_mass;
	int *passengers_leave_pending, *passengers_drop_pending;


        if((scene == NULL) || (obj_ptr == NULL))
            return(-1);

	if(SARObjGetOnBoardPtr(
	    obj_ptr,
	    &crew, &passengers, &passengers_max,
	    &passengers_mass,
	    &passengers_leave_pending, &passengers_drop_pending
	))
	    return(-2);

	/* Got passengers? */
	if(passengers != NULL)
	{
	    /* Unload them. */
	    (*passengers) = 0;

	    /* Reset passengers mass if any. */
	    if(passengers_mass != NULL)
		(*passengers_mass) = 0.0;

	    /* Since all passengers gone, no pending to leave. */
	    if(passengers_leave_pending != NULL)
		(*passengers_leave_pending) = 0;
	    if(passengers_drop_pending != NULL)
		(*passengers_drop_pending) = 0;

	    return(0);
	}
	else
	{
	    return(-2);
	}
}



/*
 *	Transfers fuel from external reserve tanks on the object
 *	(if it has any) to the main fuel tank.
 *
 *	Returns the amount transfered or 0.0 on failure.
 */
double SARSimOpTransferFuelFromTanks(
        sar_scene_struct *scene, sar_object_struct *obj_ptr
)
{
	int i;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	double *fuel = NULL, *fuel_max = NULL;
	sar_external_fueltank_struct **eft = NULL, *eft_ptr;
	int eft_total = 0;
	double fuel_transfered = 0.0;

	if((scene == NULL) || (obj_ptr == NULL))
	    return(fuel_transfered);

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		fuel = &obj_aircraft_ptr->fuel;
		fuel_max = &obj_aircraft_ptr->fuel_max;
		eft = obj_aircraft_ptr->external_fueltank;
		eft_total = obj_aircraft_ptr->total_external_fueltanks;
	    }
	    break;

/* Add support for other types of objects that have fuel here. */

	}

	/* No fuel tanks to transfer to? */
	if((fuel == NULL) || (fuel_max == NULL))
	    return(fuel_transfered);

	/* No room for fuel on main tanks? */
	if((*fuel_max) <= 0.0)
	    return(fuel_transfered);

	/* Current fuel too much? */
	if((*fuel) > (*fuel_max))
	{
	    (*fuel) = (*fuel_max);
	    return(fuel_transfered);
	}
	/* Sanitize current fuel. */
	if((*fuel) < 0.0)
	    (*fuel) = 0.0;

	/* No fuel tanks to transfer from? */
	if((eft == NULL) || (eft_total < 1))
	    return(fuel_transfered);

	/* Itterate through each external reserved fuel tank. */
	for(i = 0; i < eft_total; i++)
	{
	    eft_ptr = eft[i];
	    if(eft_ptr == NULL)
		continue;

	    /* This external fuel tank on board? */
	    if(eft_ptr->flags & SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD)
	    {
		double fuel_needed = MAX((*fuel_max) - (*fuel), 0.0);
		double fuel_to_transfer;

		/* Check if external reserved tank has enough fuel,
		 * that we need to transfer. Then get the amount of
		 * fuel we can actually transfer.
		 */
		if(eft_ptr->fuel < fuel_needed)
		    fuel_to_transfer = eft_ptr->fuel;
		else
		    fuel_to_transfer = fuel_needed;

		/* Transfer fuel. */
		(*fuel) += fuel_to_transfer;
		eft_ptr->fuel -= fuel_to_transfer;

		/* Update total amount of fuel transfered. */
		fuel_transfered += fuel_to_transfer;
	    }
	}


	return(fuel_transfered);
}


/*
 *	Drops the `next' fuel tank that is on board the given object
 *	(if any).
 *
 *	Returns -1 on error, -2 if no tanks left to drop, or the
 *	index number of the newly created fuel tank object.
 *
 *	Inputs assumed valid except for obj_ptr and obj_num.
 */
int SARSimOpDropFuelTankNext(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        int obj_num, sar_object_struct *obj_ptr
)
{
	int i, ft_obj_num;
	sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
	sar_external_fueltank_struct **eft = NULL, *eft_ptr;
	int total_efts = 0;


	if(!SARObjIsAllocated(*ptr, *total, obj_num))
	    return(-1);
	if(obj_ptr == NULL)
	    return(-1);

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		eft = obj_aircraft_ptr->external_fueltank;
		total_efts = obj_aircraft_ptr->total_external_fueltanks;
	    }
	    break;
	}

	/* Itterate through existing fuel tanks. */
	for(i = 0; i < total_efts; i++)
	{
	    eft_ptr = eft[i];
	    if(eft_ptr == NULL)
		continue;

	    /* Not droppable? */
	    if(eft_ptr->flags & SAR_EXTERNAL_FUELTANK_FLAG_FIXED)
		continue;

	    /* Already dropped? */
	    if(!(eft_ptr->flags & SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD))
                continue;

	    /* Got valid on board external fuel tank to be dropped. */
	    break;
	}
	/* No more fuel tanks to drop? */
	if((i >= total_efts) || (eft_ptr == NULL))
	{
	    return(-2);
	}


	/* Got valid fuel tank i to drop, begin dropping it. */

	/* Mark it as not on board. */
	eft_ptr->flags &= ~SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD;

	/* Create a new fuel tank object. */
	ft_obj_num = SARObjCreate(
	    scene, ptr, total, SAR_OBJ_TYPE_FUELTANK
	);
	if(ft_obj_num > -1)
	{
	    double a[3 * 1], r[3 * 1];
	    sar_direction_struct *dir;
            sar_contact_bounds_struct *cb;
	    sar_object_struct *ft_obj_ptr = (*ptr)[ft_obj_num];
	    sar_object_fueltank_struct *obj_fueltank_ptr;

	    /* New fuel tank object valid? */
	    if(ft_obj_ptr != NULL)
	    {
		/* Set up objects on fuel tank object base structure. */

		ft_obj_ptr->flags = 0;
		if(obj_ptr->flags & SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH)
		    ft_obj_ptr->flags |= SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH;

		free(ft_obj_ptr->name);
		ft_obj_ptr->name = strdup("Fueltank");

		ft_obj_ptr->range = 1000.0;	/* 1000 meters. */

		ft_obj_ptr->ground_elevation_msl = obj_ptr->ground_elevation_msl;

		/* Contact bounds. */
                if(ft_obj_ptr->contact_bounds == NULL)
                    ft_obj_ptr->contact_bounds = (sar_contact_bounds_struct *)
                        calloc(1, sizeof(sar_contact_bounds_struct));
                cb = ft_obj_ptr->contact_bounds;
                if(cb != NULL)
                {
                    cb->crash_flags = 0;        /* Reset crash flags. */
                    cb->crash_type = SAR_GRASH_TYPE_AIRCRAFT;
		    /* Spherical contact bounds. */
                    SARObjAddContactBoundsSpherical(
			ft_obj_ptr,
                        cb->crash_flags,
			cb->crash_type,
			eft_ptr->radius
		    );
                } 

                ft_obj_ptr->life_span = 0;
                ft_obj_ptr->hit_points = 1.0;
                ft_obj_ptr->hit_points_max = 1.0;

		/* Remove existing visual model on new fueltank object if
		 * it has one.
		 */
		SARVisualModelUnref(scene, ft_obj_ptr->visual_model);

		/* Transfer visual model from eft_ptr's visual model
		 * to the new fuel tank object and increment ref count.
		 */
		ft_obj_ptr->visual_model = eft_ptr->visual_model;
		SARVisualModelRef(ft_obj_ptr->visual_model);


		/* Get fuel tank object substructure. */
		obj_fueltank_ptr = (sar_object_fueltank_struct *)ft_obj_ptr->data;
		if(obj_fueltank_ptr != NULL)
		{
		    sar_position_struct *vel;

		    obj_fueltank_ptr->flags = 0;
		    obj_fueltank_ptr->ref_object = obj_num;
		    obj_fueltank_ptr->dry_mass = eft_ptr->dry_mass;
		    obj_fueltank_ptr->fuel = eft_ptr->fuel;
		    obj_fueltank_ptr->fuel_max = eft_ptr->fuel_max;

		    /* Maximum vertical velocity in meters per cycle. */
		    obj_fueltank_ptr->vel_z_max = SFMFeetToMeters(-100.0) *
			SAR_SEC_TO_CYCLE_COEFF;

		    /* Get velocity from object. */
		    vel = &obj_fueltank_ptr->vel;
		    if(obj_aircraft_ptr != NULL)
		    {
			memcpy(vel, &obj_aircraft_ptr->vel, sizeof(sar_position_struct));
		    }
		    else
		    {
			memset(vel, 0x00, sizeof(sar_position_struct));
		    }

		    obj_fueltank_ptr->speed = sqrt(
			(vel->x * vel->x) + (vel->y * vel->y) +
			(vel->z * vel->z)
		    );
		}


		/* Move new fuel tank object to proper position. */
		dir = &obj_ptr->dir;

		a[0] = eft_ptr->offset_pos.x;
                a[1] = eft_ptr->offset_pos.y;
                a[2] = eft_ptr->offset_pos.z;

                /* Rotate matrix a into r. */
                MatrixRotateBank3(a, dir->bank, r);
                MatrixRotatePitch3(r, dir->pitch, a);
                MatrixRotateHeading3(a, dir->heading, r);

		ft_obj_ptr->pos.x = obj_ptr->pos.x + r[0];
                ft_obj_ptr->pos.y = obj_ptr->pos.y + r[1];
                ft_obj_ptr->pos.z = obj_ptr->pos.z + r[2];

		memcpy(&ft_obj_ptr->dir, dir, sizeof(sar_direction_struct));

		SARSimWarpObject(
		    scene, ft_obj_ptr,
		    &ft_obj_ptr->pos, &ft_obj_ptr->dir
		);
	    }
	}

	return(i);
}


/*
 *	Generates a new smoke unit on the given smoke object if
 *	possible. The given obj_ptr must be a smoke object.
 */
void SARSimSmokeSpawn(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr
)
{
	sar_object_smoke_struct *obj_smoke_ptr;


        if((core_ptr == NULL) || (obj_ptr == NULL))
            return;

	if(obj_ptr->type == SAR_OBJ_TYPE_SMOKE)
	    obj_smoke_ptr = (sar_object_smoke_struct *)obj_ptr->data;
	else
	    obj_smoke_ptr = NULL;
	if(obj_smoke_ptr == NULL)
	    return;

	if(obj_smoke_ptr->unit != NULL)
	{
	    int i;
	    sar_object_smoke_unit_struct *smoke_unit_ptr = NULL;

	    /* Look for available smoke unit structure. */
	    for(i = 0; i < obj_smoke_ptr->total_units; i++)
	    {
		smoke_unit_ptr = &obj_smoke_ptr->unit[i];

		/* This smoke unit available? */
		if(smoke_unit_ptr->visibility <= 0.0)
		    break;
	    }
	    /* Not able to find any available smoke units? */
	    if(i >= obj_smoke_ptr->total_units)
	    {
		/* Need to overwrite oldest (first) unit and `shift' the
		 * rest of the units, leaving the last one available.
		 */
		for(i = 0; i < (obj_smoke_ptr->total_units - 1); i++)
		    memcpy(
			&obj_smoke_ptr->unit[i],
			&obj_smoke_ptr->unit[i + 1],
			sizeof(sar_object_smoke_unit_struct)
		    );
		/* Get pointer to last unit if possible. */
		if(obj_smoke_ptr->total_units > 0)
		    smoke_unit_ptr = &obj_smoke_ptr->unit[
			obj_smoke_ptr->total_units - 1
		    ];
		else
		    smoke_unit_ptr = NULL;
	    }

	    /* Got available smoke unit structure? */
	    if(smoke_unit_ptr != NULL)
	    {
		/* This smoke unit is available, update its values. */

		/* Set initial position. */
		memcpy(
		    &smoke_unit_ptr->pos, &obj_ptr->pos,
		    sizeof(sar_position_struct)
		);
		/* Apply offset to initial position. */
		smoke_unit_ptr->pos.x += obj_smoke_ptr->respawn_offset.x;
                smoke_unit_ptr->pos.y += obj_smoke_ptr->respawn_offset.y;
                smoke_unit_ptr->pos.z += obj_smoke_ptr->respawn_offset.z;

		smoke_unit_ptr->radius = obj_smoke_ptr->radius_start;
		smoke_unit_ptr->visibility = 1.0;
	    }
	}
}

/*
 *	Returns True if the object is in slew mode.
 *
 *	The object must be of type SAR_OBJ_TYPE_AIRCRAFT, otherwise
 *	it will always return False.
 */
int SARSimIsSlew(sar_object_struct *obj_ptr)
{
        sar_object_aircraft_struct *obj_aircraft_ptr;


        if(obj_ptr == NULL)
            return(False);

	switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
            if(obj_aircraft_ptr == NULL)
                break;

	    return(obj_aircraft_ptr->flight_model_type == SAR_FLIGHT_MODEL_SLEW);
	    break;

	  default:
	    break;
	}

	return(False);
}

/*
 *	Sets the object to enter slew mode if enter_slew is True.
 *	Otherwise it returns the object to its previous flight mode.
 *
 *	The object must be of type SAR_OBJ_TYPE_AIRCRAFT.
 */
void SARSimSetSlew(
        sar_object_struct *obj_ptr, int enter_slew
)
{
	sar_object_aircraft_struct *obj_aircraft_ptr;


	if(obj_ptr == NULL)
	    return;

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr == NULL)
		break;

	    /* Enter slew mode? */
	    if(enter_slew &&
	       (obj_aircraft_ptr->flight_model_type != SAR_FLIGHT_MODEL_SLEW)
	    )
	    {
		/* Record current flight model type. */
		obj_aircraft_ptr->last_flight_model_type =
		    obj_aircraft_ptr->flight_model_type;

		/* Set flight model to slew mode. */
		obj_aircraft_ptr->flight_model_type =
		    SAR_FLIGHT_MODEL_SLEW;
	    }
	    /* Leave slew mode? */
            else if(!enter_slew &&
                    (obj_aircraft_ptr->flight_model_type == SAR_FLIGHT_MODEL_SLEW)
            )
            {
		/* Restore previous flight model type. */
		obj_aircraft_ptr->flight_model_type =
		    obj_aircraft_ptr->last_flight_model_type;
	    }
	}
}

/*
 *      Moves the human object specified by obj_human_ptr (substructure
 *	only) into the object specified by obj_ptr's rescue basket.
 *
 *	Note that the given obj_human_ptr may not be the substructure
 *	of the given obj_ptr.
 *
 *	All inputs assumed valid.
 *
 *      Returns 1 if the human object was actually picked up, or 0 on
 *	failure.
 */
int SARSimDoPickUpHuman(
	sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
	sar_object_human_struct *obj_human_ptr,	/* Not data structure of obj_ptr. */
	int human_obj_num
)
{
	int n;
	int passengers = 0, passengers_max = 0;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_obj_hoist_struct *hoist_ptr = NULL;


	if(scene == NULL)
	    return(0);

	/* Handle by object type. */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
		passengers = obj_aircraft_ptr->passengers;
		passengers_max = obj_aircraft_ptr->passengers_max;
	    }
	    break;
	}

	/* Did we get a hoist structure? */
	if(hoist_ptr == NULL)
	    return(0);

	/* No more room left onboard for additional passengers? */
	if(passengers >= passengers_max)
	{
	    /* Warn if this is the player object and the scene is
	     * not currently displaying any messages.
	     */
	    if((obj_ptr == scene->player_obj_ptr) &&
               (scene->message_display_untill <= 0)
	    )
	    {
		SARMessageAdd(
		    scene,
		    SAR_MESG_NO_ROOM_LEFT_FOR_PASSENGERS
		);
	    }
	    return(0);
	}

	/* Can hoist accomidate more (just limit to 1 for now)? */
	if(hoist_ptr->total_occupants >= 1)
	    return(0);


	/* Checks passed, begin adding human to hoist's list of
	 * occupants.
	 */

	/* Sanitize total. */
	if(hoist_ptr->total_occupants < 0)
	    hoist_ptr->total_occupants = 0;

	/* Allocate more pointers. */
	n = hoist_ptr->total_occupants;
	hoist_ptr->total_occupants++;
	hoist_ptr->occupant = (int *)realloc(
	    hoist_ptr->occupant,
	    hoist_ptr->total_occupants * sizeof(int)
	);
	if(hoist_ptr->occupant == NULL)
	{
	    hoist_ptr->total_occupants = 0;
	    return(0);
	}
	else
	{
	    hoist_ptr->occupant[n] = human_obj_num;
	}

	/* Update hoist's occupants mass (in kg). */
	hoist_ptr->occupants_mass += obj_human_ptr->mass;

	/* Update nessesary flags on human object by what is on
	 * the end of the hoist rope.
	 */
	switch(hoist_ptr->deployment)
	{
	  case SAR_HOIST_DEPLOYMENT_DIVER:
            obj_human_ptr->flags = ~SAR_HUMAN_FLAG_NEED_RESCUE;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_SIT;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_SIT_DOWN;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_SIT_UP;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_LYING;
            /* Leave SAR_HUMAN_FLAG_ALERT and SAR_HUMAN_FLAG_AWARE as is. */
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_IN_WATER;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_ON_STREATCHER;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_RUN;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_RUN_TOWARDS;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_RUN_AWAY;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_PUSHING;
            obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_DIVER_CATCHER;
            obj_human_ptr->flags |= SAR_HUMAN_FLAG_GRIPPED;
	    break;

	  default:	/* SAR_HOIST_DEPLOYMENT_BASKET */
	    obj_human_ptr->flags = ~SAR_HUMAN_FLAG_NEED_RESCUE;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_SIT;
	    obj_human_ptr->flags |= SAR_HUMAN_FLAG_SIT_DOWN;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_SIT_UP;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_LYING;
	    /* Leave SAR_HUMAN_FLAG_ALERT and SAR_HUMAN_FLAG_AWARE as is. */
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_IN_WATER;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_ON_STREATCHER;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_RUN;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_RUN_TOWARDS;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_RUN_AWAY;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_PUSHING;
	    obj_human_ptr->flags &= ~SAR_HUMAN_FLAG_DIVER_CATCHER;
	    obj_human_ptr->flags |= SAR_HUMAN_FLAG_GRIPPED;
	    break;
	}

 	return(1);
}


/*
 *	This function should be called each time the hoist is pulled
 *	into the aircraft. It will delete any objects referanced as
 *	occupants in the aircraft's hoist and update the
 *	passenger count.
 *
 *	Inputs are assumed valid, returns the number of objects hoisted
 *	in (thus the number of objects deleted).
 */
int SARSimDoHoistIn(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr
)
{
	int i, passenger_obj_num, is_player = 0;
	int prev_passengers = 0;
	int *passengers = NULL, *passengers_max = NULL;
	int objects_hoisted_in = 0;
	sar_obj_hoist_struct *hoist_ptr = NULL;
	sar_object_struct *passenger_obj_ptr;
	sar_scene_struct *scene;
	sar_object_human_struct *obj_human_ptr;


	scene = core_ptr->scene;
	if(scene == NULL)
	    return(objects_hoisted_in);

	/* Is this the player object? */
	if(scene->player_obj_ptr == obj_ptr)
	    is_player = 1;

	/* Get pointers to passenger counts on the given object. */
	SARObjGetOnBoardPtr(
	    obj_ptr,
	    NULL, &passengers, &passengers_max,
	    NULL,
	    NULL, NULL
	);
	if(passengers != NULL)
	    prev_passengers = (*passengers);

	/* Get pointer to hoist structure on the given object. */
	hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
	/* If object has no hoist then give up. */
	if(hoist_ptr == NULL)
	    return(objects_hoisted_in);

	/* Delete all objects in hoist. */
	for(i = 0; i < hoist_ptr->total_occupants; i++)
	{
	    passenger_obj_num = hoist_ptr->occupant[i];
	    if(SARObjIsAllocated(
                core_ptr->object,
                core_ptr->total_objects,
                passenger_obj_num
            ))
                passenger_obj_ptr = core_ptr->object[passenger_obj_num];
            else
                continue;

	    /* Do not delete the object that has the hoist, skip it. */
	    if(passenger_obj_ptr == obj_ptr)
		continue;

	    /* Handle by object type. */
	    switch(passenger_obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_HUMAN:
		obj_human_ptr = (sar_object_human_struct *)passenger_obj_ptr->data;
		if(obj_human_ptr != NULL)
		{
		    /* If this is the player object, then print `hoist-in
		     * message' specified on the human object.
		     */
		    if(is_player)
			SARMessageAdd(scene, obj_human_ptr->mesg_enter);
		}
		break;
	    }

            /* Move the passenger object into the given object, the
             * passenger object will be deleted if this call returns
             * true.
             *
             * Passenger count and related values will be updated on
             * the obj_ptr.
             */
            if(SARSimBoardObject(core_ptr, obj_ptr, passenger_obj_num))
	    {
		passenger_obj_num = -1;
		passenger_obj_ptr = NULL;
                objects_hoisted_in++;
	    }
	}

	/* Delete occupants list on hoist structure. */
	free(hoist_ptr->occupant);
	hoist_ptr->occupant = NULL;
	hoist_ptr->total_occupants = 0;


	/* Print new number of passengers on the given object. */
	if((passengers != NULL) && (passengers_max != NULL))
	{
	    /* Got more passengers? */
	    if(prev_passengers < (*passengers))
	    {
		/* Is this the player object? */
	        if(is_player)
	        {
		    char text[256];

		    sprintf(
			text,
			"Passengers: %i(%i)",
			(*passengers),
			(*passengers_max)
		    );
		    SARMessageAdd(scene, text);
		}
		else
		{
		    char name[SAR_OBJ_NAME_MAX];
		    char text[SAR_OBJ_NAME_MAX + 256];

		    if(obj_ptr->name == NULL)
		    {
			int obj_num = SARGetObjectNumberFromPointer(
			    scene, core_ptr->object, core_ptr->total_objects,
			    obj_ptr
			);
			if(SAR_OBJ_NAME_MAX > 80)
			    sprintf(name, "Object #%i\n", obj_num);
			else
			    (*name) = '\0';
		    }
		    else
		    {
			strncpy(name, obj_ptr->name, SAR_OBJ_NAME_MAX);
		    }
		    name[SAR_OBJ_NAME_MAX - 1] = '\0';

		    sprintf(text, "%s picked up %i passengers",
			name, ((*passengers) - prev_passengers)
		    );
		    SARMessageAdd(scene, text);
		}
	    }
	}

	return(objects_hoisted_in);
}

/*
 *	Updates the source object to have taken the target object
 *	boarding it. The source object may be deleted by this function.
 *
 *	The target object must be able to hold the source object (must
 *	(be able to accept passengers and have room for more passengers).
 *
 *	For example if the target object is an aircraft and the source
 *	object is a human, then the human object will be deleted and
 *	the source object will have its passenger count increment (if
 *	it has room for more passengers).
 *
 *	Does not check if the source object is within range to board the
 *	target object.
 *
 *	Returns 0 on failure to board or 1 if the source object has
 *	boarded the target and the source object no longer exists.
 */
int SARSimBoardObject(
        sar_core_struct *core_ptr,
        sar_object_struct *tar_obj_ptr, int src_obj_num
)
{
	int *passengers, *passengers_max;
	double *passengers_mass;
	double src_mass = 0.0;
	sar_object_struct *src_obj_ptr;
	sar_object_human_struct *obj_human_ptr;


	if((core_ptr == NULL) || (tar_obj_ptr == NULL))
	    return(0);

	/* Get source object pointer. */
	src_obj_ptr = SARObjGetPtr(
	    core_ptr->object, core_ptr->total_objects, src_obj_num
	);
	/* Got source object pointer and it is not the same as the
	 * target object?
	 */
	if((src_obj_ptr == NULL) || (src_obj_ptr == tar_obj_ptr))
	    return(0);

	/* Check if the target object cannot hold passengers. */
	if(SARObjGetOnBoardPtr(
	    tar_obj_ptr,
	    NULL, &passengers, &passengers_max,
	    &passengers_mass,
	    NULL, NULL
	))
	    return(0);

	if((passengers == NULL) || (passengers_max == NULL))
	    return(0);

	/* Target object has no more room for passengers? */
	if((*passengers) >= (*passengers_max))
	    return(0);

	/* Get some information about the source object by its type. */
	switch(src_obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_HUMAN:
	    obj_human_ptr = (sar_object_human_struct *)src_obj_ptr->data;
	    if(obj_human_ptr != NULL)
	    {
		src_mass = obj_human_ptr->mass;
	    }
	    break;
/* Add support for getting mass of other object types here. */
	}


	/* Begin moving source object into target object. */

	/* Increment passengers count. */
	if((*passengers) < 0)
	    (*passengers) = 0;
	(*passengers) = (*passengers) + 1;

	/* Increment passengers mass if possible. */
	if(passengers_mass != NULL)
	{
	    if((*passengers_mass) < 0.0)
		(*passengers_mass) = 0.0;

	    (*passengers_mass) = (*passengers_mass) + src_mass;
	}

	/* Delete the source object. */
        SARObjDelete(
	    core_ptr,
	    &core_ptr->object, &core_ptr->total_objects,
	    src_obj_num
	);

	return(1);
}

/*
 *	Calculates the `fly by view' position with respect to
 *	the given object. The result of the calculation is stored
 *	in the pos_result structure.
 */
void SARSimSetFlyByPosition(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
	sar_object_struct *obj_ptr,
	sar_position_struct *pos_result
)
{
	sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_direction_struct vel_dir;
	sar_position_struct *vel, *pos;
	double r, h, d, contact_radius;


	if((obj_ptr == NULL) || (pos_result == NULL))
	    return;

        contact_radius = SARSimGetFlatContactRadius(obj_ptr);

	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = obj_ptr->data;
            if(obj_aircraft_ptr == NULL)
                break;

	    pos = &obj_ptr->pos;
	    vel = &obj_aircraft_ptr->vel;

	    d = (5 * SAR_CYCLE_TO_SEC_COEFF) *
		obj_aircraft_ptr->speed;
	    if(d < (contact_radius * 2))
		d = contact_radius * 2;

            h = (5 * SAR_CYCLE_TO_SEC_COEFF) * vel->z;

	    r = sqrt(MAX((d * d) - (h * h), 0.0));
	    if(r < (contact_radius * 2))
                r = contact_radius * 2; 

	    vel_dir.heading = SFMSanitizeRadians(
	        ((PI / 2) - atan2(vel->y, vel->x)) +
		obj_ptr->dir.heading
	    );
	    pos_result->x = (r * sin(vel_dir.heading)) + pos->x -
		(contact_radius * 2);
	    pos_result->y = (r * cos(vel_dir.heading)) + pos->y -
		(contact_radius * 2);
	    pos_result->z = h + pos->z;

	    /* Don't position lower than object's center to ground. */
	    if(pos_result->z < (pos->z + obj_aircraft_ptr->center_to_ground_height))
		pos_result->z = pos->z + obj_aircraft_ptr->center_to_ground_height;

	    break;

	  default:

	    break;
	}
}

