#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "sarreality.h"

#include "obj.h"

#include "simcb.h"
#include "simutils.h"
#include "simcontact.h"


int SARSimCrashContactCheck(
        sar_core_struct *core_ptr,
        sar_object_struct *obj_ptr
);
int SARSimRescueBasketContactCheck(
        sar_core_struct *core_ptr,
        sar_object_struct *obj_ptr
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Checks if object obj_ptr has come in contact with another object
 *      on the given core structure.
 *                  
 *      Returns the object number of the object that has come in
 *      contact with or -1 if there was no contact.
 *
 *      Proper procedure will be performed if the object has crashed.
 */
int SARSimCrashContactCheck(
        sar_core_struct *core_ptr,
        sar_object_struct *obj_ptr
)
{           
        int i, src_contact_shape;
        double dr, z, d;
        double z1_min, z1_max, z2_min, z2_max;
        sar_object_struct *obj_ptr2;
        sar_contact_bounds_struct *cb_src, *cb_tar;
        sar_position_struct *pos_src, *pos_tar;
        sar_scene_struct *scene;
                    
        
        if((core_ptr == NULL) ||
           (obj_ptr == NULL)
        )
            return(-1);
 
        scene = core_ptr->scene;
        if(scene == NULL)
            return(-1);
        
        /* Get source object contact bounds, if this is NULL then
         * that implies the object cannot cause or crash into anything
         * so do not bother checking.
         */
        cb_src = obj_ptr->contact_bounds;
        if(cb_src == NULL)
            return(-1);
  
        src_contact_shape = cb_src->contact_shape;
   
        pos_src = &obj_ptr->pos;
        
        
        for(i = 0; i < core_ptr->total_objects; i++)
        {   
            obj_ptr2 = core_ptr->object[i];
            if(obj_ptr2 == NULL)
                continue;
        
            cb_tar = obj_ptr2->contact_bounds;
            if(cb_tar == NULL)
                continue;
                    
            if(!(cb_tar->crash_flags & SAR_CRASH_FLAG_CRASH_CAUSE))
                continue;

            if(obj_ptr2 == obj_ptr)     /* Skip itself. */
                continue;
 
            pos_tar = &obj_ptr2->pos;
        
            /* Handle by source object contact shape. */
            switch(src_contact_shape)
            {
              /* **************************************************** */
              case SAR_CONTACT_SHAPE_SPHERICAL:
                /* Handle by target object contact shape. */
                switch(cb_tar->contact_shape)
                {
                  case SAR_CONTACT_SHAPE_SPHERICAL:
                    /* Spherical to spherical. */
        
                    /* Check flat r distance for contact first. */
                    dr = hypot( 
                        pos_tar->x - pos_src->x,
                        pos_tar->y - pos_src->y
                    );
                    if((dr - cb_src->contact_radius - cb_tar->contact_radius)
                        > 0.0
                    )
                        break;
        
                    /* Calculate spherical radius and then check if
                     * in contact.
                     */  
                    z = pos_tar->z - pos_src->z;
                    d = hypot(dr, z);
                    if((d - cb_src->contact_radius - cb_tar->contact_radius)
                        <= 0.0
                    )
                    {
                        SARSimObjectCollisionCB(
                            core_ptr, obj_ptr, obj_ptr2, 1.1
                        );
                        return(i);
                    }
                    break;
              
                  case SAR_CONTACT_SHAPE_CYLENDRICAL:
                    /* Spherical to cylendrical. */
                
                    /* Check flat r distance for contact first. */
                    dr = hypot(
                        pos_tar->x - pos_src->x, 
                        pos_tar->y - pos_src->y
                    );
                    if((dr - cb_src->contact_radius - cb_tar->contact_radius)
                        > 0.0
                    )
                        break;
                    
                    /* Sloppy check on vertical height. */
                    z = pos_tar->z;
                    if((pos_src->z >= (z + cb_tar->contact_h_min)) &&
                       (pos_src->z <= (z + cb_tar->contact_h_max))
                    )
                    {
                        SARSimObjectCollisionCB(
                            core_ptr, obj_ptr, obj_ptr2, 1.1
                        );
                        return(i);
                    }
                    break;
                     
                  case SAR_CONTACT_SHAPE_RECTANGULAR:
                    /* Spherical to rectangular. */

                    /* We'll be treating the source object as a square
                     * object based on its contact radius.
                     */

                    /* First check z bounds. */
                    z1_min = pos_src->z - cb_src->contact_radius;
                    z1_max = pos_src->z + cb_src->contact_radius;
                    z2_min = pos_tar->z + cb_tar->contact_z_min;  
                    z2_max = pos_tar->z + cb_tar->contact_z_max;
                    if((z1_min <= z2_max) &&
                       (z2_min <= z1_max)
                    ) 
                    {
                        /* Calculate relative deltas from source object
                         * to target object.
                         */   
                        double  dx = pos_src->x - pos_tar->x,
                                dy = pos_src->y - pos_tar->y;
                        double dx2, dy2, x_min, x_max, y_min, y_max;
                    
                        /* Rotate dx and dy about the center of the target
                         * object inversly incase target object has
                         * a rotated heading.
                         */
                        dx2 = (cb_tar->cos_heading * dx) +  
                            (cb_tar->sin_heading * dy);
                        dy2 = (cb_tar->cos_heading * dy) -
                            (cb_tar->sin_heading * dx);
                    
                        /* Calculate x and y bounds, add source object's
                         * contact bounds to increase the bounds.
                         */
                        x_min = cb_tar->contact_x_min -
                            cb_src->contact_radius;
                        x_max = cb_tar->contact_x_max +   
                            cb_src->contact_radius;
                        y_min = cb_tar->contact_y_min -
                            cb_src->contact_radius;
                        y_max = cb_tar->contact_y_max +
                            cb_src->contact_radius;
                    
                        /* Check if rotated dx2 and dy2 are in bounds. */
                        if((dx2 >= x_min) && (dx2 <= x_max) &&
                           (dy2 >= y_min) && (dy2 <= y_max)
                        )
                        {
                            SARSimObjectCollisionCB(
                                core_ptr, obj_ptr, obj_ptr2, 1.1
                            );
                            return(i);
                        }
                    }
                    break;
                }
                break;
                         
              /* **************************************************** */
              case SAR_CONTACT_SHAPE_CYLENDRICAL:
                /* Handle by target object contact shape. */
                switch(cb_tar->contact_shape)
                {
                  case SAR_CONTACT_SHAPE_SPHERICAL:
                    /* Cylendrical to spherical. */
                         
                    /* Check flat r distance for contact first. */
                    dr = hypot(
                        pos_tar->x - pos_src->x,   
                        pos_tar->y - pos_src->y
                    );
                    if((dr - cb_src->contact_radius - cb_tar->contact_radius)
                        > 0.0
                    )
                        break;
                    
                    /* Sloppy check on vertical height. */
                    z = pos_tar->z;
                    if((z >= (pos_src->z + cb_src->contact_h_min)) &&
                       (z <= (pos_src->z + cb_src->contact_h_max))
                    )
                    {
                        SARSimObjectCollisionCB(
                            core_ptr, obj_ptr, obj_ptr2, 1.1
                        );
                        return(i);
                    }
                    break;
                 
                  case SAR_CONTACT_SHAPE_CYLENDRICAL:
                    /* Cylendrical to cylendrical. */
              
                    /* Check flat r distance for contact first. */
                    dr = hypot(
                        pos_tar->x - pos_src->x,
                        pos_tar->y - pos_src->y
                    );
                    if((dr - cb_src->contact_radius - cb_tar->contact_radius)
                        > 0.0
                    )
                        break; 
                        
                    /* Get vertical bounds. */ 
                    z1_min = pos_src->z + cb_src->contact_h_min;
                    z1_max = pos_src->z + cb_src->contact_h_max;
                    z2_min = pos_tar->z + cb_tar->contact_h_min;
                    z2_max = pos_tar->z + cb_tar->contact_h_max;
                    /* Check vertical bounds. */
                    if((z1_min <= z2_max) &&
                       (z2_min <= z1_max)
                    )
                    {
                        SARSimObjectCollisionCB(
                            core_ptr, obj_ptr, obj_ptr2, 1.1
                        );
                        return(i);
                    }
                    break;
                        
                  case SAR_CONTACT_SHAPE_RECTANGULAR:
                    /* Cylendrical to rectangular. */
                 
                    /* First check z bounds. */
                    z1_min = pos_src->z + cb_src->contact_h_min;
                    z1_max = pos_src->z + cb_src->contact_h_max;
                    z2_min = pos_tar->z + cb_tar->contact_z_min;  
                    z2_max = pos_tar->z + cb_tar->contact_z_max;
                    if((z1_min <= z2_max) &&
                       (z2_min <= z1_max)
                    ) 
                    {
                        /* Calculate relative deltas from source object
                         * to target object.
                         */    
                        double  dx = pos_src->x - pos_tar->x,
                                dy = pos_src->y - pos_tar->y;
                        double dx2, dy2, x_min, x_max, y_min, y_max;
              
                        /* Rotate dx and dy about the center of the target
                         * object inversly incase target object has
                         * a rotated heading.   
                         */
                        dx2 = (cb_tar->cos_heading * dx) +
                            (cb_tar->sin_heading * dy);
                        dy2 = (cb_tar->cos_heading * dy) -
                            (cb_tar->sin_heading * dx);
                            
                        /* Check if rotated dx and dy are in bounds
                         * with the target object contact bounds.
                         * Increase the bounds by the cylendrical
                         * radius of the source object.
                         */
                        x_min = cb_tar->contact_x_min -
                            cb_src->contact_radius;  
                        x_max = cb_tar->contact_x_max +
                            cb_src->contact_radius;
                        y_min = cb_tar->contact_y_min -
                            cb_src->contact_radius;
                        y_max = cb_tar->contact_y_max +
                            cb_src->contact_radius;

                        if((dx2 >= x_min) && (dx2 <= x_max) &&
                           (dy2 >= y_min) && (dy2 <= y_max)
                        )
                        {
                            SARSimObjectCollisionCB(
                                core_ptr, obj_ptr, obj_ptr2, 1.1
                            );
                            return(i);
                        }
                    }
                }
                break;

              /* **************************************************** */
              case SAR_CONTACT_SHAPE_RECTANGULAR:
/* Need to work on this. */

                break;
            }
        }

        return(-1);
}


/*
 *      Rescue basket contact check, checks if another object on the
 *	given core structure has come in contact with this object's
 *	rescue basket. If an object has come in contact then the proper
 *	procedure will be taken.
 *
 *      Inputs assumed valid.
 *
 *	Returns the object number on the core structure that has come
 *	in contact or -1 on no match.
 */
int SARSimRescueBasketContactCheck(
        sar_core_struct *core_ptr,
        sar_object_struct *obj_ptr
)
{
	int need_break;
        int tar_obj_num, matched_obj_num = -1;
        double d, basket_contact_radius;
        sar_scene_struct *scene;
        sar_object_struct *tar_obj_ptr;
        sar_object_human_struct *human_ptr;
        sar_obj_hoist_struct *hoist_ptr;
        sar_position_struct *pos_src, *pos_tar;
        sar_contact_bounds_struct *cb_tar;


	/* Get pointer to scene structure (must be valid). */
        scene = core_ptr->scene;
        if(scene == NULL)
            return(matched_obj_num);

	/* Get pointer to first hoist on object if any. */
	hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
	if(hoist_ptr == NULL)
	    return(matched_obj_num);

        /* Hoist fully retracted? */
        if(hoist_ptr->rope_cur <= 0)
            return(matched_obj_num);

        /* Get the rescue basket's flat contact radius, first check
	 * if this is the player object which can have a rescue basket
	 * expansion option set. Multiple radius by 2.0 for easier
	 * contact.
	 */
        if(scene->player_obj_ptr == obj_ptr)
            basket_contact_radius = hoist_ptr->contact_radius * 2.0 *
                option.hoist_contact_expansion_coeff;
        else
            basket_contact_radius = hoist_ptr->contact_radius * 2.0;

        /* Get source position as the position of the rescue basket. */
        pos_src = &hoist_ptr->pos;


	/* Go through all objects on core structure. */
        for(tar_obj_num = 0; tar_obj_num < core_ptr->total_objects; tar_obj_num++)
        {
	    tar_obj_ptr = core_ptr->object[tar_obj_num];
            if(tar_obj_ptr == NULL)
                continue;

	    /* Get contact bounds for target object. */
            cb_tar = tar_obj_ptr->contact_bounds;
            if(cb_tar == NULL)
                continue;

            /* Contact shapes for target object must be cylendrical or
	     * spherical shaped.
             */
/*
            if((cb_tar->contact_shape != SAR_CONTACT_SHAPE_SPHERICAL) &&
               (cb_tar->contact_shape != SAR_CONTACT_SHAPE_CYLENDRICAL)
            )
                continue;
 */

	    /* Handle by target object type. */
            switch(tar_obj_ptr->type)
	    {
              case SAR_OBJ_TYPE_HUMAN:
                human_ptr = (sar_object_human_struct *)tar_obj_ptr->data;
                if(human_ptr == NULL)
                    break;

                /* This human want to be picked up? */
                if(!(human_ptr->flags & SAR_HUMAN_FLAG_NEED_RESCUE))
                    break;

		/* Get target position. */
                pos_tar = &tar_obj_ptr->pos;


		/* Check contact bounds by target object's contact
		 * shape. Reset need_break to 0, it will be set to 1 if
		 * there was NO contact.
		 */
		need_break = 0;
		switch(cb_tar->contact_shape)
		{
		  case SAR_CONTACT_SHAPE_SPHERICAL:
                    /* Calculate distance. */
                    d = hypot(
                        pos_tar->x - pos_src->x,
                        pos_tar->y - pos_src->y
                    );
                    /* Within rescue basket pickup distance? */
                    if((d - basket_contact_radius - cb_tar->contact_radius) >
                        0.0
                    )
		    {
			need_break = 1;
			break;
		    }
		    /* Is target object under the rescue basket? */
		    if((pos_tar->z + cb_tar->contact_radius) <
			(pos_src->z + hoist_ptr->contact_z_min)
		    )
		    {
			need_break = 1;
			break;
		    }
		    break;

		  case SAR_CONTACT_SHAPE_CYLENDRICAL:
                    /* Calculate distance. */
                    d = hypot(
                        pos_tar->x - pos_src->x,
                        pos_tar->y - pos_src->y
                    );
                    /* Within rescue basket pickup distance? */
                    if((d - basket_contact_radius - cb_tar->contact_radius) >
                        0.0
                    )
                    {
                        need_break = 1;
                        break;
                    }
		    /* Is target object under the rescue basket? */
                    if((pos_tar->z + cb_tar->contact_h_max) <
                        (pos_src->z + hoist_ptr->contact_z_min) 
                    )
		    {
                        need_break = 1;
			break;
		    }
                    break;

		  case SAR_CONTACT_SHAPE_RECTANGULAR:
                    /* Calculate distance. */
                    d = hypot(
                        pos_tar->x - pos_src->x,
                        pos_tar->y - pos_src->y
                    );
                    /* Within rescue basket pickup distance? */
/* Using target object's x bounds for now. */
                    if((d - basket_contact_radius -
			(cb_tar->contact_x_max - cb_tar->contact_x_min)) >
                        0.0
                    )
                    {
                        need_break = 1;
                        break;
                    }
		    /* Is target object under the rescue basket? */
                    if((pos_tar->z + cb_tar->contact_z_max) <
                        (pos_src->z + hoist_ptr->contact_z_min)
                    )
		    {
                        need_break = 1;
			break;
		    }
		    break;
		}
		if(need_break)
		    break;

                /* Put human in basket and set as no longer
                 * needing rescue.
                 */
		SARSimDoPickUpHuman(scene, obj_ptr, human_ptr, tar_obj_num);
		matched_obj_num = tar_obj_num;
                return(matched_obj_num);

                break;

/* Add support for other object types that can be picked up. */

            }
        }

        return(matched_obj_num);
}

