#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <sys/stat.h>

#include "../include/fio.h"
#include "../include/string.h"
#include "../include/disk.h"

#include "menu.h"
#include "weather.h"
#include "sar.h"
#include "config.h"


char *SARGetFileObjectName(const char *path);

static int SARMenuBuildStandardButton(
	sar_core_struct *core_ptr, sar_menu_struct *menu_ptr,
        double x, double y,
        int w, int h,
        const char *label,
        int id_code
);
static int SARMenuBuildButtonImage(
        sar_core_struct *core_ptr, sar_menu_struct *menu_ptr,
        double x, double y,
        int w, int h,
        const char *label,
        int id_code,
	sar_image_struct *unarmed_img,
	sar_image_struct *armed_img,
	sar_image_struct *highlighted_img
);
static int SARMenuBuildStandardSwitch(
	sar_core_struct *core_ptr, sar_menu_struct *menu_ptr,
        double x, double y,
        int w, int h,
        const char *label,
        Boolean state,
        int id_code,
        void (*change_cb)(void *, void *, int, Boolean)
);
int SARMenuBuildStandardSpin(
	sar_core_struct *core_ptr, sar_menu_struct *menu_ptr,
	double x, double y,
	double w, double h,
	const char *label,
	int id_code,
	void (*change_cb)(void *, void *, int, char *)
);
char *SARMenuBuildGetFullPath(const char *name);

void SARBuildMenusAddToList(
	sar_core_struct *core_ptr, sar_menu_struct *menu_ptr
);

void SARBuildMenusAddDirectory(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu_ptr, int list_num,
	const char *subdir
);

int SARBuildMenus(sar_core_struct *core_ptr);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))



/*
 *	Returns a statically allocated string containing the object's
 *	title (if any) or returns just the contents of path on failure.
 */
char *SARGetFileObjectName(const char *path)
{
	FILE *fp;
	char *strptr;
	const char *cstrptr;
	static char name[SAR_OBJ_NAME_MAX];


	if(path == NULL)
	    return("(null)");

	(*name) = '\0';	/* Reset name. */

#define DO_RETURN_JUST_PATH	\
{ \
 strptr = strrchr(path, '/'); \
 cstrptr = ((strptr == NULL) ? path : strptr + 1); \
 strncpy(name, cstrptr, SAR_OBJ_NAME_MAX); \
 name[SAR_OBJ_NAME_MAX - 1] = '\0'; \
}

	/* Open file for reading. */
	fp = FOpen(path, "rb");
	if(fp == NULL)
	{
	    DO_RETURN_JUST_PATH
	    return(name);
	}

        /* Name. */
        if(!FSeekToParm(fp, "name", SAR_COMMENT_CHAR, '\0'))
        {
	    strptr = FGetString(fp);
	    if(strptr != NULL)
	    {
		strncpy(name, strptr, SAR_OBJ_NAME_MAX);
		name[SAR_OBJ_NAME_MAX - 1] = '\0';

		free(strptr);
	    }
	}

	FClose(fp);

	/* Did we get a name? */
	if((*name) == '\0')
	    DO_RETURN_JUST_PATH

#undef DO_RETURN_JUST_PATH

	return(name);
}


/*
 *	Procedure to create a standard button.
 *
 *	Returns whatever SARMenuCreateButton() returns.
 */
static int SARMenuBuildStandardButton(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu_ptr,
	double x, double y,
	int w, int h,		/* Entire size of button + label. */
	const char *label,
	int id_code
)
{
	int n, bw, w2, h2;
	sar_menu_color_struct button_color;
	sar_menu_label_struct *label_ptr;
	sar_menu_button_struct *button_ptr;


        /* Button text color. */
        button_color.a = 1.0;
        button_color.r = 0.9;
        button_color.g = 0.9;
        button_color.b = 0.9;

	if(core_ptr == NULL)
	    return(-1);

	/* Calculate button width (normal size of button is 42x50). */
	bw = (int)((double)h * 42.0 / 50.0);

	/* Calculate label size. */
	w2 = MAX(w - bw, 0);
	h2 = h;
	/* Create label. */ 
	n = SARMenuCreateLabel(
	    menu_ptr, x, y,
	    w2, h2,
	    label,
	    &button_color,
	    option.menu_font,
	    core_ptr->menu_button_label_img
	);
	if(n > -1)
	{
	    label_ptr = (sar_menu_label_struct *)menu_ptr->object[n];
	    label_ptr->align = SAR_MENU_LABEL_ALIGN_CENTER;
	    label_ptr->offset_x = bw / 2;
	}

	/* Create button. */
	n = SARMenuCreateButton(
	    menu_ptr, x, y,
	    bw, h,
	    NULL,		/* No label. */
	    &button_color, option.menu_font,
	    core_ptr->menu_button_unarmed_img,
	    core_ptr->menu_button_armed_img,
	    core_ptr->menu_button_highlighted_img,
	    core_ptr, id_code,
	    SARMenuButtonCB
	);
        if(n > -1)
        {
            button_ptr = (sar_menu_button_struct *)menu_ptr->object[n];
            button_ptr->offset_x = (int)((bw / 2.0) - (w / 2.0));
        }

	return(n);
}

/*
 *	Similar to SARMenuBuildStandardButton() but you get to specify
 *	the images.
 */
static int SARMenuBuildButtonImage(
        sar_core_struct *core_ptr, sar_menu_struct *menu_ptr,
        double x, double y,
        int w, int h,
        const char *label,
        int id_code,
        sar_image_struct *unarmed_img,
        sar_image_struct *armed_img,
        sar_image_struct *highlighted_img
)
{
        sar_menu_color_struct button_color;

	/* Button text color. */
        button_color.a = 1.0;
        button_color.r = 0.9;
        button_color.g = 0.9;
        button_color.b = 0.9;

        if(core_ptr == NULL)
            return(-1);

        return(
            SARMenuCreateButton(
                menu_ptr, x, y, w, h,
                label,
                &button_color, option.menu_font,
                unarmed_img,
                armed_img,
                highlighted_img,
                core_ptr, id_code,
                SARMenuButtonCB
            )
        );
}

/*
 *	Procedure to create a standard switch.
 *
 *      Returns whatever SARMenuCreateSwitch() returns.
 */
static int SARMenuBuildStandardSwitch(
        sar_core_struct *core_ptr,
        sar_menu_struct *menu_ptr,
        double x, double y,
        int w, int h,
	const char *label,
	Boolean state,
        int id_code,
        void (*change_cb)(void *, void *, int, Boolean)
)
{
        sar_menu_color_struct switch_color;

        if(core_ptr == NULL)
            return(-1);

        switch_color.a = 1.0;
        switch_color.r = 0.9;
        switch_color.g = 0.9;
        switch_color.b = 0.9;

	return(
	    SARMenuCreateSwitch(
		menu_ptr, x, y, w, h, 
		&switch_color, option.menu_font,
		label,
                core_ptr->menu_switch_bg_img,
                core_ptr->menu_switch_off_img,
                core_ptr->menu_switch_on_img,
		state,
		core_ptr, id_code,
		change_cb
            )
        );
}


/*
 *      Procedure to create a standard spin.
 *
 *	Returns whatever SARMenuCreateSpin() returns.
 */
int SARMenuBuildStandardSpin(
	sar_core_struct *core_ptr,
        sar_menu_struct *menu_ptr,
        double x, double y,
        double w, double h,
	const char *label,
        int id_code,
	void (*change_cb)(void *, void *, int, char *)
)
{
        sar_menu_color_struct label_color, value_color;

        label_color.a = 1.0;
        label_color.r = 0.9;
        label_color.g = 0.9;
        label_color.b = 0.9;

	value_color.a = 1.0;
	value_color.r = 0.0;
	value_color.g = 0.9;
	value_color.b = 0.0;


        if(core_ptr == NULL)
            return(-1);

	return(
	    SARMenuCreateSpin(
                menu_ptr, x, y, w, h,
		&label_color, &value_color,
		option.menu_font, option.message_font, label,
                core_ptr->menu_spin_label_img,
                core_ptr->menu_spin_value_img,
                core_ptr, id_code,
		change_cb
	    )
	);
}


/*
 *	Prefixes local data dir to name and checks if file exists,
 *	if not then prefixes global data dir regardless if it exists
 *	or not.
 *
 *	Return is statically allocated.
 */
char *SARMenuBuildGetFullPath(const char *name)
{
	char *strptr;
	struct stat stat_buf;
	static char path[PATH_MAX + NAME_MAX];

	if(name == NULL)
	    return("");

	strptr = PrefixPaths(dname.local_data, name);
	if(strptr == NULL)
	    return("");

	strncpy(path, strptr, PATH_MAX + NAME_MAX);
	path[PATH_MAX + NAME_MAX - 1] = '\0';

	if(stat(path, &stat_buf))
	{
	    /* Does not exist locally, prefix global. */
	    strptr = PrefixPaths(dname.global_data, name);
	    if(strptr == NULL)
		return("");

	    strncpy(path, strptr, PATH_MAX + NAME_MAX);
            path[PATH_MAX + NAME_MAX - 1] = '\0';
	}

	return(path);
}

/*
 *	Adds the specified menu to the list of menus on the core
 *	structure.
 */
void SARBuildMenusAddToList(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu_ptr
)
{
	int n;

	if(core_ptr->total_menus < 0)
	    core_ptr->total_menus = 0;

	n = core_ptr->total_menus;

	core_ptr->total_menus++;
	core_ptr->menu = (sar_menu_struct **)realloc(
	    core_ptr->menu,
	    core_ptr->total_menus * sizeof(sar_menu_struct *)
	);
	if(core_ptr->menu == NULL)
	{
	    core_ptr->total_menus = 0;
	    return;
	}

	core_ptr->menu[n] = menu_ptr;

	return;
}

/*
 *	Adds directory listing to the specified list.
 *
 *	Local data directory listing will be added first, then global.
 */
void SARBuildMenusAddDirectory(
	sar_core_struct *core_ptr,
	sar_menu_struct *menu_ptr, int list_num,
	const char *subdir	/* Subdir to local and global data dirs. */
)
{
	int i, strc;
	char *strptr, *strptr2, *parent, *item_name;
	char **strv;
	sar_menu_list_item_data_struct *li_data_ptr;
        char tmp_path[PATH_MAX + NAME_MAX];
	struct stat stat_buf;


	if((core_ptr == NULL) ||
           (menu_ptr == NULL) ||
           (subdir == NULL)
	)
	    return;

	/* Complete path to local directory. */
	parent = dname.local_data;

        strptr = PrefixPaths(parent, subdir);
        if(strptr != NULL)
        {
            strncpy(tmp_path, strptr, PATH_MAX + NAME_MAX);
            tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

	    /* Get listing. */
            strv = GetDirEntNames(tmp_path);
            if(strv != NULL)
            {
		/* Get total number of strings. */
		for(strc = 0; strv[strc] != NULL; strc++);

		/* Sort array. */
		strv = StringQSort(strv, strc);

		/* Add each listing. */
		for(i = 0; i < strc; i++)
                {
                    strptr = strv[i];

		    /* Skip directories. */
                    if(!strcasecmp(strptr, ".") ||
                       !strcasecmp(strptr, "..")
                    )
		    {
			free(strptr);
			continue;
		    }

		    /* Get complete file name. */
		    strptr2 = PrefixPaths(tmp_path, strptr);
		    if(stat(strptr2, &stat_buf))
		    {
                        free(strptr);
                        continue;
                    }
#ifdef S_ISDIR
		    if(S_ISDIR(stat_buf.st_mode))
		    {
                        free(strptr);  
                        continue;
		    }
#endif	/* S_ISDIR */

		    /* Allocate client data. */
                    li_data_ptr = (sar_menu_list_item_data_struct *)calloc(
                        1,
                        sizeof(sar_menu_list_item_data_struct)
                    );
                    if(li_data_ptr != NULL)
                    {
			/* Add values to client data. */
                        li_data_ptr->core_ptr = core_ptr;
                        li_data_ptr->filename = StringCopyAlloc(strptr2);

			item_name = SARGetFileObjectName(li_data_ptr->filename);
                    }
		    else
		    {
			item_name = strptr;
		    }
		    /* Add item to list. */
                    SARMenuListAppendItem(
                        menu_ptr, list_num,
                        item_name,	/* Item name. */
                        li_data_ptr,	/* Client data. */
                        0		/* Flags. */
                    );

		    /* Free this entry. */
                    free(strptr);
                }

		/* Free pointer array (each pointer storage already
		 * free'ed.
		 */
                free(strv);
            }
	}

/* No global data for __MSW__. */
#ifndef __MSW__
        /* Complete path to global data directory. */
        parent = dname.global_data;

        strptr = PrefixPaths(parent, subdir);
        if(strptr != NULL)
        {
            strncpy(tmp_path, strptr, PATH_MAX + NAME_MAX);
            tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';
        
            /* Get listing. */
            strv = GetDirEntNames(tmp_path);
            if(strv != NULL)
            {
                /* Get total number of strings. */
                for(strc = 0; strv[strc] != NULL; strc++);

                /* Sort array. */
                strv = StringQSort(strv, strc);

                /* Add each listing. */
                for(i = 0; i < strc; i++)
                {
                    strptr = strv[i];

                    /* Skip directories. */
                    if(!strcasecmp(strptr, ".") ||
                       !strcasecmp(strptr, "..")
                    )
                    {
                        free(strptr);
                        continue;
		    }

                    /* Get complete file name. */
                    strptr2 = PrefixPaths(tmp_path, strptr);
                    if(stat(strptr2, &stat_buf))
                    {
                        free(strptr);
                        continue;
                    }
#ifdef S_ISDIR
                    if(S_ISDIR(stat_buf.st_mode))
                    {   
                        free(strptr);
                        continue;
                    }
#endif	/* S_ISDIR */

                    /* Allocate client data. */
                    li_data_ptr = (sar_menu_list_item_data_struct *)calloc(
                        1,
                        sizeof(sar_menu_list_item_data_struct)
                    );
                    if(li_data_ptr != NULL)
                    {
                        /* Add values to client data. */
                        li_data_ptr->core_ptr = core_ptr;
                        li_data_ptr->filename = StringCopyAlloc(strptr2);

                        item_name = SARGetFileObjectName(li_data_ptr->filename);
                    }
                    else
                    {
                        item_name = strptr;
                    }
                    /* Add item to list. */
                    SARMenuListAppendItem(
                        menu_ptr, list_num,
                        item_name,      /* Item name. */
                        li_data_ptr,    /* Client data. */
                        0               /* Flags. */
                    );

                    /* Free this entry. */
                    free(strptr);
                }

                /* Free pointer array (each pointer storage already
                 * free'ed.
                 */
                free(strv);
            }
        }
#endif	/* Not __MSW__ */

	return;
}


/*
 *	Builds menus.
 */
int SARBuildMenus(sar_core_struct *core_ptr)
{
	int list_num, spin_num, switch_num, map_num;
	sar_menu_struct *menu_ptr;
	sar_menu_list_struct *list_ptr;
	sar_menu_spin_struct *spin_ptr;
	char *strptr;
	sar_menu_color_struct	list_color, mesgbox_color,
				switch_color, progress_color;
	GWFont *font, *value_font;
	const char *img_path;
	sar_image_struct **img;

/* Standard button size is 42x50 and button label is 208x50 */
int btn_width_std = 180, btn_height_std = 40;
int btn_width_main = 180, btn_height_main = 40;
int btn_width_opt = 250, btn_height_opt = 40;
int btn_width_map = 32, btn_height_map = 32;

	list_color.a = 1.0;
	list_color.r = 0.0;
	list_color.g = 0.9;
	list_color.b = 0.0;

	mesgbox_color.a = 1.0;
	mesgbox_color.r = 0.0;
	mesgbox_color.g = 0.9;
	mesgbox_color.b = 0.0;

	switch_color.a = 1.0;
	switch_color.r = 0.9;
	switch_color.g = 0.9;
	switch_color.b = 0.9;

	progress_color.a = 1.0;
	progress_color.r = 0.9;
	progress_color.g = 0.9;
	progress_color.b = 0.9;

	font = option.menu_font;
	value_font = option.message_font;


	if(core_ptr == NULL)
	    return(-1);

	/* Load images for menus (these will be shared amoung the
	 * menu object/widgets.
	 */
#define DO_LOAD_IMAGE	\
{ \
 if((*img) != NULL) \
 { \
  SARImageDestroy(*img); \
  (*img) = NULL; \
 } \
 (*img) = SARImageLoadFromFile(img_path); \
}

	core_ptr->menu_list_bg_img = (sar_image_struct **)realloc(
	    core_ptr->menu_list_bg_img,
	    9 * sizeof(sar_image_struct *)
	);
	if(core_ptr->menu_list_bg_img != NULL)
	{
	    int i;
	    char tmp_path[PATH_MAX + NAME_MAX];

	    for(i = 0; i < 9; i++)
	    {
		sprintf(tmp_path, "images/list%i.tga", i);
		img_path = tmp_path;

		img = &core_ptr->menu_list_bg_img[i];
		(*img) = NULL;

		DO_LOAD_IMAGE
	    }
	}

        img_path = "images/button_armed.tga";
        img = &core_ptr->menu_button_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_unarmed.tga";    
        img = &core_ptr->menu_button_unarmed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_highlighted.tga";
        img = &core_ptr->menu_button_highlighted_img;
        DO_LOAD_IMAGE
        img_path = "images/button_label.tga";
	img = &core_ptr->menu_button_label_img;
        DO_LOAD_IMAGE


	img_path = "images/label_bg.tga";
	img = &core_ptr->menu_label_bg_img;
	DO_LOAD_IMAGE


        img_path = "images/label_bg.tga";	/* Use label for now. */
        img = &core_ptr->menu_switch_bg_img;
        DO_LOAD_IMAGE

	img_path = "images/switch_off.tga";
	img = &core_ptr->menu_switch_off_img;
	DO_LOAD_IMAGE

        img_path = "images/switch_on.tga";
        img = &core_ptr->menu_switch_on_img;
        DO_LOAD_IMAGE


	img_path = "images/progress_bar_bg.tga";
	img = &core_ptr->menu_progress_bg_img;
	DO_LOAD_IMAGE

	img_path = "images/progress_bar_fg.tga";
        img = &core_ptr->menu_progress_fg_img;
	DO_LOAD_IMAGE

	/* Map panning buttons. */
	img_path = "images/button_pan_up_armed.tga";
        img = &core_ptr->menu_button_pan_up_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_pan_up_unarmed.tga";
        img = &core_ptr->menu_button_pan_up_unarmed_img;
        DO_LOAD_IMAGE

        img_path = "images/button_pan_down_armed.tga";
        img = &core_ptr->menu_button_pan_down_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_pan_down_unarmed.tga";
        img = &core_ptr->menu_button_pan_down_unarmed_img;
        DO_LOAD_IMAGE

        img_path = "images/button_pan_left_armed.tga";
        img = &core_ptr->menu_button_pan_left_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_pan_left_unarmed.tga";
        img = &core_ptr->menu_button_pan_left_unarmed_img;
        DO_LOAD_IMAGE

        img_path = "images/button_pan_right_armed.tga";
        img = &core_ptr->menu_button_pan_right_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_pan_right_unarmed.tga";
        img = &core_ptr->menu_button_pan_right_unarmed_img;
        DO_LOAD_IMAGE

	/* Map zoom buttons. */
        img_path = "images/button_zoom_in_armed.tga";
        img = &core_ptr->menu_button_zoom_in_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_zoom_in_unarmed.tga";
        img = &core_ptr->menu_button_zoom_in_unarmed_img;
        DO_LOAD_IMAGE

        img_path = "images/button_zoom_out_armed.tga";
        img = &core_ptr->menu_button_zoom_out_armed_img;
        DO_LOAD_IMAGE
        img_path = "images/button_zoom_out_unarmed.tga";
        img = &core_ptr->menu_button_zoom_out_unarmed_img;
        DO_LOAD_IMAGE


	/* Spin images. */
        img_path = "images/spin_label.tga";
        img = &core_ptr->menu_spin_label_img;
        DO_LOAD_IMAGE

        img_path = "images/spin_value.tga";
        img = &core_ptr->menu_spin_value_img;
        DO_LOAD_IMAGE

	/* Menu's map icon images. */
        img_path = SAR_DEF_MISSION_MAPICON_HELIPAD_FILE;
        img = &core_ptr->menumap_helipad_img;
        DO_LOAD_IMAGE
        img_path = SAR_DEF_MISSION_MAPICON_INTERCEPT_FILE;
        img = &core_ptr->menumap_intercept_img;
        DO_LOAD_IMAGE
        img_path = SAR_DEF_MISSION_MAPICON_HELICOPTER_FILE;
        img = &core_ptr->menumap_helicopter_img;
        DO_LOAD_IMAGE
        img_path = SAR_DEF_MISSION_MAPICON_VICTIM_FILE;
        img = &core_ptr->menumap_victim_img;
        DO_LOAD_IMAGE
        img_path = SAR_DEF_MISSION_MAPICON_BOAT_FILE;
        img = &core_ptr->menumap_boat_img;
        DO_LOAD_IMAGE


#undef DO_LOAD_IMAGE

	/* ****************************************************** */
	/* Main Menu (must be the first menu) */
	strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_MAIN_FILE);
	menu_ptr = SARMenuCreate(
	    SAR_MENU_TYPE_STANDARD,
	    SAR_MENU_NAME_MAIN,
	    strptr
	);
	if(menu_ptr == NULL)
	    return(-1);
	else
	    SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Mission button. */
	SARMenuBuildStandardButton(
	    core_ptr, menu_ptr, 0.18, 0.23, btn_width_main, btn_height_main,
	    "Mission",
            SAR_OP_MENU_TO_MISSION
        );

	/* Campaign button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.18, 0.37, btn_width_main, btn_height_main,
            "Campaign",
            SAR_OP_MENU_TO_CAMPAIGN
        );

	/* Free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.18, 0.51, btn_width_main, btn_height_main,
            "Free Flight",
            SAR_OP_MENU_TO_FREE_FLIGHT
        );

        /* Options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.18, 0.65, btn_width_main, btn_height_main,
            "Options",
            SAR_OP_MENU_TO_OPTIONS
        );

        /* Exit button. */
	SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.18, 0.79, btn_width_main, btn_height_main,
            "Exit",
            SAR_OP_MENU_EXIT
        );


        /* ****************************************************** */
        /* Mission: Select Mission */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION,
            strptr
        );
        if(menu_ptr == NULL)   
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Missions list. */
        list_num = SARMenuCreateList(
            menu_ptr, 0.50, 0.42, 0.95, 0.70,
            &list_color,
            value_font,
            "Select Mission:",
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
            SARMenuListSelectCB
        );
        SARBuildMenusAddDirectory(
            core_ptr, menu_ptr, list_num,
            SAR_DEF_MISSIONS_DIR
        );
        if(list_num > -1)
        {
            /* Restore previously selected mission. */
            list_ptr = (sar_menu_list_struct *)menu_ptr->object[list_num];

            list_ptr->selected_item = option.last_selected_mission;
	    list_ptr->scrolled_item = MAX(
		list_ptr->selected_item - 1,
		0
	    );
        }

        /* Back to main button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_MAIN
        );

        /* Mission briefing button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.75, 0.92, btn_width_std, btn_height_std,
            "Briefing", SAR_OP_MENU_TO_MISSION_BRIEF
        );


        /* ****************************************************** */
        /* Mission: Briefing */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION_BRIEF,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Mission briefing message box. */
        SARMenuCreateMessageBox(
            menu_ptr, 0.50, 0.38, 0.95, 0.60,
            &mesgbox_color,
            value_font,
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
            NULL        /* No message. */
        );

        /* Back to missions list. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_MISSION
        );

        /* Mission map button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.74, btn_width_std, btn_height_std,
            "Map", SAR_OP_MENU_TO_MISSION_MAP
        );

        /* Mission begin button. */
        SARMenuBuildStandardButton(   
            core_ptr, menu_ptr, 0.75, 0.92, btn_width_std, btn_height_std,
            "Begin", SAR_OP_MENU_TO_MISSION_BEGIN
        );


        /* ****************************************************** */   
        /* Mission: Map */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION_MAP,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

	/* Map. */
	map_num = SARMenuCreateMap(
	    menu_ptr, 0.60, 0.42, 0.74, 0.7,
            value_font, &mesgbox_color,
	    (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    0.0, 0.0,
	    SAR_MAP_DEF_MTOP_COEFF,
	    SAR_MENU_MAP_SHOW_MARKING_ALL
	);

	/* Map pan buttons. */
	SARMenuBuildButtonImage(
	    core_ptr, menu_ptr, 0.12, 0.54, btn_width_map, btn_height_map,
	    NULL,
	    SAR_OP_MISSION_MAP_UP,
	    core_ptr->menu_button_pan_up_unarmed_img,
	    core_ptr->menu_button_pan_up_armed_img,
	    core_ptr->menu_button_pan_up_unarmed_img
	);
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.12, 0.66, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_DOWN,
            core_ptr->menu_button_pan_down_unarmed_img,
            core_ptr->menu_button_pan_down_armed_img,  
            core_ptr->menu_button_pan_down_unarmed_img 
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.06, 0.60, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_LEFT,
            core_ptr->menu_button_pan_left_unarmed_img,
            core_ptr->menu_button_pan_left_armed_img,  
            core_ptr->menu_button_pan_left_unarmed_img 
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.18, 0.60, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_RIGHT,
            core_ptr->menu_button_pan_right_unarmed_img,
            core_ptr->menu_button_pan_right_armed_img,
            core_ptr->menu_button_pan_right_unarmed_img
        );

        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.07, 0.25, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_ZOOM_OUT,
            core_ptr->menu_button_zoom_out_unarmed_img,
            core_ptr->menu_button_zoom_out_armed_img,
            core_ptr->menu_button_zoom_out_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.17, 0.25, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_MAP_ZOOM_IN,
            core_ptr->menu_button_zoom_in_unarmed_img,
            core_ptr->menu_button_zoom_in_armed_img,
            core_ptr->menu_button_zoom_in_unarmed_img
        );

        /* Back to mission briefing button. */ 
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_MISSION_BRIEF
        );


        /* ****************************************************** */
        /* Mission: Log map */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_MISSION_LOG_MAP,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Map. */
        map_num = SARMenuCreateMap(
            menu_ptr, 0.60, 0.42, 0.74, 0.7,
            value_font, &mesgbox_color,
	    (const sar_image_struct **)core_ptr->menu_list_bg_img,
            0.0, 0.0,
            SAR_MAP_DEF_MTOP_COEFF,
            SAR_MENU_MAP_SHOW_MARKING_SELECTED
        );

        /* Map pan buttons. */
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.12, 0.54, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_UP,
            core_ptr->menu_button_pan_up_unarmed_img,  
            core_ptr->menu_button_pan_up_armed_img,
            core_ptr->menu_button_pan_up_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.12, 0.66, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_DOWN,
            core_ptr->menu_button_pan_down_unarmed_img,
            core_ptr->menu_button_pan_down_armed_img,
            core_ptr->menu_button_pan_down_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.06, 0.60, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_LEFT,
            core_ptr->menu_button_pan_left_unarmed_img,
            core_ptr->menu_button_pan_left_armed_img,
            core_ptr->menu_button_pan_left_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.18, 0.60, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_RIGHT,
            core_ptr->menu_button_pan_right_unarmed_img,
            core_ptr->menu_button_pan_right_armed_img,
            core_ptr->menu_button_pan_right_unarmed_img
        );

        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.07, 0.25, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_ZOOM_OUT,
            core_ptr->menu_button_zoom_out_unarmed_img,
            core_ptr->menu_button_zoom_out_armed_img,
            core_ptr->menu_button_zoom_out_unarmed_img
        );
        SARMenuBuildButtonImage(
            core_ptr, menu_ptr, 0.17, 0.25, btn_width_map, btn_height_map,
            NULL,
            SAR_OP_MISSION_LOG_MAP_ZOOM_IN,
            core_ptr->menu_button_zoom_in_unarmed_img,
            core_ptr->menu_button_zoom_in_armed_img,
            core_ptr->menu_button_zoom_in_unarmed_img
        );



        /* Back to missions button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Continue", SAR_OP_MENU_TO_MISSION
        );

        /* ****************************************************** */
        /* Free Flight: Select Scenery */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

	/* Scenery list. */
	list_num = SARMenuCreateList(
            menu_ptr, 0.50, 0.20, 0.95, 0.35,
            &list_color,
            value_font,
	    "Select Theater:",
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
	    SARMenuListSelectCB
	);
	SARBuildMenusAddDirectory(
	    core_ptr, menu_ptr, list_num,
	    SAR_DEF_SCENERY_DIR
	);

        /* Locations list. */
        list_num = SARMenuCreateList(
            menu_ptr, 0.50, 0.60, 0.95, 0.30,
            &list_color,
            value_font,
            "Select Location:",
            (const sar_image_struct **)core_ptr->menu_list_bg_img,
            SARMenuListSelectCB
        );

        /* Back to main button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_MAIN
        );

        /* Select aircraft button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.75, 0.92, btn_width_std, btn_height_std,
            "Aircraft", SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT
        );

	/* Restore previously selected scene. */
	list_ptr = SARMenuGetList(menu_ptr, 0, &list_num);
	if(list_ptr != NULL)
        {
	    int i = option.last_selected_ffscene;
	    sar_menu_list_struct *loc_list_ptr;
	    int loc_list_num;
	    sar_menu_list_item_struct *item_ptr;


	    /* Get scene item pointer. */
	    item_ptr = SARMenuListGetItemByNumber(list_ptr, i);

	    /* Get pointer and number to locations list. */
	    loc_list_ptr = SARMenuGetList(menu_ptr, 1, &loc_list_num);

	    if((item_ptr != NULL) &&
               (loc_list_ptr != NULL)
	    )
	    {
		sar_menu_list_item_data_struct *item_data_ptr =
		    item_ptr->client_data;


                list_ptr->selected_item = i;
                list_ptr->scrolled_item = MAX(i - 1, 0);

		/* Fetch locations list from scene. */
                SARSceneLoadLocationsToList(
                    core_ptr,
                    menu_ptr, loc_list_ptr, loc_list_num,
                    item_data_ptr->filename
                );

		/* Select first item on locations list. */
		if(loc_list_ptr->total_items > 0)
		    loc_list_ptr->selected_item = 0;
	    }
        }


        /* ****************************************************** */
        /* Free Flight: Select Aircraft */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Aircrafts list. */
        list_num = SARMenuCreateList(
            menu_ptr, 0.50, 0.38, 0.95, 0.60,
            &list_color,
            value_font,
            "Select Aircraft:",
	    (const sar_image_struct **)core_ptr->menu_list_bg_img,
            SARMenuListSelectCB
        );
        SARBuildMenusAddDirectory(
            core_ptr, menu_ptr, list_num,   
            SAR_DEF_AIRCRAFTS_DIR
        );
        if(list_num > -1)
        {
            /* Restore previously selected aircraft. */
            list_ptr = menu_ptr->object[list_num];

            list_ptr->selected_item = option.last_selected_ffaircraft;
            list_ptr->scrolled_item = MAX(
                list_ptr->selected_item - 1,
                0
            );
        }

        /* Weather. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.74, btn_width_std, btn_height_std,
            "Weather", SAR_OP_MENU_TO_FREE_FLIGHT_WEATHER
        );

        /* Back to Free Flight: Select Scenery button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_FREE_FLIGHT
        );

        /* Begin free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.75, 0.92, btn_width_std, btn_height_std,
            "Begin", SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN
        );


        /* ****************************************************** */
        /* Free Flight: Weather */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_FREE_FLIGHT_WEATHER,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);


	/* Weather presets spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.2, 0.9, 0.0,
            "Condition",
	    0, SARMenuSpinSelectCB
        );
	if(spin_num > -1)
	{
	    int i;
	    sar_weather_data_struct *weather_data = core_ptr->weather_data;
	    sar_weather_data_entry_struct *wdp_ptr;


	    spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num];

	    spin_ptr->allow_warp = False;

	    if(weather_data != NULL)
	    {
		for(i = 0; i < weather_data->total_presets; i++)
		{
		    wdp_ptr = weather_data->preset[i];
		    if(wdp_ptr == NULL)
			continue;

		    if(wdp_ptr->name == NULL)
			continue;

		    SARMenuSpinAddValue(
			menu_ptr, spin_num,
			wdp_ptr->name
		    );
		}
	    }

	    /* Restore previously selected item. */
	    spin_ptr->cur_value = option.last_selected_ffweather;
	}

        /* Back to Free Flight: Select Aircraft button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Aircraft", SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT
        );

        /* Begin free flight button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.75, 0.92, btn_width_std, btn_height_std,
            "Begin", SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN
        );


        /* ****************************************************** */
        /* Options: General */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Back to main button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_MAIN
        );

        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.28, 0.10, btn_width_opt, btn_height_opt,
            "Simulation...", SAR_OP_MENU_TO_OPTIONS_SIMULATION
        );
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.28, 0.30, btn_width_opt, btn_height_opt,
            "Controller...", SAR_OP_MENU_TO_OPTIONS_CONTROLLER
        );
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.28, 0.50, btn_width_opt, btn_height_opt,
            "Graphics...", SAR_OP_MENU_TO_OPTIONS_GRAPHICS
        );
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.28, 0.70, btn_width_opt, btn_height_opt,
            "Sound...", SAR_OP_MENU_TO_OPTIONS_SOUNDS
        );


        /* ****************************************************** */
        /* Options: Simulation */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(  
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_SIMULATION,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_OPTIONS
        );


        /* Hoist contact. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.20, 0.9, 0.0,
            "Hoist Contact", SAR_OPT_SELECT_HOIST_CONTACT,
            SARMenuOptionsSpinCB
        );
	if(spin_num >= 0)
	{
	    spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num];
	    spin_ptr->allow_warp = False;
	}
        SARMenuSpinAddValue(menu_ptr, spin_num, "Easy");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Moderate");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Authentic");

        /* Damage resistance. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.40, 0.9, 0.0,
            "Damage Resistance", SAR_OPT_SELECT_DAMAGE_RESISTANCE,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
        SARMenuSpinAddValue(menu_ptr, spin_num, "Strong");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Nominal");  
        SARMenuSpinAddValue(menu_ptr, spin_num, "Authentic");

        /* Flight physics. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.60, 0.9, 0.0,
            "Flight Physics", SAR_OPT_SELECT_FLIGHT_PHYSICS,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
        SARMenuSpinAddValue(menu_ptr, spin_num, "Easy");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Moderate");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Realistic");


        /* ****************************************************** */
        /* Options: Controller */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_CONTROLLER,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_OPTIONS
        );

#ifdef JS_SUPPORT
	/* Joystick button mapping. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.75, 0.92, btn_width_std, btn_height_std,
            "Buttons", SAR_OP_MENU_TO_OPTIONS_CONTROLLER_JS_BTN
        );
#endif	/* JS_SUPPORT */

        /* First joystick (js0). */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.25, 0.9, 0.0,
            "JS1", SAR_OPT_SELECT_JS0_AXISES,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
#ifdef JS_SUPPORT
        SARMenuSpinAddValue(menu_ptr, spin_num, "Off");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D with throttle");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D with hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D with throttle & hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D with throttle");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D with hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D with throttle & hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "As throttle & rudder");
#else
	SARMenuSpinAddValue(menu_ptr, spin_num, "No support");
#endif	/* JS_SUPPORT */

        /* Second joystick (js1). */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.5, 0.9, 0.0,
            "JS2", SAR_OPT_SELECT_JS1_AXISES,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
#ifdef JS_SUPPORT
        SARMenuSpinAddValue(menu_ptr, spin_num, "Off");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D with throttle");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D with hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2D with throttle & hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D with throttle");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D with hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3D with throttle & hat");
        SARMenuSpinAddValue(menu_ptr, spin_num, "As throttle & rudder");
#else
	SARMenuSpinAddValue(menu_ptr, spin_num, "No support");
#endif	/* JS_SUPPORT */


        /* ****************************************************** */
        /* Options: Controller joystick button mappings*/
#ifdef JS_SUPPORT
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_CONTROLLER_JS_BTN,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Back to controller button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_OPTIONS_CONTROLLER
        );

        /* First joystick (js0) button action spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.16, 0.9, 0.0,
            "JS1 Action", SAR_OPT_SELECT_JS0_BUTTON_ACTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
        SARMenuSpinAddValue(menu_ptr, spin_num, "Rotate Modifier");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Air Brakes");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Wheel Brakes");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Zoom In");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Zoom Out");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Hoist Up");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Hoist Down");

        /* First joystick (js0) button number spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.32, 0.9, 0.0,
            "JS1 Button", SAR_OPT_SELECT_JS0_BUTTON_NUMBER,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
	/* Button values start from spin value - 1, since spin
	 * value 0 really means button -1 (none/invalid).
	 */
	SARMenuSpinAddValue(menu_ptr, spin_num, "None");
        SARMenuSpinAddValue(menu_ptr, spin_num, "1");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3");
        SARMenuSpinAddValue(menu_ptr, spin_num, "4");   
        SARMenuSpinAddValue(menu_ptr, spin_num, "5");
        SARMenuSpinAddValue(menu_ptr, spin_num, "6");
        SARMenuSpinAddValue(menu_ptr, spin_num, "7");
        SARMenuSpinAddValue(menu_ptr, spin_num, "8");
        SARMenuSpinAddValue(menu_ptr, spin_num, "9");
        SARMenuSpinAddValue(menu_ptr, spin_num, "10");
        SARMenuSpinAddValue(menu_ptr, spin_num, "11");
        SARMenuSpinAddValue(menu_ptr, spin_num, "12");
        SARMenuSpinAddValue(menu_ptr, spin_num, "13");
        SARMenuSpinAddValue(menu_ptr, spin_num, "14");
        SARMenuSpinAddValue(menu_ptr, spin_num, "15");
        SARMenuSpinAddValue(menu_ptr, spin_num, "16");


        /* Second joystick (js1) button action spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.48, 0.9, 0.0, 
            "JS2 Action", SAR_OPT_SELECT_JS1_BUTTON_ACTION,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
        SARMenuSpinAddValue(menu_ptr, spin_num, "Rotate Modifier");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Air Brakes");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Wheel Brakes");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Zoom In");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Zoom Out");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Hoist Up");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Hoist Down");

        /* Second joystick (js1) button number spin. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.64, 0.9, 0.0,
            "JS2 Button", SAR_OPT_SELECT_JS1_BUTTON_NUMBER,
            SARMenuOptionsSpinCB
        );
        /* Button values start from spin value - 1, since spin
         * value 0 really means button -1 (none/invalid).
         */
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
        SARMenuSpinAddValue(menu_ptr, spin_num, "None");
        SARMenuSpinAddValue(menu_ptr, spin_num, "1");
        SARMenuSpinAddValue(menu_ptr, spin_num, "2");
        SARMenuSpinAddValue(menu_ptr, spin_num, "3");
        SARMenuSpinAddValue(menu_ptr, spin_num, "4");
        SARMenuSpinAddValue(menu_ptr, spin_num, "5");
        SARMenuSpinAddValue(menu_ptr, spin_num, "6");
        SARMenuSpinAddValue(menu_ptr, spin_num, "7");
        SARMenuSpinAddValue(menu_ptr, spin_num, "8");
        SARMenuSpinAddValue(menu_ptr, spin_num, "9");
        SARMenuSpinAddValue(menu_ptr, spin_num, "10");
        SARMenuSpinAddValue(menu_ptr, spin_num, "11");
        SARMenuSpinAddValue(menu_ptr, spin_num, "12");
        SARMenuSpinAddValue(menu_ptr, spin_num, "13");
        SARMenuSpinAddValue(menu_ptr, spin_num, "14");
        SARMenuSpinAddValue(menu_ptr, spin_num, "15");
        SARMenuSpinAddValue(menu_ptr, spin_num, "16");



#endif	/* JS_SUPPORT */

        /* ****************************************************** */
        /* Options: Graphics */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_GRAPHICS,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_OPTIONS
        );

        /* Ground texture. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu_ptr,
	    (1.0 / 8.0) + (0.0 * 1.0 / 4.0), 0.2,
	    0, 0,
            "Ground Texture",
            False, SAR_OPT_SELECT_GROUND_TEXTURE,
            SARMenuOptionsSwitchCB 
        );
        /* Object texture. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu_ptr,
	    (1.0 / 8.0) + (1.0 * 1.0 / 4.0), 0.2,
	    0, 0,
            "Object Texture",
            False, SAR_OPT_SELECT_OBJECT_TEXTURE,
            SARMenuOptionsSwitchCB
        );
        /* Clouds. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu_ptr,
            (1.0 / 8.0) + (2.0 * 1.0 / 4.0), 0.2,
	    0, 0,
            "Clouds",
            False, SAR_OPT_SELECT_CLOUDS,
            SARMenuOptionsSwitchCB
        );
        /* Atmosphere. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu_ptr,
            (1.0 / 8.0) + (3.0 * 1.0 / 4.0), 0.2,
	    0, 0,
            "Atmosphere",
            False, SAR_OPT_SELECT_ATMOSPHERE,
            SARMenuOptionsSwitchCB
        );
        /* Dual pass depth. */
        SARMenuBuildStandardSwitch(
            core_ptr, menu_ptr,
            (1.0 / 8.0) + (0.0 * 1.0 / 4.0), 0.4,
	    0, 0,
            "Dual Pass Depth",
            False, SAR_OPT_SELECT_DUAL_PASS_DEPTH,
            SARMenuOptionsSwitchCB
        );
	/* Max visibility. */
        spin_num = SARMenuBuildStandardSpin(
            core_ptr, menu_ptr, 0.5, 0.75, 0.9, 0.0,
            "Max Visibility", SAR_OPT_SELECT_MAX_VISIBILITY,
            SARMenuOptionsSpinCB
        );
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
        SARMenuSpinAddValue(menu_ptr, spin_num, "3 miles");
        SARMenuSpinAddValue(menu_ptr, spin_num, "6 miles");
        SARMenuSpinAddValue(menu_ptr, spin_num, "9 miles");
        SARMenuSpinAddValue(menu_ptr, spin_num, "12 miles");
        SARMenuSpinAddValue(menu_ptr, spin_num, "15 miles");
        SARMenuSpinAddValue(menu_ptr, spin_num, "18 miles");
        SARMenuSpinAddValue(menu_ptr, spin_num, "21 miles");

        /* ****************************************************** */
        /* Options: Sound */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_STANDARD_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_OPTIONS_SOUND,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1);
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);
        
        /* Back to options button. */
        SARMenuBuildStandardButton(
            core_ptr, menu_ptr, 0.25, 0.92, btn_width_std, btn_height_std,
            "Back", SAR_OP_MENU_TO_OPTIONS
        );

	/* Sound level. */
	spin_num = SARMenuBuildStandardSpin(
	    core_ptr, menu_ptr, 0.5, 0.2, 0.9, 0.0,
	    "Sound Level", SAR_OPT_SELECT_SOUND_LEVEL,
	    SARMenuOptionsSpinCB
	);
        if(spin_num >= 0)
        {
            spin_ptr = (sar_menu_spin_struct *)menu_ptr->object[spin_num]; 
            spin_ptr->allow_warp = False;
        }
#ifdef HAVE_Y
        SARMenuSpinAddValue(menu_ptr, spin_num, "Off");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Events");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Events and Engine");
        SARMenuSpinAddValue(menu_ptr, spin_num, "Events, Engine, and Voice");
#else
	SARMenuSpinAddValue(menu_ptr, spin_num, "No support");
#endif
	/* Music. */
	switch_num = SARMenuBuildStandardSwitch(
            core_ptr, menu_ptr, 0.2, 0.5, 0, 0,
            "Music",
            False, SAR_OPT_SELECT_MUSIC,
            SARMenuOptionsSwitchCB
	);
	if(switch_num > -1)
	{
#if !defined(HAVE_Y)
	    sar_menu_switch_struct *switch_ptr = menu_ptr->object[switch_num];

	    switch_ptr->disabled = True;
#endif
	}

	/* ****************************************************** */
        /* Loading simulation progress menu. */
        strptr = SARMenuBuildGetFullPath(SAR_DEF_MENU_BGIMG_PROGRESS_FILE);
        menu_ptr = SARMenuCreate(
            SAR_MENU_TYPE_STANDARD,
            SAR_MENU_NAME_LOADING_SIMULATION,
            strptr
        );
        if(menu_ptr == NULL)
            return(-1); 
        else
            SARBuildMenusAddToList(core_ptr, menu_ptr);

        /* Progress bar. */
	SARMenuCreateProgress(
	    menu_ptr,
	    0.5, 0.9,
	    0, 0,
	    NULL,
            &progress_color,
            value_font,
	    core_ptr->menu_progress_bg_img,
	    core_ptr->menu_progress_fg_img,
	    0
	);


	return(0);
}
