#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "gw.h"

#include "sarreality.h"
#include "messages.h"
#include "obj.h"
#include "gctl.h"
#include "simop.h"
#include "simutils.h"
#include "textinput.h"
#include "cmd.h"
#include "sar.h"
#include "config.h"


static void SARKeyCameraRefCockpit(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyCameraRefHoist( 
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyCameraRefMap(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyCameraRefSpot(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyCameraRefTower(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyCommand(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyEscape(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);

static void SARKeyGraphicsAtmosphere(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyGraphicsTexturedClouds(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyGraphicsTexturedGround(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyGraphicsTexturedObjects(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeySoundLevel(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyHelpDisplay(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyPrintScore(
	sar_core_struct *core_ptr, gw_display_struct *display,
	sar_scene_struct *scene, Boolean state
);
static void SARKeyTimeCompression( 
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyViewNormalize(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyFlightPhysicsDifficulty(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyVisibility(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);

static void SARKeyLandingGear(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyLights(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyStrobes(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyTextColor(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyInterceptWayPoint(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyEngine(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyElevatorTrimUp(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyElevatorTrimDown(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyPitchRotors(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyHoistRopeEndSelect(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyDoor(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static int SARKeyRestart(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyRefuelRepair(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyFuel(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeySendMessage(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);
static void SARKeyTimeOfDay(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
);



void SARKey(
	sar_core_struct *core_ptr,
	int c, Boolean state, long t
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Switch camera referance on scene structure to cockpit.
 */
static void SARKeyCameraRefCockpit(
        sar_core_struct *core_ptr, 
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	if((scene == NULL) || !state)
	    return;

	scene->camera_ref = SAR_CAMERA_REF_COCKPIT;
	scene->camera_target = scene->player_obj_num;
}

/*                     
 *      Switch camera referance on scene structure to cockpit.
 */
static void SARKeyCameraRefHoist(
        sar_core_struct *core_ptr, 
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	if((core_ptr == NULL) || (scene == NULL) || !state)
	    return;

	scene->camera_ref = SAR_CAMERA_REF_HOIST;
	scene->camera_target = scene->player_obj_num;
}

/*
 *      Switch camera referance on scene structure to map.
 */
static void SARKeyCameraRefMap(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	sar_object_struct *obj_ptr;

	if((core_ptr == NULL) || (scene == NULL) || !state)
	    return;

	scene->camera_ref = SAR_CAMERA_REF_MAP;
	scene->camera_target = -1;

	obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    sar_position_struct *pos_src = &obj_ptr->pos;
	    sar_position_struct *pos_tar = &scene->camera_map_pos;

	    /* Set initial camera position at location of object. */
	    pos_tar->x = pos_src->x;
	    pos_tar->y = pos_src->y;

	    if(pos_tar->z < 1000.0)
		pos_tar->z = 1000.0;
	}
}


/*
 *      Switch camera referance on scene structure to spot.
 *
 *	If the camera referance is already set to spot then the `next'
 *	object will be set as the camera target.
 */
static void SARKeyCameraRefSpot(
        sar_core_struct *core_ptr,
        gw_display_struct *display, 
        sar_scene_struct *scene,  
        Boolean state
)
{
        if((core_ptr == NULL) || (scene == NULL) || !state)
            return;

	/* Previous camera referance is spot with set target? */
	if((scene->camera_ref == SAR_CAMERA_REF_SPOT) &&
           (scene->camera_target > -1)
	)
	{
	    /* Camera referance was already spot, so cycle camera target
	     * to the `next' object.
	     */
	    int i, n, type;
	    sar_object_struct *obj_ptr;
	    Boolean seek_backwards = display->shift_key_state;

	    /* Go through objects list twice. */
	    for(n = 0; n < 2; n++)
	    {
		/* Go through objects list, starting from the
		 * index value as the current value for the camera
		 * target + 1.
		 */
		i = scene->camera_target + (seek_backwards ? -1 : 1);
		while((i >= 0) && (i < core_ptr->total_objects))
		{
		    obj_ptr = core_ptr->object[i];
		    if(obj_ptr == NULL)
		    {
			seek_backwards ? i-- : i++;
			continue;
		    }

		    /* Get object type. */
		    type = obj_ptr->type;

		    /* Match following object types. */
		    if((type == SAR_OBJ_TYPE_STATIC) ||
		       (type == SAR_OBJ_TYPE_AUTOMOBILE) ||
		       (type == SAR_OBJ_TYPE_WATERCRAFT) ||
		       (type == SAR_OBJ_TYPE_AIRCRAFT) ||
/*
		       (type == SAR_OBJ_TYPE_RUNWAY) ||
		       (type == SAR_OBJ_TYPE_HELIPAD) ||
 */
		       (type == SAR_OBJ_TYPE_FUELTANK) ||
		       (type == SAR_OBJ_TYPE_HUMAN) ||
		       (type == SAR_OBJ_TYPE_PREMODELED)
		    )
			break;

		    /* No match, seek next index. */
		    seek_backwards ? i-- : i++;
		}
		/* Got matched object? */
		if((i >= 0) && (i < core_ptr->total_objects))
		{
		    scene->camera_target = i;
		    break;
		}
		else if(i < 0)
		{
		    /* Seeked past bottom, warp to top. */
		    scene->camera_target = core_ptr->total_objects;
		}
		else if(i >= core_ptr->total_objects)
		{
		    scene->camera_target = -1;
		}
		else
		{
		    scene->camera_target = -1;
		}
	    }
	}
	else
	{
	    /* Previous camera referance was not spot, so initially
	     * set to spot and target the player object.
	     */
	    scene->camera_ref = SAR_CAMERA_REF_SPOT;
	    scene->camera_target = scene->player_obj_num;
	}
}

/*
 *      Switch camera referance on scene structure to tower.
 *
 *	The position of the tower will be calculated by a call to
 *	SARSimSetFlyByPosition().
 */
static void SARKeyCameraRefTower(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        if((core_ptr == NULL) || (scene == NULL) || !state)
            return;

	scene->camera_ref = SAR_CAMERA_REF_TOWER;
	scene->camera_target = scene->player_obj_num;

	SARSimSetFlyByPosition(
	    scene,
	    &core_ptr->object, &core_ptr->total_objects,
	    scene->player_obj_ptr,
	    &scene->camera_tower_pos
	);
}

/*
 *	Maps the command prompt, future key events sent to SARKey()
 *	will then be forwarded to SARTextInputHandleKey() untill
 *	the command argument is typed in and processed or aborted.
 */
static void SARKeyCommand(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	if((core_ptr == NULL) || !state)
	    return;

	SARTextInputMap(
	    core_ptr,
	    "Command",
	    SARCommandTextInputCB
	);
}


/*
 *	Handles escape key depending on the situation which is checked
 *	in a specific order. The conditions are as follows:
 *
 *	1. Check is help screen is displayed, if so then hide it.
 *
 *	2. Check if any messages are currently being displayed by the
 *	   scene.
 *
 *	3. Check if a mission is in progress, and if so map the prompt
 *	   for confirmation.
 *
 *	4. All else prompt to end simulation.
 */
static void SARKeyEscape(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        if((core_ptr == NULL) || !state)
            return;

	/* Displaying help? */
	if(core_ptr->display_help > 0)
	{
	    core_ptr->display_help = 0;
	}
	/* Scene displaying message? */
	else if((scene == NULL) ? 0 :
	        (scene->message_display_untill > 0)
        )
        {
	    /* Reset message display untill time to 0 and delete
	     * all messages on the scene. This will not change
	     * the size of the message pointer array.
	     */
	    SARMessageClearAll(scene);
	}
	/* Prompt to end mission? */
	else if(core_ptr->mission != NULL)
	{
	    /* Same as end simulation, just prompt text different. */
	    SARTextInputMap(
		core_ptr,
		"Are you sure you want to abort the mission?",
		SARTextInputCBQuitSimulation
	    );
	}
	else
	{
	    SARTextInputMap(
		core_ptr,
		"Are you sure you want to quit simulation?",
		SARTextInputCBQuitSimulation
	    );
	}
}


/*
 *      Toggles atmosphere in the global options structure on/off.
 */
static void SARKeyGraphicsAtmosphere(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        const char *message = NULL;

        if((core_ptr == NULL) || (scene == NULL) || !state)
            return;

        if(option.atmosphere)
        {
            option.atmosphere = False;
            message = "Atmosphere: Off";
        }
        else
        {
            option.atmosphere = True;
            message = "Atmosphere: On";
        }

        SARMessageAdd(scene, message);
}

/*
 *	Toggles textured cloud layers in the global options structure
 *	on/off.
 *
 *	Note that this basically specifies if cloud layers should be
 *	drawn since cloud layers are always textured. If camera is
 *	above highest cloud layer then the highest cloud layer will
 *	always be drawn.
 */
static void SARKeyGraphicsTexturedClouds(
        sar_core_struct *core_ptr, 
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        const char *message = NULL;

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	/* If shift key is held then operation is alternated to be the
	 * prop wash toggle.
	 */
	if(display->shift_key_state)
	{
	    if(option.prop_wash)
	    {
                option.prop_wash = False;
                message = "Prop Wash: Off";
            }
            else
            {
                option.prop_wash = True;
                message = "Prop Wash: On";
	    }
	}
	else
	{
	    if(option.textured_clouds)
	    {
		option.textured_clouds = False;
		message = "Textured Clouds: Off";
	    }
	    else
	    {
		option.textured_clouds = True;
		message = "Textured Clouds: On";
	    }
	}

        SARMessageAdd(scene, message);
}

/*
 *	Toggles textured ground in the global options structure on/off.
 */
static void SARKeyGraphicsTexturedGround(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	const char *message = NULL;

        if((core_ptr == NULL) || (scene == NULL) || !state)
            return; 

        /* If shift key is held then operation is alternated to be the
         * dual pass depth toggle.
         */
        if(display->shift_key_state)
        {
            if(option.dual_pass_depth)
            {
                option.dual_pass_depth = False;
                message = "Dual Pass Depth: Off";
            }
            else
            {
                option.dual_pass_depth = True;
                message = "Dual Pass Depth: On";
            }
        }
        else
        {
	    if(option.textured_ground)
	    {
		option.textured_ground = False;
		message = "Ground Texture: Off";
	    }
	    else
	    {
		option.textured_ground = True;
		message = "Ground Texture: On";
	    }
	}

	SARMessageAdd(scene, message);
}

/*
 *      Toggles textured objects in the global options structure on/off.
 */
static void SARKeyGraphicsTexturedObjects(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        const char *message = NULL;

        if((core_ptr == NULL) || (scene == NULL) || !state)
            return;

        if(option.textured_objects)
        {
            option.textured_objects = False;
            message = "Object Texture: Off";
        }
        else
        {
            option.textured_objects = True;
            message = "Object Texture: On";
        }

        SARMessageAdd(scene, message);
}

/*
 *	Changes the sound level.
 */
static void SARKeySoundLevel(  
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	const char *message = NULL;
	sar_option_struct *opt = &option;

        if((core_ptr == NULL) || (scene == NULL) || !state)
            return;

	/* Was all off? */
	if(!opt->engine_sounds && !opt->event_sounds &&
	   !opt->voice_sounds
	)
	{
	    opt->event_sounds = True;
	    message = "Sound: Events";
	}
	/* Was events only? */
	else if(!opt->engine_sounds && opt->event_sounds &&
                !opt->voice_sounds 
        )
	{
	    opt->engine_sounds = True;
            message = "Sound: Events and Engine";
	}
        /* Was events and engine? */
        else if(opt->engine_sounds && opt->event_sounds &&
                !opt->voice_sounds
        )
        {
            opt->voice_sounds = True;
	    message = "Sound: Events, Engine, and Voice";
        }
	/* All else assume all on. */
	else
        {
            opt->engine_sounds = False;
	    opt->event_sounds = False;
	    opt->voice_sounds = False;
	    message = "Sound: Off";
        }

	SARSceneSoundUpdate(
	    core_ptr,
	    opt->engine_sounds,
            opt->event_sounds,
            opt->voice_sounds,
	    opt->music
	);
	SARMessageAdd(scene, message);
}

/*
 *	Toggles the value display_help on the core structure to display
 *	the next page.
 *
 *	This specifies if the help screen should be drawn or not and which
 *	page number.
 */
static void SARKeyHelpDisplay(
        sar_core_struct *core_ptr, 
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        if((core_ptr == NULL) || (display == NULL) || !state)
            return;

	/* Increment or decrement display_help value to indicate current
	 * page being displayed.
	 *
	 * Remember that help page index starts at 1 not 0.
	 *
	 * If increment exceeds maximum page, then it will be sanitized
	 * in the SARDraw*() functions (not here).
	 */
	if(display->shift_key_state)
	    core_ptr->display_help--;
	else
	    core_ptr->display_help++;

	if(core_ptr->display_help < 0)
	    core_ptr->display_help = 0;
}

/*
 *	Prints score, mission status, and passengers on board player
 *	aircraft.
 */
static void SARKeyPrintScore(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene, 
        Boolean state  
)
{
	sar_object_struct *obj_ptr;

	if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
	    return;

	/* Get player object pointer from scene structure. */
	obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    if(display->shift_key_state)
	    {
		/* Debug case, print player position stats to stdout. */
		sar_direction_struct *dir = &obj_ptr->dir;
		sar_position_struct *pos = &obj_ptr->pos;

		/* Print position (for debugging). */
		printf(
 "%i:%2i Player: (hpb) %.0f %.0f %.0f  (xyz) %.2f %.2f %.2f(%.2f feet)\n",
		    (int)(scene->tod / 3600),
		    (int)((int)((int)scene->tod / 60) % 60),
		    SFMRadiansToDegrees(dir->heading),
		    SFMRadiansToDegrees(dir->pitch),
		    SFMRadiansToDegrees(dir->bank),
		    pos->x, pos->y, pos->z,
		    SFMMetersToFeet(pos->z)
		);
	    }
	    else
	    {
		/* Print score, mission status, and occupants on
		 * the given player object obj_ptr. If core_ptr->mission
		 * is NULL then only occupants will be printed.
		 */
		SARMissionPrintStats(
		    core_ptr, scene, core_ptr->mission,
		    obj_ptr
		);
	    }
	}
}

/*
 *	Adjusts the global time compression.
 */
static void SARKeyTimeCompression(
        sar_core_struct *core_ptr, 
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	char text[256];

        if((core_ptr == NULL) || (scene == NULL) || !state)
            return;

	/* Check shift modifier. */
	if(display->shift_key_state)
	{
	    /* Shift is held, so increase time compression. */
	    if(time_compression >= 1.0)
	    {
		time_compression += 1.0;
	    }
	    else
	    {
		time_compression += 0.25;
		if(time_compression > 1.0)
		    time_compression = 1.0;
	    }
	}
	else
	{
	    /* Shift is held, so decrease time compression. */
	    if(time_compression > 1.0)
	    {
		time_compression -= 1.0;
		if(time_compression < 1.0)
		    time_compression = 1.0;
	    }
	    else
	    {
		time_compression -= 0.25;
	    }
	}

        if(time_compression < 0.25)
	    time_compression = 0.25;
	else if(time_compression > 8.0)
	    time_compression = 8.0;

	sprintf(
	    text,
	    "Time Compression %.0f%%",
	    time_compression * 100
	);
	SARMessageAdd(scene, text);
}

/*
 *	Normalizes the zoom and position of the view with respect to
 *	the current camera_ref on the scene.
 */
static void SARKeyViewNormalize(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	/* Get player object pointer from scene structure. */
	obj_ptr = scene->player_obj_ptr;

	/* Handle by camera referance specified on the scene structure. */
	switch(scene->camera_ref)
	{
	  case SAR_CAMERA_REF_HOIST:
	    if(obj_ptr != NULL)
	    {
		double contact_radius = SARSimGetFlatContactRadius(obj_ptr);
		sar_direction_struct *dir = &scene->camera_hoist_dir;

		scene->camera_hoist_dist = MAX(contact_radius, 10.0);
		dir->heading = (1.0 * PI);
		dir->pitch = (1.95 * PI);
		dir->bank = (0.0 * PI);
	    }
	    break;

	  case SAR_CAMERA_REF_MAP:
	    if(obj_ptr != NULL)
	    {
		sar_position_struct *pos_src = &obj_ptr->pos;
		sar_position_struct *pos_tar = &scene->camera_map_pos;

		/* Set camera position at location of object
		 * and keey the camera position atleast 1000.0 meters up.
		 */
		pos_tar->x = pos_src->x;
		pos_tar->y = pos_src->y;

		if(pos_tar->z < 1000.0)
		    pos_tar->z = 1000.0;
	    }
	    break;

          case SAR_CAMERA_REF_TOWER:
	    if(obj_ptr != NULL)
	    {
		SARSimSetFlyByPosition(
		    scene,
		    &core_ptr->object, &core_ptr->total_objects,
		    obj_ptr,
		    &scene->camera_tower_pos
		);
	    }
            break;

          case SAR_CAMERA_REF_SPOT:
            if(obj_ptr != NULL)
	    {
		double contact_radius = SARSimGetFlatContactRadius(obj_ptr);
                sar_direction_struct *dir = &scene->camera_spot_dir;

		scene->camera_spot_dist = MAX(contact_radius, 10.0);
		dir->heading = (1.0 * PI);
		dir->pitch = (1.95 * PI);
		dir->bank = (0.0 * PI);
	    }
	    break;

	  case SAR_CAMERA_REF_COCKPIT:
	    if(1)
            {
		sar_direction_struct *dir = &scene->camera_cockpit_dir;

		dir->heading = (0.0 * PI);
		dir->pitch = (0.0 * PI);
		dir->bank = (0.0 * PI);
	    }
	    break;
	}
}

/*
 *	Adjusts flight physics difficulty.
 */
static void SARKeyFlightPhysicsDifficulty(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	const char *mesg = SAR_MESG_FLIGHT_PHYSICS_UNSUPPORTED;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	switch(option.flight_physics)
	{
	  case FLIGHT_PHYSICS_EASY:
	    option.flight_physics = FLIGHT_PHYSICS_MODERATE;
	    mesg = SAR_MESG_FLIGHT_PHYSICS_MODERATE;
	    break;

	  case FLIGHT_PHYSICS_MODERATE:
	    option.flight_physics = FLIGHT_PHYSICS_REALISTIC;
	    mesg = SAR_MESG_FLIGHT_PHYSICS_REALISTIC;
	    break;

	  default:
	    option.flight_physics = FLIGHT_PHYSICS_EASY;
	    mesg = SAR_MESG_FLIGHT_PHYSICS_EASY;
	    break;
	}

	SARMessageAdd(scene, mesg);
}

/*
 *	Sets maximum visibility.
 */
static void SARKeyVisibility(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	char text[256];


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	option.visibility_max += ((display->shift_key_state) ?
	    1 : -1
	);
	if(option.visibility_max > 6)
	    option.visibility_max = 0;
	else if(option.visibility_max < 0)
	    option.visibility_max = 6;

	/* Equation to calculate visibility max code is
	 * miles = 3 + (option.visibility_max * 3)
	 */
	sprintf(
	    text,
	    "Visibility Max: %i Miles",
	    3 + (option.visibility_max * 3)
	);
	SARMessageAdd(scene, text);
}


/*
 *	Raise/lower landing gears on player object.
 */
static void SARKeyLandingGear(
        sar_core_struct *core_ptr, 
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    int lgear_state = SARSimGetLandingGearState(obj_ptr);

	    /* Raise/lower landing gear. */
	    SARSimOpLandingGear(
		scene,
		&core_ptr->object, &core_ptr->total_objects,
		obj_ptr,
		!lgear_state
	    );
	}
}

/*
 *      Turns lights (vector lights and spot lights) on/off on the player
 *	object.
 */
static void SARKeyLights(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    int state = SARSimGetLightsState(obj_ptr);
	    SARSimOpLights(
		scene,
		&core_ptr->object, &core_ptr->total_objects,
		obj_ptr, !state
	    );
	    SARSimOpAttenuate(
		scene,
		&core_ptr->object, &core_ptr->total_objects,
		obj_ptr, !state
	    );
	}
}

/*
 *	Turns strobe lights on/off on the player object.
 */
static void SARKeyStrobes(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
        if(obj_ptr != NULL)
        {
            int state = SARSimGetStrobesState(obj_ptr);
            SARSimOpStrobes(
                scene,
                &core_ptr->object, &core_ptr->total_objects,
                obj_ptr, !state
            );
        }
}

/*
 *	Brightens or darkens text color on scene.
 */
static void SARKeyTextColor(  
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	sar_color_struct *color;
	double g, b;			/* Gamma and base coeff. */


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	/* Adjust hud color. */
	color = &option.hud_color;
	if(display->shift_key_state)
	{
	    /* Darken. */

	    if(color->r > 0.0)
            {
                g = MAX(color->r - 0.2, 0.0);
                b = 1.0;
            }
            else
            {
                g = 0.0;
                b = MAX(color->g - 0.2, 0.0);
            }
        }
        else
        {
            /* Lighten. */

            if(color->g < 1.0)
            {
                g = 0.0;
                b = MIN(color->g + 0.2, 1.0);
            }
            else
            {
                g = MIN(color->r + 0.2, 1.0);
                b = 1.0;
            }
        }

        color->a = 1.0;
        color->r = g;
        color->g = b;
        color->b = g;

        /* Adjust message text color. */
        color = &option.message_color;
        if(display->shift_key_state)
            g = MAX(color->r - 0.1, 0.0);
        else
            g = MIN(color->r + 0.1, 1.0);

        color->a = 1.0,
        color->r = g;
        color->g = g;
        color->b = g;
}

/*
 *	Select next intercept way point.
 */
static void SARKeyInterceptWayPoint(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state   
)
{
	sar_object_struct *obj_ptr;

	if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
	    return;

	obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    int *cur_intercept = NULL, *total_intercepts = NULL;
	    sar_object_aircraft_struct *obj_aircraft_ptr;
	    char text[256];


	    (*text) = '\0';
	    switch(obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_AIRCRAFT:
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr != NULL)
		{
		    cur_intercept = &obj_aircraft_ptr->cur_intercept;
		    total_intercepts = &obj_aircraft_ptr->total_intercepts;
		}
		break;
	    }

	    if((cur_intercept != NULL) && (total_intercepts != NULL))
	    {
		/* Change intercept number. */
		if(display->shift_key_state)
		    (*cur_intercept) = (*cur_intercept) - 1;
		else
                    (*cur_intercept) = (*cur_intercept) + 1;

		/* Cycle. */
		if((*cur_intercept) >= (*total_intercepts))
		    (*cur_intercept) = 0;
		else if((*cur_intercept) < 0)
		    (*cur_intercept) = (*total_intercepts) - 1;

		if((*cur_intercept) < 0)
		    (*cur_intercept) = 0;

		if((*total_intercepts) > 0)
		    sprintf(
			text, "Selected waypoint %i.",
			(int)((*cur_intercept) + 1)
		    );
		else
		    strcpy(text, "No waypoints set.");

		if((*text) != '\0')
		    SARMessageAdd(scene, text);
	    }
	}
}

/*
 *	Turns engine off or initializes engine on the player object.
 */
static void SARKeyEngine(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
        if(obj_ptr != NULL)
	{
	    int engine_state = SARSimGetEngineState(obj_ptr);
	    if(((engine_state == SAR_ENGINE_STATE_ON) ||
	        (engine_state == SAR_ENGINE_STATE_INIT)) &&
               display->shift_key_state
	    )
		SARSimOpEngine(
		    scene,
		    &core_ptr->object, &core_ptr->total_objects,
		    obj_ptr,
		    SAR_ENGINE_STATE_OFF
		);
	    else if(!display->shift_key_state)
		SARSimOpEngine(
                    scene,
                    &core_ptr->object, &core_ptr->total_objects,
                    obj_ptr,
                    SAR_ENGINE_STATE_ON
                );
	}
}

/*
 *	Increases elevator trim on player object.
 */
static void SARKeyElevatorTrimUp(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,  
        Boolean state
)
{
        sar_object_struct *obj_ptr;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
        if(obj_ptr != NULL)
        {
            sar_object_aircraft_struct *obj_aircraft_ptr;

            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr != NULL)
                {
		    /* In slew mode, this function moves the aircraft down. */
                    if(obj_aircraft_ptr->flight_model_type == SAR_FLIGHT_MODEL_SLEW)
                    {
                        obj_ptr->pos.z -= SFMFeetToMeters(
			    (display->shift_key_state) ? 250.0 : 25.0
			);
			SARSimWarpObject(
			    scene, obj_ptr,
			    &obj_ptr->pos, &obj_ptr->dir
			);
                    }
                    /* All else is regular elevator trim adjusting. */
                    else
		    {
			if(display->ctrl_key_state)
			    obj_aircraft_ptr->elevator_trim = 0.0;
			else
			    obj_aircraft_ptr->elevator_trim -= (
				(display->shift_key_state) ? 0.1 : 0.025
			    );
			if(obj_aircraft_ptr->elevator_trim < -1.0)
			    obj_aircraft_ptr->elevator_trim = -1.0;
		    }
                }
                break;
             
            }
        }
}

/*
 *	Decreases elevator trim on player object.
 */
static void SARKeyElevatorTrimDown(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
        if(obj_ptr != NULL)
        {
            sar_object_aircraft_struct *obj_aircraft_ptr;
  
            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr != NULL)
                {
		    /* In slew mode, this function moves the aircraft up. */
		    if(obj_aircraft_ptr->flight_model_type == SAR_FLIGHT_MODEL_SLEW)
		    {
			obj_ptr->pos.z += SFMFeetToMeters(
                            (display->shift_key_state) ? 250.0 : 25.0
                        );
                        SARSimWarpObject(
                            scene, obj_ptr,
                            &obj_ptr->pos, &obj_ptr->dir
                        );
		    }
		    /* All else is regular elevator trim adjusting. */
		    else
		    {
			if(display->ctrl_key_state)
			    obj_aircraft_ptr->elevator_trim = 0.0;
			else
			    obj_aircraft_ptr->elevator_trim += (
				(display->shift_key_state) ? 0.1 : 0.025
			    );
			if(obj_aircraft_ptr->elevator_trim > 1.0)
			    obj_aircraft_ptr->elevator_trim = 1.0;
		    }
		}
		break;

	    }
	}
}


/*
 *	Pitches rotors forwards or upwards on player object.
 */
static void SARKeyPitchRotors(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	sar_object_struct *obj_ptr;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        /* If ctrl key is held then operation is alternated to be the
         * slew toggle.
         */
        if(display->ctrl_key_state)
        {
            obj_ptr = scene->player_obj_ptr;
            if((obj_ptr != NULL) ?
		(obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT) : 0
	    )
	    {
		int is_slew;
		sar_object_aircraft_struct *obj_aircraft_ptr;

		
		is_slew = SARSimIsSlew(obj_ptr);
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;

		/* Object must be an aircraft in order to toggle slew. */
		if(obj_aircraft_ptr != NULL)
		{
		    if(is_slew)
		    {
			/* Was in slew mode, now go into previous flight mode. */
			SARSimSetSlew(obj_ptr, 0);
		    }
		    else
		    {
			/* Was in some other flight mode, now wanting to
			 * enter slew mode. Make sure flight worthyness
			 * allows aircraft to go into slew mode.
			 */
			if(obj_aircraft_ptr->air_worthy_state == SAR_AIR_WORTHY_FLYABLE)
			    SARSimSetSlew(obj_ptr, 1);
		    }
		}
	    }
	}
	else
	{
	    obj_ptr = scene->player_obj_ptr;
	    if(obj_ptr != NULL)
	    {
		sar_object_aircraft_struct *obj_aircraft_ptr;

		switch(obj_ptr->type)
		{
		  case SAR_OBJ_TYPE_AIRCRAFT:
		    obj_aircraft_ptr = (sar_object_aircraft_struct *)
			obj_ptr->data;
		    if(obj_aircraft_ptr == NULL)
			break;

		    /* Can rotors pitch? */
		    if(obj_aircraft_ptr->engine_can_pitch == 1)
		    {
			/* Change flight model. */
			switch(obj_aircraft_ptr->flight_model_type)
			{
			  case SAR_FLIGHT_MODEL_AIRPLANE:
			    obj_aircraft_ptr->flight_model_type =
				SAR_FLIGHT_MODEL_HELICOPTER;
 SARMessageAdd(scene, "Helicopter Mode");
                              break;

			  /* All else assume helicopter. */
			  default:
			    obj_aircraft_ptr->flight_model_type =
                                SAR_FLIGHT_MODEL_AIRPLANE;
 SARMessageAdd(scene, "Airplane Mode");
                              break;
                         }
                    }
		    break;
		}
	    }
	}
}

/*
 *	Selects different thing (rescue basket or diver) to put at the
 *	end of the player object's rescue hoist rope.
 */
static void SARKeyHoistRopeEndSelect(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
)
{
        sar_object_struct *obj_ptr;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
        if(obj_ptr != NULL)
        {
            sar_obj_hoist_struct *hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
	    if(hoist_ptr != NULL)
	    {
		const char *mesg_ptr = NULL;

		/* Cannot select if hoist's rope is already out. */
		if(hoist_ptr->rope_cur > 0.0)
		{
		    mesg_ptr = SAR_MESG_HOIST_END_SELECT_ROPE_OUT;
		}
		else
		{

		    hoist_ptr->deployment++;

		    /* Cycle. */
		    if(hoist_ptr->deployment > 1)
			hoist_ptr->deployment = SAR_HOIST_DEPLOYMENT_BASKET;

		    switch(hoist_ptr->deployment)
		    {
		      case SAR_HOIST_DEPLOYMENT_DIVER:
			mesg_ptr = SAR_MESG_HOIST_END_SELECT_DIVER;
			break;

		      default:
			hoist_ptr->deployment = SAR_HOIST_DEPLOYMENT_BASKET;
			mesg_ptr = SAR_MESG_HOIST_END_SELECT_BASKET;
			break;
		    }
		}

		if(mesg_ptr != NULL)
		    SARMessageAdd(scene, mesg_ptr);
	    }
	}
}

/*
 *	Open/close main rescue door on player object.
 */
static void SARKeyDoor(     
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        sar_object_struct *obj_ptr;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

        obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    sar_obj_hoist_struct *hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
	    sar_obj_door_struct *door_ptr = SARObjGetDoorPtr(obj_ptr, 0, NULL);

	    if(door_ptr != NULL)
	    {
		/* Door opened and hoist out? */
		if(((hoist_ptr == NULL) ? 0 : (hoist_ptr->rope_cur > 0.0)) &&
		   (door_ptr->flags & SAR_DOOR_FLAG_STATE)
		)
		{
		    /* Door is opened and hoist is out, cannot close
		     * door.
		     */
		    SARMessageAdd(
			scene,
			SAR_MESG_CANNOT_CLOSE_DOOR_BASKET
		    );
		}
		else
		{
		    /* Check if door was opened? */
		    if(door_ptr->flags & SAR_DOOR_FLAG_STATE)
		    {
			/* Since closing door we need to abort any
			 * passengers leaving.
			 */
			SARSimOpPassengersSetLeave(
			    scene, obj_ptr, 0, 0
			);
		    }

		    /* This is an explicit door operation, so if
		     * opening the door we need to set the
		     * the SAR_DOOR_FLAG_STAY_OPEN flag on it.
		     */
		    if(door_ptr->flags & SAR_DOOR_FLAG_STATE)
		    {
			/* Door is already opened, so we're closing
			 * it. When we close the door it will 
			 * automatically remove the 
			 * SAR_DOOR_FLAG_STAY_OPEN. flag.
			 */
		    }
		    else
		    {
			/* Door is closed, open it and set the
			 * SAR_DOOR_FLAG_STAY_OPEN flag.
			 */
			door_ptr->flags |= SAR_DOOR_FLAG_STAY_OPEN;
		    }

		    /* Set up door values to begin opening or closing. */
		    SARSimOpDoor(
			scene,
			&core_ptr->object, &core_ptr->total_objects,
			obj_ptr,
			!(door_ptr->flags & SAR_DOOR_FLAG_STATE)
		    );
		}
	    }
	}
}

/*
 *      Restart from nearest restarting point or end mission.
 *
 *	Checks if a mission exists but is no longer in progress
 *	(marked success or failed), in which case will end
 *	simulation.
 *
 *	If no mission exists then checks if player object is marked
 *	crashed and moves player to nearest restarting point.
 *
 *	This function returns 1 if the event was handled or 0 if it
 *	was not or error.
 */
static int SARKeyRestart(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	int obj_num;
	sar_object_struct *obj_ptr;


	if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return(0);

	/* Is there a mission? */
	if(core_ptr->mission != NULL)
	{
	    sar_mission_struct *m = core_ptr->mission;

	    /* Handle by mission state. */
	    switch(m->state)
	    {
	      case MISSION_STATE_IN_PROGRESS:
		/* Mission still in progress, do nothing. */
		break;

	      case MISSION_STATE_FAILED:
	      case MISSION_STATE_ACCOMPLISHED:
		/* End simulation and tabulate mission results. */
                SARSimEnd(core_ptr);
		/* Return indicating event was handled. */
		return(1);
		break;

	      default:
		fprintf(
		    stderr,
 "SARKeyRestart(): Unsupported mission state `%i'\n",
		    m->state
	        );
		break;
	    }
	}
	/* No mission (assume this is a free flight), check if player
	 * object is marked crashed.
	 */
	else if(scene->player_has_crashed)
	{
	    /* Get referances to player objects. */
	    obj_num = scene->player_obj_num;
	    obj_ptr = scene->player_obj_ptr;
	    if(obj_ptr != NULL)
	    {
		/* Move player object to nearest restarting point and
		 * repair it.
		 */
		SARSimRestart(
		    core_ptr, scene,
		    &core_ptr->object, &core_ptr->total_objects,
		    obj_num, obj_ptr
		);
		/* Return indicating event was handled. */
		return(1);
	    }
	}

	/* Return indicating no event handled. */
	return(0);
}

/*
 *	Refuel and repair aircraft.
 */
static void SARKeyRefuelRepair(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	sar_object_struct *obj_ptr;

 
        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	obj_ptr = scene->player_obj_ptr;
	if(obj_ptr != NULL)
	{
	    Boolean can_refuel = False, can_repair = False;
	    sar_object_aircraft_struct *obj_aircraft_ptr;

	    switch(obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_AIRCRAFT:
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr != NULL)
		{
		    if(obj_aircraft_ptr->landed)
		    {
			/* Check if landed at a helipad with
			 * proper facilities.
			 */
			int i, hit_obj_num, list_total, *list;
			sar_object_struct *hit_obj_ptr;
			sar_object_helipad_struct *obj_helipad_ptr;

			list = SARGetGCCHitList(
			    core_ptr, scene,
			    &core_ptr->object, &core_ptr->total_objects,
			    scene->player_obj_num,
			    &list_total
			);

			for(i = 0; i < list_total; i++)
			{
			    hit_obj_num = list[i];
			    hit_obj_ptr = SARObjGetPtr(
				core_ptr->object, core_ptr->total_objects,
				hit_obj_num
			    );
			    if(hit_obj_ptr == NULL)
				continue;

			    switch(hit_obj_ptr->type)
			    {
			      case SAR_OBJ_TYPE_HELIPAD:
				obj_helipad_ptr = (sar_object_helipad_struct *)hit_obj_ptr->data;
				if(obj_helipad_ptr != NULL)
				{
				    if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_FUEL)
					can_refuel = True;
				    if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_REPAIR)
					can_repair = True;
				}
				break;
			    }
			}

			free(list);
			list = NULL;
			list_total = 0;
		    }
		    else
		    {
			/* Not landed, cannot repair or refuel. */
		    }

		    if(can_refuel)
			SARSimOpRefuel(scene, obj_ptr);

		    if(can_repair)
			SARSimOpRepair(scene, obj_ptr);
		}
		break;
	    }
	}
}

/*
 *	Prints remaining fuel and statistics, transfers fuel from reserved
 *	tanks, or drop tanks.
 */
static void SARKeyFuel(
        sar_core_struct *core_ptr, gw_display_struct *display,
        sar_scene_struct *scene, Boolean state
)
{
	int obj_num;
        sar_object_struct *obj_ptr;


        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	obj_num = scene->player_obj_num;
        obj_ptr = scene->player_obj_ptr;
        if(obj_ptr != NULL)
        {
            sar_object_aircraft_struct *obj_aircraft_ptr;

            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
                if(obj_aircraft_ptr != NULL)
                {
		    if(display->ctrl_key_state)
		    {
			/* Drop next fuel tank. */
			SARSimOpDropFuelTankNext(
			    scene,
			    &core_ptr->object, &core_ptr->total_objects,
			    obj_num, obj_ptr
			);
		    }
		    else if(display->shift_key_state)
		    {
			char text[256];

			/* Transfer fuel. */
			double fuel_transfered = SARSimOpTransferFuelFromTanks(
			    scene, obj_ptr
			);
			if(fuel_transfered > 0.0)
			{
			    sprintf(
				text,
				SAR_MESG_RESERVE_FUEL_TRANSFERED,
				SFMKGToLBS(fuel_transfered)
			    );
			}
			else
			{
			    strcpy(text, SAR_MESG_NO_RESERVE_FUEL_TO_TRANSFER);
			}
                        SARMessageAdd(scene, text);
		    }
		    else
		    {
			/* Print fuel stats. */
			int i;
			sar_external_fueltank_struct *eft_ptr;
			double fuel = obj_aircraft_ptr->fuel;	/* In kg. */
			double fuel_rate = obj_aircraft_ptr->fuel_rate;	/* In kg/cycle. */
			double aloft_time;			/* In seconds. */
			double external_fuel = 0.0, external_fuel_max = 0.0;
			char text[256];


			/* Add up fuel from external reserved tanks. */
			for(i = 0; i < obj_aircraft_ptr->total_external_fueltanks; i++)
			{
			    eft_ptr = obj_aircraft_ptr->external_fueltank[i];
			    if(eft_ptr == NULL)
				continue;

			    if(eft_ptr->flags & SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD)
			    {
				external_fuel += eft_ptr->fuel;
				external_fuel_max += eft_ptr->fuel_max;
			    }
			}

			/* Calculate time left aloft. */
			if(fuel_rate > 0.0)
			    aloft_time = ((fuel + external_fuel) / fuel_rate) * 
				SAR_CYCLE_TO_SEC_COEFF;
			else
			    aloft_time = 0.0;

			if(external_fuel_max > 0.0)
			    sprintf(
			        text,
"Int: %.0f(%.0f) lbs  Ext: %.0f(%.0f) lbs  Endur: %s",
			        SFMKGToLBS(obj_aircraft_ptr->fuel),
				SFMKGToLBS(obj_aircraft_ptr->fuel_max),
				SFMKGToLBS(external_fuel),
				SFMKGToLBS(external_fuel_max),
				SARDeltaTimeString(
				    core_ptr,
				    (time_t)aloft_time
				)
			    );
			else
                            sprintf(
                                text,
"Int: %.0f(%.0f) lbs  Endur: %s",
                                SFMKGToLBS(obj_aircraft_ptr->fuel),
                                SFMKGToLBS(obj_aircraft_ptr->fuel_max),
                                SARDeltaTimeString(
				    core_ptr,
                                    (time_t)aloft_time
                                )
                            );
			SARMessageAdd(scene, text);
                    }
                }
                break;
            }
        }
}

/*
 *	Sends a message.
 */
static void SARKeySendMessage(
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
        if((core_ptr == NULL) || (display == NULL) || !state)
            return;

	SARTextInputMap(
	    core_ptr,
            "Message",
            SARTextInputCBSendMessage
        );
}

/*
 *	Adjusts the time of day on the scene structure.
 */
static void SARKeyTimeOfDay(  
        sar_core_struct *core_ptr,
        gw_display_struct *display,
        sar_scene_struct *scene,
        Boolean state
)
{
	const char *time_str;
	char text[80];

        if((core_ptr == NULL) || (display == NULL) || (scene == NULL) || !state)
            return;

	/* Increase or decrease time of day. */
	if(display->shift_key_state)
	    scene->tod -= (0.25 * 3600.0);
	else
	    scene->tod += (0.25 * 3600.0);

	time_str = (const char *)SARTimeOfDayString(core_ptr, scene->tod);
	sprintf(
	    text,
	    "%s %s",
	    SAR_MESG_TIME_OF_DAY,
	    time_str
	);
	SARMessageAdd(scene, text);
}



/*
 *	SAR keyboard event handler. Front end for handling all in game
 *	keyboard events.
 *
 *	If the in game text input prompt is mapped then the key event
 *	will be forwarded to the prompt input handler.
 *
 *	If key event key code c does not match any key operation to be
 *	handled then the keyboard event will be forwarde to
 *	GCtlHandleKey().
 */
void SARKey(
	sar_core_struct *core_ptr,
	int c, Boolean state, long t
)
{
	gw_display_struct *display;
	sar_scene_struct *scene;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	scene = core_ptr->scene;

#define DO_HAS_NO_AUTOREPEAT	\
{ \
 GWKeyboardAutoRepeat(display, (Boolean)!state); \
}

	/* Initial simulation key checks come first. */

	/* Space bar initially `continues' if mission failed or
	 * ended, or moves player to nearest restarting point on
	 * free flights.
	 */
	if(c == ' ')
	{
	    /* Space bar always implicitly clears scene's sticky banner
	     * message.
	     */
	    SARBannerMessageAppend(scene, NULL);

	    /* Do `restart' procedure, conditions will be checked by
	     * the function. Returns positive if event was handled.
	     */
	    if(SARKeyRestart(core_ptr, display, scene, state) > 0)
	    {
		/* Event was handled, return immediatly. The simulation
		 * may have ended or other memory greatly changed.
		 */
		return;
	    }
	}


	/* Handle key as a regular simulation action. */
	switch(c)
	{
/* Debug value adjustment. */
case 'x': case 'X':
 if((scene != NULL) && (display != NULL) && state)
 {
  debug_value += ((display->shift_key_state) ? 0.25 : -0.25);
fprintf(stderr, "Debug value set to %.2f\n", debug_value);
 }
 break;

	  case 's': case 'S':	/* Print score. */
	    DO_HAS_NO_AUTOREPEAT
	    if(display->ctrl_key_state)
		SARKeySoundLevel(core_ptr, display, scene, state);
	    else
		SARKeyPrintScore(core_ptr, display, scene, state);
	    break;

	  case GWKeyBackSpace:	/* Re-center/normalize view. */
	    DO_HAS_NO_AUTOREPEAT
            SARKeyViewNormalize(core_ptr, display, scene, state);
	    break;

	  case GWKeyF1:		/* Help display toggle. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyHelpDisplay(core_ptr, display, scene, state);
	    break;

          case GWKeyF2:		/* Switch camera ref to cockpit. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyCameraRefCockpit(core_ptr, display, scene, state);
            break;

          case GWKeyF3:		/* Switch camera ref to spot. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyCameraRefSpot(core_ptr, display, scene, state);
            break;

          case GWKeyF4:		/* Switch camera ref to tower. */
	    DO_HAS_NO_AUTOREPEAT
            SARKeyCameraRefTower(core_ptr, display, scene, state);
	    break;

          case GWKeyF5:		/* Switch camera ref to hoist's rescue basket. */
            DO_HAS_NO_AUTOREPEAT
	    SARKeyCameraRefHoist(core_ptr, display, scene, state);
	    break;

	  case 'm': case 'M':	/* Switch camera ref to map. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyCameraRefMap(core_ptr, display, scene, state);
	    break;

	  case GWKeyF9:		/* Toggle textured ground or dual pass depth. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyGraphicsTexturedGround(core_ptr, display, scene, state);
            break;

          case GWKeyF10:	/* Toggle atmosphere. */
	    DO_HAS_NO_AUTOREPEAT
            SARKeyGraphicsAtmosphere(core_ptr, display, scene, state);
	    break;

          case GWKeyF11:	/* Toggle textured objects. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyGraphicsTexturedObjects(core_ptr, display, scene, state);
	    break;

          case GWKeyF12:	/* Toggle textured clouds or prop wash. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyGraphicsTexturedClouds(core_ptr, display, scene, state);
	    break;

	  case 0x1b:		/* Escape. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyEscape(core_ptr, display, scene, state);
            break;

	  case '/':		/* Command prompt map. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyCommand(core_ptr, display, scene, state);
            break;

          case 'z': case 'Z':	/* Adjust time compression. */
            DO_HAS_NO_AUTOREPEAT
            SARKeyTimeCompression(core_ptr, display, scene, state);
            break;

	  /* Raise/lower landing gears on player object. */
	  case 'g': case 'G':
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyLandingGear(core_ptr, display, scene, state);
	    break;

	  /* Turn lights on/off on player object. */
	  case 'l': case 'L':
	    DO_HAS_NO_AUTOREPEAT
            SARKeyLights(core_ptr, display, scene, state);
	    break;

          /* Turn strobe lights on/off on player object. */
          case 'o': case 'O':
            DO_HAS_NO_AUTOREPEAT
	    SARKeyStrobes(core_ptr, display, scene, state);
            break;

	  /* Adjust text color. */
	  case 'h': case 'H':
	    SARKeyTextColor(core_ptr, display, scene, state);
            break;

	  /* Select next intercept way point. */
	  case 'w': case 'W':
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyInterceptWayPoint(core_ptr, display, scene, state);
            break;

	  /* Engine off/init. */
	  case 'e': case 'E':
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyEngine(core_ptr, display, scene, state);
            break;

	  /* Elevator trim down. */
	  case GWKeyHome:
	    SARKeyElevatorTrimDown(core_ptr, display, scene, state);
            break;

          /* Elevator trim. */
          case GWKeyEnd:
            SARKeyElevatorTrimUp(core_ptr, display, scene, state);
            break;

	  /* Send message. */
	  case '\'':
	    DO_HAS_NO_AUTOREPEAT
	    SARKeySendMessage(core_ptr, display, scene, state);
            break;

	  /* Adjust time of day. */
          case 't': case 'T':
	    SARKeyTimeOfDay(core_ptr, display, scene, state);
            break;

	  /* Select rescue hoist rope end deployment type. */
	  case 'p': case 'P':
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyHoistRopeEndSelect(core_ptr, display, scene, state);
            break;

	  /* Open/close rescue door on player object. */
	  case 'd': case 'D':
	    DO_HAS_NO_AUTOREPEAT
            if((display != NULL) && state)
	    {
		if(display->ctrl_key_state)
		{
		    /* Adjust flight physics dificulty. */
		    SARKeyFlightPhysicsDifficulty(core_ptr, display, scene, state);
		}
		else
		{
		    /* Open/close door. */
		    SARKeyDoor(core_ptr, display, scene, state);
		}
	    }
	    break;

	  case 'v': case 'V':
	    SARKeyVisibility(core_ptr, display, scene, state);
	    break;

	  case 'c': case 'C':
	    DO_HAS_NO_AUTOREPEAT
/*
            if((display != NULL) && (scene != NULL) && state)
            {
		sar_object_struct *obj_ptr;

		if(!display->ctrl_key_state)
		    break;

		obj_ptr = scene->player_obj_ptr;
		if(obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT)
		{
		    sar_object_aircraft_struct *obj_aircraft_ptr =
			obj_ptr->data;
		    if(obj_aircraft_ptr != NULL)
		    {
		        obj_aircraft_ptr->engine_state =
			    SAR_ENGINE_STATE_OFF;
			obj_aircraft_ptr->air_worthy_state =
			    SAR_AIR_WORTHY_OUT_OF_CONTROL;

                        SARMessageAdd(scene, "Crash'n ya! :>");
		    }
		}
            }
 */
            break;

          case 'a': case 'A':	/* Pitch rotors or toggle slew. */
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyPitchRotors(core_ptr, display, scene, state);
            break;

	  case 'r': case 'R':
	    DO_HAS_NO_AUTOREPEAT
	    SARKeyRefuelRepair(core_ptr, display, scene, state);
	    break;

	  case 'f': case 'F':
            DO_HAS_NO_AUTOREPEAT  
	    SARKeyFuel(core_ptr, display, scene, state);
	    break;

	  default:
	    /* Some other key, let the game controller handle it. */
	    GCtlHandleKey(
		display,
		core_ptr->gctl,
		c, state,
		display->alt_key_state,
		display->ctrl_key_state,
		display->shift_key_state,
		t
	    );
	    break;
	}

/* Calling function will redraw whenever a key event is recieved
	GWPostRedraw((gw_display_struct *)ptr);
 */
}
