#include <math.h>
#include <sys/types.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif
#include <GL/gl.h>
#include <GL/glu.h>

#include "gw.h"
#include "stategl.h"
#include "obj.h"
#include "sar.h"

#include "sardrawdefs.h"


void SARDrawRunway(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_runway_struct *obj_runway_ptr,
        double distance         /* 3d distance in meters. */
);
void SARDrawRunwayMap(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_runway_struct *obj_runway_ptr,
        double icon_len
);


/*
 *      Draws a runway specified by the object obj_ptr.
 *
 *	3d distance from camera to object is specified by distance in
 *	meters.
 *
 *	The runway substructure obj_runway_ptr should be the one on
 *	the object.
 *
 * Need to add support to draw multiple styles.
 *
 *	All inputs are assumed valid.
 */
void SARDrawRunway(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_runway_struct *obj_runway_ptr,
        double distance         /* 3d distance in meters. */
) 
{
        int tex_num;
        v3d_texture_ref_struct *t;
        double x_min, x_max, y_min, y_max, len;


        /* Select current texture. */
        tex_num = obj_runway_ptr->tex_num;
        if(SARIsTextureAllocated(scene, tex_num))
            t = scene->texture_ref[tex_num];
        else
            t = NULL;

        /* Get x and y bounds. */
        len = obj_runway_ptr->length;
        y_max = len / 2;
        y_min = -y_max;
        x_max = obj_runway_ptr->width / 2;
        x_min = -x_max;

        /* Farther than 3 miles away? */
        if(distance > SFMMilesToMeters(3.0))
        {
            GLenum shade_model_mode;

            /* Draw simple, no texture with greyish background. */
            V3DTextureSelect(NULL);
            glColor4d(0.5, 0.5, 0.5, 1.0);

	    /* Set flat shading. */
            shade_model_mode = display->state_gl.shade_model_mode;
            StateGLShadeModel(&display->state_gl, GL_FLAT);

            glBegin(GL_QUADS);
            {
                glNormal3d(0, 1, 0);
                glVertex3d(x_min, 0, -y_max);
                glVertex3d(x_min, 0, -y_min);
                glVertex3d(x_max, 0, -y_min);
                glVertex3d(x_max, 0, -y_max);
            }
            glEnd();

            /* Restore to previous shade model. */
            StateGLShadeModel(&display->state_gl, shade_model_mode);
        }
        else
        {   
            /* Draw closeup, with texturing and additional details. */
            GLenum shade_model_mode;

            /* Texture for runway background available? */
            if(t == NULL)
            {
                V3DTextureSelect(NULL);
                glColor4d(0.5, 0.5, 0.5, 1.0);
            }
            else
            {
                V3DTextureSelect(t);
                glColor4d(1.0, 1.0, 1.0, 1.0);
            }

            /* Set flat shading. */
            shade_model_mode = display->state_gl.shade_model_mode;
            StateGLShadeModel(&display->state_gl, GL_FLAT);

            glBegin(GL_QUADS);
            {
                glNormal3d(0, 1, 0);
            
                glTexCoord2d(0, 1 - 0);
                glVertex3d(x_min, 0, -y_max);
             
                glTexCoord2d(0, 1 - 1);
                glVertex3d(x_min, 0, -y_min);
                
                glTexCoord2d(1, 1 - 1);
                glVertex3d(x_max, 0, -y_min);
        
                glTexCoord2d(1, 1 - 0);
                glVertex3d(x_max, 0, -y_max);
            }
            glEnd();
            
            if(t != NULL)
                V3DTextureSelect(NULL);
            
            /* Dashes. */
            glBegin(GL_QUADS);
            {
                double dash_spacing, num_dashes, dash_width_min,
                       dash_width_max, dash_cur;
        
                num_dashes = 10;
                dash_spacing = MAX(len / (num_dashes * 2), 1.0);
                dash_width_min = -1.0;
                dash_width_max = 1.0;
            
                glColor4d(1.0, 1.0, 1.0, 1.0);
                glNormal3d(0, 1, 0);
             
                dash_cur = y_min;   
                while((dash_cur + dash_spacing) <= y_max)
                {
                    glVertex3d(
                        dash_width_min, 0,
                        -(dash_cur + dash_spacing)
                    );
                    glVertex3d(
                        dash_width_min, 0,
                        -dash_cur
                    );
                    glVertex3d(
                        dash_width_max, 0,
                        -dash_cur
                    );
                    glVertex3d(
                        dash_width_max, 0,
                        -(dash_cur + dash_spacing)
                    );

                    dash_cur += (2 * dash_spacing);
                }
            }
            glEnd();

            /* Restore to previous shade model. */
            StateGLShadeModel(&display->state_gl, shade_model_mode);
        }

        /* Draw edge lighting? */
        if(obj_runway_ptr->flags & SAR_RUNWAY_FLAG_EDGE_LIGHTING)
        {
            double spacing;
            StateGLBoolean lighting = display->state_gl.lighting;

            /* Check if lighting is currently enabled, if it is then
             * disable it.
             */
            lighting = display->state_gl.lighting;
            if(lighting)
            {
                StateGLDisable(&display->state_gl, GL_LIGHTING);
                StateGLDisable(&display->state_gl, GL_LIGHT0);
            }

            /* Calculate spacing of each light, min 100 meters apart. */
            spacing = MAX(
                pow(2, floor(log(distance) / log(2.5))),
                100
            );

            glColor4d(1.0, 1.0, 1.0, 1.0);
            StateGLDisable(&display->state_gl, GL_POINT_SMOOTH);
            StateGLPointSize(&display->state_gl, 1.0);
            glBegin(GL_POINTS);
            {
                double t;

                glNormal3d(0, 1, 0);

                for(t = 0; t < len; t += spacing)
                {
                    glVertex3d(x_min, 0, -(y_min + t));
                    glVertex3d(x_max, 0, -(y_min + t));
                }

                glVertex3d(x_min, 0, -y_max);
                glVertex3d(x_max, 0, -y_max);
            }
            glEnd();

            /* Enable lighting again if it was on prior to call. */
            if(lighting)
            {
                StateGLEnable(&display->state_gl, GL_LIGHTING);
                StateGLEnable(&display->state_gl, GL_LIGHT0);
            }
        }

        return;
}

/*
 *      Draws runway for display on map.
 *
 *      All inputs are assumed valid.
 */
void SARDrawRunwayMap(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_runway_struct *obj_runway_ptr,
        double icon_len
)
{
        GLenum shade_model_mode;
	double x_min, x_max, y_min, y_max;

	/* Calculate bounds. */
	x_max = MAX(obj_runway_ptr->width / 2, icon_len / 2);
	x_min = -x_max;
	y_max = MAX(obj_runway_ptr->length / 2, icon_len / 2);
	y_min = -y_max;

	/* No texture. */
	V3DTextureSelect(NULL);

        /* Set flat shade model. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, GL_FLAT);

	glBegin(GL_QUADS);
	{
	    /* Blue background. */
	    glColor4d(0.0, 0.0, 1.0, 1.0);
	    glNormal3d(0, 1, 0);
	    glVertex3d(x_min, 0, -y_max);
	    glVertex3d(x_min, 0, -y_min);
	    glVertex3d(x_max, 0, -y_min);
	    glVertex3d(x_max, 0, -y_max);

	    /* White stripe. */
	    glColor4d(1.0, 1.0, 1.0, 1.0);
	    glVertex3d(x_min * 0.5, 0, -(y_max + (x_max * 0.5)));
	    glVertex3d(x_min * 0.5, 0, -(y_min - (x_max * 0.5)));
	    glVertex3d(x_max * 0.5, 0, -(y_min - (x_max * 0.5)));
	    glVertex3d(x_max * 0.5, 0, -(y_max + (x_max * 0.5)));
	}
	glEnd();

        /* Restore previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}
