#include <stdio.h>
#include <sys/types.h>
#include <math.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>

#include "gw.h"

#include "sarreality.h"
#include "obj.h"

#include "sar.h"


static void SARDrawPremodeledControlTower(
	gw_display_struct *display,
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodeled_ptr,
	Boolean draw_for_gcc
);
static void SARDrawPremodeledBuilding(
	gw_display_struct *display,
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodeled_ptr,
	Boolean draw_for_gcc
);

void SARDrawPremodeled(
	gw_display_struct *display,
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodeled_ptr,
        sar_position_struct *camera_pos, sar_direction_struct *camera_dir,
	Boolean draw_for_gcc
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Draws a premodeled control tower.
 */
static void SARDrawPremodeledControlTower(
	gw_display_struct *display,
        sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodeled_ptr,
	Boolean draw_for_gcc
)
{
	GLenum shade_model_mode;
        double x_min, x_max, y_min, y_max, z_max, z_roof;
        int i, tex_num;


        /* Calculate bounds. */
        x_max = obj_premodeled_ptr->length / 2;  
        x_min = -x_max;

        y_max = obj_premodeled_ptr->width / 2;
        y_min = -y_max;

        z_max = MAX(
	    obj_premodeled_ptr->height - 4.0,
	    1.0
	);
	z_roof = obj_premodeled_ptr->height;


	/* Set flat shade model state. */
	shade_model_mode = display->state_gl.shade_model_mode;
	StateGLShadeModel(&display->state_gl, GL_FLAT);


        /* Select wall texture. */
        i = 0;
        if(i < SAR_OBJ_PREMODEL_MAX_TEXTURES)
            tex_num = obj_premodeled_ptr->tex_num[i];
        else
            tex_num = -1;
  
        if(SARIsTextureAllocated(scene, tex_num))
            V3DTextureSelect(scene->texture_ref[tex_num]);
        else
            V3DTextureSelect(NULL);
        glColor4d(1.0, 1.0, 1.0, 1.0);

        glBegin(GL_QUADS);
        {
            /* North wall. */
            glNormal3d(0, 0, -1);  
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, 0, -y_max);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_max);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_max);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, 0, -y_max);

            /* West wall. */
            glNormal3d(-1, 0, 0);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_min, 0, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_min, z_max, -y_min);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_max);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, 0, -y_max);

            /* South wall. */
            glNormal3d(0, 0, 1);   
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, 0, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_min);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_min);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, 0, -y_min);

            /* East wall. */
            glNormal3d(1, 0, 0);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, 0, -y_max);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_max);
            glTexCoord2d(1, 1 - 1);   
            glVertex3d(x_max, z_max, -y_min);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_max, 0, -y_min);
        }
        glEnd();


	/* Control tower windows. */
        V3DTextureSelect(NULL);   
        glColor4d(0.0, 0.0, 0.0, 1.0);

        glBegin(GL_QUADS);
        {
	    /* North. */
	    glNormal3d(0.0, -0.2, -0.8);
            glVertex3d(x_max, z_max, -y_max);
            glVertex3d(x_min, z_max, -y_max);
            glVertex3d(x_min, z_roof, (-y_max * 1.1));
            glVertex3d(x_max, z_roof, (-y_max * 1.1));

            glNormal3d(0.6, -0.2, -0.6);
            glVertex3d(x_max, z_max, -y_max);
            glVertex3d(x_max, z_max, -y_max);
            glVertex3d(x_max, z_roof, (-y_max * 1.1));
            glVertex3d((x_max * 1.1), z_roof, -y_max);

	    /* East. */
            glNormal3d(0.8, -0.2, 0.0);
            glVertex3d(x_max * 1.1, z_roof, -y_max);
            glVertex3d(x_max * 1.1, z_roof, -y_min);
            glVertex3d(x_max, z_max, -y_min);
            glVertex3d(x_max, z_max, -y_max);

            glNormal3d(0.6, -0.2, 0.6);
            glVertex3d(x_max, z_max, -y_min);
            glVertex3d(x_max, z_max, -y_min);
            glVertex3d(x_max * 1.1, z_roof, -y_min);
            glVertex3d(x_max, z_roof, -y_min * 1.1);

	    /* South. */
            glNormal3d(0.0, -0.2, 0.8);
            glVertex3d(x_min, z_max, -y_min);
            glVertex3d(x_max, z_max, -y_min);
            glVertex3d(x_max, z_roof, -y_min * 1.1);
            glVertex3d(x_min, z_roof, -y_min * 1.1);

            glNormal3d(-0.6, -0.2, 0.6);
            glVertex3d(x_min, z_max, -y_min);
            glVertex3d(x_min, z_max, -y_min);
            glVertex3d(x_min, z_roof, -y_min * 1.1);
            glVertex3d(x_min * 1.1, z_roof, -y_min);

            /* West. */
            glNormal3d(-0.8, -0.2, 0.0);
            glVertex3d(x_min, z_max, -y_max);
            glVertex3d(x_min, z_max, -y_min);
            glVertex3d(x_min * 1.1, z_roof, -y_min);
            glVertex3d(x_min * 1.1, z_roof, -y_max);  

            glNormal3d(-0.6, -0.2, -0.6);
            glVertex3d(x_min, z_max, -y_max);
            glVertex3d(x_min, z_max, -y_max);
            glVertex3d(x_min * 1.1, z_roof, -y_max);
            glVertex3d(x_min, z_roof, -y_max * 1.1);
        }
        glEnd();

 
        /* Select roof texture. */ 
        i = 1;
        if(i < SAR_OBJ_PREMODEL_MAX_TEXTURES)
            tex_num = obj_premodeled_ptr->tex_num[i];
        else
            tex_num = -1;
            
        if(SARIsTextureAllocated(scene, tex_num))
            V3DTextureSelect(scene->texture_ref[tex_num]);
        else
            V3DTextureSelect(NULL);
        glColor4d(1.0, 1.0, 1.0, 1.0);
            
        glBegin(GL_QUADS);
        {
            /* Roof. */
            glNormal3d(0, 1, 0);

            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_max * 1.1, z_roof, -y_max);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_max, z_roof, -y_max * 1.1);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_min, z_roof, -y_max * 1.1);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_min * 1.1, z_roof, -y_max);


            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_min * 1.1, z_roof, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_min, z_roof, -y_min * 1.1);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_max, z_roof, -y_min * 1.1);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_max * 1.1, z_roof, -y_min);

            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min * 1.1, z_roof, -y_min);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_max * 1.1, z_roof, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max * 1.1, z_roof, -y_max);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_min * 1.1, z_roof, -y_max);
        }
        glEnd();

        /* Restore shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

        return;
}


/*
 *	Draws a premodeled building.
 */
static void SARDrawPremodeledBuilding(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
        sar_object_premodeled_struct *obj_premodeled_ptr,
	Boolean draw_for_gcc
)
{
	GLenum shade_model_mode;
	double x_min, x_max, y_min, y_max, z_max;
	int i, tex_num;


	/* Calculate bounds. */
	x_max = obj_premodeled_ptr->length / 2;
	x_min = -x_max;

        y_max = obj_premodeled_ptr->width / 2;
        y_min = -y_max;

        z_max = obj_premodeled_ptr->height;


        /* Set flat shade model state. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, GL_FLAT);


        /* Select wall texture. */
        i = 0;
        if(i < SAR_OBJ_PREMODEL_MAX_TEXTURES)
            tex_num = obj_premodeled_ptr->tex_num[i];
        else
            tex_num = -1;

        if(SARIsTextureAllocated(scene, tex_num))
            V3DTextureSelect(scene->texture_ref[tex_num]);
        else
            V3DTextureSelect(NULL);
        glColor4d(1.0, 1.0, 1.0, 1.0);   

	/* Draw walls. */
        glBegin(GL_QUADS);  
	{
	    /* North wall. */
            glNormal3d(0, 0, -1);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, 0, -y_max);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_max);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_max);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, 0, -y_max);

            /* West wall. */
            glNormal3d(-1, 0, 0);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_min, 0, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_min, z_max, -y_min);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_max);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, 0, -y_max);

            /* South wall. */
            glNormal3d(0, 0, 1);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, 0, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_min);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_min);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, 0, -y_min);

            /* East wall. */
            glNormal3d(1, 0, 0);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, 0, -y_max);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_max);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_max, z_max, -y_min);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_max, 0, -y_min);
	}
        glEnd();

	/* Draw walls at night? */
	if(scene->tod_code != SAR_SCENE_TOD_DAY)
	{
	    StateGLBoolean lighting = display->state_gl.lighting;


	    /* Check if lighting is on, if it is then turn it off. */
	    if(lighting)
	    {
		StateGLDisable(&display->state_gl, GL_LIGHTING);
	    }

            /* Select night wall texture. */
            i = 1;
            if(i < SAR_OBJ_PREMODEL_MAX_TEXTURES)
                tex_num = obj_premodeled_ptr->tex_num[i];
            else
                tex_num = -1;

            if(SARIsTextureAllocated(scene, tex_num))
                V3DTextureSelect(scene->texture_ref[tex_num]); 
            else
                V3DTextureSelect(NULL);
            glColor4d(1.0, 1.0, 1.0, 1.0);

            /* Draw walls. */
            glBegin(GL_QUADS);
            {
                /* North wall. */
                glNormal3d(0, 0, -1);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(x_min, 0, -y_max);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(x_min, z_max, -y_max);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(x_max, z_max, -y_max);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(x_max, 0, -y_max);

                /* West wall. */
                glNormal3d(-1, 0, 0);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(x_min, 0, -y_min);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(x_min, z_max, -y_min);
                glTexCoord2d(1, 1 - 1); 
                glVertex3d(x_min, z_max, -y_max);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(x_min, 0, -y_max);

                /* South wall. */
                glNormal3d(0, 0, 1);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(x_max, 0, -y_min);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(x_max, z_max, -y_min);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(x_min, z_max, -y_min);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(x_min, 0, -y_min);

                /* East wall. */
                glNormal3d(1, 0, 0);
                glTexCoord2d(0, 1 - 0);
                glVertex3d(x_max, 0, -y_max);
                glTexCoord2d(0, 1 - 1);
                glVertex3d(x_max, z_max, -y_max);
                glTexCoord2d(1, 1 - 1);
                glVertex3d(x_max, z_max, -y_min);
                glTexCoord2d(1, 1 - 0);
                glVertex3d(x_max, 0, -y_min);
            }
            glEnd();

            /* Turn on lighting. */
	    if(lighting)
                StateGLEnable(&display->state_gl, GL_LIGHTING);
	}


        /* Select roof texture. */
        i = 2;
        if(i < SAR_OBJ_PREMODEL_MAX_TEXTURES)
            tex_num = obj_premodeled_ptr->tex_num[i];
        else
            tex_num = -1;

        if(SARIsTextureAllocated(scene, tex_num))
            V3DTextureSelect(scene->texture_ref[tex_num]);
        else
            V3DTextureSelect(NULL);
        glColor4d(1.0, 1.0, 1.0, 1.0);

        glBegin(GL_QUADS);
        {   
            /* Roof. */
            glNormal3d(0, 1, 0);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(x_max, z_max, -y_min);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(x_max, z_max, -y_max);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(x_min, z_max, -y_max);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(x_min, z_max, -y_min);
	}
        glEnd();

        /* Restore shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}

/*
 *	This function is called from SARDraw().
 */
void SARDrawPremodeled(
	gw_display_struct *display,
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_object_premodeled_struct *obj_premodeled_ptr,
	sar_position_struct *camera_pos, sar_direction_struct *camera_dir,
	Boolean draw_for_gcc
)
{
	if(obj_premodeled_ptr == NULL)
	    return;

	switch(obj_premodeled_ptr->type)
	{
	  case SAR_OBJ_PREMODELED_BUILDING:
	    SARDrawPremodeledBuilding(
		display, scene, obj_ptr, obj_premodeled_ptr,
		draw_for_gcc
	    );
	    break;

          case SAR_OBJ_PREMODELED_CONTROL_TOWER:
            SARDrawPremodeledControlTower(
		display, scene, obj_ptr, obj_premodeled_ptr,
		draw_for_gcc
	    );
            break;
	}

	return;
}
