#include <math.h>
#include <sys/types.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif
#include <GL/gl.h>
#include <GL/glu.h>

#include "gw.h"
#include "stategl.h"
#include "obj.h"
#include "sar.h"

#include "sardrawdefs.h"


static void SARDrawHelipadEdgeLighting(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        double x_min, double x_max,
        double y_min, double y_max,
	double z_min, double z_max
);
static void SARDrawHelipadPaved(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
	v3d_texture_ref_struct *t,
	double x_min, double x_max,
	double y_min, double y_max,
	double z_min, double z_max
);
static void SARDrawHelipadBare(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
);
static void SARDrawHelipadBuilding(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
);
static void SARDrawHelipadVehicle(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
);

void SARDrawHelipad(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        double distance
);
void SARDrawHelipadMap(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        double icon_len
);


static GLfloat edge_lighting_point_size = 1.0;


/*
 *	Draws standard square edge lighting, a light in each corner.
 *
 *	Calling function needs to unselect texture.
 */
static void SARDrawHelipadEdgeLighting(
        gw_display_struct *display, sar_scene_struct *scene,
	sar_object_struct *obj_ptr,
	sar_object_helipad_struct *obj_helipad_ptr,
        double x_min, double x_max,
        double y_min, double y_max,
	double z_min, double z_max
)
{
	StateGLBoolean lighting;


        /* Unselect texture. */
	V3DTextureSelect(NULL);

	/* Check if lighting is currently enabled, if it is then
	 * disable it.
	 */
	lighting = display->state_gl.lighting;
	if(lighting)
	{
	    StateGLDisable(&display->state_gl, GL_LIGHTING);
	    StateGLDisable(&display->state_gl, GL_LIGHT0);
	}

	StateGLEnable(&display->state_gl, GL_POINT_SMOOTH);
	StateGLPointSize(&display->state_gl, edge_lighting_point_size);

	glColor4d(1.0, 1.0, 1.0, 1.0);

	glBegin(GL_POINTS);
	{
	    glNormal3d(0.0, 1.0, -0.0);
	    glVertex3d(
		x_min, z_max, -y_min
	    );
	    glVertex3d(
		x_min, z_max, -y_max
	    );
	    glVertex3d(
		x_max, z_max, -y_max
	    );
	    glVertex3d(
		x_max, z_max, -y_min
	    );
	}
	glEnd();

	/* Enable lighting again if it was on prior to call. */
        if(lighting)
        {
            StateGLEnable(&display->state_gl, GL_LIGHTING);
            StateGLEnable(&display->state_gl, GL_LIGHT0);
        }

	return;
}

/*
 *	Draws paved version of helipad.
 */
static void SARDrawHelipadPaved(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
)
{
	GLenum shade_model_mode;

        /* Texture for landable area available? */
        if(t == NULL)
        {
            /* No texture available, unselect texture and set color
             * greyish.
             */
            V3DTextureSelect(NULL);
            glColor4d(0.6, 0.6, 0.6, 1.0);
        }
        else
        {
            V3DTextureSelect(t);
            glColor4d(1.0, 1.0, 1.0, 1.0);
        }
	/* Set shade model dependant of global options. */
	shade_model_mode = display->state_gl.shade_model_mode;
	StateGLShadeModel(&display->state_gl, option.gl_shade_model);

        /* Draw main landable area. */
        glBegin(GL_QUADS);
        {
            glNormal3d(-0.33, 0.88, -0.33);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(
		x_min, z_max, -y_max
	    );

            glNormal3d(-0.33, 0.88, 0.33); 
            glTexCoord2d(0, 1 - 1);
            glVertex3d(
		x_min, z_max, -y_min
	    );

            glNormal3d(0.33, 0.88, 0.33); 
            glTexCoord2d(1, 1 - 1);
            glVertex3d(
		x_max, z_max, -y_min
	    );

            glNormal3d(0.33, 0.88, -0.33); 
            glTexCoord2d(1, 1 - 0);
            glVertex3d(
		x_max, z_max, -y_max
	    );
        }
        glEnd();

        /* Unselect texture. */
        if(t != NULL)
            V3DTextureSelect(NULL);

        /* Draw edge lighting? */
        if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_EDGE_LIGHTING)
	    SARDrawHelipadEdgeLighting(
		display, scene,
		obj_ptr, obj_helipad_ptr,
		x_min, x_max, y_min, y_max, z_min, z_max
	    );

	/* Restore to previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}

/*
 *      Draws bare version of helipad.
 */
static void SARDrawHelipadBare(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
)
{
	GLenum shade_model_mode;

        /* Texture for landable area available? */
        if(t == NULL)
        {
            /* No texture available, unselect texture and set color
             * light beige.
             */
            V3DTextureSelect(NULL);
            glColor4d(0.75, 0.72, 0.62, 1.0);
        }
        else
        {
            V3DTextureSelect(t);
            glColor4d(1.0, 1.0, 1.0, 1.0);
        }
        /* Set shade model dependant of global options. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, option.gl_shade_model);

        /* Draw main landable area. */
        glBegin(GL_QUADS);
        {
            glNormal3d(-0.33, 0.88, -0.33);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(
                x_min, z_max, -y_max
            );

            glNormal3d(-0.33, 0.88, 0.33);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(
                x_min, z_max, -y_min
            );

            glNormal3d(0.33, 0.88, 0.33);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(
                x_max, z_max, -y_min
            );

            glNormal3d(0.33, 0.88, -0.33);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(
                x_max, z_max, -y_max
            );
        }
        glEnd();

        /* Unselect texture. */
        if(t != NULL)
            V3DTextureSelect(NULL);

        /* Draw edge lighting? */
        if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_EDGE_LIGHTING)
            SARDrawHelipadEdgeLighting(
                display, scene,
                obj_ptr, obj_helipad_ptr,
                x_min, x_max, y_min, y_max, z_min, z_max
            );

        /* Restore to previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}

/*
 *      Draws building version of helipad.
 */
static void SARDrawHelipadBuilding(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
)
{
	GLenum shade_model_mode;

        V3DTextureSelect(NULL);

        /* Set shade model dependant of global options. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, option.gl_shade_model);

        /* Draw recession struts as needed. */
        if(z_min < 0.0)
        {
          glColor4d(0.3, 0.3, 0.3, 1.0);

          glBegin(GL_QUADS);
          {
	    double x[2], y[2], z[2], coeff = 0.6;

	    x[0] = x_min * coeff;
	    x[1] = x_max * coeff;
            y[0] = y_min * coeff;
            y[1] = y_max * coeff;
            z[0] = z_min;
            z[1] = z_max;

            /* North. */
            glNormal3d(0.0, 0.0, -1.0);
            glVertex3d(x[1], z[1], -y[1]);
            glVertex3d(x[1], z[0], -y[1]);
            glVertex3d(x[0], z[0], -y[1]);
            glVertex3d(x[0], z[1], -y[1]);

            /* East. */
            glNormal3d(1.0, 0.0, 0.0);
            glVertex3d(x[1], z[1], -y[0]);
            glVertex3d(x[1], z[0], -y[0]);
            glVertex3d(x[1], z[0], -y[1]);
            glVertex3d(x[1], z[1], -y[1]);

            /* South. */
            glNormal3d(0.0, 0.0, 1.0);
            glVertex3d(x[0], z[1], -y[0]);
            glVertex3d(x[0], z[0], -y[0]);
            glVertex3d(x[1], z[0], -y[0]);  
            glVertex3d(x[1], z[1], -y[0]);  

            /* West. */
            glNormal3d(-1.0, 0.0, 0.0);
            glVertex3d(x[0], z[1], -y[1]);  
            glVertex3d(x[0], z[0], -y[1]);
            glVertex3d(x[0], z[0], -y[0]);
            glVertex3d(x[0], z[1], -y[0]);
          }
	  glEnd();

	  /* Ramp down to recession on south side (this will protrude
	   * out of the helipad's bounds though.
	   */
	  glColor4d(0.75, 0.75, 0.75, 1.0);
	  glBegin(GL_QUADS);
	  {
	    double coeff = 0.2;	/* Of width. */

            glNormal3d(0.0, 0.7, 0.7);
            glVertex3d(
		x_min * coeff,
		z_max,
		-(y_min * 0.9)
	    );
            glVertex3d(
                x_min * coeff,
                z_min,
                -(y_min * 1.2)
            );
            glVertex3d(
                x_max * coeff,
                z_min,
                -(y_min * 1.2)
            );
            glVertex3d(   
                x_max * coeff,
                z_max,
                -(y_min * 0.9)
            );

            glNormal3d(0.0, 0.7, -0.7);
            glVertex3d(
                x_min * coeff,
                z_min,
                -(y_max * 1.2)   
            );
            glVertex3d(
                x_min * coeff,
                z_max,
                -(y_max * 0.9)
            );
            glVertex3d(
                x_max * coeff,
                z_max,
                -(y_max * 0.9)   
            );
            glVertex3d(
                x_max * coeff,
                z_min,
                -(y_max * 1.2)   
            );
	  }
	  glEnd();
	}

        /* Draw underside of landable area if have recession. */
        if(z_min < 0.0)
        {
            GLboolean depth_mask_flag = display->state_gl.depth_mask_flag;
            StateGLBoolean polygon_offset_fill = display->state_gl.polygon_offset_fill;


            glColor4d(0.6, 0.6, 0.6, 1.0);
            glBegin(GL_POLYGON);
            {
                double theta;

                glNormal3d(0.0, -1.0, -0.0);

                /* Draw a circular area, starting from 12 o'clock. */
                for(theta = 0.0; theta < 2.0; theta += 0.1)
                {
                    glVertex3d(
                        (x_max * sin(theta * PI)),
                        z_max,
                        -(y_max * cos(theta * PI))
                    );
                }
            }
            glEnd();

            /* Shadow of helipad. */
            StateGLDepthMask(&display->state_gl, GL_FALSE);
            StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
            StateGLPolygonOffset(
                &display->state_gl,
                (GLfloat)option.gl_polygon_offset_factor, -1.0
            );

            glColor4d(0.4, 0.4, 0.4, 1.0);
            glBegin(GL_POLYGON);
            {
                double theta;

                glNormal3d(0.0, 1.0, -0.0);

                /* Draw a circular area, starting from 12 o'clock. */
                for(theta = 2.0; theta > 0.0; theta -= 0.1)
                {
                    glVertex3d(
                        (x_max * sin(theta * PI)),
                        z_min,
                        -(y_max * cos(theta * PI))
                    );
                }
            }
            glEnd();

            /* Restore depth mask and polygon offset fill states. */
            StateGLDepthMask(&display->state_gl, depth_mask_flag);
            if(polygon_offset_fill)
                StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
            else
                StateGLDisable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
        }


        /* Texture for main landable area available? */
        if(t == NULL)
        {
            /* No texture available, unselect texture and set color
             * light grey.
             */
            glColor4d(0.84, 0.84, 0.84, 1.0);
        }
        else
        {
            V3DTextureSelect(t);
            glColor4d(1.0, 1.0, 1.0, 1.0);
        }

        /* Draw main landable area. */
        glBegin(GL_POLYGON);
        {
	    double theta;

	    /* Draw a circular area, starting from 12 o'clock. */
	    for(theta = 2.0; theta > 0.0; theta -= 0.1)
	    {
                glNormal3d(
		    0.33 * sin(theta * PI),
		    0.88,
		    -(0.33 * cos(theta * PI))
		);
		glTexCoord2d(
		    (sin(theta * PI) + 1.0) / 2.0,
		    1.0 - ((cos(theta * PI) + 1.0) / 2.0)
		);
		glVertex3d(
		    (x_max * sin(theta * PI)),
		    z_max,
		    -(y_max * cos(theta * PI))
		);
	    }
        }
        glEnd();

        /* Unselect texture. */
        if(t != NULL)
            V3DTextureSelect(NULL);

        /* Draw edge lighting? */
        if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_EDGE_LIGHTING)
	{
	    /* Draw four points of lighting at the right angle extremes
	     * instead of at the corners.
	     */
	    StateGLBoolean lighting = display->state_gl.lighting;

	    /* Check if lighting is currently enabled, if it is then
	     * disable it.
	     */
	    if(lighting)
	    {
		StateGLDisable(&display->state_gl, GL_LIGHTING);
		StateGLDisable(&display->state_gl, GL_LIGHT0);
	    }

	    StateGLEnable(&display->state_gl, GL_POINT_SMOOTH);
	    StateGLPointSize(&display->state_gl, edge_lighting_point_size);

	    glColor4d(1.0, 1.0, 1.0, 1.0);

	    glBegin(GL_POINTS);
	    {
                glNormal3d(0.0, 1.0, -0.0);
                glVertex3d(
                    0.0, z_max, -y_max
                );
                glVertex3d(
                    x_max, z_max, -0.0
                );
                glVertex3d(
                    0.0, z_max, -y_min
                );
                glVertex3d(
                    x_min, z_max, -0.0
                );
            }
            glEnd();

            /* Enable lighting again if it was on prior to call. */
            if(lighting)
            {
                StateGLEnable(&display->state_gl, GL_LIGHTING);
                StateGLEnable(&display->state_gl, GL_LIGHT0);
            }
	}

        /* Restore to previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

        return;
}

/*
 *      Draws vehicle version of helipad.
 */
static void SARDrawHelipadVehicle(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        v3d_texture_ref_struct *t,
        double x_min, double x_max,
        double y_min, double y_max,
        double z_min, double z_max
)
{
        GLenum shade_model_mode;

        V3DTextureSelect(NULL);

        /* Set shade model dependant of global options. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, option.gl_shade_model);

        /* Draw recession struts. */
        glColor4d(0.6, 0.6, 0.6, 1.0);

        glBegin(GL_QUADS);
        {
            double x[2], y[2], z[2], coeff = 0.4;

            x[0] = x_min;
            x[1] = x_max;
            y[0] = y_min;
            y[1] = y_max;
            z[0] = z_min * coeff;
            z[1] = z_max;

            /* North. */
            glNormal3d(0.0, 0.0, -1.0);   
            glVertex3d(x[1], z[1], -y[1]);
            glVertex3d(x[1], z[0], -y[1]);
            glVertex3d(x[0], z[0], -y[1]);
            glVertex3d(x[0], z[1], -y[1]);

            /* East. */
            glNormal3d(1.0, 0.0, 0.0);
            glVertex3d(x[1], z[1], -y[0]);
            glVertex3d(x[1], z[0], -y[0]);
            glVertex3d(x[1], z[0], -y[1]);
            glVertex3d(x[1], z[1], -y[1]);

            /* South. */
            glNormal3d(0.0, 0.0, 1.0);
            glVertex3d(x[0], z[1], -y[0]);
            glVertex3d(x[0], z[0], -y[0]);
            glVertex3d(x[1], z[0], -y[0]);
            glVertex3d(x[1], z[1], -y[0]);

            /* West. */
            glNormal3d(-1.0, 0.0, 0.0);
            glVertex3d(x[0], z[1], -y[1]);
            glVertex3d(x[0], z[0], -y[1]);
            glVertex3d(x[0], z[0], -y[0]);
            glVertex3d(x[0], z[1], -y[0]);


	    /* Lower struts. */
	    glColor4d(0.3, 0.3, 0.3, 1.0);

	    coeff = 0.9;
            x[0] = x_min * coeff;
            x[1] = x_max * coeff;
            y[0] = y_min * coeff;
            y[1] = y_max * coeff;
            z[0] = z_min;
            z[1] = z_max;

            /* North. */
            glNormal3d(0.0, 0.0, -1.0);   
            glVertex3d(x[1], z[1], -y[1]);
            glVertex3d(x[1], z[0], -y[1]);
            glVertex3d(x[0], z[0], -y[1]);
            glVertex3d(x[0], z[1], -y[1]);
         
            /* East. */
            glNormal3d(1.0, 0.0, 0.0);
            glVertex3d(x[1], z[1], -y[0]);
            glVertex3d(x[1], z[0], -y[0]);
            glVertex3d(x[1], z[0], -y[1]);
            glVertex3d(x[1], z[1], -y[1]);  

            /* South. */
            glNormal3d(0.0, 0.0, 1.0);
            glVertex3d(x[0], z[1], -y[0]);
            glVertex3d(x[0], z[0], -y[0]);
            glVertex3d(x[1], z[0], -y[0]);
            glVertex3d(x[1], z[1], -y[0]);

            /* West. */
            glNormal3d(-1.0, 0.0, 0.0);
            glVertex3d(x[0], z[1], -y[1]);
            glVertex3d(x[0], z[0], -y[1]);
            glVertex3d(x[0], z[0], -y[0]);
            glVertex3d(x[0], z[1], -y[0]);
        }
        glEnd();


        /* Ramp down to recession on north and south side (this will
	 * protrude out of the helipad's bounds).
         */
        glColor4d(0.7, 0.7, 0.7, 1.0);
        glBegin(GL_QUADS);
        {
            double coeff = 0.2;		/* Of width. */

            glNormal3d(0.0, 0.7, 0.7);
            glVertex3d(
                x_min * coeff,
                z_max,
                -y_min
            );
            glVertex3d(
                x_min * coeff,
                z_min,
                -(y_min * 1.1)
            );
            glVertex3d(   
                x_max * coeff,
                z_min,
                -(y_min * 1.1)
            );
            glVertex3d(
                x_max * coeff,
                z_max,
                -(y_min)
            );

            glNormal3d(0.0, 0.7, -0.7);
            glVertex3d(
                x_min * coeff,
                z_min,
                -(y_max * 1.1)
            );
            glVertex3d(
                x_min * coeff,
                z_max,
                -(y_max * 1.0)
            );
            glVertex3d(
                x_max * coeff,
                z_max,
                -(y_max * 1.0)
            );
            glVertex3d(
                x_max * coeff,
                z_min,
                -(y_max * 1.1)
            );
        }
        glEnd();


        /* Texture for landable area available? */
        if(t == NULL)
        {
            /* No texture available, unselect texture and set color
             * light grey.
             */
            V3DTextureSelect(NULL);
            glColor4d(0.84, 0.84, 0.84, 1.0);
        }
        else
        {
            V3DTextureSelect(t);
            glColor4d(1.0, 1.0, 1.0, 1.0);
        }

        /* Draw main landable area. */
        glBegin(GL_QUADS);
        {
            glNormal3d(-0.33, 0.88, -0.33);
            glTexCoord2d(0, 1 - 0);
            glVertex3d(
                x_min, z_max, -y_max
            );

            glNormal3d(-0.33, 0.88, 0.33);
            glTexCoord2d(0, 1 - 1);
            glVertex3d(
                x_min, z_max, -y_min
            );

            glNormal3d(0.33, 0.88, 0.33);
            glTexCoord2d(1, 1 - 1);
            glVertex3d(
                x_max, z_max, -y_min
            );

            glNormal3d(0.33, 0.88, -0.33);
            glTexCoord2d(1, 1 - 0);
            glVertex3d(
                x_max, z_max, -y_max
            );
        }
        glEnd();

        /* Unselect texture. */
        if(t != NULL)
            V3DTextureSelect(NULL);

	/* Draw fence. */
	if(1)
	{
	    double coeff = 1.1, zabove = z_max + 0.8;

	    glColor4d(0.3, 0.3, 0.3, 1.0);

            /* East fence. */
            glBegin(GL_LINE_LOOP);
            {
                glNormal3d(0.33, 0.88, -0.33);
                glVertex3d(
                    x_max * coeff,
                    zabove,
                    -(y_min * coeff)
                );

                glNormal3d(0.33, 0.88, 0.33);
                glVertex3d(
                    x_max * coeff,
                    zabove,
                    -(y_max * coeff)
                );

                glNormal3d(-0.33, 0.88, 0.33);
                glVertex3d(
                    x_max,
                    z_max,
                    -(y_max)
                );

                glNormal3d(-0.33, 0.88, -0.33);
                glVertex3d(
                    x_max,
                    z_max,
                    -(y_min)
                );
            }
            glEnd();

            /* West fence. */
            glBegin(GL_LINE_LOOP);
            {
                glNormal3d(-0.33, 0.88, -0.33);
                glVertex3d(
                    x_min * coeff,
                    zabove,
                    -(y_max * coeff)
                );

                glNormal3d(-0.33, 0.88, 0.33);
                glVertex3d(   
                    x_min * coeff,
                    zabove,
                    -(y_min * coeff)
                );

                glNormal3d(0.33, 0.88, 0.33);
                glVertex3d(
                    x_min,
		    z_max,
                    -(y_min)
                );

                glNormal3d(0.33, 0.88, -0.33);
                glVertex3d(
                    x_min,
                    z_max,
                    -(y_max)
                );
            }
            glEnd();
	}

        /* Draw edge lighting? */
        if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_EDGE_LIGHTING)
            SARDrawHelipadEdgeLighting(
                display, scene,
                obj_ptr, obj_helipad_ptr,
                x_min, x_max, y_min, y_max, z_min, z_max
            );

        /* Restore to previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}


/*
 *      Draws a helipad specified by the object obj_ptr.
 *
 *	3d distance from camera to object is specified by distance in
 *	meters.
 *
 *	The helipad substructure obj_helipad_ptr should be the one on
 *	the object.
 *
 * Need to add support to draw multiple styles.
 *
 *	All inputs are assumed valid.
 */
void SARDrawHelipad(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
        double distance         /* 3d distance in meters. */
) 
{
        int tex_num;
        v3d_texture_ref_struct *t; 
        double x_min, x_max, y_min, y_max, z_min, z_max;


        /* Get pointer to helipad main landable area texture. */
        tex_num = obj_helipad_ptr->tex_num;
	t = (SARIsTextureAllocated(scene, tex_num) ?
	    scene->texture_ref[tex_num] : NULL
	);

        /* Get x and y bounds of helipad, these specify the landable
         * area of the helipad.
         */
        y_max = obj_helipad_ptr->length / 2;
        y_min = -y_max;
        x_max = obj_helipad_ptr->width / 2;
        x_min = -x_max;
	z_min = -obj_helipad_ptr->recession;
	z_max = 0.0;


	/* Draw by style. */
	switch(obj_helipad_ptr->style)
	{
	  case SAR_HELIPAD_STYLE_GROUND_BARE:
            SARDrawHelipadBare(
                display, scene, obj_ptr, obj_helipad_ptr,
                t, x_min, x_max, y_min, y_max, z_min, z_max
            );
	    break;

	  case SAR_HELIPAD_STYLE_BUILDING:
            SARDrawHelipadBuilding(
                display, scene, obj_ptr, obj_helipad_ptr,
                t, x_min, x_max, y_min, y_max, z_min, z_max
            );
	    break;

	  case SAR_HELIPAD_STYLE_VEHICLE:
            SARDrawHelipadVehicle(
                display, scene, obj_ptr, obj_helipad_ptr,
                t, x_min, x_max, y_min, y_max, z_min, z_max
            );
	    break;

	  default:	/* SAR_HELIPAD_STYLE_GROUND_PAVED */
	    SARDrawHelipadPaved(
		display, scene, obj_ptr, obj_helipad_ptr,
		t, x_min, x_max, y_min, y_max, z_min, z_max
	    );
	    break;
	}

        return;
}

/*
 *	Draws helipad for display on map.
 *
 *	All inputs are assumed valid.
 */
void SARDrawHelipadMap(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_helipad_struct *obj_helipad_ptr,
	double icon_len
)
{
	GLenum shade_model_mode;
	double x_min, x_max, y_min, y_max;

	/* Calculate bounds. */
	x_min = -(icon_len / 2);
	x_max = icon_len / 2;
	y_min = -(icon_len / 2);
	y_max = icon_len / 2;

	/* No texture. */
	V3DTextureSelect(NULL);

        /* Set flat shade model. */
	shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, GL_FLAT);

	/* Begin drawing icon. */
	glBegin(GL_QUADS);
	{
	    /* Blue background quad. */
	    glColor4d(0.0, 0.0, 1.0, 1.0);
	    glNormal3d(0, 1, 0);

	    glVertex3d(x_min, 0, -y_max);
	    glVertex3d(x_min, 0, -y_min);
	    glVertex3d(x_max, 0, -y_min);
	    glVertex3d(x_max, 0, -y_max);

	    /* White `cross' composed of two quads. */
	    glColor4d(1.0, 1.0, 1.0, 1.0);
	    glVertex3d((x_min * 0.8), 0, (-y_max * 0.2));
	    glVertex3d((x_min * 0.8), 0, (-y_min * 0.2));
	    glVertex3d((x_max * 0.8), 0, (-y_min * 0.2));
	    glVertex3d((x_max * 0.8), 0, (-y_max * 0.2));

	    glVertex3d((x_min * 0.2), 0, (-y_max * 0.8));
	    glVertex3d((x_min * 0.2), 0, (-y_min * 0.8));
	    glVertex3d((x_max * 0.2), 0, (-y_min * 0.8));
	    glVertex3d((x_max * 0.2), 0, (-y_max * 0.8));
	}
	glEnd();

	/* Restore previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

	return;
}
