#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <math.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/fio.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"

#include "sfm.h"
#include "v3dhf.h"
#include "v3dtex.h"
#include "v3dmh.h"
#include "v3dmp.h"
#include "v3dmodel.h"
#include "v3dfio.h"

#include "sarreality.h"
#include "obj.h"
#include "human.h"
#include "smoke.h"
#include "fire.h"
#include "simop.h"
#include "simmanage.h"
#include "simcb.h"
#include "simutils.h"
#include "weather.h"

#include "sar.h"
#include "sarfio.h"

#include "config.h"


char *SARObjFSeekGetOpString(FILE *fp);

int SARObjLoadTranslate(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr, sar_parm_translate_struct *p_translate
);
int SARObjLoadTranslateRandom(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_parm_translate_random_struct *p_translate_random
);

int SARObjLoadSound(
        snd_recorder_struct *recorder,
        snd_play_struct **snd_play_rtn,
        const char *path, int options
);
int SARObjLoadTexture(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_texture_load_struct *p_texture_load
);
int SARObjLoadHelipad(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_new_helipad_struct *p_new_helipad
);
int SARObjLoadHuman(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_new_human_struct *p_new_human
);
int SARObjLoadFire(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_fire_struct *p_new_fire
);
int SARObjLoadSmoke(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_smoke_struct *p_new_smoke
);
int SARObjLoadHeightField(
        sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,   
        void *p, const char *filename, int line_num,
	GLuint list
);
static int SARObjLoadVisualPrimitive(
	sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        void *p, const char *filename, int line_num
);
int SARObjLoadLine(
	sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        const char *line, const char *filename, int line_num
);
int SARObjLoadFromFile(
	sar_core_struct *core_ptr,
        int n,			/* Object number, can be -1. */
        const char *filename
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define ISCOMMENT(c)	((c) == SAR_COMMENT_CHAR)
#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


static int last_begin_primitive_type;


/*
 *	Quick converter from degrees to radians without sanitization:
 */
#define DEGTORAD(d)	((d) * PI / 180)

/*
 *	Texture plane orientation codes:
 */
#define TEX_ORIENT_NONE	0
#define TEX_ORIENT_XY	1
#define TEX_ORIENT_YZ	2
#define TEX_ORIENT_XZ	3

static Boolean tex_on;
static int tex_orient;

typedef struct {

	double i, j;	/* Position of `upper left' in meters. */
	double w, h;	/* Size `to lower right' in meters. */

} tex_coord_struct;
static tex_coord_struct tex_coord;



/*
 *	Return a statically allocated string indicating the
 *	operation fetched from the pointed to fp.  Return can be
 *	an empty string if there was no op string to be found or
 *	error.
 *
 *	fp is positioned at the end of the op string which the next
 *	fetch can be used to get its argument. If a new line character
 *	is detected at the end of string on file, then the fp will be
 *	repositioned at that new line character. The next reading of fp
 *	will read that new line character.
 */
char *SARObjFSeekGetOpString(FILE *fp)
{
	int c, i;
#define len	80
	static char rtn_str[len];


	*rtn_str = '\0';
	if(fp == NULL)
	    return(rtn_str);

	/* Seek past spaces. */
	FSeekPastSpaces(fp);

	for(i = 0; i < len; i++)
	{
	    c = fgetc(fp);
	    if((c == EOF) ||
               ISBLANK(c)
	    )
	    {
                rtn_str[i] = '\0';
                break;
	    }
            /* Escape sequence? */
            else if(c == '\\')
            {
                c = fgetc(fp);
                if(c == EOF)
		{
                    rtn_str[i] = '\0';
                    break;
   		}

                if(c != '\\')
                    c = fgetc(fp);

                if(c == EOF)
                {
                    rtn_str[i] = '\0';
                    break;
                }
            }
	    /* New line? */
	    else if(ISCR(c))
	    {
		/* New line right at the end of the op string, seek
		 * back one character.
		 */
		fseek(fp, -1, SEEK_CUR);

		rtn_str[i] = '\0';
		break;
	    }

	    rtn_str[i] = (char)c;
	}

#undef len
	return(rtn_str);
}



/*
 *	Handles a translate parameter with respect to the given inputs.
 *
 *	Returns non-zero on error.
 */
int SARObjLoadTranslate(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct *obj_ptr, sar_parm_translate_struct *p_translate
)
{
	sar_position_struct new_pos;


	if((obj_ptr == NULL) || (p_translate == NULL))
	    return(-1);

	memcpy(&new_pos, &p_translate->translate, sizeof(sar_position_struct));

	/* Check certain objects that need their position offsetted
	 * internally.
	 */
	if(obj_ptr->type == SAR_OBJ_TYPE_HUMAN)
	{
	    /* Human objects fall at a constant rate, so we need to move
	     * it up twice the fall rate cycle to ensure that it
	     * `stays up' any hollow contact surfaces.
	     */
	    new_pos.z -= (2.0 * SAR_DEF_HUMAN_FALL_RATE);
	}


	/* Update the ground elevation (for more information on usage,
	 * see SARSimApplyArtificialForce() on how it uses 
	 * SARSimFindGround().
	 */

        /* Check objects of types that need to know about ground
         * elevation.
         */
        if((obj_ptr->type == SAR_OBJ_TYPE_AIRCRAFT) ||
           (obj_ptr->type == SAR_OBJ_TYPE_HUMAN) ||
           (obj_ptr->type == SAR_OBJ_TYPE_FUELTANK)
        )
        {
	    double ground_elevation = 0.0;

	    ground_elevation += SARSimFindGround(
		scene,
		core_ptr->object, core_ptr->total_objects,
		&new_pos	/* Position of our object. */
	    );

	    obj_ptr->ground_elevation_msl = ground_elevation;
	}

	/* Realize the new position. */
	SARSimWarpObject(
	    scene, obj_ptr,
	    &new_pos, NULL
	);

	return(0);
}

/*
 *	Translates the object to a random position.
 *
 *	Returns non-zero on error.
 */
int SARObjLoadTranslateRandom(
        sar_core_struct *core_ptr, sar_scene_struct *scene,  
        sar_object_struct *obj_ptr,
        sar_parm_translate_random_struct *p_translate_random
)
{
	int rand_val = SARRandom();
	double r = p_translate_random->radius_bound;
	double z = p_translate_random->z_bound;
	double rand_theta, rand_coeff;
	sar_position_struct *pos;


	rand_coeff = (double)rand_val / (double)RAND_MAX;
	rand_theta = SFMSanitizeRadians(rand_coeff * (2.0 * PI));


        if((obj_ptr == NULL) || (p_translate_random == NULL))
            return(-1);

	pos = &obj_ptr->pos;

	pos->x += (sin(rand_theta) * rand_coeff * r);
	pos->y += (cos(rand_theta) * rand_coeff * r);
	pos->z += (rand_coeff * z);

	/* Realize position. */
	SARSimWarpObject(scene, obj_ptr, pos, NULL);

	return(0);
}

/*
 *	Loads the sound object specified by path and starts playing it.
 *
 *	If the given pointer snd_play_rtn is not NULL in its pointed to
 *	location then it will be stopped first.
 *
 *	If the given path is not absolute, the global or local data
 *	directories will be prefixed to it.
 *
 *	On success the pointed to location of snd_play_rtn will be set
 *	to the new sound object.
 *
 *	Returns 0 on success or -1 on failure.
 */
int SARObjLoadSound(
	snd_recorder_struct *recorder,
	snd_play_struct **snd_play_rtn, 
	const char *path, int options
)
{
#define len	(PATH_MAX + NAME_MAX)
	char tmp_path[len];


	if((snd_play_rtn == NULL) ||
           (path == NULL)
	)
	    return(-1);

	/* Unload sound if already loaded. */
	if((*snd_play_rtn) != NULL)
	{
	     SoundStopPlay(recorder, *snd_play_rtn);
	     (*snd_play_rtn) = NULL;
	}

        /* Is given path absolute? */
        if(ISPATHABSOLUTE(path))
	{
	    /* Yes, use it as the path to load the sound object. */
	    strncpy(tmp_path, path, len);
	}
	else
        {
	    /* No, we need to prefix data dir. */
            const char *cstrptr;
            struct stat stat_buf;


            /* Prefix local data dir path first . */
            cstrptr = (const char *)PrefixPaths(dname.local_data, path);
            if(cstrptr != NULL)
            {
/* This may cause problems since the sound object may not be local. */
                /* Is it in local dir? If not then assume global. */
                if(stat(cstrptr, &stat_buf))
                    cstrptr = (const char *)PrefixPaths(dname.global_data, path);
            }
            strncpy(
		tmp_path,
		(const char *)((cstrptr == NULL) ? path : cstrptr),
		len
	    );
        }
	tmp_path[len - 1] = '\0';

	/* Load new sound object for playing. */
	(*snd_play_rtn) = SoundStartPlay(
	    recorder,
	    tmp_path,	/* Full path to object. */
	    1.0, 1.0,	/* Volume, from 0.0 to 1.0. */
	    0,		/* Applied sample rate, can be 0. */
	    options	/* Options. */
	);

#undef len
	return(((*snd_play_rtn) == NULL) ? -1 : 0);
}


/*
 *	Loads a texture from the given arguments.
 *
 *	Returns non-zero on error.
 */
int SARObjLoadTexture(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_parm_texture_load_struct *p_texture_load
)
{
	v3d_texture_ref_struct *t;
#define len	1024
        char lname[len];
	char lpath[len];


	if((core_ptr == NULL) || (scene == NULL))
	    return(-1);

	(*lname) = '\0';
	(*lpath) = '\0';

	if(p_texture_load->name != NULL)
	    strncpy(lname, p_texture_load->name, len);
        if(p_texture_load->file != NULL)
            strncpy(lpath, p_texture_load->file, len);

	/* No path? */
	if((*lpath) == '\0')
	    return(-1);

	/* Check if texture referance name is already loaded. */
	if((*lname) == '\0')
	    t = NULL;
	else
            t = SARGetTextureRefByName(scene, lname);
	if(t != NULL)
	{
            /* A texture with the same referance name is already
	     * loaded on the scene structure, so do not overload it.
	     */
	    return(0);
	}


	/* If lpath is not an absolute path, then prefix game dir. */
	if(!ISPATHABSOLUTE(lpath))
	{
	    const char *cstrptr;
	    struct stat stat_buf;

	    /* Prefix local dir lpath. */
	    cstrptr = (const char *)PrefixPaths(dname.local_data, lpath);
	    if(cstrptr != NULL)
	    {
		/* Is it in local dir, if not assume global? */
	        if(stat(cstrptr, &stat_buf))
		    cstrptr = PrefixPaths(dname.global_data, lpath);
	    }
	    if(cstrptr != NULL)
		strncpy(lpath, cstrptr, len);
	    lpath[len - 1] = '\0';
	}


	/* Load new texture. */
	t = V3DTextureLoadFromFile2D(
	    lpath, lname, V3D_TEX_FORMAT_RGBA,
	    NULL, NULL
	);
	if(t != NULL)
	{
	    int n;

	    if(scene->total_texture_refs < 0)
		scene->total_texture_refs = 0;

	    /* Increase total textures. */
	    n = scene->total_texture_refs;
	    scene->total_texture_refs = n + 1;

	    /* Allocate more pointers. */
	    scene->texture_ref = (v3d_texture_ref_struct **)realloc(
		scene->texture_ref,
		scene->total_texture_refs * sizeof(v3d_texture_ref_struct *)
	    );
	    if(scene->texture_ref == NULL)
	    {
		V3DTextureDestroy(t);
		return(-1);
	    }
	    scene->texture_ref[n] = t;

	    /* Set texture priority. */
	    V3DTexturePriority(t, p_texture_load->priority);
	}
	else
	{
	    /* Could not load texture. */
	    return(-1);
	}

#undef len
	return(0);
}

/*
 *      Loads a helipad object from the given argument line string.
 *
 *      Returns the newly created human object number or -1 on error.
 */
int SARObjLoadHelipad(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_helipad_struct *p_new_helipad
)
{
        int obj_num = -1;
	sar_object_struct *obj_ptr;
	sar_object_helipad_struct *obj_helipad_ptr;
	const char *style, *ref_obj_name;
	const char *tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	int ref_obj_num = -1;
	sar_object_struct *ref_obj_ptr = NULL;


        if((core_ptr == NULL) || (scene == NULL) ||
           (p_new_helipad == NULL)
        )
            return(obj_num);

	/* Create a new helipad object. */
	obj_num = SARObjCreate(
	    scene, &core_ptr->object, &core_ptr->total_objects,
	    SAR_OBJ_TYPE_HELIPAD
	);
	obj_ptr = ((obj_num < 0) ?
	    NULL : core_ptr->object[obj_num]
	);

	/* Create failed? */
	if(obj_ptr == NULL)
	{
	    obj_num = -1;
	    return(obj_num);
	}

	/* Get pointer to helipad data substructure. */
	obj_helipad_ptr = (sar_object_helipad_struct *)obj_ptr->data;
	if(obj_helipad_ptr == NULL)
	    return(obj_num);

	/* Begin setting values of helipad object. */

	/* Set helipad flags. */
	obj_helipad_ptr->flags = p_new_helipad->flags;

	/* Set default range. */
	obj_ptr->range = SAR_HELIPAD_DEF_RANGE;

	/* Style name not available? */
	style = p_new_helipad->style;
	if(style == NULL)
	{
	    /* Default to ground based helipad that is paved. */
	    obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_PAVED;
	}
	else
	{
	    if(!strcasecmp(style, "ground_paved") ||
               !strcasecmp(style, "standard") ||
               !strcasecmp(style, "default")
	    )
	    {
		obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_PAVED;
		tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	    }
	    else if(!strcasecmp(style, "ground_bare"))
	    {
		obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_BARE;
		tex_name = SAR_STD_TEXNAME_HELIPAD_BARE;
	    }
            else if(!strcasecmp(style, "building"))
	    {
                obj_helipad_ptr->style = SAR_HELIPAD_STYLE_BUILDING;
		tex_name = SAR_STD_TEXNAME_HELIPAD_BUILDING;
	    }
            else if(!strcasecmp(style, "vehicle"))
	    {
                obj_helipad_ptr->style = SAR_HELIPAD_STYLE_VEHICLE;
		tex_name = SAR_STD_TEXNAME_HELIPAD_VEHICLE;
	    }
	    else
	    {
		fprintf(
		    stderr,
		    "Unsupported helipad style type `%s'.\n",
		    style
		);
		obj_helipad_ptr->style = SAR_HELIPAD_STYLE_GROUND_PAVED;
		tex_name = SAR_STD_TEXNAME_HELIPAD_PAVED;
	    }
	}


	/* Landable size of helipad. */
	obj_helipad_ptr->length = MAX(p_new_helipad->length, 1.0);
        obj_helipad_ptr->width = MAX(p_new_helipad->width, 1.0);
	obj_helipad_ptr->recession = MAX(p_new_helipad->recession, 0.0);

	/* Specify contact bounds of helipad to ensure a landable
	 * surface.
	 */
	SARObjAddContactBoundsRectangular(
	    obj_ptr, SAR_CRASH_FLAG_SUPPORT_SURFACE, 0,
	    -(obj_helipad_ptr->width / 2),
	    (obj_helipad_ptr->width / 2),
	    -(obj_helipad_ptr->length / 2),
	    (obj_helipad_ptr->length / 2),
	    0.0, 0.0
	);

	/* Label. */
	obj_helipad_ptr->label = StringCopyAlloc(p_new_helipad->label);

	/* Calculate light spacing. */
	obj_helipad_ptr->light_spacing = obj_helipad_ptr->length / 10;

	/* Get texture referance number for helipad from scene
	 *  structure.
	 */
	obj_helipad_ptr->tex_num = SARGetTextureRefNumberByName(
	    scene, tex_name
	);

	/* Match ref object? */
	ref_obj_name = (const char *)p_new_helipad->ref_obj_name;
	if((ref_obj_name == NULL) ? 0 : ((*ref_obj_name) != '\0'))
	{
	    ref_obj_ptr = SARObjMatchPointerByName(
		scene, core_ptr->object, core_ptr->total_objects,
		ref_obj_name, &ref_obj_num
	    );
	    if(ref_obj_ptr == NULL)
		fprintf(
		    stderr,
 "Could not match referance object `%s' for helipad object #%i.\n",
		    ref_obj_name, obj_num
		);
	}
	/* Set referance object. */
	obj_helipad_ptr->ref_object = ref_obj_num;
	/* Make sure ref object is not helipad object itself. */
	if(ref_obj_ptr == obj_ptr)
	{
	    ref_obj_num = -1;
	    ref_obj_ptr = NULL;

	    obj_helipad_ptr->flags &= ~SAR_HELIPAD_FLAG_REF_OBJECT;
	    obj_helipad_ptr->flags &= ~SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT;

	    obj_helipad_ptr->ref_object = -1;
	}

	/* Set referance offset relative to the referance object. */
	memcpy(
	    &obj_helipad_ptr->ref_offset,
	    &p_new_helipad->ref_offset,
	    sizeof(sar_position_struct)
	);

	/* Set referance direction relative to the referance direction. */
	memcpy(
	    &obj_helipad_ptr->ref_dir,
	    &p_new_helipad->ref_dir, 
	    sizeof(sar_direction_struct)
	);

	/* Realize new position of helipad if it is relative to
	 * a refearnce object.
	 */
	if(obj_helipad_ptr->flags & SAR_HELIPAD_FLAG_FOLLOW_REF_OBJECT)
	{
	    SARSimWarpObjectRelative(
		scene, obj_ptr,
		core_ptr->object, core_ptr->total_objects, ref_obj_num,
		&obj_helipad_ptr->ref_offset,
		&obj_helipad_ptr->ref_dir
	    );
	}

	return(obj_num);
}

/*
 *	Loads a human object from the given argument line string.
 *
 *	Returns the newly created human object number or -1 on error.
 */
int SARObjLoadHuman(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_human_struct *p_new_human
)
{
	int obj_num = -1;

        if((core_ptr == NULL) || (scene == NULL) || (p_new_human == NULL))
            return(obj_num);

	/* Create new human object. */
	obj_num = SARHumanCreate(
	    core_ptr->human_data,
	    scene, &core_ptr->object, &core_ptr->total_objects,
	    p_new_human->flags,
	    p_new_human->type_name
	);

	return(obj_num);
}


/*
 *      Loads a fire object from the given argument line string.
 *
 *      Returns the newly created fire object number or -1 on error.
 */
int SARObjLoadFire( 
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_fire_struct *p_new_fire 
)
{
        int obj_num = -1;
	sar_position_struct pos;


        if((core_ptr == NULL) || (scene == NULL) || (p_new_fire == NULL))
            return(obj_num);

	memset(&pos, 0x00, sizeof(sar_position_struct));

        /* Create new fire object. */
        obj_num = FireCreate(
            scene, &core_ptr->object, &core_ptr->total_objects,
	    &pos, p_new_fire->radius, p_new_fire->height,
	    -1,
	    SAR_STD_TEXNAME_FIRE
        );
	if(obj_num > -1)
	{
	    sar_object_struct *obj_ptr = core_ptr->object[obj_num];
	    sar_object_fire_struct *obj_fire_ptr;

	    if(obj_ptr != NULL)
	    {
		obj_fire_ptr = (sar_object_fire_struct *)obj_ptr->data;
		if(obj_fire_ptr != NULL)
		{
		    /* Make animation repeat forever. */
		    obj_fire_ptr->total_frame_repeats = -1;
		}

		obj_ptr->life_span = 0;

	    }
	}

        return(obj_num);
}

/*
 *      Loads a fire object from the given argument line string.
 *
 *      Returns the newly created fire object number or -1 on error.
 */
int SARObjLoadSmoke(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_parm_new_smoke_struct *p_new_smoke
)
{
        int obj_num = -1;
        sar_position_struct pos;
	const char *tex_name = NULL;


        if((core_ptr == NULL) || (scene == NULL) || (p_new_smoke == NULL))
            return(obj_num);

        memset(&pos, 0x00, sizeof(sar_position_struct));

	/* Get texture name from color code. */
	switch(p_new_smoke->color_code)
	{
	  case 0:
	    tex_name = SAR_STD_TEXNAME_SMOKE_LIGHT;
	    break;
	  case 1:
	    tex_name = SAR_STD_TEXNAME_SMOKE_MEDIUM;
            break;
          case 2:
            tex_name = SAR_STD_TEXNAME_SMOKE_DARK;
            break;
	}

        /* Create new smoke trail object. */
	obj_num = SmokeCreate(
	    scene, &core_ptr->object, &core_ptr->total_objects,
            &pos, &p_new_smoke->offset,
	    p_new_smoke->radius_start,
	    p_new_smoke->radius_max,
	    p_new_smoke->radius_rate,	/* This might be -1.0 for autocalc. */
	    p_new_smoke->hide_at_max,
	    p_new_smoke->total_units,
	    p_new_smoke->respawn_int,
	    tex_name,
	    -1,				/* No reference object. */
	    0				/* No limit on life span. */
	);

        return(obj_num);
}

/*
 *	Loads heightfield primitive p for the given object.
 */
int SARObjLoadHeightField(
        sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        void *p, const char *filename, int line_num,
	GLuint list
)
{
	double x_length, y_length, z_length;
	int num_grids_x, num_grids_y;
	double grid_space_x, grid_space_y;
	double *zpoints = NULL;

	int status;
	char *strptr;
        sar_object_ground_struct *obj_ground_ptr = NULL;
	mp_heightfield_load_struct *mp_heightfield_load =
	    (mp_heightfield_load_struct *)p;
	v3d_hf_options_struct hfopt;
	char path[PATH_MAX + NAME_MAX];
	const int len = PATH_MAX + NAME_MAX;


        if((core_ptr == NULL) ||
           (obj_ptr == NULL) ||
           (mp_heightfield_load == NULL)
        )
            return(-1);

	/* Get path from heightfield primitive structure. */
	strncpy(
	    path,
	    (mp_heightfield_load->path == NULL) ?
		"" : mp_heightfield_load->path,
	    len
	);
	path[len - 1] = '\0';
	if((*path) == '\0')
	    return(-3);


        /* If path is not an absolute path, then prefix game dir. */
        if(!ISPATHABSOLUTE(path))
        {
            struct stat stat_buf;

            /* Prefix local dir path. */
            strptr = PrefixPaths(dname.local_data, path);
            if(strptr != NULL)
            {
                /* Is it in local dir, if not assume global? */
                if(stat(strptr, &stat_buf))
                    strptr = PrefixPaths(dname.global_data, path);
            }
            if(strptr != NULL)
                strncpy(path, strptr, len);
            path[len - 1] = '\0';
        }

	/* Get pointer to ground object structure. */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_GROUND:
	    obj_ground_ptr = (sar_object_ground_struct *)obj_ptr->data;
	    break;
	}


	/* Begin load heightfield. */

	/* Get size of heightfield from heightfield load primitive. */
	x_length = mp_heightfield_load->x_length;
        y_length = mp_heightfield_load->y_length;
        z_length = mp_heightfield_load->z_length;
	if((x_length <= 0.0) ||
           (y_length <= 0.0) ||
           (z_length <= 0.0)
	)
	    return(-2);

	glPushMatrix();
	{
	    /* Translate. */
	    glTranslated(
		mp_heightfield_load->x,
		mp_heightfield_load->z,
		-mp_heightfield_load->y
	    );

	    /* No rotations. */

	    /* Set up heightfield options. */
	    hfopt.flags = (V3D_HF_OPT_FLAG_WINDING |
		V3D_HF_OPT_FLAG_SET_NORMAL | V3D_HF_OPT_FLAG_SET_TEXCOORD
	    );
	    hfopt.winding = V3D_HF_WIND_CCW;
	    hfopt.set_normal = V3D_HF_SET_NORMAL_STREATCHED;
	    hfopt.set_texcoord = V3D_HF_SET_TEXCOORD_ALWAYS;

	    /* Load heightfield. */
	    status = V3DHFLoadFromFile(
		path,
		x_length, y_length, z_length,	/* Scaling in meters. */
		&num_grids_x, &num_grids_y,	/* Number of grids. */
		&grid_space_x, &grid_space_y,	/* Grid spacing in meters. */
		&zpoints,			/* Heightfield points as u_int8_t. */
		(void *)list,			/* Our GL list in context. */
		&hfopt
	    );
	    if(status)
	    {
		/* Load error. */
		if(zpoints != NULL)
		{
		    free(zpoints);
		    zpoints = NULL;
		}
		return(-1);
	    }
	}
	glPopMatrix();

	/* Set newly loaded values to ground object structure if possible. */
	if(obj_ground_ptr != NULL)
	{
	    /* Set new heightfield data to ground object substructure. */
	    obj_ground_ptr->x_trans = mp_heightfield_load->x;
	    obj_ground_ptr->y_trans = mp_heightfield_load->y;
	    obj_ground_ptr->z_trans = mp_heightfield_load->z;

	    obj_ground_ptr->x_len = x_length;
	    obj_ground_ptr->y_len = y_length;

	    obj_ground_ptr->grid_points_x = num_grids_x;
            obj_ground_ptr->grid_points_y = num_grids_y;
            obj_ground_ptr->grid_points_total = num_grids_x * num_grids_y;

	    obj_ground_ptr->grid_x_spacing = grid_space_x;
            obj_ground_ptr->grid_y_spacing = grid_space_y;
            obj_ground_ptr->grid_z_spacing = z_length;

	    /* Replace old heightfield z point data on ground object
	     * structure with the newly allocated one.
	     */
	    free(obj_ground_ptr->z_point_value);
	    obj_ground_ptr->z_point_value = zpoints;
	    zpoints = NULL;	/* Reset zpoints to mark it as transfered. */
	}

	/* Free allocated heightfield z point data if it was not 
	 * transfered to the ground object structure.
	 */
	if(zpoints != NULL)
	{
	    free(zpoints);
	    zpoints = NULL;
	}

	return(0);
}

/*
 *	Parses and handles the given V3D primitive p with respect to
 *	the given inputs.
 */
static int SARObjLoadVisualPrimitive(
	sar_core_struct *core_ptr,
        int obj_num, sar_object_struct *obj_ptr,
        void *p, const char *filename, int line_num
)
{
	mp_vertex_struct *ns = NULL, **nd = NULL;
	mp_vertex_struct *vs = NULL, **vd = NULL;
	mp_vertex_struct *tcs = NULL, **tcd = NULL;
	int i, ptype, v_total = 0;
	Boolean need_end_primitive = False;

	mp_point_struct *mp_point;
	mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_struct *mp_triangle;
	mp_triangle_strip_struct *mp_triangle_strip;
	mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_struct *mp_quad;
	mp_quad_strip_struct *mp_quad_strip;
	mp_polygon_struct *mp_polygon;
	mp_vertex_struct *first_normal_ptr = NULL, *n_ptr;

	double x = 0.0, y = 0.0, z = 0.0;
	double tx = 0.0, ty = 0.0;


	if((core_ptr == NULL) || (obj_ptr == NULL) || (p == NULL))
	    return(-1);

	/* Get primitive type. */
	ptype = (*(int *)p);

	/* Handle by primitive type. */
	switch(ptype)
	{
	  case V3DMP_TYPE_POINT:
	    mp_point = (mp_point_struct *)p;
	    if(last_begin_primitive_type != ptype)
	    {
		glBegin(GL_POINTS);
		last_begin_primitive_type = ptype;
	    }
	    ns = &mp_point->n[0];
	    vs = &mp_point->v[0];
	    tcs = &mp_point->tc[0];
	    v_total = V3DMP_POINT_NVERTEX;
	    break;

          case V3DMP_TYPE_LINE:
            mp_line = (mp_line_struct *)p;
            if(last_begin_primitive_type != ptype)
            {
                glBegin(GL_LINES);
                last_begin_primitive_type = ptype;
            }
	    ns = &mp_line->n[0];
            vs = &mp_line->v[0];
	    tcs = &mp_line->tc[0];
            v_total = V3DMP_LINE_NVERTEX;
            break;

          case V3DMP_TYPE_LINE_STRIP:
            mp_line_strip = (mp_line_strip_struct *)p;
            glBegin(GL_LINE_STRIP); need_end_primitive = True;
	    nd = mp_line_strip->n;
            vd = mp_line_strip->v;
	    tcd = mp_line_strip->tc;
            v_total = mp_line_strip->total;
            break;

          case V3DMP_TYPE_LINE_LOOP:
            mp_line_loop = (mp_line_loop_struct *)p;
            glBegin(GL_LINE_LOOP); need_end_primitive = True;
	    nd = mp_line_loop->n;
            vd = mp_line_loop->v;
            tcd = mp_line_loop->tc;
            v_total = mp_line_loop->total;
            break;

          case V3DMP_TYPE_TRIANGLE:
            mp_triangle = (mp_triangle_struct *)p;
            if(last_begin_primitive_type != ptype)
            {
                glBegin(GL_TRIANGLES);
                last_begin_primitive_type = ptype;
            }
            ns = &mp_triangle->n[0];
            vs = &mp_triangle->v[0];
            tcs = &mp_triangle->tc[0];
            v_total = V3DMP_TRIANGLE_NVERTEX;
            break;

          case V3DMP_TYPE_TRIANGLE_STRIP:
            mp_triangle_strip = (mp_triangle_strip_struct *)p;
            glBegin(GL_TRIANGLE_STRIP); need_end_primitive = True;
            nd = mp_triangle_strip->n;
            vd = mp_triangle_strip->v;
            tcd = mp_triangle_strip->tc;
            v_total = mp_triangle_strip->total;
            break;

          case V3DMP_TYPE_TRIANGLE_FAN:
            mp_triangle_fan = (mp_triangle_fan_struct *)p;
            glBegin(GL_TRIANGLE_FAN); need_end_primitive = True;
            nd = mp_triangle_fan->n;
            vd = mp_triangle_fan->v;
            tcd = mp_triangle_fan->tc;
            v_total = mp_triangle_fan->total;
            break;

          case V3DMP_TYPE_QUAD:
            mp_quad = (mp_quad_struct *)p;
            if(last_begin_primitive_type != ptype)
            {
                glBegin(GL_QUADS);
                last_begin_primitive_type = ptype;
            }
            ns = &mp_quad->n[0];
            vs = &mp_quad->v[0];
            tcs = &mp_quad->tc[0];
            v_total = V3DMP_QUAD_NVERTEX;
            break;

          case V3DMP_TYPE_QUAD_STRIP:
            mp_quad_strip = (mp_quad_strip_struct *)p;
            glBegin(GL_QUAD_STRIP); need_end_primitive = True;
            nd = mp_quad_strip->n;
            vd = mp_quad_strip->v;   
            tcd = mp_quad_strip->tc;
            v_total = mp_quad_strip->total;
            break;

          case V3DMP_TYPE_POLYGON:
            mp_polygon = (mp_polygon_struct *)p;
            glBegin(GL_POLYGON); need_end_primitive = True;
            nd = mp_polygon->n;
            vd = mp_polygon->v;
            tcd = mp_polygon->tc;
            v_total = mp_polygon->total;
            break;
	}


	/* Get pointer to first normal. */
	i = 0;
	if(nd != NULL)
	    first_normal_ptr = nd[i];
	else if(ns != NULL)
	    first_normal_ptr = &ns[i];
	else
	    first_normal_ptr = NULL;

	/* Go through each vertex last to first, since winding stored
	 * on V3D model file is clockwise and we need to handle it counter
	 * clockwise.
	 */
	for(i = (v_total - 1); i >= 0; i--)
	{
	    /* Get vertex values but do not set just yet, we need
	     * just the coordinates for now to set the texcoord first
	     * (incase texture is being plane oriented).
	     */
	    if(vd != NULL)
	    {
		if(vd[i] != NULL)
		{
		    x = vd[i]->x;
		    y = vd[i]->y;
		    z = vd[i]->z;
		}
	    }
	    else if(vs != NULL)
	    {
                x = vs[i].x;
                y = vs[i].y;
                z = vs[i].z;
            }

	    /* Get normal. */
	    n_ptr = NULL;
	    if(nd != NULL)
		n_ptr = nd[i];
	    else if(ns != NULL)
		n_ptr = &ns[i];

	    /* Resort to first normal if n_ptr is still NULL. */
	    if(n_ptr == NULL)
	    {
		n_ptr = first_normal_ptr;
	    }
	    else
	    {
		/* If n_ptr is a zero vector, then resort to using the
		 * first normal (which may still be zero but atleast 
		 * its the best we can do).
		 */
		if((n_ptr->x == 0.0) && (n_ptr->y == 0.0) && (n_ptr->z == 0.0))
		    n_ptr = first_normal_ptr;
	    }
	    /* Got valid normal? */
	    if(n_ptr != NULL)
	    {
		glNormal3d(n_ptr->x, n_ptr->z, -n_ptr->y);
	    }

	    /* Check if texture is on and set texture coordinates. */
	    if(tex_on)
	    {
                switch(tex_orient)
                {
                  case TEX_ORIENT_XY:
                    if(tex_coord.w > 0)
                        tx = (x - tex_coord.i) / tex_coord.w;
                    else
                        tx = 0;
                    if(tex_coord.h > 0)
                        ty = (y - tex_coord.j) / tex_coord.h;
                    else
                        ty = 0;
                    glTexCoord2d(tx, 1.0 - ty);
                    break;

                  case TEX_ORIENT_YZ:
                    if(tex_coord.w > 0)
                        tx = -(y - tex_coord.i) / tex_coord.w;
                    else
                        tx = 0;
                    if(tex_coord.h > 0)
                        ty = (z - tex_coord.j) / tex_coord.h;
                    else
                        ty = 0;
                    glTexCoord2d(tx, 1.0 - ty);
                    break;

                  case TEX_ORIENT_XZ:
                    if(tex_coord.w > 0)
                        tx = (x - tex_coord.i) / tex_coord.w;
                    else
                        tx = 0;
                    if(tex_coord.h > 0)
                        ty = (z - tex_coord.j) / tex_coord.h;
                    else
                        ty = 0;
                    glTexCoord2d(tx, 1.0 - ty);
                    break;

		  default:
		    if(tcd != NULL)
		    {
			if(tcd[i] != NULL)
			{
			    tx = tcd[i]->x;
                            ty = tcd[i]->y;
			}
		    }
		    else if(tcs != NULL)
		    {
			tx = tcs[i].x;
			ty = tcs[i].y;
		    }

                    glTexCoord2d(tx, 1.0 - ty);
		    break;
		}
	    }

	    /* Set vertex last. */
            glVertex3d(x, z, -y);
	}

	/* End primitive. */
	if(need_end_primitive)
	{
	    glEnd();
	    need_end_primitive = False;
	}

	return(0);
} 



/*
 *	Loads the line which should have come from an object file.
 *	This function is designed to be called by SARObjLoadFromFile()
 *	and inputs are assumed valid.
 *
 *	Returns 0 if the line was parsed and handled, -2 if the line
 *	is not supported or a comment, and -1 on general error.
 */
int SARObjLoadLine(
	sar_core_struct *core_ptr,
	int obj_num, sar_object_struct *obj_ptr,
	const char *line, const char *filename, int line_num
)
{
	char *strptr;
	int i, sc, sr, st;
	sar_scene_struct *scene;  
	sar_object_struct ***ptr;
	int *total;

        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
        sar_object_ground_struct *obj_ground_ptr = NULL;
	sar_obj_rotor_struct *rotor_ptr = NULL;
        sar_obj_landing_gear_struct *lgear_ptr = NULL;
	sar_external_fueltank_struct *eft_ptr = NULL;
	sar_obj_door_struct *door_ptr = NULL;
	sar_direction_struct *dir;
	sar_position_struct *pos;

        double value[16];

	char parm[256];
	const char *arg;


	if((core_ptr == NULL) ||
           (obj_ptr == NULL) ||
           (line == NULL)
	)
	    return(-1);

	scene = core_ptr->scene;
	ptr = &core_ptr->object;
	total = &core_ptr->total_objects;

	if(scene == NULL)
	    return(-1);

	dir = &obj_ptr->dir;
	pos = &obj_ptr->pos;

	/* Get pointer to substructure if any. */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		i = obj_aircraft_ptr->total_rotors;
		if(i > 0)
		    rotor_ptr = obj_aircraft_ptr->rotor[i - 1];

                i = obj_aircraft_ptr->total_landing_gears;
                if(i > 0)
                    lgear_ptr = obj_aircraft_ptr->landing_gear[i - 1];

		door_ptr = &obj_aircraft_ptr->rescue_door;
	    }
	    break;

	  case SAR_OBJ_TYPE_GROUND:
            obj_ground_ptr = obj_ptr->data;
	    break;
	}


	/* Begin handling line, seek past spaces. */
	while(ISBLANK(*line))
	    line++;

	/* Is it a comment? */
	if(ISCOMMENT(*line))
	    return(-2);

	/* Empty line? */
	if((*line) == '\0')
	    return(-2);


	/* Get parameter. */
	strncpy(parm, line, 256);
	parm[256 - 1] = '\0';
	strptr = (char *)strseekblank((char *)parm);
	if(strptr != NULL)
	    (*strptr) = '\0';


	/* Seek arg to start of arguments in line pointer. */
	arg = line;
	while(!ISBLANK(*arg) && ((*arg) != '\0'))
	    arg++;
	while(ISBLANK(*arg))
	    arg++;


	/* Begin handling parm. */
	if(1)
	{
	    /* Texture base directory. */
            if(!strcasecmp(parm, "texture_base_directory") ||
	       !strcasecmp(parm, "texture_base_dir")
            )
            {
                /* Ignore this, it is now loaded from the visual model
                 * header items.
                 */
	    }
            /* Texture load. */
            else if(!strcasecmp(parm, "texture_load"))
            {
		/* Ignore this, it is now loaded from the visual model
		 * header items.
		 */
            }

            /* Name. */
            else if(!strcasecmp(parm, "name"))
            {
		free(obj_ptr->name);
		obj_ptr->name = strdup(arg);
	    }
            /* Type. */
            else if(!strcasecmp(parm, "type"))
            {
/* Ignore type set within the object file, the object type
   should already be set prior to calling this function.
 */
	    }
            /* Visable range (how far can visual model be seen). */
            else if(!strcasecmp(parm, "range"))
            {
		st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
		for(sc = sr; sc < st; sc++)
		    value[sc] = 0.0;

		obj_ptr->range = value[0];
		if(value[0] < 0)
		    fprintf(stderr,
 "%s: Line %i: Warning: range is negative.\n",
                        filename, line_num
                    );
	    }
            /* Visable far range, for displaying of the far visual model
	     * when the distance is greater than this far range.
	     */
            else if(!strcasecmp(parm, "range_far"))
            {
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                obj_ptr->range_far = value[0];
                if(value[0] < 0)
                    fprintf(stderr,
 "%s: Line %i: Warning: range_far is negative.\n",
                        filename, line_num
                    );
            }
            /* No depth test performed while drawing this
	     * object's model(s).
	     */
            else if(!strcasecmp(parm, "no_depth_test"))
	    {
		obj_ptr->flags |= SAR_OBJ_FLAG_NO_DEPTH_TEST;
	    }
	    /* Smooth light model shading. */
	    else if(!strcasecmp(parm, "shade_model_smooth"))
	    {
		obj_ptr->flags |= SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH;
	    }
            /* Flat light model shading. */
            else if(!strcasecmp(parm, "shade_model_flat"))
            {
		obj_ptr->flags &= ~SAR_OBJ_FLAG_SHADE_MODEL_SMOOTH;
            }
            /* Offset polygons. */
            else if(!strcasecmp(parm, "offset_polygons"))
            {
                obj_ptr->flags |= SAR_OBJ_FLAG_POLYGON_OFFSET;
            }
	    /* Show night model at dawn. */
	    else if(!strcasecmp(parm, "show_night_model_at_dawn"))
            {
		obj_ptr->flags |= SAR_OBJ_FLAG_NIGHT_MODEL_AT_DAWN;
	    }
            /* Show night model at dusk. */
            else if(!strcasecmp(parm, "show_night_model_at_dusk"))
            {
                obj_ptr->flags |= SAR_OBJ_FLAG_NIGHT_MODEL_AT_DUSK;
            }
	    /* Show far model only during day. */
	    else if(!strcasecmp(parm, "show_far_day_only"))
            {
                obj_ptr->flags |= SAR_OBJ_FLAG_FAR_MODEL_DAY_ONLY;
            }
            /* Crash flags. */
            else if(!strcasecmp(parm, "crash_flags"))
            {
                /* Arguments are: <crash_into_other> <causes_crash>
                 * <support_surface> <crash_type>
                 */
		sar_contact_bounds_struct *cb;
                st = 4;
                sr = sscanf(arg, "%lf %lf %lf %lf",
		    &value[0], &value[1], &value[2], &value[3]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		/* Allocate contact bounds structure on object as needed. */
		if(obj_ptr->contact_bounds == NULL)
		    obj_ptr->contact_bounds = (sar_contact_bounds_struct *)
			calloc(1, sizeof(sar_contact_bounds_struct));
		cb = obj_ptr->contact_bounds;
		if(cb != NULL)
		{
		    cb->crash_flags = 0;	/* Reset crash flags. */

		    /* Crash into other objects? */
		    if((int)value[0])
			cb->crash_flags |= SAR_CRASH_FLAG_CRASH_OTHER;

		    /* Other objects can crash into this object? */
		    if((int)value[1])
			cb->crash_flags |= SAR_CRASH_FLAG_CRASH_CAUSE;

                    /* Other objects can land or walk on top of object? */
                    if((int)value[2])
                        cb->crash_flags |= SAR_CRASH_FLAG_SUPPORT_SURFACE;

		    /* Crash type code. */
		    cb->crash_type = (int)value[3];
		}
            }
            /* Contact bounds, spherical shape. */
            else if(!strcasecmp(parm, "contact_spherical"))
            {
                /* Arguments are: <radius>
                 */
                sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                SARObjAddContactBoundsSpherical(
                    obj_ptr,
                    (cb == NULL) ? 0 : cb->crash_flags,
                    (cb == NULL) ? 0 : cb->crash_type, 
                    value[0]			/* Radius. */
                );
	    }
            /* Contact bounds, cylendrical shape. */
            else if(!strcasecmp(parm, "contact_cylendrical"))
            {
                /* Arguments are: <radius> <height_min> <height_max>
                 */
                sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                st = 3; 
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0; 

		SARObjAddContactBoundsCylendrical(
		    obj_ptr,
		    (cb == NULL) ? 0 : cb->crash_flags,
		    (cb == NULL) ? 0 : cb->crash_type,
		    value[0],			/* Radius. */
		    value[1], value[2]		/* Height min/max. */
		);
            }
            /* Contact bounds, rectangular shape. */
            else if(!strcasecmp(parm, "contact_rectangular"))
            {
                /* Arguments are: <x_min> <x_max> <y_min> <y_max>
		 * <z_min> <z_max>
                 */
                sar_contact_bounds_struct *cb = obj_ptr->contact_bounds;
                st = 6;
                sr = sscanf(arg, "%lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
		    &value[3], &value[4], &value[5]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                SARObjAddContactBoundsRectangular(
                    obj_ptr,
                    (cb == NULL) ? 0 : cb->crash_flags,
                    (cb == NULL) ? 0 : cb->crash_type,
		    value[0], value[1],		/* X min/max. */
		    value[2], value[3],		/* Y min/max. */
		    value[4], value[5]		/* Z min/max. */
                );
            }
            /* Speed bounds. */
            else if(!strcasecmp(parm, "speed"))
            {
                /* Arguments are: <speed_stall> <speed_max>
                 * <speed_max_expected> <min_drag>
		 * All units from miles per hour.
                 */
                st = 4;
                sr = sscanf(arg, "%lf %lf %lf %lf",
                    &value[0], &value[1], &value[2], &value[3]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {   
                    obj_aircraft_ptr->speed_stall =
                        SFMMPHToMPC(value[0]);
                    obj_aircraft_ptr->speed_max =
                        SFMMPHToMPC(value[1]);
                    obj_aircraft_ptr->speed_max_expected =
                        SFMMPHToMPC(value[2]);
                    obj_aircraft_ptr->min_drag =
                        SFMMPHToMPC(value[3]);
                }
            }
            /* Helicopter acceleration responsiveness. */
            else if(!strcasecmp(parm, "helicopter_accelresp") ||
                    !strcasecmp(parm, "helicopter_acceleration_responsiveness")
            )
            {
                /* Arguments are: <i> <j> <k>
                 */
                sar_position_struct *ar = NULL;

                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(obj_aircraft_ptr != NULL)
		    ar = &obj_aircraft_ptr->accel_responsiveness;

		if(ar != NULL)
                {
                    ar->x = value[0];
                    ar->y = value[1];
                    ar->z = value[2];
                }
            }
            /* Airplane acceleration responsiveness. */
            else if(!strcasecmp(parm, "airplane_accelresp") ||
                    !strcasecmp(parm, "airplane_acceleration_responsiveness")
            )
            {
                /* Arguments are: <i> <j> <k>
                 */
                sar_position_struct *ar = NULL;

                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(obj_aircraft_ptr != NULL)
                    ar = &obj_aircraft_ptr->airplane_accel_responsiveness;

		if(ar != NULL)
                {
                    ar->x = value[0];
                    ar->y = value[1];
                    ar->z = value[2];
                }
            }
            /* Cockpit offset. */
            else if(!strcasecmp(parm, "cockpit_offset"))
            {
                /* Arguments are: <x> <y> <z>
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    obj_aircraft_ptr->cockpit_offset_pos.x = value[0];
                    obj_aircraft_ptr->cockpit_offset_pos.y = value[1];
                    obj_aircraft_ptr->cockpit_offset_pos.z = value[2];
                }
            }
            /* Belly height (belly to object's xy plane). */
            else if(!strcasecmp(parm, "belly_height"))
            {
                /* Arguments are: <height> */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(value[0] < 0)
                    fprintf(stderr,
 "%s: Line %i: Warning: belly_height value is negative.\n",
                        filename, line_num
                    );

                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->belly_height = value[0];
            }
            /* Landing gear height (height of landing gear). */
            else if(!strcasecmp(parm, "gear_height"))
            {
                /* Arguments are: <height> */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(value[0] < 0)
                    fprintf(stderr,
 "%s: Line %i: Warning: gear_height value is negative.\n",
                        filename, line_num
                    );

                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->gear_height = value[0];
            }
            /* Ground turning. */
            else if(!strcasecmp(parm, "ground_turning"))
            {
                /* Arguments are: <turn_rad> <turn_vel_opt>
		 * <turn_vel_max>
		 * Units from miles per hour.
		 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Turn radius in meters, distance from farthest
                     * non-turnable wheel to turnable wheel.
                     * Can be negative or 0.0 for no turning.
                     */
                    obj_aircraft_ptr->gturn_radius = value[0];

                    /* Optimul ground turning velocity along
                     * aircraft's y axis in meters per cycle.
                     */
                    obj_aircraft_ptr->gturn_vel_opt =
                        SFMMPHToMPC(value[1]);

                    /* Maximum ground turning velocity along
                     * aircraft's y axis in meters per cycle.
                     */
                    obj_aircraft_ptr->gturn_vel_max =
                        SFMMPHToMPC(value[2]);
                }
            }
            /* Dry mass. */
            else if(!strcasecmp(parm, "dry_mass"))
            {
                /* Arguments are: <mass>
		 * Units from kg.
		 */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(value[0] < 0)
                    fprintf(stderr,
 "%s: Line %i: Warning: dry_mass value is negative.\n",
                        filename, line_num
                    );
                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->dry_mass = value[0];
            }
            /* Fuel. */
            else if(!strcasecmp(parm, "fuel"))
            {
                /* Arguments are: <consumption_rate> <fuel_init> 
                 * <fuel_max>
                 * Units from kg per second, kg, and kg.
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Convert fuel rate from (kg / sec) to (kg / cycle). */
                    obj_aircraft_ptr->fuel_rate = MAX(
			value[0] * SAR_SEC_TO_CYCLE_COEFF, 0.0
		    );

                    /* Fuel in kg. */
                    obj_aircraft_ptr->fuel = MAX(value[1], 0.0);
                    obj_aircraft_ptr->fuel_max = MAX(value[2], 0.0);
                }
            }
            /* Crew. */
            else if(!strcasecmp(parm, "crew"))
            {
                /* Arguments are: <crew> <passengers>
                 * <passengers_max>
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    obj_aircraft_ptr->crew = (int)value[0];
                    obj_aircraft_ptr->passengers = (int)value[1];
                    obj_aircraft_ptr->passengers_max = (int)value[2];
                }
            }
            /* Engine. */
            else if(!strcasecmp(parm, "engine"))
            {
                /* Arguments are: <can_pitch> <init_pitch>
                 * <power> <collective_range>
		 * Power from units of kg * m / cycle^2.
                 */
                st = 4;
                sr = sscanf(arg, "%lf %lf %lf %lf",
                    &value[0], &value[1], &value[2], &value[3]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Can pitch? */
                    obj_aircraft_ptr->engine_can_pitch = (((int)value[0]) ?
			1 : 0
		    );

                    /* Initial rotor pitch state, this determines the
		     * the value for member flight_model_type.
		     */
		    switch((int)value[1])
		    {
		      case 1:
			obj_aircraft_ptr->flight_model_type =
			    SAR_FLIGHT_MODEL_AIRPLANE;
			break;

		      /* All else assume helicopter. */
		      default:
			obj_aircraft_ptr->flight_model_type =
                            SAR_FLIGHT_MODEL_HELICOPTER;
                        break;
		    }
		    /* Explicitly set previous flight model type to the
		     * same type as the current one since this is the
		     * first time this is being set for this object.
		     */
		    obj_aircraft_ptr->last_flight_model_type =
			obj_aircraft_ptr->flight_model_type;

                    /* Engine power (in kg * m / cycle^2). */
                    obj_aircraft_ptr->engine_power = value[2];

		    /* Collective range coefficient (from 0.0 to 1.0). */
                    obj_aircraft_ptr->collective_range = value[3];
                }
            }
            /* Engine inside sound. */
            else if(!strcasecmp(parm, "engine_inside_sound"))
            {
		char **snd_path = NULL;

                if(obj_aircraft_ptr != NULL)
		{
		    snd_path = (char **)
			&obj_aircraft_ptr->engine_inside_sndpath;
		}

		if(snd_path != NULL)
		{
		    free(*snd_path);
		    (*snd_path) = strdup(arg);
		}
            }
            /* Engine outside sound. */
            else if(!strcasecmp(parm, "engine_outside_sound"))
            {
		char **snd_path = NULL;

                if(obj_aircraft_ptr != NULL)
		{
                    snd_path = (char **)
                        &obj_aircraft_ptr->engine_outside_sndpath;
		}

                if(snd_path != NULL)
                {
                    free(*snd_path);
                    (*snd_path) = strdup(arg);
                }
            }
            /* Service ceiling. */  
            else if(!strcasecmp(parm, "service_ceiling"))
            {
                /* Arguments are: <altitude>
                 * Units from feet.
                 */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    obj_aircraft_ptr->service_ceiling =
                        SFMFeetToMeters(value[0]);
            }
            /* Attitude change rates. */
            else if(!strcasecmp(parm, "attitude_change_rate"))
            {
                /* Arguments are: <heading> <pitch> <bank>
                 * Units from degrees per second.
		 */
		sar_direction_struct *acr = NULL;
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                    acr = &obj_aircraft_ptr->attitude_change_rate;

                if(acr != NULL)
                {
                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing, values are from degrees per
                     * second.
                     */
                    acr->heading = DEGTORAD(value[0]) *
                        SAR_SEC_TO_CYCLE_COEFF;  
                    acr->pitch = DEGTORAD(value[1]) *   
                        SAR_SEC_TO_CYCLE_COEFF;
                    acr->bank = DEGTORAD(value[2]) *   
                        SAR_SEC_TO_CYCLE_COEFF;
                }
            }
            /* Attitude leveling. */
            else if(!strcasecmp(parm, "attitude_leveling"))
            {
                /* Arguments are: <heading> <pitch> <bank>
                 * Units from degrees per second.
                 */
                st = 3;
                sr = sscanf(arg, "%lf %lf %lf",
                    &value[0], &value[1], &value[2]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_aircraft_ptr != NULL)
                {
                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing, values are from degrees per
                     * second.
                     */
                    /* No heading leveling. */
                    obj_aircraft_ptr->pitch_leveling =
                        DEGTORAD(value[1]) * SAR_SEC_TO_CYCLE_COEFF;
                    obj_aircraft_ptr->bank_leveling =
                        DEGTORAD(value[2]) * SAR_SEC_TO_CYCLE_COEFF;
                }
            }
            /* Create new light. */
            else if(!strcasecmp(parm, "light_new"))
            {
                /* Arguments are: <x> <y> <z> <r> <g> <b> <a>
		 * <radius> <init_on?> <type> <on_int> <off_int>
                 * Units from radius are in pixels and intervals are
		 * in milliseconds.
                 */
                int n;
                sar_light_struct *light_ptr;
                st = 12;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
                    &value[9], &value[10], &value[11]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                while(1)
                {
                    /* Allocate a new light on the object. */
                    if(obj_ptr->total_lights < 0)
                        obj_ptr->total_lights = 0;

                    n = obj_ptr->total_lights;
                    obj_ptr->total_lights++;

                    obj_ptr->light = (sar_light_struct **)realloc(
                        obj_ptr->light,
                        obj_ptr->total_lights * sizeof(sar_light_struct *)
                    );
                    if(obj_ptr->light == NULL)
                    {
                        obj_ptr->total_lights = 0;
			break;
                    }

                    obj_ptr->light[n] = (sar_light_struct *)calloc( 
                        1,
                        sizeof(sar_light_struct)
                    );
                    if(obj_ptr->light[n] == NULL)
                        break;

                    light_ptr = obj_ptr->light[n];

                    /* Position (values 0 to 2). */
                    light_ptr->pos.x = value[0];
                    light_ptr->pos.y = value[1];
                    light_ptr->pos.z = value[2];

                    /* Color (values 3 to 6). */
                    light_ptr->color.r = value[3];
                    light_ptr->color.g = value[4];
                    light_ptr->color.b = value[5];
                    light_ptr->color.a = value[6];

                    /* Radius (in pixels, value 7). */
                    light_ptr->radius = (int)MAX(value[7], 0.0);

                    /* Initially on (value 8)? */
                    if((int)value[8])  
                        light_ptr->flags |= SAR_LIGHT_FLAG_ON;

                    /* Type (value 9). */
                    switch((int)value[9])
                    {
                      case 2:
                        light_ptr->flags |= SAR_LIGHT_FLAG_ATTENUATE;
                        break;

                      case 1:
                        light_ptr->flags |= SAR_LIGHT_FLAG_STROBE;
                        break;
                    }

                    /* Strobe on and off intervals (values 10 to 11). */
                    light_ptr->int_on = (time_t)value[10];
                    light_ptr->int_off = (time_t)value[11];

		    break;
                }
	    }
            /* Create new rotor/propellar. */
            else if(!strcasecmp(parm, "rotor_new") ||
                    !strcasecmp(parm, "propellar_new")
            )
            {
                /* Arguments are: <x> <y> <z>
		 * <heading> <pitch> <bank> <radius> <has_prop_wash>
		 * <follow_control_pitch_bank> <blurs_when_fast>
                 */
                st = 10;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
		    &value[9]
		);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		rotor_ptr = NULL;

                if(obj_aircraft_ptr != NULL)
                {
                    int n = SARObjCreateRotor(
                        scene,
                        &obj_aircraft_ptr->rotor,
                        &obj_aircraft_ptr->total_rotors
                    );
                    rotor_ptr = ((n < 0) ? NULL :
                        obj_aircraft_ptr->rotor[n]
                    );
                }
                if(rotor_ptr == NULL)
                {
                    fprintf(stderr,
 "%s: Line %i: Error: Cannot create rotor/propellar (check object type).\n",
                        filename, line_num
                    );
                }
                else
		{
		    pos = &rotor_ptr->pos;
                    dir = &rotor_ptr->dir;      

                    pos->x = value[0];
                    pos->y = value[1];
                    pos->z = value[2];

                    dir->heading = SFMDegreesToRadians(value[3]);
                    dir->pitch = SFMDegreesToRadians(value[4]);
                    dir->bank = SFMDegreesToRadians(value[5]);

                    rotor_ptr->radius = MAX(value[6], 0);

                    /* Has rotor wash? */
                    if((int)value[7])
                    {
                        rotor_ptr->rotor_wash_tex_num =
                            SARGetTextureRefNumberByName(
                                scene,
                                SAR_STD_TEXNAME_ROTOR_WASH
                            );
                    } 
                    else
                    {
                        rotor_ptr->rotor_wash_tex_num = -1;
                    }

		    /* Follow controls for pitch and bank? */
		    if((int)value[8])
		    {
			rotor_ptr->flags |= SAR_ROTOR_FLAG_FOLLOW_PB;
		    }

		    /* Blurs when fast? */
                    if((int)value[9] == 1)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_BLADES_BLUR_FAST;
                    }
		    /* Blurs always? */
                    else if((int)value[9] == 2)
                    {
                        rotor_ptr->flags |= SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS;
                    }

                }
	    }
            /* Rotor blured color. */
            else if(!strcasecmp(parm, "rotor_blur_color") ||
                    !strcasecmp(parm, "propellar_blur_color")
            )
            {
                /* Arguments are: <r> <b> <g> <a> */
                st = 4;
                sr = sscanf(arg,
		    "%lf %lf %lf %lf",
		    &value[0], &value[1], &value[2], &value[3]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		if(rotor_ptr != NULL)
		{
		    if(!(rotor_ptr->flags & SAR_ROTOR_FLAG_BLADES_BLUR_FAST) &&
		       !(rotor_ptr->flags & SAR_ROTOR_FLAG_BLADES_BLUR_ALWAYS)
		    )
		    {
			fprintf(
			    stderr,
 "%s: Line %i: Warning: Rotor is not set to blur at all, blur color not set.\n",
			    filename, line_num
			);
		    }
		    else
                    {
			sar_color_struct *c = &rotor_ptr->blades_blur_color;

			c->r = CLIP(value[0], 0.0, 1.0);
                        c->g = CLIP(value[1], 0.0, 1.0); 
                        c->b = CLIP(value[2], 0.0, 1.0); 
                        c->a = CLIP(value[3], 0.0, 1.0); 
                    }
		}
	    }
            /* Create new landing gear. */
            else if(!strcasecmp(parm, "landing_gear_new"))
            {
                /* Arguments are: <x> <y> <z>
                 * <heading> <pitch> <bank> <up?> <anim_rate>
		 * <fixed?> <ski?>
                 */
                st = 10;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
		    &value[9]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                lgear_ptr = NULL;

                if(obj_aircraft_ptr != NULL)
                {
                    int n = SARObjCreateLandingGear(
                        scene,
                        &obj_aircraft_ptr->landing_gear,
                        &obj_aircraft_ptr->total_landing_gears
                    );
                    lgear_ptr = ((n < 0) ? NULL :
                        obj_aircraft_ptr->landing_gear[n]
                    );
                }
                /* Was a landing gear created and/or found? */
                if(lgear_ptr == NULL)
                {
                    fprintf(stderr,
 "%s: Line %i: Error: Cannot create landing gear (check object type).\n",
                        filename, line_num
                    );
                }
                else
                {
                    pos = &lgear_ptr->pos;
                    dir = &lgear_ptr->extended_dir;

                    pos->x = value[0];
                    pos->y = value[1];
                    pos->z = value[2];

                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing.
                     */
                    dir->heading = DEGTORAD(value[3]);
                    dir->pitch = DEGTORAD(value[4]);
                    dir->bank = DEGTORAD(value[5]);

                    lgear_ptr->anim_rate = (sar_grad_anim_t)value[6];
                    lgear_ptr->flags = 0;

                    /* Landing gear up or down? */
                    if((int)value[7])
                    {
                        /* Landing gear initially up. */
                        lgear_ptr->anim_pos = 0;
                        lgear_ptr->flags |= SAR_LGEAR_FLAG_STATE;
                    }
                    else
                    {
                        /* Landing gear initially down. */
                        lgear_ptr->anim_pos = (sar_grad_anim_t)-1;   
                        lgear_ptr->flags &= ~SAR_LGEAR_FLAG_STATE;
                    } 
                    /* Landing gear fixed? */
                    if((int)value[8])
                    {
                        /* Landing gear is fixed. */
                        lgear_ptr->anim_pos = (sar_grad_anim_t)-1;
                        lgear_ptr->flags &= ~SAR_LGEAR_FLAG_STATE;
                        lgear_ptr->flags |= SAR_LGEAR_FLAG_FIXED;
                    }
                    else
                    {
                        /* Landing gear is retractable. */
                        lgear_ptr->flags &= ~SAR_LGEAR_FLAG_FIXED;
                    }  
                    /* Landing gear is a ski? */
                    if((int)value[9])
                    {
                        /* Landing gear is fixed. */
                        lgear_ptr->flags |= SAR_LGEAR_FLAG_SKI;
                    }
                    else
                    {
                        /* Landing gear is retractable. */
                        lgear_ptr->flags &= ~SAR_LGEAR_FLAG_SKI;
                    }
                }
	    }
            /* External reserved fuel tank on aircrafts. */
            else if(!strcasecmp(parm, "fueltank_new") ||
                    !strcasecmp(parm, "fuel_tank_new")
	    )
            {
                /* Arguments are: <x> <y> <z> <radius>
                 * <dry_mass> <fuel> <fuel_max> <droppable>
                 */
                st = 8;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                eft_ptr = NULL;

                if(obj_aircraft_ptr != NULL)
                {
                    int n = SARObjCreateExternalFuelTanks(
                        scene,
                        &obj_aircraft_ptr->external_fueltank,
                        &obj_aircraft_ptr->total_external_fueltanks
                    );
                    eft_ptr = ((n < 0) ? NULL :
                        obj_aircraft_ptr->external_fueltank[n]
                    );
		}
                /* Was a fuel tank created and/or found? */
                if(eft_ptr == NULL)
                {
                    fprintf(stderr,
 "%s: Line %i: Error: Cannot create fuel tank (check object type).\n",
                        filename, line_num
                    );
                }
                else
                {
                    pos = &eft_ptr->offset_pos;

                    pos->x = value[0];
                    pos->y = value[1];
                    pos->z = value[2];

		    eft_ptr->radius = MAX(value[3], 0.0);

		    eft_ptr->dry_mass = MAX(value[4], 0.0);
                    eft_ptr->fuel_max = MAX(value[6], 0.0);
                    eft_ptr->fuel = CLIP(value[5], 0.0, eft_ptr->fuel_max);

		    if((int)value[7])
			eft_ptr->flags |= SAR_EXTERNAL_FUELTANK_FLAG_FIXED;
		    else
                        eft_ptr->flags &= ~SAR_EXTERNAL_FUELTANK_FLAG_FIXED;

		    /* Mark as initially onboard (not jettesoned). */
		    eft_ptr->flags |= SAR_EXTERNAL_FUELTANK_FLAG_ONBOARD;
		}
	    }
            /* Rescue door (on aircrafts and other objects that can
	     * have one).
	     */
            else if(!strcasecmp(parm, "rescue_door_new"))
            {
                /* Arguments are:
		 * <x_closed> <y_closed> <z_closed>
                 * <x_opened> <y_opened> <z_opened>
		 * <h_opened> <p_opened> <b_opened>
		 * <x_thres> <y_thres> <z_thres>
		 * <anim_rate> <opened?>
                 */
                st = 14;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8],
                    &value[9], &value[10], &value[11],
		    &value[12], &value[13]
                );
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		door_ptr = SARObjGetDoorPtr(obj_ptr, 0, NULL);
                if(door_ptr == NULL)  
                {
                    fprintf(stderr,
 "%s: Line %i: Error: Cannot create door (check object type).\n",
                        filename, line_num
                    ); 
                }
                else
                {
                    /* Closed position. */
                    pos = &door_ptr->pos_closed;
                    pos->x = value[0];
                    pos->y = value[1];  
                    pos->z = value[2];

                    /* Opened position. */
                    pos = &door_ptr->pos_opened;
                    pos->x = value[3];
                    pos->y = value[4];
                    pos->z = value[5];

                    /* Do not call SFMDegreesToRadians() since we do not
                     * want sanitizing.
                     */
                    dir = &door_ptr->dir_opened;
                    dir->heading = DEGTORAD(value[6]);
                    dir->pitch = DEGTORAD(value[7]);
                    dir->bank = DEGTORAD(value[8]);

		    /* Threshold position. */
                    pos = &door_ptr->pos_threshold;
                    pos->x = value[9];
                    pos->y = value[10];
                    pos->z = value[11];

		    /* Get animation rate. */
                    door_ptr->anim_rate = (sar_grad_anim_t)value[12];

		    /* Set initial flags. */
                    door_ptr->flags = 0;
                    /* Door opened or closed? */
                    if((int)value[13])
                    {
                        /* Door initially open. */
                        door_ptr->anim_pos = (sar_grad_anim_t)-1;
                        door_ptr->flags |= SAR_DOOR_FLAG_STATE;
			door_ptr->flags |= SAR_DOOR_FLAG_STAY_OPEN;
                    }
                    else
                    {
                        /* Door initially down. */
                        door_ptr->anim_pos = (sar_grad_anim_t)0;
                        door_ptr->flags &= ~SAR_DOOR_FLAG_STATE;
                    }
                }
            }
            /* Hoist. */
            else if(!strcasecmp(parm, "hoist"))
            {
                /* Arguments are: <x> <y> <z>
                 * <rope_max> <rope_rate> <capacity>
		 * <radius> <z_min> <z_max>
                 */
                sar_obj_hoist_struct *hoist_ptr;
                st = 9;
                sr = sscanf(arg,
 "%lf %lf %lf %lf %lf %lf %lf %lf %lf",
                    &value[0], &value[1], &value[2],
                    &value[3], &value[4], &value[5],
                    &value[6], &value[7], &value[8]
                );  
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

		hoist_ptr = SARObjGetHoistPtr(obj_ptr, 0, NULL);
		if(hoist_ptr == NULL)
		{

		}
		else
		{
		    sar_human_data_entry_struct *hde_ptr;

                    hoist_ptr->offset.x = value[0];
                    hoist_ptr->offset.y = value[1];
                    hoist_ptr->offset.z = value[2];

                    hoist_ptr->rope_max = value[3];
                    hoist_ptr->rope_rate = value[4];
                    hoist_ptr->capacity = value[5];

                    hoist_ptr->contact_radius = value[6];
                    hoist_ptr->contact_z_min = value[7];
                    hoist_ptr->contact_z_max = value[8];

		    hoist_ptr->deployment = SAR_HOIST_DEPLOYMENT_BASKET;

                    /* Set hoist texture referance numbers to
                     * match textures on scene structure.
                     */ 
                    hoist_ptr->side_tex_num =
                        SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_BASKET_SIDE
			);
                    hoist_ptr->end_tex_num =
                        SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_BASKET_END
			);
                    hoist_ptr->bottom_tex_num =
                        SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_BASKET_BOTTOM
			);
		    hoist_ptr->water_ripple_tex_num =
			SARGetTextureRefNumberByName(
			    scene, SAR_STD_TEXNAME_WATER_RIPPLE
			);

		    /* Set diver color. */
		    hde_ptr = SARHumanMatchEntryByName(
			core_ptr->human_data, SAR_HUMAN_PRESET_NAME_DIVER
		    );
		    if(hde_ptr == NULL)
			hde_ptr = SARHumanMatchEntryByName(
			    core_ptr->human_data, SAR_HUMAN_PRESET_NAME_STANDARD
			);
		    if(hde_ptr != NULL)
		    {
			memcpy(
			    hoist_ptr->diver_color,
			    hde_ptr->color,
			    SAR_HUMAN_COLORS_MAX * sizeof(sar_color_struct)
			);
		    }

		    /* Animation position and rate. */
		    hoist_ptr->anim_pos = 0;
		    /* Use human animation rate for now. */
		    hoist_ptr->anim_rate = SAR_HUMAN_ANIM_RATE;
		}
	    }
            /* Ground elevation. */
            else if(!strcasecmp(parm, "ground_elevation"))
            {
                /* Arguments are: <altitude>
		 * Units from feet.
                 */
                st = 1;
                sr = sscanf(arg, "%lf", &value[0]);
                for(sc = sr; sc < st; sc++)
                    value[sc] = 0.0;

                if(obj_ground_ptr == NULL)
                {
                    fprintf(stderr, 
 "%s: Line %i: Warning: Setting ground_elevation for object not type `%i'.\n",
                        filename, line_num, SAR_OBJ_TYPE_GROUND
                    );
                }
                else
                {
                    if(value[0] < 0)
                        fprintf(stderr,
 "%s: Line %i: Warning: ground_elevation value is negative.\n",
                            filename, line_num
                        );

                    obj_ground_ptr->elevation =
                        SFMFeetToMeters(value[0]);
                }
            }   




	    /* Unsupported parameter. */
	    else
	    {
		/* Ignore unsupported parameter. */
		fprintf(stderr,
 "%s: Line %i: Warning: Unsupported parameter `%s'.\n",
		    filename, line_num, parm
		);
	    }
	}


	return(0);
}



/*
 *      Loads object model data from file. Object can be already allocated  
 *      if n is not -1. Otherwise a new object will be appended to the given
 *      objects pointer array.
 */
int SARObjLoadFromFile(
        sar_core_struct *core_ptr,
        int n,                          /* Object number. */
        const char *filename
)
{
        FILE *fp;
        char *strptr, *line_ptr;
        int i, status, line_num, ptype, *total;
	void *p;

	void **vmh_item = NULL;
	int vmh_item_num, total_vmh_items = 0;

	v3d_model_struct *vmodel_ptr, **vmodel = NULL;
	int vmodel_num, total_vmodels = 0;

	mh_comment_struct *mh_comment;
	mh_texture_base_directory_struct *mh_tbd;
	mh_texture_load_struct *mh_texture_load;
	mh_color_specification_struct *mh_color_spec;
	sar_parm_texture_load_struct *p_texture_load;

	mp_comment_struct *mp_comment;
	Boolean blend_state = False;
	mp_color_struct *mp_color;
	mp_texture_select_struct *mp_texture_select;
	mp_texture_orient_xy_struct *mp_texture_xy;
	mp_texture_orient_yz_struct *mp_texture_yz;
	mp_texture_orient_xz_struct *mp_texture_xz;
	mp_heightfield_load_struct *mp_heightfield_load;

        sar_scene_struct *scene;
        sar_object_struct ***ptr, *obj_ptr;
        sar_object_aircraft_struct *obj_aircraft_ptr = NULL;
        sar_object_ground_struct *obj_ground_ptr = NULL;

	int total_rotors = 0;
        sar_obj_rotor_struct *rotor_ptr = NULL;
	int total_lgears = 0;
        sar_obj_landing_gear_struct *lgear_ptr = NULL;
	int total_doors = 0;
        sar_obj_door_struct *door_ptr = NULL;
	int total_external_fueltanks = 0;
	sar_external_fueltank_struct *eft_ptr = NULL;

        sar_direction_struct *dir;
        sar_position_struct *pos;
	char *sar_vmodel_name;
	sar_visual_model_struct **sar_vmodel;	/* SAR Visual Model. */

        GLuint list;
        struct stat stat_buf;
        char tmp_path[PATH_MAX + NAME_MAX];


        if((core_ptr == NULL) ||
           (filename == NULL)
        )
            return(-1);

        scene = core_ptr->scene;
        ptr = &core_ptr->object;
        total = &core_ptr->total_objects;

        if(scene == NULL)
            return(-1);

        /* If path is not an absolute path, then prefix game dir. */
        if(ISPATHABSOLUTE(filename))
        {
            strncpy(tmp_path, filename, PATH_MAX + NAME_MAX);
        }
        else
        {
            /* Prefix local dir path. */
            strptr = PrefixPaths(dname.local_data, filename);
            if(strptr != NULL)
            {
                /* Is it in local dir, if not assume global? */
                if(stat(strptr, &stat_buf))
                    strptr = PrefixPaths(dname.global_data, filename);
            }
            if(strptr != NULL)
                strncpy(tmp_path, strptr, PATH_MAX + NAME_MAX);
        }
        tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

	/* File exists? */
        if(stat(tmp_path, &stat_buf))
        {
            fprintf(stderr,
 "%s: No such object model file (checked locally and globally).\n",
                filename
            );
            return(-1);
        }
#ifdef S_ISDIR
	if(S_ISDIR(stat_buf.st_mode))
	{
            fprintf(stderr,
 "%s: Cannot open model file, it is a directory.\n",
                filename
            );
            return(-1);
        }
#endif	/* S_ISDIR */

	/* Open file. */
        fp = FOpen(tmp_path, "rb");
        if(fp == NULL)
            return(-1);

        /* Append new object? */
        if(n < 0)
        {
            /* Create new object, type will be set later. */
            n = SARObjCreate(scene, ptr, total, SAR_OBJ_TYPE_STATIC);
        }
        if(SARObjIsAllocated(*ptr, *total, n))
        {
            obj_ptr = (*ptr)[n];

	    /* Set default values on object. */

	    /* Set default object flags. */
	    obj_ptr->flags |= (	SAR_OBJ_FLAG_HIDE_DAWN_MODEL |
				SAR_OBJ_FLAG_HIDE_DUSK_MODEL |
				SAR_OBJ_FLAG_HIDE_NIGHT_MODEL
	    );


            /* Get pointer to substructure. */
            switch(obj_ptr->type)
            {
              case SAR_OBJ_TYPE_AIRCRAFT:
                obj_aircraft_ptr = obj_ptr->data;
                if(obj_aircraft_ptr == NULL)
                    break;

                /* Reset values. */
                obj_aircraft_ptr->air_worthy_state = SAR_AIR_WORTHY_FLYABLE;
                obj_aircraft_ptr->engine_state = SAR_ENGINE_STATE_ON;
                obj_aircraft_ptr->next_engine_on = 0;

                obj_aircraft_ptr->rotor_wash_tex_num =
		    SARGetTextureRefNumberByName(
			scene, SAR_STD_TEXNAME_ROTOR_WASH
		    );

                break;  
                
              case SAR_OBJ_TYPE_GROUND:
                obj_ground_ptr = obj_ptr->data;
                break;
            }

	    pos = &obj_ptr->pos;
	    dir = &obj_ptr->dir;
        }
        else    
        {
            FClose(fp);
            return(-1);
        }

        /* Begin reading visual model file. */
	status = V3DLoadModel(
	    NULL, fp,
	    &vmh_item, &total_vmh_items,
	    &vmodel, &total_vmodels,
	    NULL, NULL
        );

        /* Close the file. */
        FClose(fp);

	if(status)
	{
	    /* Load error. */
	    V3DMHListDeleteAll(&vmh_item, &total_vmh_items);
	    V3DModelListDeleteAll(&vmodel, &total_vmodels);

	    return(-1);
	}


	/* Begin parsing loaded visual model header. */
	for(vmh_item_num = 0; vmh_item_num < total_vmh_items; vmh_item_num++)
	{
	    p = vmh_item[vmh_item_num];
	    if(p == NULL)
		continue;

	    switch(*(int *)p)
	    {
	      case V3DMH_TYPE_COMMENT:
		mh_comment = p;

		break;

	      case V3DMH_TYPE_TEXTURE_BASE_DIRECTORY:
		mh_tbd = p;

		break;

	      case V3DMH_TYPE_TEXTURE_LOAD:
		mh_texture_load = p;
		p_texture_load = (sar_parm_texture_load_struct *)SARParmNew(
		    SAR_PARM_TEXTURE_LOAD
		);
		if(p_texture_load != NULL)
		{
		    /* Copy data from model header item to sar parm. */
		    free(p_texture_load->name);
		    p_texture_load->name = StringCopyAlloc(mh_texture_load->name);
		    free(p_texture_load->file);
		    p_texture_load->file = StringCopyAlloc(mh_texture_load->path);
		    p_texture_load->priority = mh_texture_load->priority;
		    /* Load texture using the sar parm. */
		    SARObjLoadTexture(
			core_ptr, scene,
			p_texture_load
		    );
		    /* Deallocate sar parm for loading texture, not needed
		     * anymore.
		     */
		    SARParmDelete(p_texture_load);
		}
		break;

	      case V3DMH_TYPE_COLOR_SPECIFICATION:
		mh_color_spec = p;

		break;
	    }
	}


	/* Begin parsing loaded visual model data and creating glLists. */
	for(vmodel_num = 0; vmodel_num < total_vmodels; vmodel_num++)
	{
	    vmodel_ptr = vmodel[vmodel_num];
	    if(vmodel_ptr == NULL)
		continue;

	    /* Get new pointer values from new vmodel. */
	    if(obj_aircraft_ptr != NULL)
	    {
		total_rotors = obj_aircraft_ptr->total_rotors;
		rotor_ptr = ((total_rotors > 0) ?
		    obj_aircraft_ptr->rotor[total_rotors - 1] : NULL
		);

		total_lgears = obj_aircraft_ptr->total_landing_gears;
                lgear_ptr = ((total_lgears > 0) ?
		    obj_aircraft_ptr->landing_gear[total_lgears - 1] : NULL
		);

		total_external_fueltanks = obj_aircraft_ptr->total_external_fueltanks;
                eft_ptr = ((total_external_fueltanks > 0) ?
		    obj_aircraft_ptr->external_fueltank[total_external_fueltanks - 1]
		    : NULL
                );

		total_doors = 1;
		door_ptr = &obj_aircraft_ptr->rescue_door;
	    }
/* Add fetching of substructure pointers support for other object
 * types here.
 */
	    else
	    {
		total_rotors = 0;
		rotor_ptr = NULL;

		total_lgears = 0;
		lgear_ptr = NULL;

		total_external_fueltanks = 0;
		eft_ptr = NULL;

		total_doors = 0;
		door_ptr = NULL;
	    }

	    /* Go through each `other data' line on the vmodel. */
	    for(line_num = 0;
                line_num < vmodel_ptr->total_other_data_lines;
                line_num++
	    )
	    {
		line_ptr = vmodel_ptr->other_data_line[line_num];
		if(line_ptr == NULL)
		    continue;

		SARObjLoadLine(
		    core_ptr, n, obj_ptr,
		    line_ptr, filename, line_num
		);
	    }

	    /* Skip if this model does not contain primitives. */
	    if(vmodel_ptr->type != V3D_MODEL_TYPE_STANDARD)
		continue;

	    /* Create model by vmodel name. */
	    strptr = vmodel_ptr->name;
	    if(strptr == NULL)
		continue;

	    /* Reset SAR Visual Model and name pointer to NULL. */
	    sar_vmodel = NULL;
	    sar_vmodel_name = NULL;

	    /* Handle by visual model type. */
	    /* Standard model. */
	    if(!strcasecmp(strptr, "standard"))
	    {
		sar_vmodel = &obj_ptr->visual_model;

		free(sar_vmodel_name);
		sar_vmodel_name = strdup("standard");
	    }
            /* Standard far model. */
            else if(!strcasecmp(strptr, "standard_far"))
            {
                sar_vmodel = &obj_ptr->visual_model_far;

                free(sar_vmodel_name);
                sar_vmodel_name = strdup("standard_far");
            }
            /* Standard dusk model. */ 
            else if(!strcasecmp(strptr, "standard_dusk"))
            {
                sar_vmodel = &obj_ptr->visual_model_dusk;

                free(sar_vmodel_name);
                sar_vmodel_name = strdup("standard_dusk");
            }
            /* Standard night model. */
            else if(!strcasecmp(strptr, "standard_night"))
            {
                sar_vmodel = &obj_ptr->visual_model_night;

                free(sar_vmodel_name);
                sar_vmodel_name = strdup("standard_night");
            }
            /* Standard dawn model. */
            else if(!strcasecmp(strptr, "standard_dawn"))
            {
                sar_vmodel = &obj_ptr->visual_model_dawn;

                free(sar_vmodel_name);
                sar_vmodel_name = strdup("standard_dawn");
            }
            /* Rotor/propellar */
            else if(!strcasecmp(strptr, "rotor") ||
                    !strcasecmp(strptr, "propellar")
            )
            {
                if(rotor_ptr != NULL)
		{
                    sar_vmodel = &rotor_ptr->visual_model;

		    free(sar_vmodel_name);
		    sar_vmodel_name = (char *)malloc(256 * sizeof(char *));
		    if(sar_vmodel_name != NULL)
			sprintf(sar_vmodel_name, "rotor%i",
			    total_rotors - 1
			);
		}
            }
            /* Landing gear. */
            else if(!strcasecmp(strptr, "landing_gear"))
            {
                if(lgear_ptr != NULL)
		{
                    sar_vmodel = &lgear_ptr->visual_model;

		    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(256 * sizeof(char *));
                    if(sar_vmodel_name != NULL)
                        sprintf(sar_vmodel_name, "landing_gear%i",
                            total_lgears - 1
                        );
		}
            }
            /* Door. */
            else if(!strcasecmp(strptr, "door"))
            {
                if(door_ptr != NULL)
		{
                    sar_vmodel = &door_ptr->visual_model;

		    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(256 * sizeof(char *));
                    if(sar_vmodel_name != NULL)
                        sprintf(sar_vmodel_name, "door%i",
                            total_doors - 1
                        );
		}
            }
            /* External fuel tank */
            else if(!strcasecmp(strptr, "fueltank") ||
                    !strcasecmp(strptr, "fuel_tank")
            )
            {
                if(eft_ptr != NULL)
                {
                    sar_vmodel = &eft_ptr->visual_model;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(256 * sizeof(char *));
                    if(sar_vmodel_name != NULL)
                        sprintf(sar_vmodel_name, "fueltank%i",
                            total_external_fueltanks - 1
                        );
                }
            }
            /* Cockpit */
            else if(!strcasecmp(strptr, "cockpit"))
            {
                if(obj_aircraft_ptr != NULL)
		{
                    sar_vmodel = &obj_aircraft_ptr->visual_model_cockpit;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(256 * sizeof(char *));
                    if(sar_vmodel_name != NULL)
                        sprintf(sar_vmodel_name, "cockpit%i",
                            0
                        );
		}
            }
            /* Shadow */   
            else if(!strcasecmp(strptr, "shadow"))
            {
                if(obj_aircraft_ptr != NULL)
		{
                    sar_vmodel = &obj_aircraft_ptr->visual_model_shadow;

                    free(sar_vmodel_name);
                    sar_vmodel_name = (char *)malloc(256 * sizeof(char *));
                    if(sar_vmodel_name != NULL)
                        sprintf(sar_vmodel_name, "shadow%i",
                            0
                        );
		}
            }

	    /* Is SAR Visual Model pointer NULL? */
	    if(sar_vmodel == NULL)
	    {
		/* This implies we got a visual model type that is not
		 * supported.
		 */
		fprintf(
		    stderr,
"%s: Warning: Model type `%s' not supported.\n",
		    filename, strptr
		);

		/* Deallocate SAR Visual Model name. */
		free(sar_vmodel_name);
		sar_vmodel_name = NULL;

		/* Do not go further. */
		continue;
	    }

            /* Is SAR Visual Model pointer's pointed to location not
	     * NULL?
	     */
            if((*sar_vmodel) != NULL)
            {
		/* Implies it's already pointing to a created visual model
		 * so print warning before we create a new one.
		 */
                fprintf(
		    stderr,
 "%s: Warning: Visual Model type `%s' redefined.\n",
                    filename, strptr
                );

		/* Unref the SAR Visual Model. */
		SARVisualModelUnref(scene, *sar_vmodel);
		(*sar_vmodel) = NULL;
            }

            /* Create new visual model. */
            (*sar_vmodel) = SARVisualModelNew(
		scene, filename, sar_vmodel_name
	    );

	    /* Deallocate SAR Visual Model name (don't need it anymore). */
	    free(sar_vmodel_name);
	    sar_vmodel_name = NULL;

	    /* If there is only one referance count then that implies we
	     * need to generate the SAR Visual Model's GL list.
	     */
	    if(SARVisualModelGetRefCount(*sar_vmodel) == 1)
	    {
		/* (Re)generate GL list on visual model structure. */
		list = (GLuint)SARVisualModelNewList(*sar_vmodel);
		if(list != 0)
		{
                    /* Mark SAR Visual Model as done loading. */
                    (*sar_vmodel)->load_state = SAR_VISUAL_MODEL_LOADING;

		    /* Begin recording GL list. */
		    glNewList(list, GL_COMPILE);

		    /* Reset texture and color variables and GL states. */
		    V3DTextureSelect(NULL);
		    tex_on = False;
		    tex_orient = TEX_ORIENT_NONE;
		    glColor4d(1.0, 1.0, 1.0, 1.0);

		    /* Reset global last begin primitive type. */
		    last_begin_primitive_type = -1;

		    /* Itterate through each V3D model primitive. */
		    for(i = 0; i < vmodel_ptr->total_primitives; i++)
		    {
			p = vmodel_ptr->primitive[i];
			if(p == NULL)
			    continue;

			/* Get primitive type. */
                        ptype = (*(int *)p);

			/* Check if last begin primitive differs from this
			 * one. If it does then glEnd() needs to be called
			 * to end a prior glBegin() that was not ended.
			 */
			if((last_begin_primitive_type != ptype) &&
			   (last_begin_primitive_type > -1)
			)
			{
			    glEnd();
			    last_begin_primitive_type = -1;
			}

			switch(ptype)
			{
			  case V3DMP_TYPE_COMMENT:
			    mp_comment = (mp_comment_struct *)p;
			    if(mp_comment->total_lines > 0)
			    {
				int cline_num;
				char *cline_ptr;

				for(cline_num = 0;
				    cline_num < mp_comment->total_lines;
				    cline_num++
				)
				{
				    cline_ptr = mp_comment->line[cline_num];
				    if(cline_ptr == NULL)
					continue;

				    SARObjLoadLine(
					core_ptr, n, obj_ptr,
					cline_ptr, filename, 0
				    );
				}
			    }
			    break;

			  case V3DMP_TYPE_POINT:
	                  case V3DMP_TYPE_LINE:
	                  case V3DMP_TYPE_LINE_STRIP:
	                  case V3DMP_TYPE_LINE_LOOP:
	                  case V3DMP_TYPE_TRIANGLE:
	                  case V3DMP_TYPE_TRIANGLE_STRIP:
        	          case V3DMP_TYPE_TRIANGLE_FAN:
	                  case V3DMP_TYPE_QUAD:
	                  case V3DMP_TYPE_QUAD_STRIP:
	                  case V3DMP_TYPE_POLYGON:
			    SARObjLoadVisualPrimitive(
	                        core_ptr, n, obj_ptr,
	                        p, filename, 0
	                    );
			    break;

			  case V3DMP_TYPE_COLOR:
			    mp_color = (mp_color_struct *)p;
			    glColor4d(
				mp_color->r,
				mp_color->g,
				mp_color->b,
				mp_color->a
			    );
/* Enable and mark blend_state if alpha is below 1.0? */
			    break;

	                  case V3DMP_TYPE_TEXTURE_SELECT:
	                    mp_texture_select = (mp_texture_select_struct *)p;
			    strptr = mp_texture_select->name;

			    if(strptr == NULL)
			    {
	                        /* NULL name pointer implies unselect texture. */
	                        V3DTextureSelect(NULL);
				tex_on = False;
	                        tex_orient = TEX_ORIENT_NONE;
			    }
	                    else if((*strptr) == '\0')
	                    {
	                        /* Empty string implies unselect texture. */
	                        V3DTextureSelect(NULL);
				tex_on = False;
	                        tex_orient = TEX_ORIENT_NONE;
	                    }
	                    else
	                    {
	                        /* Select texture. */
	                        v3d_texture_ref_struct *t =
				    SARGetTextureRefByName(scene, strptr);
	                        if(t == NULL)
	                        {
	                            fprintf(stderr,
 "%s: Error: Texture `%s' not defined.\n",
	                                filename, strptr
	                            );
				    tex_on = False;
	                            tex_orient = TEX_ORIENT_NONE;
	                        }
	                        else
				{
				    tex_on = True;
	                            V3DTextureSelect(t);
				}
			    }
	                    break;

			  case V3DMP_TYPE_TEXTURE_ORIENT_XY:
			    mp_texture_xy = p;
	                    tex_orient = TEX_ORIENT_XY;
	                    tex_coord.i = mp_texture_xy->x;  
	                    tex_coord.j = mp_texture_xy->y;
	                    tex_coord.w = mp_texture_xy->dx;
	                    tex_coord.h = mp_texture_xy->dy;
			    break;

			  case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
			    mp_texture_yz = p;
	                    tex_orient = TEX_ORIENT_YZ;
	                    tex_coord.i = mp_texture_yz->y;
	                    tex_coord.j = mp_texture_yz->z;
	                    tex_coord.w = mp_texture_yz->dy;
	                    tex_coord.h = mp_texture_yz->dz;
			    break;

			  case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
			    mp_texture_xz = p;
	                    tex_orient = TEX_ORIENT_XZ;
	                    tex_coord.i = mp_texture_xz->x;
	                    tex_coord.j = mp_texture_xz->z;
	                    tex_coord.w = mp_texture_xz->dx;
	                    tex_coord.h = mp_texture_xz->dz;
			    break;

			  case V3DMP_TYPE_TEXTURE_OFF:
	                    V3DTextureSelect(NULL);
			    tex_on = False;
	                    tex_orient = TEX_ORIENT_NONE;
			    break;

			  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
			    mp_heightfield_load = p;
			    SARObjLoadHeightField(
	                        core_ptr, n, obj_ptr,
	                        p, filename, 0,
				list
			    );
			    break;
			}
                    }	/* Itterate through each V3D model primitive. */

		    /* Check if last_begin_primitive_type is valid,
		     * if it is then that means we need to call glEnd()
		     * to end a prior glBegin().
		     */
		    if(last_begin_primitive_type > -1)
		    {
			glEnd();
			last_begin_primitive_type = -1;
		    }

		    /* Check if blending was still enabled while 
		     * generating the list.
		     */
		    if(blend_state)
		    {

		    }


		    /* End gl list. */
	            glEndList();

		    /* Mark SAR Visual Model as done loading. */
		    (*sar_vmodel)->load_state = SAR_VISUAL_MODEL_LOADED;

		}	/* (Re)generate GL list on visual model structure. */
	    }
	}


	/* Unload all data from V3D model and header item structures. */
	V3DMHListDeleteAll(&vmh_item, &total_vmh_items);
	V3DModelListDeleteAll(&vmodel, &total_vmodels);


        /* Create flight dynamics model if this is an aircraft. */
        if(obj_aircraft_ptr != NULL)
        {
	    if(scene->realm == NULL)
	    {
		fprintf(
		    stderr,
 "%s: Warning: Cannot create SFM for NULL realm structure on scene.\n",
		    filename
		);
	    }
	    else
	    {
		obj_aircraft_ptr->fdm = SFMModelAllocate();
		SFMModelAdd(scene->realm, obj_aircraft_ptr->fdm);
	    }
        }


	/* Need to realize translation and rotation values on object. */
	SARSimWarpObject(
	    scene, obj_ptr,
	    &obj_ptr->pos, &obj_ptr->dir
	);

	return(0);
}
