#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "v3dtex.h"

#include "obj.h"
#include "explosion.h"
#include "sar.h"
#include "config.h"


int ExplosionCreate(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
        sar_position_struct *pos,
        double radius,
        int ref_object,
        const char *tex_name
);

int SplashCreate(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        sar_position_struct *pos, double radius,
        int ref_object,
        const char *tex_name
);

/*
 *	Creates an explision at the specified position referenced by
 *	ref_object.
 *
 *	The explosion will follow the ref_object or remain stationary
 *	if ref_object is -1.
 */
int ExplosionCreate(
	sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        sar_position_struct *pos,
        double radius,
	int ref_object,
	const char *tex_name
)
{
	int obj_num;
	sar_object_struct *obj_ptr;
	sar_object_explosion_struct *obj_explosion_ptr;


	if((scene == NULL) || (pos == NULL))
	    return(-1);

	/* Create explosion object. */
	obj_num = SARObjCreate(
	    scene, ptr, total,
	    SAR_OBJ_TYPE_EXPLOSION
	);
	if(obj_num < 0)
	    return(-1);

	obj_ptr = (*ptr)[obj_num];
	if(obj_ptr == NULL)
	    return(-1);


	/* Begin setting new values. */

	/* Initial position. */
	memcpy(
	    &obj_ptr->pos, pos, sizeof(sar_position_struct)
	);

        /* Set visible range based on radius. */
        if(radius >= 20.0)
            obj_ptr->range = SFMMilesToMeters(14.0);
	else if(radius >= 12.0)
            obj_ptr->range = SFMMilesToMeters(10.0);
        else if(radius >= 4.0)
            obj_ptr->range = SFMMilesToMeters(6.0);
        else
            obj_ptr->range = SFMMilesToMeters(3.0);


	/* Get pointer to explosion sub structure. */
	obj_explosion_ptr = (sar_object_explosion_struct *)obj_ptr->data;
	if(obj_explosion_ptr == NULL)
	    return(-1);

	/* Set size of explosion. */
	obj_explosion_ptr->radius = radius;

	/* Color emission. */
	obj_explosion_ptr->color_emission = SAR_EXPLOSION_COLOR_EMISSION_IS_LIGHT;

	/* Regular explosions always have center of gravity at object's
	 * center.
	 */
	obj_explosion_ptr->gravity_offset = SAR_EXPLOSION_GRAVITY_CENTER;

	/* Set frame animation values. */
        obj_explosion_ptr->cur_frame = 0;
        obj_explosion_ptr->frame_inc_int = option.explosion_frame_int;
        obj_explosion_ptr->next_frame_inc = cur_millitime +
            obj_explosion_ptr->frame_inc_int;

	obj_explosion_ptr->cur_frame = 0;
        obj_explosion_ptr->frame_repeats = 0;
        obj_explosion_ptr->total_frame_repeats = 1;

	/* Get texture. */
	obj_explosion_ptr->tex_num = SARGetTextureRefNumberByName(
	    scene, tex_name
	);

	/* Set referance object. */
	obj_explosion_ptr->ref_object = ref_object;

	/* Return newly created explosion object. */
	return(obj_num);
}

/*
 *	Creates a splash at the specified position referenced by
 *      ref_object.
 *
 *      The splash will follow the ref_object or remain stationary
 *      if ref_object is -1.
 */
int SplashCreate(
        sar_scene_struct *scene,
        sar_object_struct ***ptr, int *total,
        sar_position_struct *pos,
        double radius,
        int ref_object,
        const char *tex_name
)
{
        int obj_num;
        sar_object_struct *obj_ptr;
        sar_object_explosion_struct *obj_explosion_ptr;


        if((scene == NULL) || (pos == NULL))
            return(-1);

        /* Create explosion object. */
        obj_num = SARObjCreate(
            scene, ptr, total,
            SAR_OBJ_TYPE_EXPLOSION
        );
        if(obj_num < 0)
            return(-1);

        obj_ptr = (*ptr)[obj_num];
        if(obj_ptr == NULL)
            return(-1);


        /* Begin setting new values. */

        /* Initial position. */
        memcpy(
            &obj_ptr->pos, pos, sizeof(sar_position_struct)
	);

        /* Set visible range based on radius. */
	if(radius >= 8.0)
            obj_ptr->range = SFMMilesToMeters(10.0);
        else if(radius >= 4.0)
            obj_ptr->range = SFMMilesToMeters(6.0);
	else
            obj_ptr->range = SFMMilesToMeters(3.0);

        /* Get pointer to explosion sub structure. */
        obj_explosion_ptr = (sar_object_explosion_struct *)obj_ptr->data;
        if(obj_explosion_ptr == NULL)
            return(-1);

        /* Set size of explosion. */
        obj_explosion_ptr->radius = radius;

        /* Color emission. */
        obj_explosion_ptr->color_emission = SAR_EXPLOSION_COLOR_EMISSION_NONE;

        /* `Splash' explosions always have center of gravity at object's
         * base, so that it's kept above water.
         */
        obj_explosion_ptr->gravity_offset = SAR_EXPLOSION_GRAVITY_BASE;

        /* Set frame animation values. */
	obj_explosion_ptr->frame_inc_int = option.splash_frame_int;
        obj_explosion_ptr->next_frame_inc = cur_millitime +
            obj_explosion_ptr->frame_inc_int;

        obj_explosion_ptr->cur_frame = 0;
        obj_explosion_ptr->frame_repeats = 0;
        obj_explosion_ptr->total_frame_repeats = 1;

        /* Get texture. */
        obj_explosion_ptr->tex_num = SARGetTextureRefNumberByName(
            scene, tex_name
        );

        /* Set referance object. */
        obj_explosion_ptr->ref_object = ref_object;

        /* Return newly created explosion object. */
        return(obj_num);
}
