#	Color.pm
#
#	A package for manipulating SDL_Color *
#
#	Copyright (C) 2002 David J. Goehrig

package SDL::Color;
use strict;
use SDL;

use vars qw/ $white $black /;

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self = {};
	my (%options) = @_;
	if ($options{-pixel} && $options{-surface}) {
		(@$self{-r,-g,-b}) = @{SDL::GetRGB($options{-surface},$options{-pixel})};
	} else {
		$self->{-r} = $options{-r} || 0;
		$self->{-g} = $options{-g} || 0;
		$self->{-b} = $options{-b} || 0;
	}
	$self->{-color} = SDL::NewColor($self->{-r},$self->{-g},$self->{-b});
	die "Could not create color, ", SDL::GetError(), "\n"
		unless ($self->{-color});
	bless $self,$class;
	return $self;
}

sub DESTROY {
	my $self = shift;
	SDL::FreeColor($self->{-color});
}

sub r {
	my $self = shift;
	return SDL::ColorR($self->{-color},@_);	
}

sub g {
	my $self = shift;
	return SDL::ColorG($self->{-color},@_);
}

sub b {
	my $self = shift;
	return SDL::ColorB($self->{-color},@_);
}

sub pixel {
	my ($self,$surface) = @_;
	return SDL::MapRGB($surface->{-surface},$self->{-r},$self->{-g},$self->{-b});
}

$white = new SDL::Color -r => 0xff, -g => 0xff, -b => 0xff;
$black = new SDL::Color -r => 0x0, -g => 0x0, -b => 0x0;


1;

__END__;

=head1 NAME

SDL::Color - a SDL perl extension

=head1 SYNOPSIS

  $rect = new SDL::Color ( -r => 0xde, -g => 0xad, -b =>c0 );

=head1 DESCRIPTION

C<SDL::Color> is a wrapper for display format independent color
representations. 

=head2 new ( -surface => $app, -pixel => 0x000000 )

=head2 new ( -r => 0xff, -g => 0xff, -b => 0xff )

=head2 r ( [ red ] )

=head2 g ( [ green ] )

=head2 b ( [ blue ] )

=head2 pixel ()

=head1 AUTHOR

David J. Goehrig

=head1 SEE ALSO

perl(1) SDL::Surface(3) SDL::TTFont(3)

=cut

