/*
    Copyright (C) 2003-2006 Teus Benschop.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


#include "libraries.h"
#include "utilities.h"
#include "constants.h"


ustring current_book;
vector<ustring> books;
vector<ustring> original_chapter;
vector<ustring> original_verse;
vector<ustring> hebrew_greek_chapter;
vector<ustring> hebrew_greek_verse;


void mapping_get_systems ();
void mapping_parse_data (const ustring & system);
void mapping_get_books (const ustring& system);
void mapping_get_chapters (const ustring& system, const ustring& book);
void mapping_get_verses (const ustring& system, const ustring& book, const ustring& chapter);
void mapping_get_verse (const ustring& system, const ustring& book, const ustring& chapter, const ustring& verse);


int main (int argc, char *argv[])
{
  ustring system;
  ustring book;
  ustring chapter;
  ustring verse;
  // Process command line arguments.
  for (int i = 1; i < argc; i++) {
    ustring argument;
    argument = argv[i];
    if (argument.length() > 2) {
      if (argument.substr (0, 2) == "--") {
        argument.erase (0, 2);
        if (argument == "name")
          system = argv[++i];
        if (argument == "book")
          book = argv[++i];
        if (argument == "chapter")
          chapter = argv[++i];
        if (argument == "verse")
          verse = argv[++i];
        continue;
      }
    }
  }
  if (argc == 1) {
    cout << "sc-mapping provides mapping information on stdout." << endl;
    cout << "--names" << endl;
    cout << "  Available mapping systems" << endl;
    cout << "--name <name>>" << endl;
    cout << "  Available books in this mapping system, e.g. -- name Original" << endl;
    cout << "--name <name> --book <book>" << endl;
    cout << "  Provides maps for the book, e.g. -- name Original --book Genesis" << endl;
    cout << "--name <name> --book <book> --chapter <chapter>" << endl;
    cout << "  Provides maps for the chapter" << endl;
    cout << "--name <name> --book <book> --verse <verse>" << endl;
    cout << "  Provides maps for the verse" << endl;
    cout << "These last three options provide maps in a four-line format." << endl;
    cout << "1: Version's chapter." << endl;
    cout << "2: Version's verse." << endl;
    cout << "3: Corresponding Hebrew/Greek chapter." << endl;
    cout << "4: Corresponding Hebrew/Greek verse." << endl;
    cout << "And this repeats for every available mapping." << endl;
  }
  if (argc == 2) {
    // Show all available systems.
    mapping_get_systems ();
  }
  if (argc == 3) {
    // Show books in this mapping.
    mapping_get_books (system);
  }
  if (argc == 5) {
    // Give maps for the book.
    mapping_get_chapters (system, book);
  }
  if (argc == 7) {
    // Provides maps for the chapter.
    mapping_get_verses (system, book, chapter);
  }
  if (argc == 9) {
    // Provides maps for the verse.
    mapping_get_verse (system, book, chapter, verse);
  }
  return 0;
}


void mapping_get_systems ()
// Lists the available mapping systems the library knows about.
{
  ReadFiles rf (PACKAGE_DATA_DIR, "map_", ".txt");
  for (unsigned int i = 0; i < rf.files.size(); i++) {
    ustring filename;
    filename = gw_build_filename (PACKAGE_DATA_DIR, rf.files[i]);
    ReadText rt (filename, true, true);
    cout << rt.lines[0] << endl;
  }    
}


void mapping_parse_data (const ustring& system)
{
  // Read the mapping file.
  string s;
  s = system;
  for (unsigned int i = 0; i < s.length(); i++)
    if (s[i] == ' ')
      s[i] = '_';
  ustring filename (s);
  filename = "map_" + filename.casefold() + ".txt";
  filename = gw_build_filename (PACKAGE_DATA_DIR, filename);
  ReadText rt (filename, true, false);
  // Go through all lines, but skip first as that one contains the name of the mapping.
  for (unsigned int i = 1; i < rt.lines.size(); i++) {
    // If colon found, then it's a mapping, else a book name.
    size_t position;
    position = rt.lines[i].find (":");
    if (position != string::npos) {
      // We found a mapping, like: 31:55=32:1
      books.push_back (current_book);
      original_chapter.push_back (rt.lines[i].substr (0, position));
      rt.lines[i].erase (0, ++position);
      position = rt.lines[i].find ("=");
      original_verse.push_back (rt.lines[i].substr (0, position));
      rt.lines[i].erase (0, ++position);
      position = rt.lines[i].find (":");
      hebrew_greek_chapter.push_back (rt.lines[i].substr (0, position));
      rt.lines[i].erase (0, ++position);
      hebrew_greek_verse.push_back (rt.lines[i]);
    } else {
      current_book = rt.lines[i];
    }
  }
}


void mapping_get_books (const ustring& system)
{
  mapping_parse_data (system);
  ustring previous_book;
  for (unsigned int i = 0; i < books.size(); i++) {
    if (books[i] != previous_book) {
      cout << books[i] << endl;
      previous_book = books[i];
    }
  }
}


void mapping_get_chapters (const ustring& system, const ustring& book)
{
  mapping_parse_data (system);
  for (unsigned int i = 0; i < books.size(); i++) {
    if (book == books[i]) {
      cout << original_chapter[i] << endl;
      cout << original_verse[i] << endl;
      cout << hebrew_greek_chapter[i] << endl;
      cout << hebrew_greek_verse[i] << endl;
    }
  }
}


void mapping_get_verses (const ustring& system, const ustring& book, const ustring& chapter)
{
  mapping_parse_data (system);
  for (unsigned int i = 0; i < books.size(); i++) {
    if (book == books[i]) {
      if (chapter == original_chapter[i]) {
        cout << original_chapter[i] << endl;
        cout << original_verse[i] << endl;
        cout << hebrew_greek_chapter[i] << endl;
        cout << hebrew_greek_verse[i] << endl;
      }
    }
  }
}


void mapping_get_verse (const ustring& system, const ustring& book, const ustring& chapter, const ustring& verse)
{
  mapping_parse_data (system);
  for (unsigned int i = 0; i < books.size(); i++) {
    if (book == books[i]) {
      if (chapter == original_chapter[i]) {
        if (verse == original_verse[i]) {
          cout << original_chapter[i] << endl;
          cout << original_verse[i] << endl;
          cout << hebrew_greek_chapter[i] << endl;
          cout << hebrew_greek_verse[i] << endl;
        }
      }
    }
  }
}
