/*  Screem:  support.c,
 *  some useful functions
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <errno.h>
#include <gnome.h>
#include <stdio.h>
#include <string.h>

#include <glade/glade.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#endif

#include "support.h"

#include "noicon.xpm"
#include "broken.xpm"

extern GtkWidget *app;

static GList *icon_list = NULL;

static void file_sel_delete( GtkWidget *widget, GtkWidget **file_sel );
static void file_selected( GtkWidget *widget, gboolean *selected );

/**
 * change_state:
 *
 * toggles widget sensitivity
 *
 * return values: none
 */
void change_state( GtkWidget *widget, gpointer data )
{
        gtk_widget_set_sensitive( widget, !GTK_WIDGET_IS_SENSITIVE( widget ) );
}

/**
 * file_select:
 *
 * handles selection of files/directories modally
 * the return value is NULL if the select was cancelled, otherwise it
 * returns the filename
 *
 * return values: a string
 */
gchar* file_select( gchar *title )
{
	GtkWidget *file_sel;
	gchar *filename;
	gboolean selected = FALSE;

	file_sel = gtk_file_selection_new( title );
	gtk_window_set_modal( GTK_WINDOW( file_sel ), TRUE );

	gtk_window_set_wmclass( GTK_WINDOW( file_sel ), "file_select", 
				"Screem" );

	gtk_signal_connect( GTK_OBJECT( file_sel ), "destroy", 
                            GTK_SIGNAL_FUNC( file_sel_delete ), &file_sel );
	
        gtk_signal_connect( GTK_OBJECT( GTK_FILE_SELECTION( file_sel )->cancel_button ), "clicked", GTK_SIGNAL_FUNC( file_sel_delete ), &file_sel );
        
        gtk_signal_connect( GTK_OBJECT( GTK_FILE_SELECTION( file_sel )->ok_button ), "clicked", GTK_SIGNAL_FUNC( file_selected ), &selected );

        gtk_widget_show( file_sel );

	while( ( ! selected ) && ( file_sel ) )
		gtk_main_iteration();

	/* canceled or window was closed */
	if( ! selected )
		return NULL;

	/* ok */
	filename = g_strdup( gtk_file_selection_get_filename( GTK_FILE_SELECTION( file_sel ) ) );
	gtk_widget_destroy( file_sel );
	return filename;
}

static void file_sel_delete( GtkWidget *widget, GtkWidget **file_sel )
{
	gtk_widget_destroy( *file_sel );
	*file_sel = NULL;
}

static void file_selected( GtkWidget *widget, gboolean *selected )
{
	*selected = TRUE;
}

gint execute_command( const gchar *command )
{
        FILE *results;
        GtkWidget *text;
	GtkWidget *etext;
        gchar buffer[ 1024 ];
        gint retval;
        gint size;

	GladeXML *xml;

	xml = glade_get_widget_tree( app );
	text = glade_xml_get_widget( xml, "message_text" );
	etext = glade_xml_get_widget( xml, "error_text" );

        results = popen( command, "r" );

        gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL,
                         "Running: ", -1 );
        gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL, command,
                         -1  );
        gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL,
                         "\n\n", -1 );
	if( results ) {
		while( ( size = fread( buffer, 1, 1024, results ) ) ) {
			gtk_text_insert( GTK_TEXT( text ), NULL, NULL, NULL,
					 buffer, size );
		}
		retval = pclose( results );
	} else {
		gtk_text_insert( GTK_TEXT( etext ), NULL, NULL, NULL,
				 "Unable to execute: ", -1 );
		gtk_text_insert( GTK_TEXT( etext ), NULL, NULL, NULL,
				 command, strlen( command ) );
		retval = -1;
	}

        return retval;
}

gchar* escape_path( gchar *pathname )
{
	gchar *temp;
	gchar *path;
	gchar *temp2;
	gint i = 0;
	
	/* we need to escape any _'s in the path */
	temp2 = g_strdup( pathname );
	while( i < strlen( temp2 ) ) {
		if( temp2[ i ] == '_' ) {
			temp = g_strndup( temp2, ++i );
			path = g_strconcat( temp, "_", &temp2[ i ],
					    NULL );
			g_free( temp );
			g_free( temp2 );
			temp2 = path;
		}
		i ++;
	}
	
	return temp2;
}

Icons *icon_from_mime_type( const gchar *filename, const gchar *mime_type )
{
	GList *list;
	Icons *icon = NULL;
	const gchar *file;

	GdkPixbuf *pixbuf;

	static Icons *notfound = NULL;
	static Icons *brokenlink = NULL;

	if( ! mime_type && filename )
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type( filename );
#else
	mime_type = gnome_vfs_mime_type_from_name( filename );
#endif

	if( ! mime_type )
		return NULL;

	for( list = icon_list; list; list = list->next ) {
		icon = list->data;
		if( ! strcmp( icon->mime_type, mime_type ) ) {
			break;
		} else if( filename && icon->filename &&
			   ! strcmp( filename, icon->filename ) ) {
			break;
		}
		icon = NULL;
	}

	if( ! icon ) {
		/* add an icon type */
#ifndef HAVE_GNOME_VFS
		file = gnome_mime_get_value( mime_type, 
					     "icon-filename" );
#else
		file = gnome_vfs_mime_get_icon( mime_type );
#endif
		icon = g_new0( Icons, 1 );
		icon->mime_type = g_strdup( mime_type );

		if( I_LIKE_WASTING_RESOURCES && filename && 
		    ! strncmp( "image/", mime_type, strlen( "image/" ) ) ) {
			/* make the icon a mini version of the image */
			pixbuf = gdk_pixbuf_new_from_file( filename );
			if( pixbuf ) {
				icon->pixbuf = 
					gdk_pixbuf_scale_simple( pixbuf,
								 40, 40,
								 GDK_INTERP_NEAREST );
				gdk_pixbuf_unref( pixbuf );
			}
			icon->filename = g_strdup( filename );
		} else if( file )
			icon->pixbuf = gdk_pixbuf_new_from_file( file );
		else if( ! strcmp( mime_type, "broken-link" ) ) {
			if( ! brokenlink ) {
				brokenlink = icon;
				icon->pixbuf =
					gdk_pixbuf_new_from_xpm_data( (const char**)broken_xpm );
			} else {
				g_free( icon );
				icon = brokenlink;
			}
		} else if( ! notfound ) {
			notfound = icon;
			icon->pixbuf = 
				gdk_pixbuf_new_from_xpm_data( (const char**)noicon );
		} else {
			g_free( icon );
			icon = notfound;
		}
	}
	
	if( pixbuf ) {
		pixbuf = gdk_pixbuf_scale_simple( icon->pixbuf, 20, 20,
						  GDK_INTERP_NEAREST );
		gdk_pixbuf_render_pixmap_and_mask( pixbuf,
						   &icon->open,
						   &icon->open_mask,
						   100 );
		gdk_pixbuf_unref( pixbuf );
	}
	icon_list = g_list_append( icon_list, icon );

	return icon;
}

/* from GNOME Panel */
/* from gnome panel code, slightly modified for screem */
GdkFilterReturn
grab_key_filter(GdkXEvent *gdk_xevent, GdkEvent *event, gpointer data)
{
        XEvent *xevent = (XEvent *)gdk_xevent;
        GtkEntry *entry;
	GtkWidget *grab_dialog;
        char *key;

	grab_dialog = gtk_object_get_data( GTK_OBJECT( data ), "dialog" );

        if (xevent->type != KeyPress && xevent->type != KeyRelease)
                puts("EXIT X");
        if (event->type != GDK_KEY_PRESS && event->type != GDK_KEY_RELEASE)
                puts("EXIT GDK");

        if (xevent->type != KeyPress && xevent->type != KeyRelease)
        /*if (event->type != GDK_KEY_PRESS && event->type != GDK_KEY_RELEASE)*/
                return GDK_FILTER_CONTINUE;
        
        entry = GTK_ENTRY (data);

        key = convert_keysym_state_to_string (event->key.keyval,
                                              event->key.state);
        gtk_entry_set_text (entry, key?key:"");
        g_free(key);

        gdk_keyboard_ungrab (GDK_CURRENT_TIME);
        gtk_widget_destroy (grab_dialog);
        gdk_window_remove_filter (GDK_ROOT_PARENT(),
                                  grab_key_filter, data);

        return GDK_FILTER_REMOVE;
}

gchar * convert_keysym_state_to_string( guint keysym, guint state )
{
        gchar *key;
	gchar *temp;

        if( keysym == 0 )
                return g_strdup( _("Disabled") );

        gdk_error_trap_push();
        key = gdk_keyval_name( keysym );
        gdk_error_trap_pop();
        if( ! key )
		return NULL;


	key = g_strdup( "" );

        if( state & GDK_CONTROL_MASK )
		key = g_strdup( "Control-" );

        if( state & GDK_LOCK_MASK ) {
		temp = key;
		key = g_strconcat( key, "Lock-", NULL );
		g_free( temp );
        }
        if( state & GDK_SHIFT_MASK ) {
		temp = key;
		key = g_strconcat( key, "Shift-", NULL );
		g_free( temp );
        }
        if( state & GDK_MOD1_MASK ) {
		temp = key;
		key = g_strconcat( key, "Mod1-", NULL );
		g_free( temp );
        }
        if( state & GDK_MOD2_MASK ) {
               	temp = key;
		key = g_strconcat( key, "Mod2-", NULL );
		g_free( temp );
        }
        if( state & GDK_MOD3_MASK ) {
              	temp = key;
		key = g_strconcat( key, "Mod3-", NULL );
		g_free( temp );
        }
        if( state & GDK_MOD4_MASK ) {
             	temp = key;
		key = g_strconcat( key, "Mod4-", NULL );
		g_free( temp );
        }
        if( state & GDK_MOD5_MASK ) {
		temp = key;
		key = g_strconcat( key, "Mod5-", NULL );
		g_free( temp );
        }

        temp = key;
	if( *key ) {
		key = g_strconcat( key, gdk_keyval_name( keysym ), NULL );
		g_free( temp );
	}

	return key;
}

void key_list_grab_key( GtkWidget *widget )
{
	GtkWidget *frame;
        GtkWidget *box;
        GtkWidget *label;
	GtkWidget *grab_dialog;
	gpointer data;
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );

	data = glade_xml_get_widget( xml, "key_name" );

        grab_dialog = gtk_window_new (GTK_WINDOW_POPUP);

	gtk_object_set_data( GTK_OBJECT( data ), "dialog", grab_dialog );

        gdk_keyboard_grab (GDK_ROOT_PARENT(), TRUE, GDK_CURRENT_TIME);
        gdk_window_add_filter (GDK_ROOT_PARENT(), grab_key_filter, data);

        gtk_window_set_policy (GTK_WINDOW (grab_dialog), FALSE, FALSE, TRUE);
        gtk_window_set_position (GTK_WINDOW (grab_dialog), GTK_WIN_POS_CENTER);
        gtk_window_set_modal (GTK_WINDOW (grab_dialog), TRUE);

        frame = gtk_frame_new (NULL);
        gtk_container_add (GTK_CONTAINER (grab_dialog), frame);

        box = gtk_hbox_new (0, 0);
        gtk_container_set_border_width (GTK_CONTAINER (box), 20);
        gtk_container_add (GTK_CONTAINER (frame), box);

        label = gtk_label_new (_("Press a key..."));
        gtk_container_add (GTK_CONTAINER (box), label);
        
        gtk_widget_show_all (grab_dialog);
}
/* End of Gnome panel code */

GList* add_recent( GList *list, const gchar *entry )
{
	if( ( g_list_length( list ) ) &&
	    ( ! g_list_find_custom( list, (gchar*)entry, 
				    (GCompareFunc)strcmp ) ) )
		list = g_list_append( list, g_strdup( entry ) );
	else if( ! list )
		list = g_list_append( list, g_strdup( entry ) );

	return list;
}

GString *load_text_file_with_dir( const gchar *dir, const gchar *file )
{
	gchar *filename = g_strconcat( dir, G_DIR_SEPARATOR_S, file, NULL );

	GString *data = load_text_file( filename );

	g_free( filename );

	return data;
}


GString* load_text_file( const gchar *path )
{
	FILE *in;
	GString *tmp;

	in = fopen( path, "r" );

	if( ! in )
		return NULL;

	tmp = load_text_file_from_stream( in );

	fclose( in );

	return tmp;
}

GString* load_text_file_from_stream( FILE *in )
{
	GString *tmp;
	gchar *buffer;

	buffer = g_new0( gchar, BUFSIZ );
	tmp = g_string_new( "" );

	while( fread( buffer, 1, BUFSIZ - 1, in ) ) 
		g_string_append( tmp, buffer );
	
	g_free( buffer );

	return tmp;	
}

void screem_set_upload_flags( Site *site, GladeXML *xml, gchar *path )
{
	gboolean active;
	GtkWidget *check;

	active = screem_site_is_excluded( site, path );
	check = glade_xml_get_widget( xml, "exclude" );
	gtk_check_menu_item_set_show_toggle( GTK_CHECK_MENU_ITEM( check ), 
					     TRUE );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( check ), active );
	
	active = screem_site_is_ascii( site, path );
	check = glade_xml_get_widget( xml, "ascii" );
	gtk_check_menu_item_set_show_toggle( GTK_CHECK_MENU_ITEM( check ), 
					     TRUE );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( check ), active );
	
	active = screem_site_is_ignored( site, path );
	check = glade_xml_get_widget( xml, "ignore" );
	gtk_check_menu_item_set_show_toggle( GTK_CHECK_MENU_ITEM( check ), 
					     TRUE );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( check ), active );
}

void screem_do_show_error( gchar *m, gboolean fatal )
{
	GtkWidget *message;
	GladeXML *xml;
	GtkWidget *widget;

	static gint lines = 0;

	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	xml = glade_get_widget_tree( widget );
	message = glade_xml_get_widget( xml, "error_text" );

	if( ! fatal )
		gnome_app_warning( GNOME_APP( app ), m );
	else
		gnome_app_error( GNOME_APP( app ), m );

	lines ++;

	/* remove old messages if we have too many */
	if( lines > 10 ) {

		lines --;
	}
#ifdef I_COULD_BE_BOTHERED
	gtk_text_insert( GTK_TEXT( message ), NULL, NULL, NULL, "\n", -1 );

	gtk_text_insert( GTK_TEXT( message ), NULL, NULL, NULL, m, -1 );
#endif
}

void screem_show_message( gchar *m )
{
	GtkWidget *message;
	GladeXML *xml;
	GtkWidget *widget;

	static gint lines = 0;

	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	xml = glade_get_widget_tree( widget );
	message = glade_xml_get_widget( xml, "message_text" );

	gnome_app_flash( GNOME_APP( app ), m );

	lines ++;

	/* remove old messages if we have too many */
	if( lines > 10 ) {
	

		lines --;
	}
#ifdef I_COULD_BE_BOTHERED
	gtk_text_insert( GTK_TEXT( message ), NULL, NULL, NULL, "\n", -1 );

	gtk_text_insert( GTK_TEXT( message ), NULL, NULL, NULL, m, -1 );
#endif
}
