/*  Screem:  siteTree.c,
 *  non UI part of the site file tree, builds an xml representation of
 *  a directory heirachy, and adds relevant meta data such as mime type
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <dirent.h>
#include <glib.h>
#include <string.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnome/libgnome.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "siteTree.h"
#include "site.h"
#include "page.h"
#include "xml.h"

extern Site *current_site;

/*
 * build_directory_tree:
 * @site:     the site who we are building a tree for, NULL for 
 *            current_site
 * @dir_name: the name of the directory to start at
 * @node:     the parent node in the tree, NULL if the tree hasn't
 *            been started
 *
 * Builds an xml tree of the directory/sub directory contents of 
 * dir_name,
 * xml is being used rather than using Gnodes as it makes it simple to
 * add any metadata.
 *
 * return values: an xmlDocPtr
 */
xmlDocPtr build_directory_tree( Site *site, const gchar *dir_name, 
				xmlNodePtr node )
{
        DIR *dir = NULL;
        gchar *temp = NULL;
        gchar *next_dir_name = NULL;
        struct dirent *entry = NULL;
	struct stat s;

	xmlNodePtr tree;
	xmlDocPtr doc = NULL;

	gboolean is_dir;
	const gchar *site_path;
	const gchar *type;

	gchar *name;
	gchar *size;

	if( site )
		site_path = screem_site_get_pathname( site );
	else
		site_path = RESOURCE_PATH;

	/* if node == NULL then we are just starting so create the doc */
	if( ! node ) {
		doc = xmlNewDoc( XML_DEFAULT_VERSION );
		doc->root = xmlNewDocNode( doc, NULL, "directory", NULL );
		node = doc->root;
		xmlSetProp( node, "type", "directory" );
	}

	if( stat( dir_name, &s ) < 0 )
		return doc;

	if( ( is_dir = S_ISDIR(s.st_mode) ) )
		type = "directory";
	else
#ifndef HAVE_GNOME_VFS
		type = gnome_mime_type_or_default( dir_name, "text/plain" );
#else
	type = gnome_vfs_mime_type_from_name_or_default( dir_name, "text/plain" );
#endif

	if( is_dir && strcmp( site_path, dir_name ) )
		name = g_strconcat( dir_name, G_DIR_SEPARATOR_S, NULL );
	else
		name = g_strdup( dir_name );
       
	tree = xmlNewChild( node, NULL, "node", name );
	xmlSetProp( tree, "type", type );

	if( ! is_dir ) {
		/* add other meta info from the stat struct */
		size = g_strdup_printf( "%d", (int)s.st_size );
		xmlSetProp( tree, "size", size );
		g_free( size );
	}

        dir = opendir( dir_name );
	while( dir && ( entry = readdir( dir ) ) ) {
		temp = entry->d_name;
		if( strcmp( temp, ".." ) && strcmp( temp, "." ) ) {
			next_dir_name = g_strconcat( name, temp, NULL );
			build_directory_tree( site, next_dir_name, tree );
			g_free( next_dir_name );
		}
	}
	if( dir )
		closedir( dir );
        g_free( name );

        return doc;
}

/*
 * indentify_pages:
 * @site: the site who we are id'ing the pages
 * @doc:  the complete tree
 * @node: an xml tree node holding the directory structure
 *
 * Adds all text documents to the sites list of pages
 *
 * return values: none
 */
void identify_pages( Site *site, xmlDocPtr doc, xmlNodePtr node )
{
	gchar *mime_type;
	gchar *path;

	Page *page;

	/* get the nodes pathname */
	path = xmlNodeListGetString(doc, node->childs, 1);
	if( path ) {
		/* is it a text file? */
		mime_type = xml_get_value( node, "type" );

		if( screem_page_is_mime_type_page( mime_type ) ) {
			/* its a text file, add it to the sites list if its not
			   already there that is */
			page = screem_site_locate_page( site, (gchar*)path );
			if( ! page ) {
				/* it wasn't there so add it */
				screem_site_add_page( site, (gchar*)path );
			}
		} 
		g_free( mime_type );
		g_free( path );
	}
	if( node->childs )
		identify_pages( site, doc, node->childs );
	
	if( node->next )
		identify_pages( site, doc, node->next );
}
