/*  Screem:  preferences.c,
 *  This file deals with the programs settings, loading/saving them and
 *  editing them. 
 *
 *  Copyright (C) 1999, 2000, 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>
#include <dirent.h>
#include <gmodule.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include <glade/glade.h>

#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "fileops.h"
#include "support.h"
#include "preferences.h"
#include "helpers.h"
#include "tag_tree.h"
#include "editor.h"
#include "xml.h"
#include "site.h"
#include "page.h"
#include "dtd.h"
#include "preview.h"

/* these are placed into the key actions combo, and must correspond in order
   to the KeyAction values.  ADD_CLOSE = 1, so the text for that is
   the first element */
gchar *key_actions[] = {
        "Close current tag, and place cursor between the open and close tags",
        NULL
};

static GladeXML *xml;  /* the preferences glade xml tree */

extern GList *helpers;
extern GnomeClient *client;
extern GtkWidget *app;
extern Preferences *cfg;

static void misc_tab( void );
static void keys_tab( void );
static void colour_tab( void );
static void helper_tab( void );
static void printing_tab( void );
static void java_tab( void );

static KeyAction is_action( gchar *text );

static void syntax_files_io( gboolean save );
static void load_syntax_file( gchar *file );
static void save_syntax_file( gchar *file );


#define HELPER_NAME "Helpers/pathname"
#define HELPER_TYPE "Helpers/mimetype"
#define HELPER_ICON "Helpers/icon"
#define HELPER_DEFAULT "invalid"
#define DEFAULT_FONT "-*-helvetica-bold-r-normal--12-*-*-*-*-*-*-*"
#define SITE_DEFAULT "invalid"

void preferences_new()
{
	cfg = g_new0( Preferences, 1 );
	cfg->mpage = g_new0( MiscPage, 1 );
	cfg->kpage = g_new0( KeysPage, 1 );
	cfg->cpage = g_new0( ColourPage, 1 );
	cfg->hpage = g_new0( HelperPage, 1 );
	cfg->ppage = g_new0( PrintPage, 1 );
	cfg->jpage = g_new0( JavaPage, 1 );
}

void load_open()
{
	gint n;
	gchar *p;
	gchar *path;
	/* don't auto load if the user doesn't want to */
	if( ! cfg->mpage->auto_load )
		return;

	gnome_config_push_prefix( "/screem/" );

	if( gnome_config_has_section( "open_sites" ) ) {
		/* 1 as there can't be a site0 variable as that would be
		   the fake site, which doesn't get listed */
		n = 1;
		p = "";
		/* read the paths in and place them in the list */
		while( strcmp( p, "invalid" ) ) {
			path = g_strdup_printf( "open_sites/site%i=%s",
						n, SITE_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			if( strcmp( p, "invalid" ) ) {
				screem_site_open_with_filename( p );
				g_free( p );
			} else {
				g_free( p );
				p = "invalid";
			}
			n ++;
		}
	}

	gnome_config_pop_prefix();
}

void save_open( GList *sites )
{
	GList *list;
	GList *l;
	gint count;
	gint c;
	gchar *path;
	const gchar *pathname;
	Site *site;
	Page *page;

	gnome_config_push_prefix( "/screem/" );

	gnome_config_clean_section( "open_sites" );
	gnome_config_clean_section( "open_pages" );

	for( count = 0, list = sites; list; list = list->next, count ++ ) {
		site = (Site*)list->data;
		pathname = screem_site_get_pathname( site );
		if( ! screem_site_get_fake_flag( site ) ) {
			path = g_strdup_printf( "open_sites/site%i", count );
			gnome_config_set_string( path, pathname );
			g_free( path );
		} else {
			l = screem_site_get_pages( site );
			for( c = 0; l; l = l->next, c ++ ) {
				path = g_strdup_printf( "open_pages/page%i", 
							c );
				page = (Page*)l->data;
				pathname = screem_page_get_pathname( page );
				gnome_config_set_string( path, pathname );
				g_free( path );
			}
		}
	}

	gnome_config_sync ();

	gnome_config_pop_prefix();
}


void load_preferences()
{
	MiscPage   *mpage;
	KeysPage   *kpage;
	ColourPage *cpage;
	HelperPage *hpage;
	PrintPage  *ppage;
	JavaPage   *jpage;

	gchar *p = "";
	gchar *m = "";
	gchar *i = "";
	gchar *path;
	gint n;
	Helper *helper;
	KeyCombo *combo;

	gnome_config_push_prefix( "/screem/" );

	preferences_new();

	mpage = cfg->mpage;
	kpage = cfg->kpage;
	cpage = cfg->cpage;
	hpage = cfg->hpage;
	ppage = cfg->ppage;
	jpage = cfg->jpage;

	cfg->dock_string = gnome_config_get_string( "Placement/Dock=" );
	cfg->tag_width = 200;//gnome_config_get_int( "Config/tag_width=200" );
	cfg->tag_height = gnome_config_get_int( "Config/tag_height=125" );
	cfg->file_width = gnome_config_get_int( "Config/file_width=200" );
        cfg->file_height = gnome_config_get_int( "Config/file_height=300" );
	cfg->editor_width = gnome_config_get_int( "Config/editor_width=500" );
	cfg->editor_height =
		gnome_config_get_int( "Config/editor_height=425" );
	cfg->preview_width = 
		gnome_config_get_int( "Config/preview_width=500" );
	cfg->preview_height = 
		gnome_config_get_int( "Config/preview_height=425" );
	cfg->messages_height = 
		gnome_config_get_int( "Config/messages_height=150" );

	/* misc page settings */
	mpage->auto_load =
		gnome_config_get_int("Config/auto_load=1");
	mpage->auto_open =
		gnome_config_get_int("Config/auto_open=1");
	mpage->parse =
		gnome_config_get_int("Config/parse=1");
	mpage->highlighting = 
		gnome_config_get_int( "Config/tag_highlighting=1" );
	mpage->inline_tagging = 
		gnome_config_get_int( "Config/inline_tagging=1" );
	mpage->auto_indent =
		gnome_config_get_int( "Config/auto_indent=1" );
	mpage->intelliclose =
		gnome_config_get_int( "Config/intelligent_close=1" );
	mpage->notebookbar =
		gnome_config_get_int( "Config/notebook_toolbar=1" );
	mpage->viewbar =
		gnome_config_get_int( "Config/viewbar=1" );
	mpage->font_name = 
		gnome_config_get_string( "Config/font_name="DEFAULT_FONT );
	if( mpage->font_name )
		mpage->font = gdk_font_load( mpage->font_name );
       	mpage->ent = gnome_config_get_int( "Config/ent_insert=1" );
	mpage->default_filename =
		gnome_config_get_string("Config/default_filename=index.html" );
	mpage->default_charset =
		gnome_config_get_string( "Config/default_charset=ISO-8859-1" );
	mpage->default_dtd =
		gnome_config_get_string( "Config/default_dtd=-//W3C//DTD HTML 4.01 Transitional//EN" );

	cfg->hints = gnome_config_get_int( "Config/hints=1" );
	cfg->last_hint = gnome_config_get_int( "Config/last_hint=0" );


	/* colours page settings */
	cpage->back[ 0 ] =
                gnome_config_get_int( "Config/back_red=65535" );
        cpage->back[ 1 ] =
                gnome_config_get_int( "Config/back_green=65535" );
	cpage->back[ 2 ] =
                gnome_config_get_int( "Config/back_blue=65535" );
        cpage->fore[ 0 ] =
                gnome_config_get_int( "Config/fore_red=0" );
        cpage->fore[ 1 ] =
                gnome_config_get_int( "Config/fore_green=0" );
        cpage->fore[ 2 ] =
                gnome_config_get_int( "Config/fore_blue=0" );

	cpage->colour_classes = NULL;
	/* load colour classes */
	syntax_files_io( FALSE );

	if( ! cpage->colour_classes )
		g_error( "No Syntax files found, check your installation\n" );
       
	/* java page settings */
	jpage->compiler =
		gnome_config_get_string( "Config/java_compiler=javac" );


	/* load recent sites */
	cfg->recent_sites = NULL;
	if( gnome_config_has_section( "Recent_sites" ) ) {
		n = 0;
		/* read the paths in and place them in the list */
		for( ;; ) {
			path = g_strdup_printf( "Recent_sites/path%i=%s",
						n, HELPER_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			if( strcmp( p, "invalid" ) )
				cfg->recent_sites = 
					g_list_append( cfg->recent_sites, p );
			else {
				g_free( p );
				break;
			}
			n ++;
		}
	}

	/* load recent pages */
	cfg->recent_pages = NULL;
	if( gnome_config_has_section( "Recent_pages" ) ) {
		n = 0;
		p = "";
		/* read the paths in and place them in the list */
		for( ;; ) {
			path = g_strdup_printf( "Recent_pages/path%i=%s",
						n, HELPER_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			if( strcmp( p, "invalid" ) )
				cfg->recent_pages = 
					g_list_append( cfg->recent_pages, p );
			else {
				g_free( p );
				break;
			}
			n ++;
		}
	}

	/* load global tag tree */
	cfg->tag_tree = screem_tag_tree_load();

	/* load helpers */
	hpage->helpers = NULL;
	if( gnome_config_has_section( "Helpers" ) ) {
		n = 0;
		p = "";
		/* read the helpers in and place them in the list */
		for( ;; ) {
			path = g_strdup_printf( "%s%i=%s", HELPER_NAME, n,
						HELPER_DEFAULT );
			p = gnome_config_get_string( path );
			g_free( path );
			path = g_strdup_printf( "%s%i=%s", HELPER_TYPE, n,
						HELPER_DEFAULT );
			m = gnome_config_get_string( path );
			g_free( path );
			path = g_strdup_printf( "%s%i=%s", HELPER_ICON, n,
						HELPER_DEFAULT );
			i = gnome_config_get_string( path );
			g_free( path );

			/* do we create a new Helper and add it? */
			if( strcmp( p, HELPER_DEFAULT ) && 
			    strcmp( m, HELPER_DEFAULT ) ){
				helper = screem_helper_new();
				helper->path = p;
				helper->mime_type = m;
				if( strcmp( i, HELPER_DEFAULT ) )
					helper->icon = i;
				screem_helper_add( helper );
			} else {
				g_free( m );
				g_free( i );
				break;
			}
			n ++;
		}
	}

	/* Key combos */
	kpage->keys = NULL;
	if( gnome_config_has_section( "Key_Combos" ) ) {
		n = 0;
		m = p = "";
		/* read the helpers in and place them in the list */
		for( ;; ) {
			path = g_strdup_printf("Key_Combos/keys%i=invalid", n);
			p = gnome_config_get_string( path );
			g_free( path );
			path = g_strdup_printf("Key_Combos/text%i=invalid", n);
			m = gnome_config_get_string( path );
			g_free( path );

			/* do we create a new  and add it? */
			if( strcmp( p, "invalid" ) && strcmp( m, "invalid" ) ){
				combo = g_new( KeyCombo, 1 );
				combo->key_name = p;
				switch( ( combo->action = is_action( m ) ) ) {
				case TEXT:
					combo->text = m;
					break;
				default:
					g_free( m );
					combo->text = g_strdup( key_actions[ combo->action - 1 ] );
					break;
				} 
				kpage->keys = g_list_append( kpage->keys,
							     combo );
			} else {
				g_free( p );
				g_free( m );
				break;
			}
			n ++;
		}
	}

	/* Print page settings */
	ppage->printwrap = gnome_config_get_bool ("Config/printwrap=TRUE");
        ppage->printheader = gnome_config_get_bool ("Config/printheader=TRUE");
        ppage->printlines = gnome_config_get_int ("Config/printlines=0");
        ppage->papersize = gnome_config_get_string ("Config/papersize");
	ppage->orientation = 
		gnome_config_get_int( "Config/printorientation=1" );
#ifdef ENABLE_PRINTING
        if( ! ppage->papersize )
                ppage->papersize = g_strdup( gnome_paper_name_default() );
        if( strlen( ppage->papersize ) < 1 ) {
                g_free( ppage->papersize );
                strcpy( ppage->papersize, gnome_paper_name_default() );
        }
#endif

	/* get the dock hide status string */
	mpage->hide_string = gnome_config_get_string( "Placement/Hidden=" );

	gnome_config_pop_prefix();
}

void save_preferences()
{
	MiscPage   *mpage;
	KeysPage    *kpage;
	ColourPage *cpage;
	HelperPage *hpage;
	PrintPage  *ppage;
	JavaPage   *jpage;

	GList *list;
	Helper *helper;
	gchar *path;
	gchar *pathname;
	gint n;
	KeyCombo *combo;
	GtkWidget *widget;

	GladeXML *wxml;

    	gint mh;
	gint sw;

	mpage = cfg->mpage;
	kpage = cfg->kpage;
	cpage = cfg->cpage;
	hpage = cfg->hpage;
	ppage = cfg->ppage;
	jpage = cfg->jpage;

        gnome_config_push_prefix( gnome_client_get_config_prefix( client ) );

	/* save open sites */

	widget = gtk_object_get_data( GTK_OBJECT( app ), "sitebook" );
	wxml = glade_get_widget_tree( widget );

	widget = glade_xml_get_widget( wxml, "resources_window" );
	gnome_config_set_int( "Config/tag_width", widget->allocation.width );
        gnome_config_set_int( "Config/tag_height",widget->allocation.height );

	widget = glade_xml_get_widget( wxml, "messages_book" );
	mh =  widget->allocation.height;

	if( GTK_WIDGET_VISIBLE( widget ) ) {
		gnome_config_set_int( "Config/messages_height", mh );
		mh = 0;
	}

	widget = glade_xml_get_widget( wxml, "sitebook" );
	sw = widget->allocation.width;

	if( GTK_WIDGET_VISIBLE( widget ) ) {
		gnome_config_set_int( "Config/file_width", sw );
		gnome_config_set_int( "Config/file_height",  
				      widget->allocation.height - mh );
		sw = 0;
	}

	gnome_config_set_int("Config/editor_width", 
			     screem_editor_get_width() - sw );
	gnome_config_set_int( "Config/editor_height",
			      screem_editor_get_height() - mh );

	gnome_config_set_int("Config/preview_width", 
			     screem_preview_get_width() - sw );
	gnome_config_set_int( "Config/preview_height",
			      screem_preview_get_height() - mh );

	gnome_config_set_int( "Config/auto_load", mpage->auto_load );
	gnome_config_set_int( "Config/auto_open", mpage->auto_open );
	gnome_config_set_int( "Config/parse", mpage->parse );
	gnome_config_set_int( "Config/tag_highlighting", mpage->highlighting );
	gnome_config_set_int( "Config/inline_tagging", mpage->inline_tagging );
	gnome_config_set_int( "Config/auto_indent", mpage->auto_indent );
	gnome_config_set_int( "Config/intelligent_close", 
			      mpage->intelliclose );
	gnome_config_set_int( "Config/notebook_toolbar", mpage->notebookbar );
	gnome_config_set_int( "Config/viewbar", mpage->viewbar );
	gnome_config_set_string( "Config/font_name", mpage->font_name );
	gnome_config_set_int( "Config/ent_insert", mpage->ent );
	gnome_config_set_string( "Config/default_filename", 
				 mpage->default_filename );
	gnome_config_set_string( "Config/default_charset",
				 mpage->default_charset );
	gnome_config_set_string( "Config/default_dtd",
				 mpage->default_dtd );

	gnome_config_set_int( "Config/hints", cfg->hints );
	gnome_config_set_int( "Config/last_hint", cfg->last_hint );

	gnome_config_set_int( "Config/back_red", cpage->back[ 0 ] );
	gnome_config_set_int( "Config/back_green", cpage->back[ 1 ] );
	gnome_config_set_int( "Config/back_blue", cpage->back[ 2 ] );
	gnome_config_set_int( "Config/fore_red", cpage->fore[ 0 ] );
	gnome_config_set_int( "Config/fore_green", cpage->fore[ 1 ] );
	gnome_config_set_int( "Config/fore_blue",  cpage->fore[ 2 ] );

	gnome_config_set_string( "Config/java_compiler",jpage->compiler );

	/* save recent sites */
	gnome_config_clean_section( "Recent_sites" );
	gnome_config_sync ();
	for( n = 0, list = cfg->recent_sites; list; list = list->next ) {
		pathname = (gchar*)list->data;
		path = g_strdup_printf( "Recent_sites/path%i", n );
		gnome_config_set_string( path, pathname );
		g_free( path );
		n ++;
	}

	/* save recent pages */
	gnome_config_clean_section( "Recent_pages" );
	gnome_config_sync ();
	for( n = 0, list = cfg->recent_pages; list; list = list->next ) {
		pathname = (gchar*)list->data;
		path = g_strdup_printf( "Recent_pages/path%i", n );
		gnome_config_set_string( path, pathname );
		g_free( path );
		n ++;
	}

	/* save helpers */
	gnome_config_clean_section( "Helpers" );
	gnome_config_sync ();
	for( n = 0, list = hpage->helpers; list; list = list->next, n ++ ) {
		helper = (Helper*)list->data;
		path = g_strdup_printf( "%s%i", HELPER_NAME, n );
		gnome_config_set_string( path, helper->path );
		g_free( path );
		path = g_strdup_printf( "%s%i", HELPER_TYPE, n );
		gnome_config_set_string( path, helper->mime_type );
		g_free( path );
		path = g_strdup_printf( "%s%i", HELPER_ICON, n );
		gnome_config_set_string( path, helper->icon );
		g_free( path );
	}

	/* save colours */
	syntax_files_io( TRUE );

	/* save key combos */
	gnome_config_clean_section( "Key_Combos" );
	gnome_config_sync();
	for( n = 0, list = kpage->keys; list; list = list->next, n++ ) {
		combo = (KeyCombo*)list->data;
		path = g_strdup_printf( "Key_Combos/keys%i", n );
		gnome_config_set_string( path, combo->key_name );
		g_free( path );
		path = g_strdup_printf( "Key_Combos/text%i", n );
		gnome_config_set_string( path, combo->text );
		g_free( path );
	}

	/* save printer settings */
	gnome_config_set_bool( "Config/printwrap", ppage->printwrap );
        gnome_config_set_bool( "Config/printheader", ppage->printheader );
        gnome_config_set_int( "Config/printlines", ppage->printlines );
        gnome_config_set_string( "Config/papersize", ppage->papersize );
	gnome_config_set_int( "Config/printorientation", ppage->orientation );


	{
		/* maybe this will help sort out the dock layout
		   going wrong sometimes */
		GtkWidget *dock = gnome_app_get_dock( GNOME_APP( app ) );
		GtkWidget *layout = gnome_dock_get_layout( GNOME_DOCK( dock ) );
		gchar *temp = gnome_dock_layout_create_string( GNOME_DOCK_LAYOUT( layout ) );
		gchar **items = g_strsplit( temp, "\\", 256 );
		gint i;
		guint junk;
		GString *hide_string = g_string_new( "" );

		gnome_config_set_string( "Placement/Dock", temp );

		/* set the hidden status of all dockitems */
		for( i = 0; items[ i ]; i += 2 ) {
			/* items[ i ] == item name */
			widget = gnome_dock_get_item_by_name( GNOME_DOCK( dock ),
							      items[ i ], &junk,
							      &junk, &junk, 
							      &junk );
			g_string_append( hide_string, items[ i ] );
			g_string_append( hide_string, "\\" );
			if( GTK_WIDGET_VISIBLE( widget ) )
				g_string_append( hide_string, "0" );
			else
				g_string_append( hide_string, "1" );
			if( items[ i + 2 ] )
				g_string_append( hide_string, "\\" );
		}
		gnome_config_set_string( "Placement/Hidden", hide_string->str );
		g_string_free( hide_string, TRUE );

		g_strfreev( items );

		g_free( temp );
	}

	gnome_config_sync();
	gnome_config_pop_prefix();
}

void edit_preferences()
{
	GtkWidget *tree;
	GtkCTreeNode *node;

	gint i;
	/* must be the same order the pages are in the notebook */
	const gchar *names[] = {
		N_("Misc"), N_("Keys"), N_("Colours"), N_("Helpers"),
		N_("Printing"),	N_("Java"), NULL
	};
	Tabfn tabs[] = {
		misc_tab, keys_tab, colour_tab, helper_tab,
		printing_tab, java_tab,	NULL
	};

	gchar *name[ 2 ] = { NULL, NULL };

	if( cfg->box ) {
                gdk_window_raise( cfg->box->window );
                gdk_window_show( cfg->box->window );
                return;
        }

	xml = glade_xml_new( cfg->glade_path, "preferences_dialog" );
	cfg->box = glade_xml_get_widget( xml, "preferences_dialog" );

	/* fill in the view names */
	tree = glade_xml_get_widget( xml, "view_tree" );
	gtk_clist_clear( GTK_CLIST( tree ) );
	gtk_clist_freeze( GTK_CLIST( tree ) );
	for( i = 0; names[ i ]; i ++ ) {
		name[ 0 ] = g_strdup( names[ i ] );
		node = gtk_ctree_insert_node( GTK_CTREE( tree ), NULL, NULL,
					      name, 3, NULL, NULL, NULL, NULL,
					      FALSE, FALSE );
		g_free( name[ 0 ] );
	}
	gtk_clist_thaw( GTK_CLIST( tree ) );

	for( i = 0; tabs[ i ]; i ++ )
		tabs[ i ]();

	glade_xml_signal_autoconnect( xml );
	gtk_widget_show_all( cfg->box );
}

/* now handle the pages of the notebook */

/* misc tab fill in details */
static void misc_tab()
{
	GtkWidget *widget;
	MiscPage *mpage = cfg->mpage;
	gint i;

	gpointer toggles[] = {
		"syntax_highlighting", NULL,
		"inline_tagging", NULL,
		"auto_indent",  NULL,
		"intelligent_close", NULL,
		"notebook_check", NULL,
		"viewbar_check", NULL,
		"auto_load", NULL,
		"auto_open", NULL,
		"parse", NULL,
		NULL
	};
	gchar *ent_modes[] = {
		"entity_off", "entity_on", "entity_by_char", NULL
	};

	toggles[ 1 ] = GINT_TO_POINTER( mpage->highlighting );
	toggles[ 3 ] = GINT_TO_POINTER( mpage->inline_tagging );
	toggles[ 5 ] = GINT_TO_POINTER( mpage->auto_indent );
	toggles[ 7 ] = GINT_TO_POINTER( mpage->intelliclose );
	toggles[ 9 ] = GINT_TO_POINTER( mpage->notebookbar );
	toggles[ 11 ] = GINT_TO_POINTER( mpage->viewbar );
	toggles[ 13 ] = GINT_TO_POINTER( mpage->auto_load );
	toggles[ 15 ] = GINT_TO_POINTER( mpage->auto_open );
	toggles[ 17 ] = GINT_TO_POINTER( mpage->parse );

	widget = glade_xml_get_widget( xml, "default_filename" );
	gtk_entry_set_text( GTK_ENTRY( widget ), mpage->default_filename );

	widget = glade_xml_get_widget( xml, "default_charset" );
	gtk_entry_set_text( GTK_ENTRY( widget ), mpage->default_charset );

	widget = glade_xml_get_widget( xml, "dtd_combo" );
	screem_dtd_fill_combo( widget );
	widget = glade_xml_get_widget( xml, "default_dtd" );
	gtk_entry_set_text( GTK_ENTRY( widget ), mpage->default_dtd );

	if( mpage->font_name ) {
		widget = glade_xml_get_widget( xml, "font_name_entry" );
		gtk_entry_set_text( GTK_ENTRY( widget ), mpage->font_name );
		widget = glade_xml_get_widget( xml, "fontpicker1" );
		gnome_font_picker_set_font_name( GNOME_FONT_PICKER( widget ),
						 mpage->font_name );
	}

	for( i = 0; toggles[ i ]; i ++ ) {
		widget = glade_xml_get_widget( xml, toggles[ i ] );
		gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON( widget ),
					     (gboolean)toggles[ i + 1 ] );
		i ++;
	}

	widget = glade_xml_get_widget( xml, ent_modes[ mpage->ent ] );
	gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON( widget ), TRUE );
}
/* misc tab callbacks */
void font_changed( GtkWidget *widget )
{
	gchar *font;
	GtkWidget *entry;

	entry = glade_xml_get_widget( xml, "font_name_entry" );
	font = gnome_font_picker_get_font_name( GNOME_FONT_PICKER( widget ) );
	gtk_entry_set_text( GTK_ENTRY( entry ), font );
}


/* keys tab fill in details */
static void keys_tab()
{
	GtkWidget *widget;
	GList *list;
	KeyCombo *combo;
	KeysPage *kpage;
	gchar *item[ 3 ] = { NULL, NULL, NULL };
	gint i;

	kpage = cfg->kpage;
	widget = glade_xml_get_widget( xml, "key_list" );

	gtk_clist_clear( GTK_CLIST( widget ) );
	for( list = kpage->keys; list; list = list->next ) {
		combo = (KeyCombo*)list->data;
		item[ 0 ] = combo->key_name;
		item[ 1 ] = combo->text;
		gtk_clist_append( GTK_CLIST( widget ), item );
	}

	gtk_object_set_data( GTK_OBJECT( widget ), "row", (gpointer)-1 );

	widget = glade_xml_get_widget( xml, "key_actions" );
	
	for( list = g_list_alloc(), i = 0; key_actions[ i ]; i ++ )
		list = g_list_append( list, key_actions[ i ] );
	gtk_combo_set_popdown_strings( GTK_COMBO( widget ), list );
	g_list_free( list );
}
/* callbacks for keys tab */
void key_list_select( GtkCList *list, gint row, gint column, GdkEvent *event )
{
	GtkWidget *name;
	GtkWidget *text;

	gchar *n;
	gchar *t;

	gtk_object_set_data( GTK_OBJECT( list ), "row", (gpointer)row );

	/* insert details */
	name = glade_xml_get_widget( xml, "key_name" );
	text = glade_xml_get_widget( xml, "text_to_insert" );

	gtk_clist_get_text( GTK_CLIST( list ), row, 0, &n );
	gtk_clist_get_text( GTK_CLIST( list ), row, 1, &t );

	gtk_entry_set_text( GTK_ENTRY( name ), n );
	gtk_entry_set_text( GTK_ENTRY( text ), t );
}

void key_list_unselect( GtkCList *list, gint row, gint column, GdkEvent *event)
{
	gint r;

	r = (gint)gtk_object_get_data( GTK_OBJECT( list ), "row" );

	if( row == r )
		gtk_object_set_data( GTK_OBJECT( list ), "row", (gpointer)-1 );

}

void key_list_add_key()
{
	gchar *item[ 3 ] = { NULL, NULL, NULL };
	GtkWidget *name;
	GtkWidget *text;
	GtkWidget *list;

	/* insert details */
	name = glade_xml_get_widget( xml, "key_name" );
	text = glade_xml_get_widget( xml, "text_to_insert" );

	item[ 0 ] = gtk_entry_get_text( GTK_ENTRY( name ) );
	item[ 1 ] = gtk_entry_get_text( GTK_ENTRY( text ) );

	list = glade_xml_get_widget( xml, "key_list" );

	gtk_clist_append( GTK_CLIST( list ), item );

        preferences_changed();
}

void key_list_remove_key()
{
	gint row;
	GtkWidget *list;

	list = glade_xml_get_widget( xml, "key_list" );
	row = (gint)gtk_object_get_data( GTK_OBJECT( list ), "row" );

	gtk_clist_remove( GTK_CLIST( list ), row );

        preferences_changed();
}

void key_list_update_key()
{
	gint row;
	GtkWidget *list;
	GtkWidget *name;
	GtkWidget *text;

	gchar *n;
	gchar *t;

	list = glade_xml_get_widget( xml, "key_list" );
	row = (gint)gtk_object_get_data( GTK_OBJECT( list ), "row" );

	/* get details */
	name = glade_xml_get_widget( xml, "key_name" );
	text = glade_xml_get_widget( xml, "text_to_insert" );

	n = gtk_entry_get_text( GTK_ENTRY( name ) );
	t = gtk_entry_get_text( GTK_ENTRY( text ) );

	gtk_clist_set_text( GTK_CLIST( list ), row, 0, n );
	gtk_clist_set_text( GTK_CLIST( list ), row, 1, t );

        preferences_changed();
}

/* colour tab fill in details */
static void colour_tab()
{
	GtkStyle *style;
	GdkColor *c;
	GtkWidget *widget;
	GtkWidget *picker;
	GtkWidget *box;

	GList *list;
	GList *list2;
	GList *tags = NULL;
     	ColourClass *cc;
	ColourGroup *cg;

	ColourPage *cpage;
	gint i;

	GladeXML *hxml;

	cpage = cfg->cpage;

	/* the editor colours */
	style = screem_editor_get_style();

	c = &style->base[ 0 ];
	widget = glade_xml_get_widget( xml, "editor_background" );
	gnome_color_picker_set_i16( GNOME_COLOR_PICKER( widget ),
				    c->red, c->green, c->blue, 0 );
	c = &style->text[ 0 ];
	widget = glade_xml_get_widget( xml, "editor_text" );
	gnome_color_picker_set_i16( GNOME_COLOR_PICKER( widget ),
				    c->red, c->green, c->blue, 0 );

	/* store colour backups, and init gui editing for each class */
	for( list = cpage->colour_classes; list; list = list->next ) {
		cc = (ColourClass*)list->data;
		tags = NULL;
		for( list2 = cc->groups; list2; list2 = list2->next ) {
			cg = (ColourGroup*)list2->data;
			tags = g_list_append( tags, cg->name );
			for( i = 0; i < 5; i ++ ) {
				cg->bakfore[ i ] = cg->fore[ i ];
				cg->bakback[ i ] = cg->back[ i ];
			}
		}
		cg = cc->groups->data;
		cc->current = cg;

       		hxml = glade_xml_new( cfg->glade_path, "highlighting_frame" );
		widget = glade_xml_get_widget( hxml, "groupcombo" );
		gtk_combo_set_popdown_strings( GTK_COMBO( widget ), tags );
		g_list_free( tags );
		
		/* set initial fore colour and set active if needed */
		widget = glade_xml_get_widget( hxml, "usefore" );
		picker = glade_xml_get_widget( hxml, "fore" );
		gtk_signal_connect_object( GTK_OBJECT( widget ), "toggled",
					   GTK_SIGNAL_FUNC( change_state ), 
					   (gpointer)picker );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      cc->current->fore[ 0 ] );
		gnome_color_picker_set_i16( GNOME_COLOR_PICKER( picker ),
					    cc->current->fore[ 1 ],
					    cc->current->fore[ 2 ],
					    cc->current->fore[ 3 ],
					    cc->current->fore[ 4 ] );
		
		/* set initial html back colour for unknown tags 
		   and set active if needed */
		widget = glade_xml_get_widget( hxml, "useback" );
		picker = glade_xml_get_widget( hxml, "back" );
		gtk_signal_connect_object( GTK_OBJECT( widget ), "toggled",
					   GTK_SIGNAL_FUNC( change_state ), 
					   (gpointer)picker );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      cc->current->back[ 0 ] );
		gnome_color_picker_set_i16( GNOME_COLOR_PICKER( picker ),
					    cc->current->back[ 1 ],
					    cc->current->back[ 2 ],
					    cc->current->back[ 3 ],
					    cc->current->back[ 4 ] );

		box = glade_xml_get_widget( xml, "colours_box" );
		widget = glade_xml_get_widget( hxml, "highlighting_frame" );
		gtk_frame_set_label( GTK_FRAME( widget ), cc->types->data );
		gtk_box_pack_start( GTK_BOX( box ), widget, FALSE, TRUE, 4 );
		glade_xml_signal_autoconnect( hxml );
	}
}
/* callbacks for colour tab */
void tag_colour_sel( GtkEntry *widget )
{
	gchar *group;
        GList *list;
	GtkWidget *frame;
	GtkWidget *fore;
	GtkWidget *fore_check;
	GtkWidget *back;
	GtkWidget *back_check;

	ColourPage *cpage;
	ColourClass *cc;
	ColourGroup *cg;
	
	GladeXML *hxml;

	hxml = glade_get_widget_tree( GTK_WIDGET( widget ) );
        group = gtk_entry_get_text( widget );

	frame = glade_xml_get_widget( hxml, "highlighting_frame" );

	cpage = cfg->cpage;
	cc = colour_class_from_type( GTK_FRAME( frame )->label );

	for( cg = NULL, list = cc->groups; list; list = list->next ) {
		cc->current = cg = list->data;
		if( ! strcmp( group, cg->name ) )
			break;
	}

	g_assert( cg != NULL );

	fore = glade_xml_get_widget( hxml, "fore" );
  	fore_check = glade_xml_get_widget( hxml, "usefore" );
	back = glade_xml_get_widget( hxml, "back" );
  	back_check = glade_xml_get_widget( hxml, "useback" );

        /* now set the colour selectors/checkboxes */
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( fore_check ),
				      cg->fore[ 0 ] );
        gnome_color_picker_set_i16( GNOME_COLOR_PICKER( fore ),
                                    cg->fore[ 1 ],
                                    cg->fore[ 2 ],
                                    cg->fore[ 3 ],
                                    cg->fore[ 4 ] );
        gtk_widget_set_sensitive( fore, cg->fore[ 0 ] );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( back_check ),
				      cg->back[ 0 ] );
        gnome_color_picker_set_i16( GNOME_COLOR_PICKER( back ),
                                    cg->back[ 1 ],
                                    cg->back[ 2 ],
                                    cg->back[ 3 ],
                                    cg->back[ 4 ] );
        gtk_widget_set_sensitive( back, cg->back[ 0 ] );
}

void tag_use_change( GtkWidget *widget  )
{
	GtkWidget *check;
	GtkWidget *frame;
	gboolean state;
	ColourPage *cpage;
	ColourClass *cc;
	ColourGroup *cg;
	GladeXML *hxml;

	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	hxml = glade_get_widget_tree( widget );
    
	frame = glade_xml_get_widget( hxml, "highlighting_frame" );

	cpage = cfg->cpage;
	cc = colour_class_from_type( GTK_FRAME( frame )->label );
	cg = cc->current;

	check = glade_xml_get_widget( hxml, "usefore" );

        if( widget == check )
                cg->fore[ 0 ] = state;
        else 
                cg->back[ 0 ] = state;

        preferences_changed();
}

void tag_col_change( GnomeColorPicker *cp, guint r, guint g, guint b,
		     guint a )
{
	GtkWidget *fore;
     	ColourPage *cpage;
	ColourClass *cc;
	ColourGroup *cg;
	GladeXML *hxml;
	GtkWidget *frame;

	hxml = glade_get_widget_tree( GTK_WIDGET( cp ) );
    
	frame = glade_xml_get_widget( hxml, "highlighting_frame" );

	cpage = cfg->cpage;
	cc = colour_class_from_type( GTK_FRAME( frame )->label );
	cg = cc->current;

	fore = glade_xml_get_widget( hxml, "fore" );

        if( GTK_WIDGET( cp ) == fore ) {
                cg->fore[ 1 ] = r;
                cg->fore[ 2 ] = g;
                cg->fore[ 3 ] = b;
                cg->fore[ 4 ] = a;
        } else {
                cg->back[ 1 ] = r;
                cg->back[ 2 ] = g;
                cg->back[ 3 ] = b;
                cg->back[ 4 ] = a;
        }

        preferences_changed();
}

/* helper tab fill in details */
static void helper_tab()
{
	GtkWidget *widget;
	GList *list;
	Helper *helper;
	gchar *entry[ 2 ];
	gint row;

	HelperPage *hpage;

	hpage = cfg->hpage;

	widget = glade_xml_get_widget( xml, "helpers_list" );

	/* add current helpers to the clist */
	for( row = 0, list = hpage->helpers; list; list = list->next ) {
		helper =  (Helper*)list->data;
		entry[ 0 ] = helper->path;
		entry[ 1 ] = helper->mime_type;
		gtk_clist_append( GTK_CLIST( widget ), entry );
		gtk_clist_set_row_data( GTK_CLIST( widget ), row++, helper );
	}
}
/* callbacks for helper tab */
void helper_clicked( GtkCList *list, gint row, gint column,
		     GdkEventButton *event )
{
	GtkWidget *path;
	GtkWidget *mime_type;
	gchar *entry[ 2 ];

	path = glade_xml_get_widget( xml, "helper_path" );
	mime_type = glade_xml_get_widget( xml, "helper_mime_type" );

	gtk_object_set_data( GTK_OBJECT( list ), "row", (gpointer)row );

	gtk_clist_get_text( GTK_CLIST( list ), row, 0, &entry[ 0 ] );
	gtk_clist_get_text( GTK_CLIST( list ), row, 1, &entry[ 1 ] );

	gtk_entry_set_text( GTK_ENTRY( path ), entry[ 0 ] );
	gtk_entry_set_text( GTK_ENTRY( mime_type ), entry[ 1 ] );
}

void helper_add( GtkWidget *button, gpointer data )
{
	GtkWidget *list;
	GtkWidget *path;
	GtkWidget *mime_type;
	Helper *helper;
	gchar *entry[ 2 ] = { NULL, NULL };

	list = glade_xml_get_widget( xml, "helpers_list" );
	path = glade_xml_get_widget( xml, "helper_path" );
	mime_type = glade_xml_get_widget( xml, "helper_mime_type" );

	entry[ 0 ] = gtk_entry_get_text( GTK_ENTRY( path ) );
	entry[ 1 ] = gtk_entry_get_text( GTK_ENTRY( mime_type ) );
	gtk_clist_append( GTK_CLIST( list ), entry );

	helper = screem_helper_new();
	helper->path = g_strdup( entry[ 0 ] );
	helper->mime_type = g_strdup( entry[ 1 ] );
	screem_helper_add( helper );

	preferences_changed();
}

void helper_update( GtkWidget *button, gpointer data )
{
	GtkWidget *list;
	GtkWidget *path;
	GtkWidget *mime_type;
	gint row;
    	gchar *temp;

	list = glade_xml_get_widget( xml, "helpers_list" );
	path = glade_xml_get_widget( xml, "helper_path" );
	mime_type = glade_xml_get_widget( xml, "helper_mime_type" );
	row = (gint)gtk_object_get_data( GTK_OBJECT( list ), "row" );

	temp = gtk_entry_get_text( GTK_ENTRY( path ) );
	gtk_clist_set_text( GTK_CLIST( list ), row, 0, temp );
	temp = gtk_entry_get_text( GTK_ENTRY( mime_type ) );
	gtk_clist_set_text( GTK_CLIST( list ), row, 1, temp );
}

void helper_remove( GtkWidget *button, gpointer data )
{
	GtkWidget *list;
	GtkWidget *path;
	gint row;
	gchar *p;

	list = glade_xml_get_widget( xml, "helpers_list" );
	path = glade_xml_get_widget( xml, "helper_path" );
	row = (gint)gtk_object_get_data( GTK_OBJECT( list ), "row" );
	p = gtk_entry_get_text( GTK_ENTRY( path ) );

	gtk_clist_remove( GTK_CLIST( list ), row );

	screem_helper_remove( p );

        preferences_changed();
}

/* printing tab fill in details */
static void printing_tab()
{
	PrintPage *ppage;
	GtkWidget *widget;

	ppage = cfg->ppage;

	widget = glade_xml_get_widget( xml, "add_header" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      ppage->printheader );

	widget = glade_xml_get_widget( xml, "word_wrap" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      ppage->printwrap );

	widget = glade_xml_get_widget( xml, "line_numbers" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      ppage->printlines );

	widget = glade_xml_get_widget( xml, "lines" );
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
				    (gfloat)ppage->printlines );
	gtk_widget_set_sensitive( widget, ppage->printlines );

	widget = glade_xml_get_widget( xml, "portrait" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      ppage->orientation );
}
/* callbacks for printing tab */

/* java tab fill in details */
static void java_tab()
{
	GtkWidget *widget;
	JavaPage *jpage;

	jpage = cfg->jpage;

	widget = glade_xml_get_widget( xml, "compiler" );
	gtk_entry_set_text( GTK_ENTRY( widget ), jpage->compiler );
}

/* callbacks for java tab */


/* preference change notification + application functions */
void preferences_changed()
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( xml, "ok_button" );
	gtk_widget_set_sensitive( widget, TRUE );
	widget = glade_xml_get_widget( xml, "apply_button" );
	gtk_widget_set_sensitive( widget, TRUE );
}

void preferences_apply()
{
	GtkWidget *widget;

	/* init mpage here as we need it for the toggles array */
	MiscPage   *mpage = cfg->mpage;
	KeysPage   *kpage;
	ColourPage *cpage;
	HelperPage *hpage;
	PrintPage  *ppage;
	JavaPage   *jpage;

	GtkWidget *colourpicker;
	GtkStyle  *style;
	GdkColor  *c;
	guint16 junk;
    	GList *list;
	GList *list2;
	ColourClass *cc;
	ColourGroup *cg;
	gint i;

	KeyCombo *combo;
	gint rows;

	gpointer toggles[] = {
		"syntax_highlighting", &mpage->highlighting,
		"inline_tagging", &mpage->inline_tagging,
		"auto_indent", &mpage->auto_indent,
		"intelligent_close", &mpage->intelliclose,
		"notebook_check", &mpage->notebookbar,
		"viewbar_check", &mpage->viewbar,
		"auto_load", &mpage->auto_load,
		"auto_open", &mpage->auto_open,
		"parse", &mpage->parse,
		NULL
	};

	kpage = cfg->kpage;
	cpage = cfg->cpage;
	hpage = cfg->hpage;
	ppage = cfg->ppage;
	jpage = cfg->jpage;

	widget = glade_xml_get_widget( xml, "ok_button" );
	gtk_widget_set_sensitive( widget, FALSE );
	widget = glade_xml_get_widget( xml, "apply_button" );
	gtk_widget_set_sensitive( widget, FALSE );

	/* default filename */
	widget = glade_xml_get_widget( xml, "default_filename" );
	mpage->default_filename = 
		g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

	/* default charset */
	widget = glade_xml_get_widget( xml, "default_charset" );
	mpage->default_charset = 
		g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

	/* default dtd */
	widget = glade_xml_get_widget( xml, "default_dtd" );
	mpage->default_dtd = 
		g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

	/* font */
	widget = glade_xml_get_widget( xml, "font_name_entry" );
	g_free( mpage->font_name );
	mpage->font_name = 
		g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
      	mpage->font = gdk_font_load( mpage->font_name );
	
	/* toggle button settings */
	for( i = 0; toggles[ i ]; i ++ ) {
		widget = glade_xml_get_widget( xml, toggles[ i++ ] );
		if( widget )
			*((gboolean*)toggles[ i ]) = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	}

	/* entity insertion */
	widget = glade_xml_get_widget( xml, "entity_on" );
	mpage->ent = ON;
	if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		mpage->ent = OFF;
		widget = glade_xml_get_widget( xml, "entity_off" );
       		if( ! gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
			mpage->ent = BY_SET;
	}

	/* set the new key combos */
	widget = glade_xml_get_widget( xml, "key_list" );
	for( list = kpage->keys; list; ) {
		combo = (KeyCombo*)list->data;
		g_free( combo->key_name );
		g_free( combo->text );
		list = g_list_remove( list, combo );
		g_free( combo );
	}
	kpage->keys = NULL;
	for( rows = GTK_CLIST( widget )->rows; rows; rows -- ) {
		combo = g_new( KeyCombo, 1 );
		gtk_clist_get_text( GTK_CLIST( widget ), rows -1, 0, 
				    &combo->key_name );
		gtk_clist_get_text( GTK_CLIST( widget ), rows -1, 1, 
				    &combo->text );
		combo->key_name = g_strdup( combo->key_name );

		/* set action: if combo->text matches one of the
		   presets set it to that, else its just text */
		combo->action = is_action( combo->text );
		if( combo->action == TEXT )
			combo->text = g_strdup( combo->text );
		else
			combo->text = g_strdup(key_actions[combo->action - 1]);
		kpage->keys = g_list_append( kpage->keys, combo );
	}

  	/* background colour */
	colourpicker = glade_xml_get_widget( xml, "editor_background" );
     	style = screem_editor_get_style();
	style->font = mpage->font;
	c = &style->base[ 0 ];

	gnome_color_picker_get_i16( GNOME_COLOR_PICKER( colourpicker ),
				    &c->red, &c->green, &c->blue, &junk );
	cpage->back[ 0 ] = c->red;
	cpage->back[ 1 ] = c->green;
	cpage->back[ 2 ] = c->blue;

	/* text colour */
	colourpicker = glade_xml_get_widget( xml, "editor_text" );

	c = &style->text[ 0 ];

	gnome_color_picker_get_i16( GNOME_COLOR_PICKER( colourpicker ),
				    &c->red, &c->green, &c->blue, &junk );
	cpage->fore[ 0 ] = c->red;
	cpage->fore[ 1 ] = c->green;
	cpage->fore[ 2 ] = c->blue;

	screem_editor_set_style( style );

	/* set the new highlighting colours */
	for( list = cpage->colour_classes; list; list = list->next ) {
		cc = (ColourClass*)list->data;
		for( list2 = cc->groups; list2; list2 = list2->next ) {
			cg = (ColourGroup*)list2->data;
			for( i = 0; i < 5; i ++ ) {
				cg->bakfore[ i ] = cg->fore[ i ];
				cg->bakback[ i ] = cg->back[ i ];
			}
			cg->fg.red = cg->fore[ 1 ];
			cg->fg.green = cg->fore[ 2 ];
			cg->fg.blue = cg->fore[ 3 ];
			cg->bg.red = cg->back[ 1 ];
			cg->bg.green = cg->back[ 2 ];
			cg->bg.blue = cg->back[ 3 ];
		}
	}

	/* set the printer settings */
	widget = glade_xml_get_widget( xml, "add_header" );
	ppage->printheader = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "word_wrap" );
	ppage->printwrap = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "lines" );
	ppage->printlines = 0;
	if( GTK_WIDGET_IS_SENSITIVE( widget ) )
		ppage->printlines = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "portrait" );
	ppage->orientation = 
		gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
       
	/* get the java compiler */
	widget = glade_xml_get_widget( xml, "compiler" );
	g_free( jpage->compiler );
	jpage->compiler = g_strdup( gtk_entry_get_text( GTK_ENTRY(widget) ) );

	/* save them */
	save_preferences();

	/* if we are editing a page then we need to reinsert it all in
	   the possibly new font / highlighting colours */
	screem_editor_changed();

	/* update helper toolbar */
	screem_helpers_build_toolbar();
}

void preferences_done()
{
	GList *list;
	GList *list2;
	GtkWidget *widget;
	ColourClass *cc;
	ColourGroup *cg;
	gint i;
	
	/* restore the highlighting colours */
	for( list = cfg->cpage->colour_classes; list; list = list->next ) {
		cc = (ColourClass*)list->data;
		for( list2 = cc->groups; list2; list2 = list2->next ) {
			cg = (ColourGroup*)list2->data;
			for( i = 0; i < 5; i ++ ) {
				cg->fore[ i ] = cg->bakfore[ i ];
				cg->back[ i ] = cg->bakback[ i ];
			}
		}
	}

	widget = glade_xml_get_widget( xml, "preferences_dialog" );

	gtk_widget_destroy( widget );
	cfg->box = NULL;
}

void preferences_clicked( GtkWidget *widget, gint button )
{
	if( button < 2 )
		preferences_apply();
	if( button == 0 || button == 2 )
		preferences_done();
}

void preferences_page_selected( GtkWidget *list, gint row, gint col )
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( xml, "prefsbook" );
        gtk_notebook_set_page( GTK_NOTEBOOK( widget ), row );

}

ColourClass* colour_class_from_type( const gchar *type )
{
	ColourClass *class;
	GList *list;

	class = NULL;
	
	for( list = cfg->cpage->colour_classes; list; list = list->next ) {
		class = (ColourClass*)list->data ;
		if( g_list_find_custom( class->types, (gpointer)type, 
					(GCompareFunc)strcmp ) )
			break;
		class = NULL;
	}

	return class;
}

static KeyAction is_action( gchar *text )
{
	gint i;

	for( i = 0; key_actions[ i ]; i ++ )
		if( ! strcmp( text, key_actions[ i ] ) )
			return (KeyAction)(i + 1);

	return TEXT;
}

static void syntax_files_io( gboolean save )
{
           struct dirent **namelist;
	   gint num;
	   const gchar *type;
	   gchar *user_syntax_path;
	   struct stat s;

	   gchar cwd[ 16384 ];

	   getcwd( cwd, 16384 );

	   user_syntax_path = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
					   ".screem", G_DIR_SEPARATOR_S,
					   "syntax", G_DIR_SEPARATOR_S, NULL );
	   if( stat( user_syntax_path, &s ) < 0 ) {
		   if( errno != ENOENT ) {
			   g_free( user_syntax_path );
			   return;
		   }
		   if( ! copy_dir( SYNTAX_PATH, user_syntax_path, FALSE ) ) {
			   g_print( "Failed to install %s as %s\n",
				    SYNTAX_PATH, user_syntax_path );
			   g_free( user_syntax_path );
			   return;
		   }
	   }

	   chdir( user_syntax_path );
	   g_free( user_syntax_path );

	   num = scandir( ".", &namelist, 0, alphasort );

	   while( num > 0 ) {
		   num --;
#ifndef HAVE_GNOME_VFS
		   type = gnome_mime_type_or_default( namelist[ num ]->d_name,
						      "no" );
#else
		   type = gnome_vfs_mime_type_from_name_or_default( namelist[ num ]->d_name, "no" );
#endif
		   if( ! strcmp( "application/x-screem-syntax-file", type ) ) {
			   if( ! save ) {
				   load_syntax_file( namelist[ num ]->d_name );
			   } else {
				   /* save it then */
				   save_syntax_file( namelist[ num ]->d_name );
			   }
		   }
		   g_free( namelist[ num ] );
	   }

	   chdir( cwd );
}

static void load_syntax_file( gchar *file )
{
	xmlDocPtr doc;
	xmlNodePtr node;
	xmlNodePtr subnode;
	ColourClass *cc;
	ColourGroup *cg;
	ColourPage *cpage;
	gchar *value;

	ScreemEditorMenu *menu;
	GtkWidget *menuitem;
	GtkWidget *menu2;
	gchar *name;
	gchar *callback;
	gchar *data;
	gpointer cbfunc;
	GModule *self;

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	doc = xmlParseFile( file );

	if( ! doc ) {
		g_print( "Corrupt syntax file: %s\n", file );
		return;
	}

	cpage = cfg->cpage;

	/* get types */
	node = xml_search_child( doc->root, "type" );

	if( ! node ) {
		g_print( "Corrupt syntax file: %s (no types)\n", file );
		xmlFreeDoc( doc );
		return;
	}

	cc = g_new0( ColourClass, 1 );
	do {
		cc->types = g_list_append( cc->types,
					   xmlNodeGetContent( node ) );
		node = node->next;
	} while( node && ! strcmp( "type", node->name ) );

	/* read in the menus */
	node = xml_search_child( doc->root, "menu" );
	while( node && ! strcmp( "menu", node->name ) ) {
		menu = g_new0( ScreemEditorMenu, 1 );
		menu->name = name = xml_get_value( node, "name" );
		menu->menu = gtk_menu_item_new_with_label( g_basename(name) );
		menu2 = gtk_menu_new();
		gtk_menu_item_set_submenu( GTK_MENU_ITEM( menu->menu ),
					   menu2 );
		gtk_widget_show( menu->menu );
		gtk_widget_ref( menu->menu );
		if( menu->name ) {
			cc->menus = g_list_append( cc->menus, menu );
			/* get the items */
			subnode = xml_search_child( node, "item" );
			while( subnode ) {
				name = xml_get_value( subnode, "name" );
				callback = xml_get_value( subnode, "callback");
				data = xml_get_value( subnode, "data" );
				g_module_symbol( self, callback, &cbfunc );
				g_free( callback );

				menuitem = gtk_menu_item_new_with_label(name);
				g_free( name );
				gtk_signal_connect( GTK_OBJECT( menuitem ),
						    "activate",
						    GTK_SIGNAL_FUNC( cbfunc ),
						    data );
				gtk_widget_show( menuitem );
				gtk_menu_append( GTK_MENU( menu2 ),
						 menuitem );
				subnode = subnode->next;
			}
		}
		node = node->next;
	}

	/* read in keys */
	node = xml_search_child( doc->root, "keys" );
	while( node && ! strcmp( "keys", node->name ) ) {
		callback = xml_get_value( node, "callback" );
		if( ! callback ) {
			g_print( "Corrupt syntax file: %s (no callback for key press)\n", file );
			break;
		}
		if( strcmp( "guile-script", callback ) )
			cc->keys = g_list_append( cc->keys, callback );
		else {
			/* key press callback is a guile script */
			g_free( callback );
			callback = xmlNodeGetContent( node );

			g_free( callback );  /* we don't do anything with it yet */
		}
		node = node->next;
	}

	/* read in tips */
	node = xml_search_child( doc->root, "tips" );
	while( node && ! strcmp( "tips", node->name ) ) {
		callback = xml_get_value( node, "callback");
		cc->tips = g_list_append( cc->tips, callback );
		node = node->next;
	}

	/* read in the groups */
	node = xml_search_child( doc->root, "group" );

	if( ! node ) {
		g_print( "Corrupt syntax file: %s (no groups)\n", file );
		g_free( cc );
		xmlFreeDoc( doc );
		return;
	}


	while( node ) {
		if( strcmp( node->name, "group" ) ) {
			break;
		}
		cg = g_new0( ColourGroup, 1 );
		cg->parent = cc;
		/* get the name */
		cg->name = xml_get_value( node, "name" );
		/* read in colours */
		subnode = xml_search_child( node, "fore" );
		if( subnode ) {
			value = xml_get_value( subnode, "use" );
			cg->fore[ 0 ] = atoi( value );
			g_free( value );
			value = xml_get_value( subnode, "red" );
			cg->fore[ 1 ] = atoi( value );
			g_free( value );
			value = xml_get_value( subnode, "green" );
			cg->fore[ 2 ] = atoi( value );
			g_free( value );
			value = xml_get_value( subnode, "blue" );
			cg->fore[ 3 ] = atoi( value );
			g_free( value );
			cg->fg.red = cg->fore[ 1 ];
			cg->fg.green = cg->fore[ 2 ];
			cg->fg.blue = cg->fore[ 3 ];
		}
		subnode = xml_search_child( node, "back" );
		if( subnode ) {
			value = xml_get_value( subnode, "use" );
			cg->back[ 0 ] = atoi( value );
			g_free( value );
			value = xml_get_value( subnode, "red" );
			cg->back[ 1 ] = atoi( value );
			g_free( value );
			value = xml_get_value( subnode, "green" );
			cg->back[ 2 ] = atoi( value );
			g_free( value );
			value = xml_get_value( subnode, "blue" );
			cg->back[ 3 ] = atoi( value );
			g_free( value );
			cg->bg.red = cg->back[ 1 ];
			cg->bg.green = cg->back[ 2 ];
			cg->bg.blue = cg->back[ 3 ];
		}
		/* read in end pattern */
		subnode = xml_search_child( node, "end" );
		if( subnode ) {
			value = xmlNodeGetContent( subnode );
			value = g_strstrip( value );
			cg->end = value;
		} else
			cg->end = NULL;
		/* read in patterns */
		subnode = xml_search_child( node, "pattern" );
		while( subnode ) {
			if( strcmp( "pattern", subnode->name ) )
				break;
			value = xmlNodeGetContent( subnode );
			value = g_strstrip( value );
			cg->patterns = g_list_append( cg->patterns, value );
			subnode = subnode->next;
		}

		cc->groups = g_list_append( cc->groups, cg );

		/* next group */
		node = node->next;
	}

	/* load in context groups 
	 * 
	 * context groups are used for highlighting sub modes, such as
	 * PHP or javascript content in an html document
	 *
	 */
	node = xml_search_child( doc->root, "contextgroup" );
	while( node ) {
		if( strcmp( node->name, "contextgroup" ) ) {
			break;
		}
		xml_get_value( node, "name" );
		xml_get_value( node, "mode" );
		xml_get_value( node, "start" );
		xml_get_value( node, "end" );
		node = node->next;
	}
		
	cc->current = cc->groups->data;

	xmlFreeDoc( doc );

	cpage->colour_classes = g_list_append( cpage->colour_classes, cc );
}

static void save_syntax_file( gchar *file )
{
	xmlDocPtr doc;
	xmlNodePtr node;
	xmlNodePtr fore;
	xmlNodePtr back;
	gchar *type;
	ColourPage *cpage;
	ColourClass *cc;
	ColourGroup *cg;
	gchar *value;
	gint i;
	GList *list;

	const gchar *attr[] = {
		"use", "red", "green", "blue"
	};

	doc = xmlParseFile( file );

	if( ! doc ) {
		g_print( "Corrupt syntax file: %s\n", file );
		return;
	}

	cpage = cfg->cpage;

	type = xml_get_value( doc->root, "type" );

	if( ! type ) {
		g_print( "Corrupt syntax file: %s (no type)\n", file );
		xmlFreeDoc( doc );
		return;
	}

	for(cc = NULL, list = cpage->colour_classes; list; list = list->next) {
		cc = (ColourClass*)list->data;
		if( g_list_find_custom( cc->types, (gpointer)type, 
					(GCompareFunc)strcmp ) )
			break;
	}

	g_free( type );

	if( ! list ) {
		g_print( "Unknown syntax file: %s (must be new)\n", file );
		xmlFreeDoc( doc );
		return;	
	}

	/* have the file for the sytax type, set the colours and save,
	   the groups will be in the same order so no need to check
	   the names match (unless the user has been editing them while
	   Screem is running in which case they deserve what they get) */
	node = xml_search_child( doc->root, "group" );
	for( list = cc->groups; list; list = list->next ) {
		cg = (ColourGroup*)list->data;
		fore = xml_search_child( node, "fore" );
		back = xml_search_child( node, "back" );
		for( i = 0; i < 4; i ++ ) {
			value = g_strdup_printf( "%i", cg->fore[ i ] );
			xmlSetProp( fore, attr[ i ], value );
			g_free( value );
			value = g_strdup_printf( "%i", cg->back[ i ] );
			xmlSetProp( back, attr[ i ], value );
			g_free( value );
		}
		node = node->next;
	}

	xmlSaveFile( file, doc );
	xmlFreeDoc( doc );
}
