/*  Screem:  linkWizard.c,
 *  The link wizard, converted to being a g_module  
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gmodule.h>
#include <gnome.h>

#include <glade/glade.h>

#include "site.h"
#include "page.h"
#include "editor.h"

#include "link.xpm"

extern GtkWidget *app;
extern Site *current_site;

static GladeXML *xml;
static GtkWidget *dialog = NULL;

void linkWizard( void );
void link_wizard_delete( GtkWidget *widget, GdkEvent *event, gpointer data );
void link_wizard_clicked( GtkWidget *widget, gint button, gpointer data );

#define TAG "a"
#define WIZARD_NAME "link"

static gboolean popup = FALSE;
static gchar *unknown_attributes;

/* the following is for identifying this as a wizard for the <a> tag, this
   allows it to be opened from the popup menu, and have its entries filled
   with the current settings */
G_MODULE_EXPORT const gchar *html_tag( gchar **wizard_name )
{
	*wizard_name = g_strdup( WIZARD_NAME ); 
	return g_strdup( TAG );
}

G_MODULE_EXPORT void popup_wizard( GtkMenuItem *menu, GList *data )
{
	GList *list;

	gint start, len;
	gchar *name;
	gchar *value;
	gchar *temp;

  	GtkWidget *widget;

	g_return_if_fail( data != NULL );

	popup = TRUE;
	unknown_attributes = g_strdup( "" );

	/* display wizard */
	linkWizard();

	widget = glade_xml_get_widget( xml, "frame1" );
	change_state( widget );

	/* force modal */
	gtk_window_set_modal( GTK_WINDOW( dialog ), TRUE );
	/* fill in fields: to avoid using a struct the list is formatted as
	   follows:

	   list->data = start pos;
	   list->data = length;

	   from then on:

	   list->data = attribute 1 name
	   list->next->data = attribute 1 value
	   list->next->next->data = attribute 2 name
	   list->next->next->next->data = attribute 2 value
	   .
	   .
	   .
	   etc.
	*/
	start = (gint)data->data;
	data = data->next;
	len = (gint)data->data;

	screem_editor_select_region( start, len );

	for( list = data->next; list; list = list->next ) {
		name = list->data;
		list = list->next;
		value = list->data;
		
		if( ! strcmp( name, "href" ) ) {
			temp = NULL;
			if( !strncmp( "http://", value, 
				      strlen( "http://" ) ) ) {
				temp = "http://";
				widget = glade_xml_get_widget( xml, "http" );
			} else if( ! strncmp( "ftp://", value, 
					      strlen( "ftp://" ) ) ) {
				temp = "ftp://";
				widget = glade_xml_get_widget( xml, "ftp" );
			} else if( ! strncmp( "mailto:", value, 
					      strlen( "mailto:" ) ) ) {
				temp = "mailto:";
				widget = glade_xml_get_widget( xml, "mailto" );
			} else {
				widget = glade_xml_get_widget( xml, "other" );
			}
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget),
						     TRUE);
			if( temp )
				value += strlen( temp );
			widget = glade_xml_get_widget( xml, "link_to" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );

		} else if( ! strcmp( name, "target" ) ) {
			widget = glade_xml_get_widget( xml, "target_text" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else {
			/* not handled by the wizard, add to
			   unknown_attributes variable */
			temp = unknown_attributes;
			if( ! temp )
				temp = g_strdup( "" );
			if( ! value ) {
				unknown_attributes = g_strconcat( temp, " ", 
								  name, NULL );
			} else {
				unknown_attributes = g_strconcat( temp, " ", 
								  name, "=\"",
								  value, "\"",
								  NULL);
			}
			g_free( temp );
		}
	}

	/* now we want to sit in a loop allowing gtk to continue.
	   We don't want to return as it will remove the highlight on the
	   text */
	while( dialog ) {
		if( gtk_events_pending() )
			gtk_main_iteration();
	}

	g_free( unknown_attributes );
	unknown_attributes = NULL;
	popup = FALSE;
}

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	g_print("linkWizard: check-init\n");
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
	g_print( "linkWizard: unloaded\n" );
}

G_MODULE_EXPORT void init() 
{
	GtkWidget *linkButton;
	GtkWidget *toolbar;

	GnomeUIInfo menuinfo[] = { 
		{
			GNOME_APP_UI_ITEM, N_("Link..."),
			N_("Insert a link"),
			linkWizard, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK,
			GNOME_STOCK_MENU_BLANK,
			0,
			GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_END
	};


	toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );

	/* place menu item after image under insert */
	gnome_app_insert_menus( GNOME_APP( app ),
				_("_Insert/"), menuinfo);

	/* place a button on the wizards toolbar */
	linkButton = gnome_pixmap_new_from_xpm_d( LINK_XPM );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
				 _("Link Wizard"), "", linkButton ,
				 linkWizard, 0 );

	g_print( "linkWizard: initialised\n" );
}

/******************************************************************/
void link_wizard_delete( GtkWidget *widget, GdkEvent *event, gpointer data )
{
        dialog = NULL;
}
/******************************************************************/
void link_wizard_clicked( GtkWidget *widget, gint button, gpointer data )
{
	gchar *text;
	gchar *protocol = NULL;
	gchar *target_text;
	gint pos;

	GtkWidget *toggle;
	GtkWidget *ltext;
	GtkWidget *link;
	GtkWidget *target;

	if( button < 2 ) { /* applied or ok'ed */
		toggle = glade_xml_get_widget( xml, "http" );
		if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON( toggle )) )
			protocol = "http://";

		toggle = glade_xml_get_widget( xml, "ftp" );
		if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON( toggle )) )
			protocol = "ftp://";

		toggle = glade_xml_get_widget( xml, "mailto" );
		if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON( toggle ) ))
			protocol = "mailto:";
		
		if( ! protocol )
			protocol = "";

		link = glade_xml_get_widget( xml, "link_to" );
		ltext = glade_xml_get_widget( xml, "link_text" );
		target = glade_xml_get_widget( xml, "target_text" );

		target_text = gtk_entry_get_text( GTK_ENTRY( target ) );
		if( strcmp( "", target_text ) ) {
			target_text = g_strdup_printf( " target=\"%s\"",
						       target_text );
		} else {
			target_text = g_strdup( "" );
		}

		if( ! popup ) {
			text = g_strdup_printf( "<a href=\"%s%s\"%s>%s</a>", protocol, gtk_entry_get_text( GTK_ENTRY(link) ), target_text, gtk_entry_get_text(GTK_ENTRY(ltext)) );
		} else {
			text = g_strdup_printf( "<a href=\"%s%s\"%s%s>", protocol, gtk_entry_get_text( GTK_ENTRY( link ) ), target_text, unknown_attributes );
			screem_editor_clear_selection();
		}

		pos = screem_editor_get_pos();
		screem_editor_insert( pos, text );
		screem_editor_set_pos( pos + strlen( text ) );
		g_free( text );
		g_free( target_text );
	}
	if( button == 2 || button == 0 ) { /* ok or close clicked */
                gtk_widget_destroy( widget );
		dialog = NULL;
		g_free( data );
        }
}
/******************************************************************/
void linkWizard()
{
  	Page *page;

	page = screem_site_get_current_page( current_site );
		
	g_return_if_fail( page != NULL );
	
        if( dialog ) {
                gdk_window_raise( dialog->window );
                gdk_window_show( dialog->window );
                return;
        }

	xml = glade_xml_new( GLADE_PATH"/linkWizard.glade", 
			     "link_wizard_dialog" );
	glade_xml_signal_autoconnect( xml );

	dialog = glade_xml_get_widget( xml, "link_wizard_dialog" );
}
