#ifndef EDITOR_H
#define EDITOR_H

#include <sys/types.h>
#include <regex.h>

#include "page.h"
#include "preferences.h"

typedef struct ScreemHighlightEntry {
	gchar *start;
	regex_t *rstart;

	gchar *end;
	regex_t *rend;

	gchar *name;
	GdkFont *font;
	ColourGroup *cg;

	GList *sub_entries;   /* a list of ScreemHighlightEntry 's for
				 submatches in this pattern */
} ScreemHighlightEntry;

typedef struct ScreemHighlightTable {
	GList *entries;
	regex_t *all;
} ScreemHighlightTable;

typedef struct ScreemEditorMenu {
	gchar *name;
	GtkWidget *menu;
} ScreemEditorMenu;

typedef struct ScreemEditorMode {
	GList *mime_types;  /* a list of mime types for which this
			       handles highlighting for */
	ScreemHighlightTable *table;  /* the highlighting table */
	GList *menus;      /* a list of ScreemEditorMenu structs */
	GList *keys;       /* a list of callback names to process key presses
			      with, called sequentially if not processed by
			      the previous one */
	GList *tips;      /* a list of callback names for editor tips */
} ScreemEditorMode;

typedef gboolean (*KeysCallback)(GdkEventKey *event, KeyCombo *combo );

GtkWidget *screem_editor_new( void );
void screem_editor_normal_cursor( void );
void screem_editor_close( void );
gboolean screem_editor_popup( GtkWidget *widget, GdkEventButton *event );
void screem_editor_remove_syntax_menus( Page *page );
void screem_editor_display_page( gpointer page );
void screem_editor_insert( gint pos, const gchar *text );
void screem_editor_clear( void );
GtkStyle *screem_editor_get_style( void );
void screem_editor_set_style( GtkStyle *style );
gint screem_editor_get_width( void );
gint screem_editor_get_height( void );
gint screem_editor_get_length( void );
gchar *screem_editor_get_text( gint from, gint len );
gboolean screem_editor_has_selection( gint *start, gint *end );
void screem_editor_select_region( gint start, gint len );
void screem_editor_cut( void );
void screem_editor_copy( void );
void screem_editor_paste( void );
void screem_editor_paste_encoded( void );
void screem_editor_clear_selection( void );
void screem_editor_set_pos( gint pos );
gint screem_editor_get_pos( void );
void screem_editor_delete_forward( gint pos, gint len );
void screem_editor_enable( void );
void screem_editor_disable( void );
void screem_editor_buffer_text( void );
void screem_editor_insert_markup( gchar *open_element, gchar *close_element );
void screem_editor_insert_attribute( gchar *attribute );
void screem_editor_insert_file( gchar *filename );
void screem_editor_changed( void );

void screem_editor_text_inserted( GtkEditable *editable, const gchar *text,
				  gint length, gint *pos );
void screem_editor_text_removed( GtkEditable *editable, gint start, gint end );
void screem_editor_undo( void );
void screem_editor_redo( void );

void screem_editor_select_context( void );


void screem_editor_build_highlight_tables( void );

ScreemEditorMode* screem_highlight_mode_new( ScreemHighlightTable *table,
						GList *types );
ScreemHighlightTable* screem_highlight_table_new( GList *entries );
GList *screem_highlight_entry_new( GList *list, ColourGroup *cg );
regex_t* screem_highlight_compile_regex( const gchar *pattern, gint flags );

ScreemEditorMode* screem_editor_get_mode( const gchar *mime_type);

void screem_editor_indent( gchar *text, gint pos );
void screem_editor_goto_line( void );

void screem_editor_encode_text( void );

gchar* screem_editor_get_word( gint pos, gchar *text );

void screem_editor_set_charset( Page *page );

#endif
