/*  Screem:  skel-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */
#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include <glade/glade.h>
#include <string.h>

#include "screem-skel-plugin.h"

/* setup your plugin here */

/* plugin name should only include a-zA-Z */
static const gchar *plugin_name = "ScreemSsiWizard";

static const gchar *authors[] = {
	"David A Knight <david@screem.org>",
	NULL
};
static const gchar *displayed_name = N_( "Screem SSI Wizard" );
static const gchar *description = N_( "A wizard for inserting server side include directives" );
static const gchar *version = "2.0.0";

/* add any per instance data items here */
struct ScreemSkelPluginPrivate {
	GladeXML *xml;
	GtkWidget *dialog;
};

typedef enum {
	CONFIGURATION = 0,
	VARIABLES,
	EXECUTE,
	ECHO,
	INCLUDE,
	FILE_INFORMATION,
	CONDITIONALS
} SsiDirectives;

static void ssi_wizard_display( GtkAction *action, gpointer user_data );
static void ssi_wizard_create( ScreemPlugin *plugin );
static void ssi_wizard_clicked( GtkWidget *dialog, gint button, 
		gpointer data );

/**
 * setup:
 * 
 * this function will be called once for each window,
 * you should add any actions / ui here, eg.
 * 
 * screem_plugin_add_action( plugin, name, label, tip, stock_id,
 * 				callback, error );
 * screem_plugin_add_menu( plugin, path, action, error );
 * screem_plugin_add_toolbar( plugin, path, action, error );
 *
 *
 * to insert text into the current page being edited your callbacks
 * should make use of
 * screem_plugin_get_cursor_position( plugin )
 * screem_plugin_set_cursor_position( plugin, pos )
 * screem_plugin_insert( plugin, pos, text, length, indent )
 * 
 **/
static gboolean setup( ScreemPlugin *plugin )
{
	GError *error;
	gboolean ret;

	error = NULL;
	
	ret = screem_plugin_add_interface( plugin, "SsiWizard",
				_( "SSI Wizard" ),
				_( "Insert a Server Side Include directive" ),
				GTK_STOCK_ADD,
				G_CALLBACK( ssi_wizard_display ),
				&error  );
	if( ! ret ) {
		g_print( "Add interface error: %s\n", error->message );
		g_error_free( error );
	}	
	return ret;

}

/**
 * cleanup:
 *
 * this function will be called once for each window when
 * it is closed, you should cleanup any data items you
 * have in ScreemSkelPluginPrivate here
 **/
static void cleanup( ScreemSkelPluginPrivate *priv )
{
	gtk_widget_destroy( priv->dialog );
	g_object_unref( priv->xml );
}

static void ssi_wizard_display( GtkAction *action, gpointer user_data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	ScreemPage *page;
		
	plugin = SCREEM_PLUGIN( user_data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	page = screem_plugin_get_current_document( plugin );

	if( page ) {
		ssi_wizard_create( plugin );
		if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) {
			screem_plugin_restore_from_session( plugin,
					priv->dialog );
		}
		gtk_widget_show_all( priv->dialog );
		gdk_window_raise( priv->dialog->window );
	}
}

static void ssi_wizard_create( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;
	GtkWidget *widget;
	gint i;

	static const gchar *sizes[] = {
		"Bytes", "bytes",
		"Kb or Mb", "abbrev",
		NULL
	};

	static const gchar *execute[] = {
		"CGI", "cgi",
		"Shell", "cmd",
		NULL
	};

	static const gchar *echo[] = {
		"Date (GMT)", "DATE_GMT",
		"Date (Local)", "DATE_LOCAL",
		"Document Name", "DOCUMENT_NAME",
		"Document URI", "DOCUMENT_URI",
		"Last Modified", "LAST_MODIFIED",
		NULL
	};
	GtkListStore *model;
	GtkTreeIter it;
	GtkCellRenderer *rend;
	
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;

	if( ! priv->xml ) {
		priv->xml = glade_xml_new( GLADE_PATH"/ssiwizard.glade",
				"ssiwizard", NULL );
		priv->dialog = glade_xml_get_widget( priv->xml,
				"ssiwizard" );
		g_signal_connect( G_OBJECT( priv->dialog ), "response",
				G_CALLBACK( ssi_wizard_clicked ),
				plugin );
		
		widget = glade_xml_get_widget( priv->xml, 
				"directives" );
		gtk_notebook_set_show_tabs( GTK_NOTEBOOK( widget ), 
				FALSE );

		widget = glade_xml_get_widget( priv->xml, 
				"directive_type" );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );

		widget = glade_xml_get_widget( priv->xml, 
				"config_size" );
		model = gtk_list_store_new( 2, 
				G_TYPE_STRING, G_TYPE_STRING );
		for( i = 0; sizes[ i ]; i += 2 ) {
			gtk_list_store_append( model, &it );
			gtk_list_store_set( model, &it,
					0, sizes[ i ],
					1, sizes[ i + 1 ],
					-1 );
		}
		gtk_combo_box_set_model( GTK_COMBO_BOX( widget ),
				GTK_TREE_MODEL( model ) );
		g_object_unref( G_OBJECT( model ) );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
		rend = gtk_cell_renderer_text_new();
		gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), 
				rend, TRUE );
		gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
				rend, "text", 0, NULL );

		widget = glade_xml_get_widget( priv->xml, 
				"execute_menu" );
		model = gtk_list_store_new( 2, 
				G_TYPE_STRING, G_TYPE_STRING );
		for( i = 0; execute[ i ]; i += 2 ) {
			gtk_list_store_append( model, &it );
			gtk_list_store_set( model, &it,
					0, execute[ i ],
					1, execute[ i + 1 ],
					-1 );
		}
		gtk_combo_box_set_model( GTK_COMBO_BOX( widget ),
				GTK_TREE_MODEL( model ) );
		g_object_unref( G_OBJECT( model ) );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
		rend = gtk_cell_renderer_text_new();
		gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), 
				rend, TRUE );
		gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
				rend, "text", 0, NULL );

		widget = glade_xml_get_widget( priv->xml, "echo_menu" );
		model = gtk_list_store_new( 2, 
				G_TYPE_STRING, G_TYPE_STRING );
		for( i = 0; echo[ i ]; i += 2 ) {
			gtk_list_store_append( model, &it );
			gtk_list_store_set( model, &it,
					0, echo[ i ],
					1, echo[ i + 1 ],
					-1 );
		}
		gtk_combo_box_set_model( GTK_COMBO_BOX( widget ),
				GTK_TREE_MODEL( model ) );
		g_object_unref( G_OBJECT( model ) );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
		rend = gtk_cell_renderer_text_new();
		gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), 
				rend, TRUE );
		gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
				rend, "text", 0, NULL );

		glade_xml_signal_autoconnect( priv->xml );
	}
}

void ssi_wizard_type_switch( GtkWidget *widget )
{
	GladeXML *xml;
	gint page;

	xml = glade_get_widget_tree( widget );
	page = gtk_combo_box_get_active( GTK_COMBO_BOX( widget ) );

	widget = glade_xml_get_widget( xml, "directives" );
	gtk_notebook_set_current_page( GTK_NOTEBOOK( widget ), page );
}

static void ssi_wizard_clicked( GtkWidget *dialog, gint button, 
		gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	
	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( button == GTK_RESPONSE_APPLY || button == GTK_RESPONSE_OK ) {
		GtkWidget *widget;
		gint page;
		const gchar *path;
		gchar *mode;
		const gchar *name;
		const gchar *value;
		const gchar *msg;
		const gchar *timemsg;
		gchar *sizefmt;
		const gchar *expr;
		gchar *text;
		GladeXML *xml;
		GtkTreeModel *model;
		GtkTreeIter it;

		xml = priv->xml;
		text = NULL;
		widget = glade_xml_get_widget( xml, "directives" );
		page = gtk_notebook_get_current_page( GTK_NOTEBOOK( widget ) );
		switch( page ) {
		case CONFIGURATION:
			widget = glade_xml_get_widget( xml, "error_message" );
			msg = gtk_entry_get_text( GTK_ENTRY( widget ) );

			widget = glade_xml_get_widget( xml, "time_format" );
			timemsg = gtk_entry_get_text( GTK_ENTRY( widget ) );

			widget = glade_xml_get_widget( xml, "config_size" );
			model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
			gtk_combo_box_get_active_iter( GTK_COMBO_BOX( widget ), &it );
			gtk_tree_model_get( model, &it, 
					1, &sizefmt, -1 );
			text = g_strconcat( "<!--#config errmsg=\"", msg,
					    "\" sizefmt=\"", sizefmt,
					    "\" timefmt=\"", timemsg,
					    "\" -->", NULL );
			g_free( sizefmt );
			break;
		case VARIABLES:
			widget = glade_xml_get_widget( xml, "print_var" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strdup( "<!--#printenv -->" );
			} else {
				widget = glade_xml_get_widget( xml,
							       "var_name" );
				name = gtk_entry_get_text( GTK_ENTRY(widget) );
				widget = glade_xml_get_widget( xml,
							       "var_value" );
				value = gtk_entry_get_text( GTK_ENTRY(widget));
				text = g_strconcat( "<!--#set var=\"",
						    name,
						    " \" value=\"",
						    value,
						    "\" -->", NULL );
			}
			break;
		case EXECUTE:
			widget = glade_xml_get_widget( xml, "exec_path" );
			path = gtk_entry_get_text( GTK_ENTRY( widget ) );
			widget = glade_xml_get_widget( xml, "execute_menu" );
			model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
			gtk_combo_box_get_active_iter( GTK_COMBO_BOX( widget ), &it );
			gtk_tree_model_get( model, &it, 
					1, &mode, -1 );
			text = g_strconcat( "<!--#exec ", mode,
					    "=\"", path, "\" -->", NULL );
			g_free( mode );
			break;
		case ECHO:
			widget = glade_xml_get_widget( xml, "echo_menu" );
			model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
			gtk_combo_box_get_active_iter( GTK_COMBO_BOX( widget ), &it );
			gtk_tree_model_get( model, &it, 
					1, &mode, -1 );
			text = g_strconcat( "<!--#echo var=\" ", mode,
					    "\" -->", NULL );
			g_free( mode );
			break;
		case INCLUDE:
			widget = glade_xml_get_widget( xml, "include_path" );
			path = gtk_entry_get_text( GTK_ENTRY( widget ) );
			text = g_strconcat( "<!--#include virtual=\"",
					    path,
					    "\" -->", NULL );
			break;
		case FILE_INFORMATION:
			widget = glade_xml_get_widget( xml, "info_path" );
			path = gtk_entry_get_text( GTK_ENTRY( widget ) );
			widget = glade_xml_get_widget( xml, "info_size" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				mode = "fsize";
			} else {
				mode = "flastmod";
			}
			text = g_strconcat( "<!--#", mode,
					    " virtual=\"", path, "\" -->",
					    NULL );
			break;
		case CONDITIONALS:
			widget = glade_xml_get_widget( xml, "endif" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strdup( "<!--#endif -->" );
			}
			widget = glade_xml_get_widget( xml, "else" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strdup( "<!--#else -->" );
			}
			widget = glade_xml_get_widget( xml, "expression" );
			expr = gtk_entry_get_text( GTK_ENTRY( widget ) );
			widget = glade_xml_get_widget( xml, "elif" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strconcat( "<!--#elif expr=\"",
						    expr, "\" -->", NULL );
			}
			widget = glade_xml_get_widget( xml, "if" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				text = g_strconcat( "<!--#if expr=\"",
						    expr, "\" -->", NULL );
			}
			break;
		default:
			text = NULL;
			break;
		}
		if( text ) {
			guint pos;
			
			pos = screem_plugin_get_cursor_position( plugin );
			screem_plugin_insert( plugin, pos, text,
					strlen( text ), TRUE );
			pos += strlen( text );
			screem_plugin_set_cursor_position( plugin, pos );
			g_free( text );
		}
	}
	screem_plugin_store_in_session( plugin, dialog );
	
	if( button != GTK_RESPONSE_APPLY ) {
		gtk_widget_hide( dialog );
        }
}

/* There should be no need to change any code below here */
enum {
	ARG_0
};

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass );
static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin );
static void screem_skel_plugin_finalize( GObject *object );

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_PLUGIN

static gpointer parent_class;

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_skel_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );
}

static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin )
{
	skel_plugin->priv = g_new0( ScreemSkelPluginPrivate, 1 );
	SCREEM_PLUGIN( skel_plugin )->setup = setup;
}

static void screem_skel_plugin_finalize( GObject *object )
{
	ScreemSkelPlugin *skel_plugin;
	ScreemSkelPluginPrivate *priv;
	
	skel_plugin = SCREEM_SKEL_PLUGIN( object );
	priv = skel_plugin->priv;

	cleanup( priv );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static GType screem_skel_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSkelPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_skel_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSkelPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_skel_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       plugin_name,
					       &info, 0 );
	}

	return type;
}

static ScreemSkelPlugin *screem_skel_plugin_new( void )
{
	ScreemSkelPlugin *skel_plugin;

	skel_plugin = SCREEM_SKEL_PLUGIN( g_object_new( SCREEM_TYPE_SKEL_PLUGIN, 
				"name", plugin_name,
				NULL ) );

	return skel_plugin;
}

G_MODULE_EXPORT void get_details( ScreemPluginDetails **ret )
{
	ScreemPluginDetails *details;

	details = g_new0( ScreemPluginDetails, 1 );
	details->name = plugin_name;
	details->displayed_name = displayed_name;
	details->authors = authors;
	details->description = description;
	details->version = version;
	details->create = screem_skel_plugin_new;
	details->api_version = SCREEM_PLUGIN_REQUIRED_VERSION;

	*ret = details;
}

