//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [N] = CL_colMult(coef,M)
// Multiplication of column vectors by coefficients
//
// Calling Sequence
// [N] = CL_colMult(coef,M)
//
// Description
// <itemizedlist><listitem>
// Multiplies column vectors by coefficients. 
// <para> The output matrix <emphasis role="bold">N</emphasis> is such that: 
// <emphasis role="bold">N</emphasis>(:,j) = <emphasis role="bold">coef</emphasis>(j) * <emphasis role="bold">M</emphasis>(:,j) </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// coef: Coefficients (1 x N)
// M: Matrix (P x N)
// N: Matrix (P x N) such that: N(:,j) =  coef(j) * M(:,j)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_unitVector
// CL_colNorm
//
// Examples
// coef = [ 1 2 3];
// v = [ 1 2 3 ; 4 5 6 ; 7 8 9];
// w = CL_colMult(coef,v)


// Declarations:


// Code:

if (typeof(M) ~= 'constant')
  CL__error('Invalid type of first argument');
end

if (size(coef,2) ~= size(M,2))
  CL__error("Invalid number of columns in input arguments (coef or v)");
end

if (size(coef,1) ~= 1)
  CL__error("Invalid number of rows in input arguments (coef)");
end

N = M .* (ones(M(:,1)) * coef);

endfunction
