//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [ecc,pom] = CL_op_frozenOrbit(sma,inc,er,mu,zonals)
// Eccentricity and argument of perigee for a frozen orbit
//
// Calling Sequence
// [ecc,pom] = CL_op_frozenOrbit(sma,inc,[,er[,mu[,zonals]]])
//
// Description
// <itemizedlist><listitem>
// This function computes the eccentricity and argument of periapsis of a "frozen" orbit (so that the mean value of  
// eccentricity and argument of periapsis remain constant over time). 
// Freezing the orbit is possible by balancing the effects of harmonic even and odd terms of the potential (J1=0, J2, J3, ...).  
// The argument of periapis found is 90 or 270 degrees. The eccentricity is usually small: of the order of 1.e-3 for the Earth. 
// <para> Note that up 50 first zonals terms are used due to limited algorithm validation even if a greater number of terms is given as input.</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// sma: semi major axis [m] (1xN or 1x1) 
// inc: inclination [rad] (1xN or 1x1)
// er: (optional) equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) vector of zonals coefficients J1 to Jn to be used (default is %CL_j1jn(1:3)) (1 x n)
// ecc: mean eccentricity (1xN)
// pom: mean argument of periapsis [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// sma = [%CL_eqRad+350.e3 %CL_eqRad+700.e3];
// inc = CL_deg2rad([51.6 91.6]);
// [ecc,pom] = CL_op_frozenOrbit(sma,inc)



	function [eccg,pomg] = CL_op_perGelJ2(a,inc,ipom,er,mu,j2,j3)	
	
	//calcul du vecteur eccentricite correspondant a un perigee gele au sens
	//des termes J2 et J3 du potentiel terrestre

	//initialisation



	eccg = zeros(a);  //creation of variable eccg with arbitrary values
	pomg = zeros(a);  //creation of variable pomg with arbitrary values
	sininc = sin(inc);  //sinus of inclinations
	cosinc = cos(inc);  //cosinus of inclinations

	//argument of periapsis
	i1 = find(ipom==1); //find argument of periapsis 90deg
	i2 = find(ipom==2); //find argument of periapsis 270deg
	i3 = find(ipom~=1 & ipom~=2); //invalid argument of periapsis
	pomg(i1) = CL_deg2rad(90);
	pomg(i2) = CL_deg2rad(270);
	pomg(i3) = 0; //for invalid argument of periapsis set values to 0 and return (0 values are set at end of function)
	eccg(i3) = 0;

	//computation of coefficients a1, a2, a3, a4 for
	//eccentricity equation a1*e**3+a2*e**2+a3*e+a4=0
	xn = CL_kp_params('mm',a,mu); //mean motion
	a1 = -0.75*xn .* (er./a).^2 .* j2 .* sininc .* (1-5*(cosinc).^2);
	a2 = 1.5*xn .* (er./a).^3 .* j3 .* (1-8.75*(sininc).^2 .*(cosinc).^2);
	a3 = -a1;
	a4 = 1.5*xn .* (er./a).^3 .* j3 .* (sininc).^2 .* (1.25*(sininc).^2-1);

	//argument of periapsis at 270 deg (ipom=2)
	a2(i2) = -a2(i2);
	a4(i2) = -a4(i2);

	//computation of coefficients xp and xq for equation
	//of reduced eccentricity e1**3+3*xp*e1+2*xq=0
	xp = (3*a1 .* a3 - a2.^2) ./ (9*a1.^2);
	xq = a2.^3 ./ (27*a1.^3) - (a2.*a3) ./ (6*a1) + a4./(2*a1);

	discr = xp.^3+xq.^2;  //calculation of discriminant

	//calculation of equation roots
	xr = abs(xq)./xq.*sqrt(abs(xp));  //parameter xr allowing roots calculation
	d1 = find(discr>0); //positif discriminant -> single real root
	d2 = find(~(discr>0));  //negatif discriminant -> three real roots

	phi0 = zeros(a);
	phi = zeros(a);
	ecc0 = zeros(a);
	ecc1 = zeros(a);
	ecc2 = zeros(a);
	ecc3 = zeros(a);

	//if discr>0

	  phi0(d1) = (xq(d1)./xr(d1).^3);
	  phi(d1) = log(phi0(d1)+sqrt(phi0(d1).^2-1));
	  ecc0(d1) = (-2*xr(d1).*cosh(phi(d1)./3));
	  //test on validity of results
	  i_test_ok = find((ecc0>0)&(ecc0<1)   &(discr>0));
	  i_test_notok = find(~((ecc0>0)&(ecc0<1))   &(discr>0));
	  eccg(i_test_ok) = ecc0(i_test_ok);
	  pomg(i_test_notok) = 0; //if test not ok set values to 0 and return (0 values are set at end of function)
	  eccg(i_test_notok) = 0;

	//else if ~discr>0

	  phi(d2) = acos(xq(d2)./xr(d2).^3);
	  ecc1(d2) = (-2*xr(d2).*cos(phi(d2)./3));
	  ecc2(d2) = (2*xr(d2).*cos((%pi-phi(d2))./3));
	  ecc3(d2) = (2*xr(d2).*cos((%pi+phi(d2))./3));
	  //calculation of results (lower value between 0 and 1)
	  lecc1 = ((ecc1>0)&(ecc1<1)   & ~(discr>0));
	  lecc2 = ((ecc2>0)&(ecc2<1)   & ~(discr>0));
	  lecc3 = ((ecc3>0)&(ecc3<1)   & ~(discr>0));

	  i_B = find( (lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) ); //B
		if i_B~=[]
		  eccg(i_B) = min( ecc1(i_B) , ecc2(i_B) , ecc3(i_B) );
		end
	  i_D = find( ~(lecc2) & ~(lecc1) & ~(lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) );//D
		eccg(i_D) = ecc3(i_D);
	  i_E = find( ~(lecc3) & lecc2 & ~(lecc1) & ~(lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) );//E
		eccg(i_E) = ecc2(i_E);
	  i_eE = find( lecc3 & lecc2 & ~(lecc1) & ~(lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) );//eE
		if i_eE~=[]
		  eccg(i_eE) = min( ecc2(i_eE) , ecc3(i_eE) );
		end
	  i_G = find( ~(lecc3) & ~(lecc2) & lecc1 & ~(lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) );//G
		eccg(i_G) = ecc1(i_G);
	  i_eG = find( lecc3 & ~(lecc2) & lecc1 & ~(lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) );//eG
		if i_eG~=[]
		  eccg(i_eG) = min( ecc1(i_eG) , ecc3(i_eG) );
		end
	  i_eF = find( lecc2 & lecc1 & ~(lecc1&lecc2&lecc3) & (lecc1|lecc2|lecc3) & ~(discr>0) );//eF
		if i_eF~=[]
		  eccg(i_eF) = min( ecc1(i_eF) , ecc1(i_eF) );
		end
	  i_eA = find( ~(lecc1|lecc2|lecc3) & ~(discr>0) );//eA
		pomg(i_eA) = 0;
		eccg(i_eA) = 0;

	//end if discr

	//set to 0 values of test not ok (i_test_notok), unknown periapsis argument (i3) and i_eA
	if i_eA~=[]
	  pomg(i_eA) = 0;
	  eccg(i_eA) = 0;
	  warning('0 values given (test e>0 and e<1 false for negative discriminant of reduced eccentricity equation)');
	end
	if i_test_notok~=[]
	  pomg(i_test_notok) = 0;
	  eccg(i_test_notok) = 0;
	  warning('0 values given (test e>0 and e<1 false for positive discriminant of reduced eccentricity equation)');
	end
	if i3~=[]
	  pomg(i3) = 0;
	  eccg(i3) = 0;
	  warning('0 values given for invalid arguments or periapsis (not equal to 90 or 270 deg)');
	end

	//NOT VECTORIZED VERSION:
	//Correspondance between indexs found (i_A,i_eA,d1,d2,...) and if and else is shown.

	//if discr>0  //positif discriminant -> single real root  //d1
	//
	//  phi0 = (xq/xr^3);
	//  phi = log(phi0+sqrt(phi0^2-1));
	//  ecc0 = (-2*xr*cosh(phi/3));
	//  //test on validity of results
	//  if ((ecc0>0)&(ecc0<1))  //i_test_ok
	//    eccg = ecc0;
	//  else  //i_test_notok
	//    pomg = 0;
	//    eccg = 0;
	//    return
	//  end //end test
	//
	//else  //negatif discriminant -> three real roots  //d2
	//
	//  phi = acos(xq/xr^3);
	//  ecc1 = (-2*xr*cos(phi/3));
	//  ecc2 = (2*xr*cos((%pi-phi)/3));
	//  ecc3 = (2*xr*cos((%pi+phi)/3));
	//  //calculation of results (lower value between 0 and 1)
	//  lecc1 = ((ecc1>0)&(ecc1<1));
	//  lecc2 = ((ecc2>0)&(ecc2<1));
	//  lecc3 = ((ecc3>0)&(ecc3<1));
	//  if ((lecc1)|(lecc2)|(lecc3))  //i_A
	//    if ((lecc1)&(lecc2)&(lecc3))  //i_B
	//      eccg = min(ecc1,ecc2,ecc3);
	//    else //i_eB
	//      if ~lecc1 //i_C
	//        if ~lecc2 //i_D
	//          eccg = ecc3;
	//        else  //i_eD
	//          if ~lecc3 //i_E
	//            eccg = ecc2;
	//          else  //i_eE
	//            eccg = min(ecc2,ecc3);
	//          end
	//        end
	//      else  //i_eC
	//        if ~lecc2 //i_F
	//          if ~lecc3 //i_G
	//            eccg = ecc1;
	//          else  //i_eG
	//            eccg = min(ecc1,ecc3);
	//          end
	//        else  //i_eF
	//          eccg = min(ecc1,ecc2);
	//        end
	//      end
	//    end
	//  else  //i_eA
	//    pomg = 0;
	//    eccg = 0;
	//    return;
	//  end
	//
	//end //end discr


	endfunction

	function [eccg,pom] = CL_op_perGelJN(dga,xi,nmaxdeg,er,jn)  //acabada de fer pero no provada

	//calcule l'eccentricite correspondant a un perigee gele
	//au sense des termes J2 a Jn du potentiel terrestre pour une orbite (a,i) donnee


	// Declarations:


	// Code:

	nj_max = 50

	if nmaxdeg>nj_max | nmaxdeg<3 then CL__error('not valid nmaxdeg'); end

	ta = zeros(nj_max,nj_max/2+1)
	te = zeros(nj_max,nj_max/2)
	td = zeros(nj_max,nj_max/2)

	eccg = zeros(dga)
	pom = CL_deg2rad(90)*ones(dga)

	xj = jn(1:nmaxdeg)

	sini = sin(xi)
	cosi2 = cos(xi).^2

	den = zeros(dga)
	xnum = zeros(dga)

	rapa = er./dga

	//calcul de eg:
	for i=2:nmaxdeg

	  if pmodulo(i,2)==0

		k = i./2
		sx = 0
		sy = 0
		sz = 0
		for l=0:k

		  f1 = factorial(2*(i-l))
		  f2 = factorial(l)
		  f3 = factorial(i-l)
		  f4 = factorial(i./2-l)
		  dif = 2.*(i-l)
		  expo2 = 2.^dif

		  prepa = ((-1).^l).*f1 ./ (f2.*f3.*expo2)
		  ta(i,l+1) = prepa./f4.^2

		  expsi = sini.^(i-2.*l)
		  sy = sy + expsi.*ta(i,l+1)
		  if l~=k
			expsi2 = sini.^(i-2.*l-2)
			sx = sx + (k-l).*expsi2.*ta(i,l+1)
			if k~=1
			  f5 = factorial((i-2)./2-l)
			  f6 = factorial((i+2)./2-l)
			  te(i,l+1) = prepa./(f5.*f6)
			  sz = sz + expsi.*te(i,l+1)
			end
		  end

		end

		sx = 2.*cosi2.*sx
		sy = -(k.*(2.*k+1))*sy
		sz = -((k-1).*(2*k-1)).*sz

		den = den + (sx+sy+sz).*(rapa.^i).*xj(i)

	  else

		k = (i-1)./2
		som = 0

		for l=0:k

		  expsi3 = sini.^(i-2*l)
		  f1 = factorial(2*(i-l))
		  f2 = factorial(l)
		  f3 = factorial(i-l)
		  dif = 2*(i-l)
		  expo2 = 2.^dif
		  f4 = factorial((i-1)./2-l)
		  f5 = factorial((i+1)./2-l)
		  td(i,l+1) = (((-1).^l).*f1) ./ (f2.*f3.*f4.*f5.*expo2)
		  som = som + expsi3.*td(i,l+1)
		end

		xnum = xnum + 2.*k.*som.*(rapa.^i).*xj(i)

	  end //end if

	  if i>=3
		eccg = xnum./den
	  end

	end

	endfunction


// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_mu')) then global %CL_mu; end;
if(~exists('%CL_j1jn')) then global %CL_j1jn; end;

// Code:


Na = size(sma,2); //number of semi major axis given
Ninc = size(inc,2); //number of inclinations given

N = max(Na,Ninc); //max of sizes

coherence = (Na==N|Na==1) & (Ninc==N|Ninc==1); //check coherence for vectorization
if ~coherence then CL__error('bad size of input arguments'); end

if N~=1
  if Na==1 then sma=sma*ones(1,N); end
  if Ninc==1 then inc=inc*ones(1,N); end
end

if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:3); end

Nzonals = length(zonals);
if(Nzonals > 3)
  if(Nzonals>50)
    Nzonals=50;
  end
  model='j2jn';
elseif(Nzonals ==3)
  model='j2j3'
elseif(Nzonals <3)
  CL__error('zonals must be a vector of size 3 or more');
end

//calculations
if model=='j2j3'
  [ecc,pom] = CL_op_perGelJ2(sma,inc,ones(1,N),er,mu,zonals(2),zonals(3))
elseif model=='j2jn'
  [ecc,pom] = CL_op_perGelJN(sma,inc,Nzonals,er,zonals(1:Nzonals))
end


endfunction
