//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [M] = CL_rot_quat2matrix(q)
// Quaternion to transformation matrix
//
// Calling Sequence
// [M]=CL_rot_quat2matrix(q)
//
// Description
// <itemizedlist><listitem>
// Computes the transformation matrix that corresponds to a given quaternion. 
// </listitem>
// <listitem>
// Notes: 
// <para> - See <link linkend="CL_rot_angles2matrix">CL_rot_angles2matrix</link> for 
// conventions regarding the definition of the transformation matrix. </para>
// <para> - See <link linkend="Conventions">Conventions</link> or <link linkend="CL_rot_defQuat">CL_rot_defQuat</link> for more details on quaternions. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// q: Quaternion (dim N)
// M: Transformation matrix (3x3xN)
//
// Bibliography
// 1 Mecanique Spatiale - CNES Cepadues 1995, Tome I, 7.2.2.3 Les quaternions, eq.15
// 2 James R. Wertz, Spacecraft attitude determination and control (volume 73 of Astrophyisics and Space Science Library), D. Reidel Publishing Company, 1980, appendix D-E
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
// CL_rot_matrix2quat
// CL_rot_angles2matrix
//
// Examples
// ang = CL_deg2rad(10)
// M1 = CL_rot_angles2matrix(3,ang)
// q1 = CL_rot_matrix2quat(M1)
// q2 = CL_rot_eul2quat(0,0,ang)  // same as q1
// M2 = CL_rot_quat2matrix(q2)    // same as M1
//

// Declarations:


// Code:

[lhs,rhs] = argn();

if (typeof(q) ~= "CLquat")
  M=[];
  CL__error("Input is not a quaternion.");
end

q4 = q.r;
q1 = q.i(1,:);
q2 = q.i(2,:);
q3 = q.i(3,:);

n = size(q);

M = hypermat([3 3 n], ...
       [ q1.^2 - q2.^2 - q3.^2 + q4.^2; 2*(q1.*q2 - q3.*q4); ...
         2*(q1.*q3 + q2.*q4); 2*(q1.*q2 + q3.*q4); ...
         (-q1.^2 + q2.^2 - q3.^2 + q4.^2); ...
         2*(q2.*q3 - q1.*q4); 2*(q1.*q3 - q2.*q4); ...
         2*(q2.*q3 + q1.*q4); ...
         (-q1.^2 - q2.^2 + q3.^2 + q4.^2)]);

if ( n == 1 ) 
  M = M(:,:,1);  // avoids hypermat if only one quaternion
end;  

endfunction


