/* ---------------------------------------------------------------------
*
*  -- ScaLAPACK routine (version 1.0) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     November 17, 1996
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "pblas.h"

void pstrmv_( uplo, trans, diag, n, A, ia, ja, desc_A, X, ix, jx,
              desc_X, incx )
/*
*  .. Scalar Arguments ..
*/
   F_CHAR      diag, trans, uplo;
   int         * ia, * incx, * ix, * ja, * jx, * n;
/* ..
*  .. Array Arguments ..
*/
   int         desc_A[], desc_X[];
   float       A[], X[];
{
/*
*  Purpose
*  =======
*
*  PSTRMV performs one of the distributed matrix-vector operations
*
*  sub( X ) := sub( A )*sub( X )   or   sub( X ) := sub( A )'*sub( X ),
*
*  where sub( A ) denotes A(IA:IA+N-1,JA:JA+N-1),
*
*        sub( X ) denotes X(IX,JX:JX+N-1) if INCX = M_X,
*                         X(IX:IX+N-1,JX) if INCX = 1 and INCX <> M_X,
*
*  sub( X ) is an N element vector and  sub( A ) is an N-by-N unit, or
*  non-unit, upper or lower triangular distributed matrix.
*
*  Notes
*  =====
*
*  Each global data object is described by an associated description
*  vector.  This vector stores the information required to establish
*  the mapping between an object element and its corresponding process
*  and memory location.
*
*  Let A be a generic term for any 2D block cyclicly distributed array.
*  Such a global array has an associated description vector descA.
*  In the following comments, the character _ should be read as
*  "of the global array".
*
*  NOTATION        STORED IN      EXPLANATION
*  --------------- -------------- --------------------------------------
*  DT_A   (global) descA[ DT_ ]   The descriptor type.  In this case,
*                                 DT_A = 1.
*  CTXT_A (global) descA[ CTXT_ ] The BLACS context handle, indicating
*                                 the BLACS process grid A is distribu-
*                                 ted over. The context itself is glo-
*                                 bal, but the handle (the integer
*                                 value) may vary.
*  M_A    (global) descA[ M_ ]    The number of rows in the global
*                                 array A.
*  N_A    (global) descA[ N_ ]    The number of columns in the global
*                                 array A.
*  MB_A   (global) descA[ MB_ ]   The blocking factor used to distribu-
*                                 te the rows of the array.
*  NB_A   (global) descA[ NB_ ]   The blocking factor used to distribu-
*                                 te the columns of the array.
*  RSRC_A (global) descA[ RSRC_ ] The process row over which the first
*                                 row of the array A is distributed.
*  CSRC_A (global) descA[ CSRC_ ] The process column over which the
*                                 first column of the array A is
*                                 distributed.
*  LLD_A  (local)  descA[ LLD_ ]  The leading dimension of the local
*                                 array.  LLD_A >= MAX(1,LOCr(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCr( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCc( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCr() and LOCc() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC:
*          LOCr( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCc( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*  An upper bound for these quantities may be computed by:
*          LOCr( M ) <= ceil( ceil(M/MB_A)/NPROW )*MB_A
*          LOCc( N ) <= ceil( ceil(N/NB_A)/NPCOL )*NB_A
*
*  Because vectors may be seen as particular matrices, a distributed
*  vector is considered to be a distributed matrix.
*
*  It is required that the row and column block sizes of A are equal,
*  i.e MB_A = NB_A as well as MOD(IA-1,MB_A) = MOD(JA-1,NB_A).
*
*  If INCX = M_X, the process column having the first entries of
*  sub( X ) must also contain the first block of sub( A ). Moreover, the
*  column blocksize of X must be equal to the column blocksize of A, i.e
*  NB_X = NB_X = NB_A. Finally, the column offset of sub( X ) must be
*  equal to the row and column offset of sub( A ), i.e MOD(JX-1,NB_X) =
*  MOD(JA-1,NB_A) = MOD(IA-1,MB_A).
*
*  If INCX = 1 and INCX <> M_X, the process row having the first entries
*  of sub( X ) must also contain the first block of sub( A ).  Moreover,
*  the row blocksize of X must be equal to the column blocksize of A,
*  i.e MB_X = MB_X = NB_A. Finally, the row offset of sub( X ) must be
*  equal to the row and column offset of A, i.e MOD(IX-1,MB_X) =
*  MOD(JA-1,NB_A) = MOD(IA-1,MB_A).
*
*  Parameters
*  ==========
*
*  UPLO    (global input) pointer to CHARACTER
*          On entry, UPLO specifies whether the distributed matrix
*          sub( A ) is an upper or lower triangular matrix as follows:
*
*             UPLO = 'U' or 'u'   sub( A ) is an upper triangular
*                                 distributed matrix.
*
*             UPLO = 'L' or 'l'   sub( A ) is a lower triangular
*                                 distributed  matrix.
*
*  TRANS   (global input) pointer to CHARACTER
*          On entry, TRANS specifies the operation to be performed as
*          follows:
*
*             TRANS = 'N' or 'n'   sub( x ) := sub( A )*sub( x ).
*
*             TRANS = 'T' or 't'   sub( x ) := sub( A )'*sub( x ).
*
*             TRANS = 'C' or 'c'   sub( x ) := sub( A )'*sub( x ).
*
*  DIAG    (global input) pointer to CHARACTER
*          On entry, DIAG specifies whether or not sub( A ) is unit
*          triangular as follows:
*
*             DIAG = 'U' or 'u'   sub( A ) is assumed to be unit
*                                 triangular.
*
*             DIAG = 'N' or 'n'   sub( A)  is not assumed to be unit
*                                 triangular.
*
*  N       (global input) pointer to INTEGER
*          The order of the distributed matrix sub( A ). N >= 0.
*
*  A       (local input) REAL pointer into the local memory
*          to an array of dimension (LLD_A,LOCc(JA+N-1) containing the
*          local pieces of the distributed matrix sub( A ). Before
*          entry with  UPLO = 'U' or 'u', the leading N-by-N upper
*          triangular part of the distributed matrix sub( A ) must
*          contain the upper triangular distributed matrix and the
*          strictly lower triangular part of sub( A ) is not referenced.
*          Before entry with UPLO = 'L' or 'l', the leading N-by-N
*          lower triangular part of the distributed matrix sub( A ) must
*          contain the lower triangular distributed matrix and the
*          strictly upper triangular part of sub( A ) is not referenced.
*          Note that when  DIAG = 'U' or 'u', the diagonal elements of
*          sub( A ) are not referenced either, but are assumed to be
*          unity.
*
*  IA      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix A to operate on.
*
*  JA      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix A to operate on.
*
*  DESCA   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix A.
*
*  X       (local input/local output) REAL array
*          containing the local pieces of a distributed matrix of
*          dimension of at least
*              ( (JX-1)*M_X + IX + ( N - 1 )*abs( INCX ) )
*          This array contains the entries of the distributed vector
*          sub( X ). On exit sub( X ) is overwritten with the
*          tranformed distributed vector sub( X ).
*
*  IX      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix X to operate on.
*
*  JX      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix X to operate on.
*
*  DESCX   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix X.
*
*  INCX    (global input) pointer to INTEGER
*          The global increment for the elements of X. Only two values
*          of INCX are supported in this version, namely 1 and M_X.
*
*  =====================================================================
*
*  .. Local Scalars ..
*/
   char        DiagA, TrA, UploA, xdist;
   int         iacol, iarow, icoffa, ictxt, iia, iix, info, iroffa,
               iszcmp, ixcol, ixpos, ixrow, jja, jjx, lcm, lcmp, lcmq,
               locincx, mullen, mycol, myrow, nca, nn, np0, nprow,
               npcol, nra, nq0, szcmp, tmp1, tmp2, wksz;
/* ..
*  .. PBLAS Buffer ..
*/
   float       * buff;
/* ..
*  .. External Functions ..
*/
   void        blacs_gridinfo_();
   void        pbchkmat();
   void        pbchkvect();
   void        pberror_();
   char        * getpbbuf();
   F_VOID_FCT  pbstrmv_();
   F_INTG_FCT  ilcm_();
/* ..
*  .. Executable Statements ..
*
*  Get grid parameters
*/
   ictxt = desc_A[CTXT_];
   blacs_gridinfo_( &ictxt, &nprow, &npcol, &myrow, &mycol );
/*
*  Test the input parameters
*/
   info = 0;
   if( nprow == -1 )
      info = -(800+CTXT_+1);
   else
   {
      DiagA = Mupcase( F2C_CHAR( diag )[0] );
      UploA = Mupcase( F2C_CHAR( uplo )[0] );
      TrA = Mupcase( F2C_CHAR( trans )[0] );
      iroffa = (*ia-1) % desc_A[MB_];
      icoffa = (*ja-1) % desc_A[NB_];
      pbchkmat( *n, 4, *n, 4, *ia, *ja, desc_A, 8, &iia, &jja,
                &iarow, &iacol, nprow, npcol, myrow, mycol,
                &nra, &nca, &info );
      pbchkvect( *n, 4, *ix, *jx, desc_X, *incx, 12, &iix, &jjx,
                 &ixrow, &ixcol, nprow, npcol, myrow, mycol, &info );
      if( info == 0 )
      {
         if( ( UploA != 'U' ) && ( UploA != 'L' ) )
            info = -1;
         else if( ( TrA != 'N' ) && ( TrA != 'T' ) && ( TrA != 'C' ) )
            info = -2;
         else if( ( DiagA != 'U' ) && ( DiagA != 'N' ) )
            info = -3;
         else if( iroffa != icoffa )
            info = -7;
         else if( desc_A[NB_] != desc_A[MB_] )
            info = -(800+NB_+1);
         if( *incx == desc_X[M_] )
         {
            if( ( ( (*jx-1) % desc_X[NB_] ) != icoffa ) ||
                ( ixcol != iacol ) )
               info = -7;
            else if( desc_A[NB_] != desc_X[NB_] )
               info = -(1200+NB_+1);
         }
         else if( ( *incx == 1 ) && ( *incx != desc_X[M_] ) )
         {
            if( ( ( (*ix-1) % desc_X[MB_] ) != icoffa ) ||
                ( ixrow != iarow ) )
               info = -6;
            else if( desc_A[NB_] != desc_X[MB_] )
               info = -(1200+MB_+1);
         }
         else
         {
            info = -13;
         }
         if( ictxt != desc_X[CTXT_] )
            info = -(1200+CTXT_+1);
      }
   }
   if( info )
   {
      pberror_( &ictxt, "PSTRMV", &info );
      return;
   }
/*
*  Quick return if possible.
*/
   if( *n == 0 )
      return;
/*
*  Figure out the arguments to be passed to pbstrmv
*/
   nn = *n + iroffa;
   mullen = MULLENFAC * desc_A[MB_];
   lcm = ilcm_( &nprow, &npcol );
   if( *incx == desc_X[M_] )
   {
      lcmp = lcm / nprow;
      lcmq = lcm / npcol;
      ixpos = ixrow;
      xdist = 'R';
      locincx = desc_X[LLD_];
      tmp1 = nn / desc_A[MB_];
      np0 = MYROC0( tmp1, nn, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );

      tmp2 = lcmq * desc_A[MB_];
      iszcmp = CEIL( mullen, tmp2 );
      szcmp = iszcmp * iszcmp * lcmq*desc_A[MB_] * lcmp*desc_A[MB_];

      tmp1 = np0 / desc_A[MB_];
      tmp1 = MYROC0( tmp1, np0, desc_A[MB_], lcmp );
      wksz = np0 + nq0 + MAX( szcmp, tmp1 );
   }
   else
   {
      lcmp = lcm / nprow;
      lcmq = lcm / npcol;
      ixpos = ixcol;
      xdist = 'C';
      locincx = 1;
      tmp1 = nn / desc_A[MB_];
      np0 = MYROC0( tmp1, nn, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );

      tmp2 = lcmq * desc_A[MB_];
      iszcmp = CEIL( mullen, tmp2 );
      szcmp = iszcmp * iszcmp * lcmq*desc_A[MB_] * lcmp*desc_A[MB_];

      tmp1 = nq0 / desc_A[NB_];
      tmp1 = MYROC0( tmp1, nq0, desc_A[NB_], lcmq );
      wksz = np0 + nq0 + MAX( szcmp, tmp1 );
   }
   buff = (float *)getpbbuf( "PSTRMV", wksz*sizeof(float) );
/*
*  Call PB-BLAS routine
*/
   pbstrmv_( &ictxt, uplo, trans, diag, C2F_CHAR( &xdist ), n,
             &desc_A[MB_], &iroffa, &A[iia-1+(jja-1)*desc_A[LLD_]],
             &desc_A[LLD_], &X[iix-1+(jjx-1)*desc_X[LLD_]], &locincx,
             &iarow, &iacol, &ixpos, C2F_CHAR( NO ), &mullen, buff );
}
