// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "ruler.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Ruler::Ruler(const Ruler& a) : BaseLine(a)
{
  p3 = a.p3;
  coordSystem = a.coordSystem;
  skyFrame = a.skyFrame;
  distSystem = a.distSystem;
  distFormat = a.distFormat;
  dist = a.dist;
}

Ruler::Ruler(FrameBase* p, const Vector& ptr1, const Vector& ptr2,
	     CoordSystem sys, SkyFrame sky, 
	     CoordSystem distsys, SkyFormat distfor,
	     const char* clr, int wth, const char* fnt, const char* txt, 
	     unsigned short prop, const char* cmt,
	     const List<Tag>& tg, const List<CallBack>& cb) 
  : BaseLine(p, ptr1, ptr2, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  coordSystem = sys;
  skyFrame = sky;
  distSystem = distsys;
  distFormat = distfor;
  dist = 0;

  strcpy(type,"ruler");
  handle = new Vector[2];
  numHandle = 2;

  updateBBox();
}

void Ruler::updateBBox()
{
  FitsImage* ptr = parent->findFits(center);
  center = (p2-p1)/2 + p1;
  angle  = (p2-p1).angle();
  if (parent->isIIS())
    angle = -angle;

  // calc P3
  Vector a = ptr->mapFromRef(p1,coordSystem,skyFrame);
  Vector b = ptr->mapFromRef(p2,coordSystem,skyFrame);
  p3 = ptr->mapToRef(Vector(b[0],a[1]),coordSystem,skyFrame);

  // calc Dist
  dist = ptr->mapDistFromRef(p2, p1, distSystem, distFormat);

  // bound marker
  bbox = BBox(p1 * parent->refToCanvas, p2 * parent->refToCanvas);
  bbox.bound(p3 * parent->refToCanvas);

  // generate handles in canvas coords
  handle[0] = p1 * parent->refToCanvas;
  handle[1] = p2 * parent->refToCanvas;

  // make room for text
  if (font) {
    Vector v = (p2-p1)/2 + p1;

    ostringstream str;
    distToStr(str);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    char* buf = (char*)str.str().c_str();
    int width = Tk_TextWidth(font, buf, strlen(buf));

    Vector ll = v * Translate(-width/2.,-metrics.descent)*parent->refToCanvas;
    Vector ur = v * Translate(width/2., metrics.ascent) * parent->refToCanvas;
    
    bbox.bound(ll);
    bbox.bound(ur);
  }

  // make room for handles/arrows
  bbox.expand(5);

  // calculate overall bbox
  calcAllBBox();
}

void Ruler::updateCoords(const Matrix& mx)
{
  p3*=mx;
  BaseLine::updateCoords(mx);
}

void Ruler::ps(int mode)
{
  Marker::ps(mode);

  Vector a = p1 * parent->refToCanvas;
  Vector b = p2 * parent->refToCanvas;
  Vector c = p3 * parent->refToCanvas;

  ostringstream str;

  str << "[] 0 setdash" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "[8 3] 0 setdash" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "[8 3] 0 setdash" << endl;

  str << "newpath " 
      << b.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  psArrow(a, (b-a).normalize());
  psArrow(b, (a-b).normalize());

  psDist(mode, (b-a)/2 + a);
}

int Ruler::isOn(const Vector& v, const Vector& v1, const Vector& v2)
{
  // v : canvas coords
  // v1: ref coords
  // v2: ref coords

  // do this in canvas coords, not ref coords

  Vector l1 = v1 * parent->refToCanvas;
  Vector l2 = v2 * parent->refToCanvas;
  double a = (l2-l1).angle();

  Matrix m = Translate(-l1) * Rotate(a);
  Vector end = l2*m;
  Vector vv = v*m;
  return (vv[0]>0 && vv[0]<end[0] && vv[1]>-3 && vv[1]<3);
}

int Ruler::isIn(const Vector& vv)
{
  // test to see if point is on edge, if so, considered inside

  if (isOn(vv,p1,p2) || isOn(vv,p1,p3) || isOn(vv,p2,p3))
    return 1;

  /*
    v[0]-- x value of point being tested
    v[1]-- y value of point being tested

    This algorithm is from "An Introduction to Ray Tracing", Academic Press,
    1989, edited by Andrew Glassner, pg 53
    -- a point lies in a polygon if a line is extended from the point to 
    infinite
    in any direction and the number of intersections with the polygon is odd.
    This is valid for both concave and convex polygons.
    Points on a vertex are considered inside.
    Points on a edge are considered inside.
  */

  // analysis in ref coords
  Vector v = vv * parent->canvasToRef;

  int crossings = 0;   // number of crossings
  int sign;

  // for all edges

  for (int i=0; i<3; i++) {
    Vector v1, v2;
    switch (i) {
    case 0:
      v1 = p1-v;
      v2 = p2-v;
      sign = ((v1[1])>=0) ? 1 : -1; // init sign
      break;
    case 1:
      v1 = p2-v;
      v2 = p3-v;
      break;
    case 2:
      v1 = p3-v;
      v2 = p1-v;
      break;
    }

    int nextSign = (v2[1]>=0) ? 1 : -1; // sign holder for p2

    if (sign != nextSign) {
      if (v1[0]>0 && v2[0]>0)
	crossings++;
      else if (v1[0]>0 || v2[0]>0) {
	if (v1[0]-(v1[1]*(v2[0]-v1[0])/(v2[1]-v1[1])) > 0)
	  crossings++;
      }
      sign = nextSign;
    }
  }

  return fmod(float(crossings),float(2)) ? 1 : 0; // if odd, point is inside
}

void Ruler::setCoordSystem(CoordSystem sys, SkyFrame sky, 
			   CoordSystem d, SkyFormat f)
{
  coordSystem = sys;
  skyFrame = sky;
  distSystem = d;
  distFormat = f;
  updateBBox();
}

// Private

void Ruler::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }

  Vector a = (p1 * mx).round();
  Vector b = (p2 * mx).round();
  Vector c = (p3 * mx).round();

  setLineNoDash();
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	    (int)b[0], (int)b[1]);
  renderArrow(drawable, a, (b-a).normalize());
  renderArrow(drawable, b, (a-b).normalize());

  setLineDash();
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], 
	    (int)c[0], (int)c[1]);
  XDRAWLINE(display, drawable, gc, (int)b[0], (int)b[1], 
	    (int)c[0], (int)c[1]);

  renderDist(drawable, (b-a)/2 + a, mode);
}

void Ruler::renderDist(Drawable drawable, Vector v, RenderMode mode)
{
  ostringstream str;
  distToStr(str);

  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }
  setLineNoDash();

  XSetFont(display, gc, Tk_FontId(font));
  if (font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    char* buf = (char*)str.str().c_str();
    int width = Tk_TextWidth(font, buf, strlen(buf));

    Vector t = v * Translate(-width/2., -metrics.descent);
    XDrawString(display, drawable, gc, (int)t[0], (int)t[1], 
		buf, strlen(buf));
  }
}

void Ruler::psDist(int mode, Vector v)
{
  ostringstream vstr;
  ostringstream str;

  distToStr(vstr);

  // set color
  psColor(mode, colorName);

  if (font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
    Tcl_DStringFree(&psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    char* vbuf = (char*)vstr.str().c_str();
    int width = Tk_TextWidth(font, vbuf, strlen(vbuf));

    Vector t = v;
    str << t.TkCanvasPs(parent->canvas) << " moveto" << endl
	<< '(' << psQuote(vbuf) << ')' << endl
	<< "dup stringwidth pop 2 div 0 exch sub " 
	<< metrics.descent << " rmoveto show" << endl
	<< ends;
  }
  str << ends;

  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

void Ruler::distToStr(ostream& str)
{
  FitsImage* ptr = parent->findFits(center);
  switch (distSystem) {
  case IMAGE:
    str << dist << " img";
    break;
  case PHYSICAL:
    str << dist << " phy";
    break;
  case AMPLIFIER:
    str << dist << " amp";
    break;
  case DETECTOR:
    str << dist << " det";
    break;
  default:
    if (ptr->hasWCS(distSystem)) {
      if (ptr->hasWCSEqu(distSystem)) {
	switch (distFormat) {
	case DEGREES:
	  str << dist << " deg";
	  break;
	case ARCMIN:
	  str << dist << '\'';
	  break;
	case ARCSEC:
	  str << dist << '"';
	  break;
	}
      }
      else
	str << dist << " lin";
    }
  }
}

// list

void Ruler::list(ostream& str, CoordSystem sys, SkyFrame sky,
		 SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      {
	Vector v1 = ptr->mapFromRef(p1,sys);
	Vector v2 = ptr->mapFromRef(p2,sys);
	str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] << ','
	    << v2[0] << ',' << v2[1] << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSEqu(sys)) {
	  switch (format) {
	  case DEGREES:
	    {
	      Vector v1 = ptr->mapFromRef(p1,sys,sky);
	      Vector v2 = ptr->mapFromRef(p2,sys,sky);
	      str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
		  << ',' << v2[0] << ',' << v2[1] << ')';
	    }
	    break;
	  case SEXAGESIMAL:
	    {
	      char buf[64];
	      char ra1[16], ra2[16];
	      char dec1[16], dec2[16];
	      {
		ptr->mapFromRef(p1,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra1 >> dec1;
	      }
	      {
		ptr->mapFromRef(p2,sys,sky,format,buf,64);
		string x(buf);
		istringstream wcs(x);
		wcs >> ra2 >> dec2;
	      }
	      str << type << '(' << ra1 << ',' << dec1 << ',' 
		  << ra2 << ',' << dec2 << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v1 = ptr->mapFromRef(p1,sys);
	  Vector v2 = ptr->mapFromRef(p2,sys);
	  str << type << '(' << setprecision(8) << v1[0] << ',' << v1[1] 
	      << ',' << v2[0] << ',' << v2[1] << ')';
	}
      }
    }

    listPost(str, conj, strip);
  }
}

void Ruler::listPost(ostream& str, int conj, int strip)
{
  if (conj)
    str << " ||";

  FitsImage* ptr = parent->findFits(center);
  str << " ruler=";
  listCoordSystem(str, coordSystem, skyFrame, ptr->hasWCSEqu(coordSystem));
  str << ' ';
  listDistSystem(str, distSystem, distFormat, ptr->hasWCSEqu(distSystem));
  listProperties(str, 0);
}
