/*
 *	Copyright (c) 1999-2003 Smithsonian Astrophysical Observatory
 */

#include <funtools.h>
#include <wcs.h>
#include <tl.h>
#include <word.h>

#define MAXROW 8192
static int maxrow=MAXROW;

extern char *optarg;
extern int optind;

typedef struct evstruct{
  double x, y;
  double ox, oy;
  short nfile;
} *Ev, EvRec;

typedef struct filestruct{
  struct filestruct *next;
  Fun fun;
  struct WorldCoor *wcs;
  int tltyp[2];
  double tlmin[2];
  double tlmax[2];
  double binsiz[2];
  int nfile;
} *MFile, MFileRec;

#ifdef ANSI_FUNC
void ConvertEvents(MFile ifile, MFile ofile, MFile file1, Ev ebuf, int nrow,
		   char *wmode, char *fmode, char *iomode, int imwcs, int put)
#else
void ConvertEvents(ifile, ofile, file1, ebuf, nrow,
		   wmode, fmode, iomode, imwcs, put)
     MFile ifile;
     MFile ofile;
     MFile file1;
     Ev ebuf;
     int nrow;
     char *wmode;
     char *fmode;
     char *iomode;
     int imwcs;
     int put;
#endif
{
  int i;
  int offscl;
  double dval1, dval2;
  Ev ev=NULL;

  for(i=0; i<nrow; i++){
    if( wmode || fmode ){
      ev = ebuf+i;
    }
    /* do wcs translation, if necessary */
    if( wmode ){
      /* save old coord, might have to write it back out */
      ev->ox = ev->x;
      ev->oy = ev->y;
      /* wcs convert all but the first file (which is the base file) */
      if( ifile != file1 ){
	if( imwcs ){
	  /* convert physical pixels to image */
	  ev->x = tlp2i(ev->x,
			ifile->tlmin[0], ifile->binsiz[0], ifile->tltyp[0]);
	  ev->y = tlp2i(ev->y,
			ifile->tlmin[1], ifile->binsiz[1], ifile->tltyp[1]);
	}
	/* convert image pixels to ra/dec using wcs */
	pix2wcs(ifile->wcs, ev->x, ev->y, &dval1, &dval2);
	/* convert ra/dec to image pixels using base wcs */
	wcs2pix(file1->wcs, dval1, dval2, &(ev->x), &(ev->y), &offscl);
	if( imwcs ){
	  /* convert pixels back to physical in base system */
	  ev->x = tli2p(ev->x,
			file1->tlmin[0], file1->binsiz[0], file1->tltyp[0]);
	  ev->y = tli2p(ev->y,
			file1->tlmin[1], file1->binsiz[1], file1->tltyp[1]);
	}
	/* if output coords are not float, we need to round before update */
	switch( ofile->tltyp[0] ){
	case 'D':
	case 'E':
	  break;
	default:
	  if( ev->x >=0 ) ev->x += 0.5;
	  else ev->x -= 0.5;
	  break;
	}
	switch( ofile->tltyp[1] ){
	case 'D':
	case 'E':
	  break;
	default:
	  if( ev->y >=0 ) ev->y += 0.5;
	  else ev->y -= 0.5;
	  break;
	}
      }
    }
    /* save file number, if necessary */
    if( fmode ){
      ev->nfile = ifile->nfile;
    }
  }
  if( put ){
    /* write to output file */
    if( (put=FunTableRowPut(ofile->fun, ebuf, nrow, 0, iomode)) != nrow ){
      gerror(stderr, "expected to write %d events; only wrote %d\n",
	     nrow, put);
    }
  }
}

#ifdef ANSI_FUNC
int 
main (int argc, char **argv)
#else
int
main(argc, argv)
     int argc;
     char **argv;
#endif
{
  int i;
  int c;
  int got;
  int imwcs;
  int iargs;
  int nrow;
  int nfile;
  int itype;
  int tlen, maxlen;
  int bin[2];
  char *bincols[2];
  char onames[2][SZ_LINE];
  char colname[SZ_LINE];
  char *s;
  char *iname, *oname, *curname, *tname=NULL;
  char *mode=NULL, *fmode=NULL, *xmode=NULL, *wmode=NULL;
  double tlmin0, tlmin1, tlmax0, tlmax1;
  Ev ev, ebuf;
  EvRec tev;
  Fun ifun, ofun, cfun;
  MFile ifiles, ofile, cfile, tfile;

  /* exit on gio errors */
  if( !getenv("GERROR")  )
    setgerror(2);

  /* get maxrow,if user-specified */
  if( (s=getenv("FUN_MAXROW")) != NULL )
    maxrow = atoi(s);

  /* we want the args in the same order in which they arrived, and
     gnu getopt sometimes changes things without this */
  putenv("POSIXLY_CORRECT=true");

  /* set colname to something silly. it won't actually be added to the
     file unless -c <name> is on the command line, because the mode is
     NULL by default */
  strcpy(colname, "???");
  strcpy(onames[0], "???");
  strcpy(onames[1], "???");

  /* assume we will use image wcs */
  imwcs = 1;

  /* process switch arguments */
  while ((c = getopt(argc, argv, "f:wxIT")) != -1){
    switch(c){
    case 'f':
      fmode = "w";
      strcpy(colname, optarg);
      break;
    case 'w':
      wmode = "rw";
      break;
    case 'x':
      xmode = "w";
      wmode = "rw";
      break;
    case 'I':
      imwcs = 1;
      break;
    case 'T':
      imwcs = 0;
      break;
    default:
      break;
    }
  }

  iargs = argc - optind - 1;
  /* make sure we have something to do */
  if( iargs < 1 ){
    fprintf(stderr,
	"usage: %s [-w|-x] [-f colname] iname1 iname2 ...  oname\n", argv[0]);
    exit(1);
  }

  /* input files are in separate args -- concat them to make one input */
  maxlen = SZ_LINE;
  tname = xmalloc(maxlen);
  if( mode ){
    strcpy(tname, mode);
    strcat(tname, " ");
    tlen = strlen(tname);
  }
  else{
    *tname = '\0';
    tlen = 0;
  }
  for(i=optind; i<optind+iargs; i++){
    tlen += strlen(argv[i])+1;
    if( tlen >= maxlen ){
      maxlen += SZ_LINE;
      tname = xrealloc(tname, maxlen);
    }
    strcat(tname, argv[i]);
    strcat(tname, " ");
  }
  iname = tname;
  oname = argv[argc-1];

  /* open and preprocess the input files */
  if( !(ifun = FunOpen(iname, "rc", NULL)) )
    gerror(stderr, "can't FunOpen input file (or find extension): %s\n",
	   iname);

  /* get data type */
  FunInfoGet(ifun, FUN_TYPE, &itype, 0);

  /* open the output FITS image, preparing to copy input params */
  if( !(ofun = FunOpen(oname, "w", ifun)) )
    gerror(stderr, "can't FunOpen output file: %s\n", oname);

  /* allocate record for the base input file */
  ifiles = (MFile)xcalloc(1, sizeof(MFileRec));
  ifiles->fun = ifun;
  ifiles->nfile = 1;
  /* allocate record for the output file */
  ofile = (MFile)xcalloc(1, sizeof(MFileRec));
  ofile->fun = ofun;
  /* get info for wcs translations */
  if( wmode ){
    FunInfoGet(ifun, FUN_BINOFFS, bin, FUN_FNAME, &curname, 0);
    /* get WCS for the base input file */
    if( imwcs )
      FunInfoGet(ifun, FUN_WCS, &(ifiles->wcs), 0);
    else
      FunInfoGet(ifun, FUN_WCS0, &(ifiles->wcs), 0);
    if( !ifiles->wcs )
      gerror(stderr, "no WCS found in %s\n", curname);
    for(i=0; i<=1; i++){
      if( bin[i] >=0 ){
	FunColumnLookup(ifun, NULL, bin[i], &(bincols[i]),
			&(ifiles->tltyp[i]), NULL, NULL, NULL, NULL);
	ifiles->tlmin[i] = 
	  FunParamGetd(ifun, "TLMIN", bin[i]+1, 0.0, &got);
	ifiles->tlmax[i] = 
	  FunParamGetd(ifun, "TLMAX", bin[i]+1, 0.0, &got);
	ifiles->binsiz[i] = 
	  FunParamGetd(ifun, "TDBIN", bin[i]+1, 1.0, &got);
	if( xmode )
	  snprintf(onames[i], SZ_LINE, "OLD_%s", bincols[i]);
      }
    }
  }
  /* get info for the rest of the files */
  cfile = ifiles;
  nfile = 1;
  /* set temp tlmin variables */
  tlmin0 = ifiles->tlmin[0];
  tlmin1 = ifiles->tlmin[1];
  tlmax0 = ifiles->tlmax[0];
  tlmax1 = ifiles->tlmax[1];
  while( 1 ){
    /* get next file in list */
    FunInfoGet(cfile->fun, FUN_NEXT, &cfun, 0);
    /* break if we have no more */
    if( !cfun ) break;
    /* make a new file record */
    cfile->next = (MFile)xcalloc(1, sizeof(MFileRec));
    /* link it to previous */
    cfile = cfile->next;
    /* fill in the blanks */
    cfile->fun = cfun;
    /* file number */
    cfile->nfile = ++nfile;
    if( wmode ){
      FunInfoGet(cfile->fun, FUN_BINOFFS, bin, FUN_FNAME, &curname, 0);
      if( imwcs )
	FunInfoGet(cfile->fun, FUN_WCS, &(cfile->wcs), 0);
      else
	FunInfoGet(cfile->fun, FUN_WCS0, &(cfile->wcs), 0);
      if( !cfile->wcs )
	gerror(stderr, "no WCS found in %s\n", curname);
      for(i=0; i<=1; i++){
	if( bin[i] >=0 ){
	  FunColumnLookup(cfile->fun, NULL, bin[i], &(bincols[i]),
			  &(cfile->tltyp[i]), NULL, NULL, NULL, NULL);
	  cfile->tlmin[i] = 
	    FunParamGetd(cfile->fun, "TLMIN", bin[i]+1, 0.0, &got);
	  cfile->tlmax[i] = 
	    FunParamGetd(cfile->fun, "TLMAX", bin[i]+1, 0.0, &got);
	  cfile->binsiz[i] = 
	    FunParamGetd(cfile->fun, "TDBIN", bin[i]+1, 1.0, &got);
	}
      }
      /* adjust the TLMIN/TLMAX of the base file to accommodate all of the
	 possible position values from this event file. This effectively
	 enlargens the size of the output image. */
      ev = &tev;
      /* this is the smallest x,y we can have */
      ev->x = cfile->tlmin[0];
      ev->y = cfile->tlmin[1];
      ConvertEvents(cfile, ofile, ifiles, ev, 1, wmode, fmode, mode, imwcs,0);
      if( ev->x < tlmin0 ){
	tlmin0 = ev->x;
        FunParamPutd(ofun, "TLMIN", bin[0]+1, tlmin0, 7, "Min. axis value",1);
      }
      if( ev->y < tlmin1 ){
	tlmin1 = ev->y;
	FunParamPutd(ofun, "TLMIN", bin[1]+1, tlmin1, 7, "Min. axis value",1);
      }
      /* this is the largest x,y we can have */
      ev->x = cfile->tlmax[0];
      ev->y = cfile->tlmax[1];
      ConvertEvents(cfile, ofile, ifiles, ev, 1, wmode, fmode, mode, imwcs,0);
      if( ev->x > tlmax0 ){
        tlmax0 = ev->x;
	FunParamPutd(ofun, "TLMAX", bin[0]+1, tlmax0, 7, "Max. axis value",1);
      }
      if( ev->y > tlmax1 ){
        tlmax1 = ev->y;
	FunParamPutd(ofun, "TLMAX", bin[1]+1, tlmax1, 7, "Max. axis value",1);
      }
    }
  }

  /* type-specific preprocessing */
  switch(itype){
  case FUN_TABLE:
  case FUN_EVENTS:
    /* if wmode, add the $x, $y col as input/output */
    if( fmode )
      FunParamPuts(ofun, "FUNFILES", 0, iname, "funmerge input filenames", 1);
    /* select columns for processing */
    if( fmode || wmode ){
      FunColumnSelect(ifun, sizeof(EvRec), "merge=update",
		      "$X",       "D", 	wmode,  FUN_OFFSET(Ev, x),
		      "$Y",       "D", 	wmode,  FUN_OFFSET(Ev, y),
		      onames[0],  "D", 	xmode,  FUN_OFFSET(Ev, ox),
		      onames[1],  "D", 	xmode,  FUN_OFFSET(Ev, oy),
		      colname,    "I", 	fmode,  FUN_OFFSET(Ev, nfile),
		      NULL);
    }
    /* there are no user components */
    else{
      /* to avoid transfering anything into user space, we don't call
	 FunColumnSelect() on the input file, but we must explicitly make
	 a select call to output the raw data */
      FunColumnSelect(ofun, 0, "merge=update", NULL);
      /* flag that we do not convert data on input or output */
      mode = "convert=false";
    }
    /* extract events from this file */
    while( (ebuf = FunTableRowGet(ifun, NULL, maxrow, mode, &nrow)) ){
      /* get fun handle for current file */
      FunInfoGet(ifun, FUN_CURRENT, &cfun, 0);
      /* find corresponding file/wcs record */
      for(cfile=ifiles; cfile; cfile=cfile->next){
	if( cfile->fun == cfun ){
	  /* translate (if necessary) and output events */
	  ConvertEvents(cfile, ofile, ifiles, ebuf, nrow, wmode, fmode, mode,
			imwcs, 1);
	  xfree(ebuf);
	  ebuf = NULL;
	  break;
	}
      }
    }
    break;
  default:
    gerror(stderr, "Sorry, only binary tables can be merged (thus far)\n");
    break;
  }

  /* close output before input, so end flush is done automatically  */
  FunClose(ofun);
  FunClose(ifun);

  /* clean up */
  for(cfile=ifiles; cfile;){
    tfile = cfile->next;
    if( cfile ) xfree(cfile);
    cfile = tfile;
  }
  if( ofile ) xfree(ofile);
  if( tname ) xfree(tname);
  return(0);
}
