 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@t-online.de)

   This file was initially copied from the hp3300 backend.
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RTS88xx chipset,
    such as HP4400C, HP4470C.
    Parts of this source were inspired by other backends.

		History:

		Version 0.18  21.11.04 13.alpha,
				- source sorted,
				- now only SANEI_USB_SUPPORT for a better overview(xfermodules removed)
				- read and verify the MainBoardID 
		Version 0.17p 02.11.04 12.alpha, source sorted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17p 02.11.04 12.alpha, source sourted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
		Version 0.17  09.03.04 9. alpha, HP3500 included
		Version 0.16  06.02.04 8. alpha, wait counting on LCD
		Version 0.15a 29.01.04 7. alpha, CCD switch moved to config file
		Version 0.15  11.01.04 6. alpha, a second CCD implemented
		Version 0.13a 21.11.04 4. alpha, an little fix included
		Version 0.12  22.10.03 third alpha, Backend name changed to HP_RTS88xx
		Version 0.11  30.08.03 second alpha
		Version 0.10  19.07.03 first alpha
*/

/*
    Core HP44x0c functions.
*/

#include <stdio.h>  /* fopen, fread, fwrite, fclose etc */
#include <stdarg.h> /* va_list for vfprintf */
#include <string.h> /* memcpy, memset */
#include <unistd.h> /* unlink */
#include <stdlib.h> /* malloc, free */
#include <math.h>   /* exp, pow */

#include "hp_rts_xfer.h"
#include "hp_rts_44x0c.h"


/****************************************************************************/
SANE_Bool Hp44x0_Wakeup(THWParams *pHWParams,TScanParams *pParams)
/****************************************************************************/
{
	SANE_Byte Reg_10, Reg_11;
	SANE_Int iHandle;

	DBG(DBG_SCAN, "Hp44x0_Wakeup....\n");
	/* prevent compiler from complaining about unused parameters */
	pParams->mode = pParams->mode;
	iHandle = pHWParams->iXferHandle;

	if (iHandle < 0) {
		DBG(DBG_SCAN, "Hp44x0_Wakeup, wrong iHandle\n");
		return SANE_FALSE;
	}
	/* write magic startup sequence */
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	/*  test_reg( 0xb2,0x00,20 );*/
	if ( !(Hp_rts_RegWrite(iHandle,0xb2,0x02))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb1,0x00))) return SANE_FALSE;
	if (pParams->optXPA)
		Reg_11 = 0x08;
	else
		Reg_11 = 0x28;
	Reg_10 = 0x28;
	if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, Reg_10, Reg_11 ))) return SANE_FALSE;
	usleep(1000);
	/*  test_double_reg( 0x20,0x3a,0xf2,21 );*/
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x01))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x01))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_Set_double_reg(iHandle,0x12,0xff,0x20))) return SANE_FALSE;
	if ( !(Hp_rts_Set_double_reg(iHandle,0x14,0xf8,0x28))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, Reg_10, Reg_11 ))) return SANE_FALSE;
	usleep(1000);
	if ( !(Hp_rts_RegWrite(iHandle,0xd9,0x80))) return SANE_FALSE;
	memcpy(pHWParams->regs,Hp44x0_switch_on_regs,254);
	pHWParams->regs[0x10] = Reg_10;
	pHWParams->regs[0x11] = Reg_11;
	if ( !(Hp_rts_BulkWrite (iHandle, 0x00,pHWParams->regs, 0xb3,SANE_TRUE))) return SANE_FALSE;
	return (Hp_rts_BulkWrite (iHandle, 0xb4,pHWParams->regs+0xb3, 0x3f,SANE_TRUE));

	/*  need we a new callibriation ? */
}


/****************************************************************************/
SANE_Bool Hp44x0_Down(SANE_Int iHandle)
/****************************************************************************/
{
	DBG(DBG_SCAN, "Hp44x0_Down....\n");
	if (iHandle <= 0) {
		return SANE_FALSE;
	}
	/* switch off the little green LEDs for color selector */
	if ( !(Hp_rts_Set_double_reg(iHandle,LEDSTATUS,0x08,0x28))) return SANE_FALSE;
	usleep(10000);
	/*test_reg( 0xb2, 0x02,13 ); */
	/* switch off the green power LED */
	if ( !(Hp_rts_RegWrite(iHandle,0xb2,0x00))) return SANE_FALSE;
	/* ?? */
	if ( !(Hp_rts_RegWrite(iHandle,0xb1,0x00))) return SANE_FALSE;
	/* switch off lamp, LCD and power LED */
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x04))) return SANE_FALSE;
	return (Hp_rts_RegWrite(iHandle,0xb3,0x04));
}

/****************************************************************************/
SANE_Bool Hp44x0_patch_and_send_values(SANE_Int iHandle, SANE_Byte *my_values){
/****************************************************************************/
	/* read registers, patch and write out them */
	SANE_Byte v1,v2;

	DBG(DBG_SCAN, "Hp44x0_patch_and_send_values....\n");
	Hp_rts_BulkReadall(iHandle,registers);
	v1 = *my_values++;
	v2 = *my_values++;
	registers[v1] = v2;
	while(*my_values){
		v1 = *my_values++;
		v2 = *my_values++;
		registers[v1] = v2;
	}
	if ( !(Hp_rts_BulkWrite(iHandle,0x00,registers,0xb3,SANE_TRUE))) return SANE_FALSE;
	return (Hp_rts_BulkWrite(iHandle,0xb4,registers+0xb3,0x3f,SANE_TRUE));
}

/****************************************************************************/
void Hp44x0_patch_regs(THWParams *pHWParams,SANE_Byte *my_values){
/****************************************************************************/
	/* read registers, patch and write out them */
	SANE_Byte v1,v2;

	DBG(DBG_SCAN, "Hp44x0_patch_regs....\n");
	v1 = *my_values++;
	v2 = *my_values++;
	pHWParams->regs[v1] = v2;
	while(*my_values){
		v1 = *my_values++;
		v2 = *my_values++;
		pHWParams->regs[v1] = v2;
	}
}

/****************************************************************************/
SANE_Bool Hp44x0_set_display( SANE_Int iHandle, SANE_Byte val )
/****************************************************************************/
{
	static SANE_Word LcdValues_h []=
	/* left char       0,      1,     2,     3 */
			{ 0xa18f2,0x0890,0x10b3,0x18b1};
	static SANE_Word LcdValues_l []=
	/* right char       0,      1,     2,     3      4 */
			{ 0xa68c,0x2280,0xe484,0x6684,0x6288,
	/*                  5,      6,     7,     8      9 */
			  0x468c,0xc68c,0x2284,0xe68c,0x668c};
	SANE_Byte reg_20,reg_21;

	DBG(DBG_SCAN, "Hp44x0_set_display.to %d\n",val);
	if (val < 40){
		reg_20 = (LcdValues_l [val%10]  >> 8) & 0x00ff;
		reg_21 =  LcdValues_l [val%10] & 0x00ff;
		reg_20 = reg_20 | (LcdValues_h [val/10]  >> 8) & 0x00ff;
		reg_21 = reg_21 |  LcdValues_h [val/10] & 0x00ff;
		if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
		return (Hp_rts_Set_double_reg(iHandle, LCD_CONTROL1, reg_20, reg_21 ));
	};
	return SANE_TRUE;
}

/****************************************************************************/
SANE_Bool Hp44x0_write_cal_bytes( THWParams *pHWParams, SANE_Byte b ) {
/****************************************************************************/
  SANE_Word i,n;
  SANE_Int iHandle;

	DBG(DBG_SCAN, "Hp44x0_write_cal_bytes....\n");
  iHandle = pHWParams->iXferHandle;
  callibration_buffer[0]=0x89;
  callibration_buffer[1]=0x00;
  callibration_buffer[2]=0x00;
  callibration_buffer[3]=0x20;
  callibration_buffer[4]=0;
  n = 2;
  for(i=5;i<36;i++) {
     callibration_buffer[i]   = b;
     callibration_buffer[i+1] = n;
     i++;
     n = n + 2;
  }
  return (Hp_rts_BulkWrite(iHandle,0x00,callibration_buffer,36,SANE_FALSE));
}


/****************************************************************************/
SANE_Bool Hp44x0_cal_scanner(THWParams *pHWParams, TScanParams *pParams) {
/****************************************************************************/
  SANE_Word i,n;
  SANE_Int iHandle;
  SANE_Byte Reg_10, Reg_11;

	DBG(DBG_SCAN, "Hp44x0_cal_scanner....\n");
  iHandle = pHWParams->iXferHandle;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  memcpy(registers,Hp44x0_init_1,0xf4);
  if (pParams->mode == COLOR){
     Reg_10 = 0x28;
  } else {
     Reg_10 = 0x18;}
  if (pParams->optXPA){
     Reg_11 = 0x1b;
  } else {
     Reg_11 = 0x3b;}
  registers[0x10] = Reg_10;
  registers[0x11] = Reg_11;
  if ( !(Hp_rts_BulkWrite(iHandle,0x00,registers,0xb3,SANE_TRUE))) return SANE_FALSE;
  if ( !(Hp_rts_BulkWrite(iHandle,0xb4,registers+0xb3,0x3f,SANE_TRUE))) return SANE_FALSE;
  if ( !(Hp44x0_patch_and_send_values(iHandle,Hp44x0_patch_2))) return SANE_FALSE;
  if ( !(Hp44x0_patch_and_send_values(iHandle,Hp44x0_patch_3))) return SANE_FALSE;

/*  if (pParams->mode == COLOR){*/
     if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x20, 0x28))) return SANE_FALSE;
/*     usleep(10000);*/
     if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x28, 0x28))) return SANE_FALSE;
/*
  } else {
     if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x10, 0x28))) return SANE_FALSE;
     usleep(10000);
     if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x18, 0x28))) return SANE_FALSE;
  }
*/
  usleep(10000);
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x02))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x02))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  if ( !(Hp44x0_patch_and_send_values(iHandle,Hp44x0_patch_4))) return SANE_FALSE;

  for (n=0;n<3;n++){
    switch (n){
      case 0: if ( !(Hp_rts_RegWrite(iHandle,0x93,0x06))) return SANE_FALSE; break;
      case 1: if ( !(Hp_rts_RegWrite(iHandle,0x93,0x02))) return SANE_FALSE; break;
      case 2: if ( !(Hp_rts_RegWrite(iHandle,0x93,0x01))) return SANE_FALSE; break;
      default:;
    }
    if ( !(Hp_rts_Set_double_reg(iHandle,0x91, 0x81, 0x00))) return SANE_FALSE;
    callibration_buffer[0]=0x89;
    callibration_buffer[1]=0x00;
    callibration_buffer[2]=0x08;
    callibration_buffer[3]=0x18;
    callibration_buffer[4]=0;
    for(i=5;i<0x818;i++) {
       callibration_buffer[i]=callibration_buffer[i-1]+1;
       if(callibration_buffer[i]==0x61) callibration_buffer[i]=0;
    };
    if ( !(Hp_rts_BulkWrite(iHandle,0x00,callibration_buffer,0x818+4,SANE_FALSE))) return SANE_FALSE;
    if ( !(Hp35x0c_set_sram_page(iHandle,0x81))) return SANE_FALSE;
/*    Hp_rts_Set_double_reg(iHandle,0x91, 0x81, 0x00);*/
    if ( !(Hp_rts_Read_Sram (iHandle, callibration_buffer,0x0818))) return SANE_FALSE;
#if 0
    read_bulk_callibriation_data(callibration_buffer,0x0818);
/*    read_bulk_callibriation_data(buffer2,0x0818); and verify it*/
#endif
	};
  memcpy(registers,Hp44x0_init_5,0xf4);
/*
  if (pParams->mode == COLOR){
     registers[LEDSTATUS] = 0x28;
  } else {
     registers[LEDSTATUS] = 0x18;
  }*/
  if ( !(Hp_rts_BulkWrite(iHandle,0x00,registers,0xb3,SANE_TRUE))) return SANE_FALSE;
  if ( !(Hp_rts_BulkWrite(iHandle,0xb4,registers+0xb3,0x3f,SANE_TRUE))) return SANE_FALSE;

  for(i=0;i<5;i++) {
    if ( !(Hp44x0_write_cal_bytes( pHWParams, i ))) return SANE_FALSE;
    if ( !(Hp_rts_Set_double_reg(iHandle, 0x91, 0x00, 0x10 ))) return SANE_FALSE;
/*    read_bulk_callibriation_data(buffer2,0x20); and verify it*/
  };
  if ( !(Hp_rts_RegWrite(iHandle,LAMPREG,0x80))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMPREG,0xad))) return SANE_FALSE;
  if ( !(Hp_rts_Set_double_reg(iHandle, 0x14, 0x78, 0x28 ))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa0))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMPREG,0x8d))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMPREG,0xad))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,BUTTON_1,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,BUTTON_2,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,BUTTON_1,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,BUTTON_2,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa0))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;
  if ( !(Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00))) return SANE_FALSE;

  if (pParams->mode == COLOR){
     if ( !(Hp_rts_Set_double_reg(iHandle, 0x10, 0x28, 0x3f))) return SANE_FALSE;
  } else {
     if ( !(Hp_rts_Set_double_reg(iHandle, 0x10, 0x18, 0x3f))) return SANE_FALSE;
  }
  usleep(10000);
  if ( !(Hp_rts_RegWrite(iHandle,LAMPREG,0xad))) return SANE_FALSE;
  usleep(10000);
/*  if (pParams->mode == COLOR){*/
     if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x20, 0x3f))) return SANE_FALSE;
/*  } else {
     if ( !(Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x10, 0x3f))) return SANE_FALSE;
  }*/
  usleep(10000);
  return(SANE_TRUE);
}


/****************************************************************************/
SANE_Int Hp44x0_park_to_home(THWParams *pHWParams,TScanParams *pParams){
/* Moves scanner to home position */
/****************************************************************************/
	SANE_Int n,iHandle;
	SANE_Byte	buffer[0xffc0];

	iHandle = pHWParams->iXferHandle;

	DBG(DBG_SCAN, "Hp44x0_park_to_home....");
	/* make shure the scaner is stoped */
	if ( !(Hp_rts_stop_moving(iHandle))) return SANE_FALSE;

  if (Hp35x0c_is_rewound(iHandle )){
		DBG(DBG_SCAN,"i'm home\n");
/*	if(read_reg(iHandle,MICROSWITCH) & 0x02) {
		DBG(DBG_MSG,"i'm home\n");*/
		return(SANE_TRUE);
	} else {
		DBG(DBG_SCAN,"i must moving\n");
		memcpy(pHWParams->regs,Hp44x0_move_back,255);
/*
		pHWParams->regs[0x32] = 0x80;
		pHWParams->regs[0x33] = 0x81;
		pHWParams->regs[0x34] = 0x10;
		pHWParams->regs[0x35] = 0x10;
		pHWParams->regs[0x36] = 0x21; *0x24;*
		pHWParams->regs[0x39] = 0x02; * ??*
		pHWParams->regs[0x3A] = 0x0e;
		pHWParams->regs[0x60] = 0x40;
		pHWParams->regs[0x61] = 0x1f;
		pHWParams->regs[0x62] = 0x41;
		pHWParams->regs[0x63] = 0x1f;
		pHWParams->regs[0xb2] = 0x16;
		pHWParams->regs[0xe2-1] = 0x17; * StepSize */
		#if 1
		if (pParams->optXPA)
			pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
		else
			pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;
		#endif
		if ( !(Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE))) return SANE_FALSE;
		if ( !(Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE))) return SANE_FALSE;
		if ( !(Hp_rts_RegWrite(iHandle,0xd3,0x00))) return SANE_FALSE;
		if ( !(Hp_rts_start_moving(iHandle))) return SANE_FALSE;
		usleep(1000);
		while (!Hp35x0c_is_rewound(iHandle)  &&
/*			(Hp_rts_data_ready(iHandle,&n) ||*/
			 Hp_rts_is_moving(iHandle) > 0)
		{
			/*if ( !(*/Hp_rts_data_ready(iHandle,&n);/*)) return SANE_FALSE;*/
			if (n)
			{
				if (n > (SANE_Word)sizeof(buffer))
					n = sizeof(buffer);
				/*if ( !(*/Hp_rts_read_data(iHandle, n, buffer);/*)) return SANE_FALSE;*/
			}
			else
			{
				usleep(10000);
			}
		}
		return(Hp_rts_stop_moving(iHandle));
	}
}

/****************************************************************************/
SANE_Int Hp44x0_move_to_pos(THWParams *pHWParams, TScanParams *pParams){
/* Moves scanner to position */
/****************************************************************************/
	SANE_Int iHandle;
	SANE_Word pos;
	SANE_Int ypos;
	SANE_Byte r;

	iHandle = pHWParams->iXferHandle;

	ypos = MM_TO_PIXEL(pHWParams->iTopLeftY,pParams->iDpi);
	DBG(DBG_MSG,"Hp44x0_move_to_pos: %d dpi pParams->iY = %d pHWParams->iTopLeftY = %d ypos=%d \n",
							pParams->iDpi,(SANE_Int)pParams->iY,pHWParams->iTopLeftY,ypos);

	if (pParams->iDpi == 600)
		pos = ((pParams->iY) / 2 )+ ypos;
	else
		pos = (pParams->iY) + ypos;
	pos = pos / 2;

/*	DBG(DBG_MSG,"Hp44x0_move_to_pos: move to pos %d (pixel) %x (hex)\n",pos,pos-1);*/

	pHWParams->regs[0x33] = 0x81; /* 81 move fast, 82 lower */
	pHWParams->regs[0x36] = 0x29; /*29 fast; * 2C slow forward, 24 reverse */
	pHWParams->regs[0x3A] = 0x1b; /* 1b scan, 0e quick move */
	pHWParams->regs[0x60] = 0x01;
	pHWParams->regs[0x61] = 0x00;
	pHWParams->regs[0x62] = (pos+1) & 0xff;
	pHWParams->regs[0x63] = ((pos+1) >> 8) & 0xff;
	pHWParams->regs[0x65] = 0x00;  /* switch to move only (no scan)*/
	pHWParams->regs[0xE2-1] = 0x07 ; /* Stepsize */
	if (pParams->optXPA)
		pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
	else
		pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;

	usleep(10000);
#if 0
		printf("start_scan: start the moving!\n");
		Hp_rts_DumpHex(pHWParams->regs,252,16,SANE_TRUE);
#endif
	if ( !(Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE))) return SANE_FALSE;
	if ( !(Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xd3,0x03))) return SANE_FALSE;

	if ( !(Hp_rts_start_moving(iHandle))) return SANE_FALSE;
	usleep(1000);
	
	/*DBG(DBG_MSG,"move_to_pos: check moving\n");*/
	r = 0x08;
	if ( Hp_rts_Check_Moving (iHandle)){
		while( r == 0x08){
			usleep(10000);
			usleep(10000);
			Hp_rts_RegRead(iHandle,MOVE_START_STOP, &r);
		}
		/*DBG(DBG_MSG,"move_to_pos: move finish\n");*/
	}else
	{
		DBG(DBG_ERR,"start_scan: Error, stop the moving!\n");
		Hp_rts_stop_moving(iHandle);
		return(SANE_FALSE);
	} 
	return(SANE_TRUE);
}


/****************************************************************************/
SANE_Int Hp44x0_wait_for_WarmUp(SANE_Int iHandle,THWParams *pHWParams){
/****************************************************************************/
  /* Reads 11 bytes and check the last one */
SANE_Char i;
#ifdef DEBUG_L
SANE_Char c;
#endif
	DBG(DBG_SCAN, "Hp44x0_wait_for_WarmUp....\n");
  /* we must here implement a wait loop for the lamp */
  if ( !(Hp_rts_BulkRead(iHandle,0x84,read_buffer,0x0b,SANE_TRUE))) return SANE_FALSE;
#ifdef DEBUG_L
  c = read_buffer[0xa];
  DBG(DBG_MSG,"wait_for_WarmUp: read %x\n",c);
  i = 1;
#else
  i = 15;
#endif
  if ( pHWParams->lamp_warm == SANE_FALSE ){
    while (i--){
      if ( !(Hp44x0_set_display( iHandle, i ))) return SANE_FALSE;
      sleep(1);
    }
    pHWParams->lamp_warm = SANE_TRUE;
  }
  return( /*(read_buffer[0xa] & 0x60) == 0x60*/ SANE_TRUE );
}

/****************************************************************************/
SANE_Bool Hp44x0_start_scan(THWParams *pHWParams, TScanParams *pParams,
								TDataPipe *pDataPipe){
/****************************************************************************/
#ifdef DEBUG
	SANE_Word lengtha1,lengtha2;
#endif
	SANE_Int brightness, iHandle;
	SANE_Int ires/*,result*/;
	SANE_Word x1,x2,length;

	iHandle = pHWParams->iXferHandle;
	brightness = 0;

	DBG(DBG_SCAN, "Hp44x0_start_scan....\n");
	DBG(DBG_MSG,"Hp44x0_start_scan: check resolution %d\n",pParams->iDpi);
	for (ires = 0; Hp44x0c_resparms[ires].resolution &&
		 Hp44x0c_resparms[ires].resolution != pParams->iDpi; ++ires);
	if (Hp44x0c_resparms[ires].resolution == 0){
		DBG(DBG_MSG,"start_scan:  did not found this resolution %d\n",pParams->iDpi);
		return SANE_FALSE;
	}

	DBG(DBG_MSG,"start_scan: Use ires entry %d (%ddpi) for the resolution %d dpi\n",ires,
			Hp44x0c_resparms[ires].resolution, pParams->iDpi);

	memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/
	
	if ( !(Hp44x0_move_to_pos(pHWParams,pParams))) return SANE_FALSE;
	x1 = pParams->iX;
	x2 = pParams->iWidth + pParams->iX;
	length = pParams->iLenght * 2;
	pDataPipe->iScanned = pParams->iLenght;
	DBG(DBG_MSG,"start_scan: calculate scandata x1 = %d; x2 = %d; lenght = %d\n",x1,x2,length);

	switch (pParams->mode) {

		case BLACK_WHITE:
			DBG(DBG_MSG,"Copy 300 DPI BW Regs\n");
			if (pParams->oCCD_Type){
				DBG(DBG_MSG,"Use CCD_Type 1 Regs\n");
				memcpy(pHWParams->regs,Hp44x0_300_BW_1,254);  /* CCD_Type 1*/
			}else{
				DBG(DBG_MSG,"Use CCD_Type 0 Regs\n");
				memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/
			}
			pHWParams->regs[0xD2 ]= 0x03;
			pHWParams->regs[0xD3 ]= 0x05;
			pHWParams->regs[0xE4 ]= 0x1c;
			pHWParams->regs[0xE5 ]= 0x10;
			pHWParams->regs[0xd5 ]= 0xab;/* ab gray */
			break;

		case GRAY:
			DBG(DBG_MSG,"Copy 300 DPI Regs\n");
			brightness = 0;
			if (pParams->oCCD_Type){
				DBG(DBG_MSG,"Use CCD_Type 1 Regs\n");
				memcpy(pHWParams->regs,Hp44x0_300_true_1_3,254); /* CCD_Type 1*/
			}else{
				DBG(DBG_MSG,"Use CCD_Type 0 Regs\n");
				memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/
			}
			if (pParams->iDpi == 600){
				DBG(DBG_MSG,"Patch to 600 DPI\n");
				Hp44x0_patch_regs(pHWParams,Hp44x0_patch_600);
				brightness = 43;
			}
			pHWParams->regs[0xd5 ]= 0xab;/* ab gray */
#if 1
			pHWParams->regs[0x08] = brightness + pParams->brightness;
			pHWParams->regs[0x09] = brightness + pParams->brightness;
			pHWParams->regs[0x0a] = brightness + pParams->brightness;
#endif
			#if 1
			if (pParams->optXPA)
				pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
			else
				pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;
			#endif
			break;

		case COLOR:
			DBG(DBG_MSG,"Copy 300 DPI Regs\n");
			if (pParams->oCCD_Type)
				memcpy(pHWParams->regs,Hp44x0_300_true_1_3,254); /* CCD_Type 1*/
			else
				memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/

			if (pParams->iDpi == 200){
				DBG(DBG_MSG,"Patch to 200 DPI\n");
				memcpy(pHWParams->regs,Hp44x0_200_TRUE_1,254);  /* CCD_Type 0*/
/*				Hp44x0_patch_regs(pHWParams,Hp44x0_200_TRUE_1);*/
			}

			if (pParams->iDpi == 300){
				DBG(DBG_MSG,"Patch to 300 DPI\n");
/*				memcpy(pHWParams->regs,Hp44x0_300_TRUE_1,254);  * CCD_Type 1*/
				Hp44x0_patch_regs(pHWParams,Hp44x0_patch_600);
			}

			if (pParams->iDpi == 600){
				DBG(DBG_MSG,"Patch to 600 DPI\n");
				Hp44x0_patch_regs(pHWParams,Hp44x0_patch_600);
			}
			pHWParams->regs[0xd5 ]= 0x0f;/* 0f color*/
			#if 1
			if (pParams->optXPA)
				pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
			else
				pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;
			#endif
#if 1
			pHWParams->regs[0x32 ]=  Hp44x0c_resparms[ires].reg_32_value;
			pHWParams->regs[0x33 ]=  Hp44x0c_resparms[ires].reg_33_value;
#endif
#if DEBUG_L
/*			pHWParams->regs[0xd5 ]= 0x1B;* 0f color*/
			/* horizon resolution */
			pHWParams->regs[0x10 ]=  Hp44x0c_resparms[ires].reg_10_value;
			pHWParams->regs[0x32 ]=  Hp44x0c_resparms[ires].reg_32_value;
			pHWParams->regs[0x33 ]=  Hp44x0c_resparms[ires].reg_33_value;
			pHWParams->regs[0x36 ]=  Hp44x0c_resparms[ires].reg_36_value;
			printf("pHWParams->regs[0xE1]= 0x%x to 0x%x\n",pHWParams->regs[0xE1],Hp44x0c_resparms[ires].step_size);
			pHWParams->regs[0xE1]= Hp44x0c_resparms[ires].step_size;
			#if 1
			printf("pHWParams->regs[0xE4]= 0x%x to 0x%x\n",pHWParams->regs[0xE4],Hp44x0c_resparms[ires].reg_E4_value);
			pHWParams->regs[0xE4]=  Hp44x0c_resparms[ires].reg_E4_value;
			printf("pHWParams->regs[0xE5]= 0x%x to 0x%x\n",pHWParams->regs[0xE5],Hp44x0c_resparms[ires].reg_E5_value);
			pHWParams->regs[0xE5]=  Hp44x0c_resparms[ires].reg_E5_value;
			#endif
			#if 1
			/* color releated */
			pHWParams->regs[0x72 ]=  Hp44x0c_resparms[ires].reg_72_value;
			pHWParams->regs[0x73 ]=  Hp44x0c_resparms[ires].reg_73_value;
			pHWParams->regs[0x74 ]=  Hp44x0c_resparms[ires].reg_74_value;
			#endif
			#if 1
			pHWParams->regs[0x11 ]=  Hp44x0c_resparms[ires].reg_11_value;
			pHWParams->regs[0x3a ]= 0x0e;/* for test*/
			/*	pHWParams->regs[0xCC ]=  Hp44x0c_resparms[ires].reg_CC_value;
			pHWParams->regs[0xCD ]=  Hp44x0c_resparms[ires].reg_CD_value;
			pHWParams->regs[0xCE ]=  Hp44x0c_resparms[ires].reg_CE_value;
			pHWParams->regs[0xCF ]=  Hp44x0c_resparms[ires].reg_CF_value;
			pHWParams->regs[0xd0 ]=  Hp44x0c_resparms[ires].reg_D0_value;*/
			/*	pHWParams->regs[0xd7 ]=  Hp44x0c_resparms[ires].reg_D7_value;
			pHWParams->regs[0xC4 ]=  Hp44x0c_resparms[ires].reg_C4_value; * colors ? *
			pHWParams->regs[0xC5 ]=  Hp44x0c_resparms[ires].reg_C5_value; * colors ? *
			pHWParams->regs[0xd5 ]=  Hp44x0c_resparms[ires].reg_D5_value;

			pHWParams->regs[0x35 ]=  Hp44x0c_resparms[ires].reg_35_value;
			pHWParams->regs[0x3A ]=  Hp44x0c_resparms[ires].reg_3A_value; */
			#endif
			/* vertical resolution */
			#if 0
			pHWParams->regs[0x3A ]=  Hp44x0c_resparms[ires].reg_3A_value;
			#endif
/*			pHWParams->regs[0x02 ]= 0xbb;
			pHWParams->regs[0x03 ]= 0x7d;
			pHWParams->regs[0x04 ]= 0x3e;
			pHWParams->regs[0x05 ]= 0xbb;
			pHWParams->regs[0x06 ]= 0x7d;
			pHWParams->regs[0x07 ]= 0x3e;

			pHWParams->regs[0x08 ]= 0x00;
			pHWParams->regs[0x09 ]= 0x00;
			pHWParams->regs[0x0a ]= 0x00;*/

			pHWParams->regs[0xBF ]=  Hp44x0c_resparms[ires].reg_BF_value;
			pHWParams->regs[0xC0 ]=  Hp44x0c_resparms[ires].reg_C0_value;
			pHWParams->regs[0xC1 ]=  Hp44x0c_resparms[ires].reg_C1_value;
			pHWParams->regs[0xc2 ]= 0xff;/* for test*/
			pHWParams->regs[0xc3 ]= 0x00;/* for test*/
			pHWParams->regs[0xc4 ]= 0xf0;/* for test*/
			pHWParams->regs[0xc5 ]= 0x00;/* for test*/
			pHWParams->regs[0xc6 ]= 0x7f;/* for test*/
			pHWParams->regs[0xc7 ]= 0x0f;/* for test*/
/*			pHWParams->regs[0xc9 ]= 0xff;* for test*/
			pHWParams->regs[0xca ]= 0xf1;/* for test*/
			pHWParams->regs[0xcb ]= 0xff;/* for test*/
			pHWParams->regs[0xcc ]= 0x00;/* for test*/
			pHWParams->regs[0xcd ]= 0xf0;/* for test*/
			pHWParams->regs[0xce ]= 0xed;/* for test*/
			pHWParams->regs[0xcf ]= 0xef;/* for test*/
			pHWParams->regs[0xd0 ]= 0xe2;/* for test*/
/*			pHWParams->regs[0xd1 ]=  Hp44x0c_resparms[ires].reg_D1_value;
			pHWParams->regs[0xd2 ]=  Hp44x0c_resparms[ires].reg_D2_value;
			pHWParams->regs[0xd3 ]=  Hp44x0c_resparms[ires].reg_D3_value;*/
			pHWParams->regs[0xd1 ]= 0x03;/* for test*/
			pHWParams->regs[0xd2 ]= 0x17;/* for test*/
			pHWParams->regs[0xd3 ]= 0x01;/* for test*/
			pHWParams->regs[0xd6 ]= 0x10;/* for test*/

			pHWParams->regs[0x80 ]= 0x2b;/* for test*/
			pHWParams->regs[0x81 ]= 0x02;/* for test*/
			pHWParams->regs[0x82 ]= 0x2C;/* for test*/
			pHWParams->regs[0x83 ]= 0x02;/* for test*/
			pHWParams->regs[0x85 ]= 0x18;/* for test*/
			pHWParams->regs[0x86 ]= 0x1b;/* for test*/
			pHWParams->regs[0x87 ]= 0x30;/* for test*/
			pHWParams->regs[0x88 ]= 0x30;/* for test*/
			pHWParams->regs[0x89 ]= 0x2d;/* for test*/
			pHWParams->regs[0x8a ]= 0x02;/* for test*/
			pHWParams->regs[0x8d ]= 0x21;/* for test*/
			pHWParams->regs[0x8e ]= 0x60;/* for test*/
#endif
		break;

		default:
			DBG(DBG_ERR, "Hp44x0_init_scan: Invalid parameter\n");
			return SANE_FALSE;
	}/* switch */




#if 0
	if (pParams->optXPA)
		pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
	else
		pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;
#endif

#if 0
	if (pParams->mode == COLOR)
		pHWParams->regs[0xd5 ]= 0x0f;/* 0f color*/
	else
		pHWParams->regs[0xd5 ]= 0xab;/* ab gray */
#endif

	/* modify reg 60 + 61 */
	Hp_rts_set_noscan_distance(pHWParams->regs, 0x01 );
	/* modify reg 62 + 63 */
#if 1
	Hp_rts_set_total_distance(pHWParams->regs, length+1 );
#endif
#if 0
	Hp_rts_set_total_distance(pHWParams->regs, 0x02 );
#endif

	pHWParams->regs[0x65]= 0x20; /* Schan mode on */

	/* modify reg 66 + 67 */
	Hp_rts_set_scanline_start(pHWParams->regs, x1 );
	/* modify reg 6C + 6D */
	Hp_rts_set_scanline_end(pHWParams->regs, x2 );

	#ifdef DEBUG
		lengtha1 = 0;
		lengtha2 = 0;
		lengtha1 = pHWParams->regs[0x60];
		lengtha1 = lengtha1 + ((pHWParams->regs[0x61] <<8) & 0xFF00);
		lengtha2 = pHWParams->regs[0x62];
		lengtha2 = lengtha2 + ((pHWParams->regs[0x63] <<8) & 0xFF00);
		DBG(DBG_MSG,"start_scan: Init scan_region: dpi=%d iLenght=%d, iScanned=%d lenght=%d mode=%x xPDA=%x\n",
		pParams->iDpi,pParams->iLenght,pDataPipe->iScanned,length,pParams->mode,pParams->optXPA);
		#endif

#if 0
	pHWParams->regs[0xDA-1]= pParams->intensity2; /* Lamp density */
#endif


#if 0  /* do not use by graymode ! */
		pHWParams->regs[0x05] = 0x20 - pParams->GainR;
		pHWParams->regs[0x06] = 0x20 - pParams->GainG;
		pHWParams->regs[0x07] = 0x20 - pParams->GainB;
#endif

#if 0
		pHWParams->regs[0x08] = 0x20 - pParams->GainR;
		pHWParams->regs[0x09] = 0x20 - pParams->GainG;
		pHWParams->regs[0x0a] = 0x20 - pParams->GainB;
#endif

#if 0  /* moved to graymode */
	pHWParams->regs[0x08] = /*0x20 -*/ pParams->brightness;
	pHWParams->regs[0x09] = /*0x20 -*/ pParams->brightness;
	pHWParams->regs[0x0a] = /*0x20 -*/ pParams->brightness;
#endif
#if 0
		printf("start_scan: start the scanner!\n");
		Hp_rts_DumpHex(pHWParams->regs,255,16,SANE_TRUE);
#endif

	if ( !(Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],pHWParams->regs[0x11]))) return SANE_FALSE;
	usleep(10000);

	if ( !(Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE))) return SANE_FALSE;
	if ( !(Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE))) return SANE_FALSE;
#if 0
	if (pParams->mode == BLACK_WHITE)
		Hp_rts_RegWrite(iHandle,0xd3,0x17);
	else
		Hp_rts_RegWrite(iHandle,0xd3,0x03);
#endif
#if 0
	DBG(DBG_MSG,"pHWParams->regs[0xd3]= 0x%x to 0x%x\n",
		pHWParams->regs[0xd3-1],pHWParams->regs[0xd3-1]);
#endif	
	if ( !(Hp_rts_RegWrite(iHandle,0xd3,pHWParams->regs[0xd3-1]))) return SANE_FALSE;

	if ( !(Hp_rts_start_moving(iHandle))) return SANE_FALSE;
	usleep(10000);
	/* wait for moving and check it */
	return(Hp_rts_Check_Moving (iHandle));
}


/****************************************************************************/
SANE_Bool Hp44x0_init_scan(THWParams *pHWParams, TScanParams *pParams,
													TDataPipe *pDataPipe){
/****************************************************************************/

	SANE_Int iHandle;
	SANE_Int	ires;

	iHandle = pHWParams->iXferHandle;
	DBG(DBG_MSG,"Hp44x0_init_scan: check resolution %d\n",pParams->iDpi);
	for (ires = 0; Hp44x0c_resparms[ires].resolution &&
								 Hp44x0c_resparms[ires].resolution != pParams->iDpi; ++ires);
	if (Hp44x0c_resparms[ires].resolution == 0){
		DBG(DBG_MSG,"Hp44x0_init_scan: this resolution %d not found\n",pParams->iDpi);
		return SANE_FALSE;
	}

	if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa0))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x01))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x01))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_Set_double_reg(iHandle, 0x12, 0xff, 0x20 ))) return SANE_FALSE;
	if ( !(Hp_rts_Set_double_reg(iHandle, 0x14, 0xf8, 0x28 ))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;

	memcpy(pHWParams->regs,Hp44x0_init_1,0xf4);
	pHWParams->regs[0x10 ]=  Hp44x0c_resparms[ires].reg_10_value;
	pHWParams->regs[0x11 ]=  Hp44x0c_resparms[ires].reg_11_value;

	if (pParams->optXPA)
		pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
	else
		pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;

	if ( !(Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],
																		pHWParams->regs[0x11]))) return SANE_FALSE;
	usleep(10000);
	if ( !(Hp_rts_RegWrite(iHandle,0xd9,0x80))) return SANE_FALSE;

	#ifndef WITH_TSTBACKEND
	if ( !(Hp44x0_wait_for_WarmUp(iHandle,pHWParams))) return SANE_FALSE;
	#endif

	if ( !(Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE))) return SANE_FALSE;
	if ( !(Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE))) return SANE_FALSE;

	if ( !(Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xd9,0x8d))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xd9,0xad))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	if ( !(Hp_rts_RegWrite(iHandle,0xb3,0x00))) return SANE_FALSE;
	/*  test_double_reg( LEDSTATUS, 0x28, 0x3f, 7 );  */
	usleep(10000);
	if ( !(Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],
																		pHWParams->regs[0x11]))) return SANE_FALSE;
	/*  if ( !(test_double_reg( LEDSTATUS, 0x28, 0x3f, 8 );  */
	if ( !(Hp_rts_RegWrite(iHandle,0xd9,0xad))) return SANE_FALSE;

	return(Hp44x0_start_scan(pHWParams,pParams,pDataPipe));
}



/****************************************************************************/
/*  Lamp control functions                                                  */
SANE_Bool Hp44x0_GetLamp(SANE_Int iHandle, SANE_Bool *pfLampIsOn)
/****************************************************************************/
{
	SANE_Byte bData;

	DBG(DBG_SCAN, "Hp44x0_GetLamp....\n");
	if ( !(Hp_rts_RegRead(iHandle, LAMPREG, &bData))) return SANE_FALSE;
	*pfLampIsOn = ((bData & 0x80) != 0);
	return SANE_TRUE;
}


/****************************************************************************/
SANE_Bool Hp44x0_SetLamp(SANE_Int iHandle, SANE_Bool fLampOn)
/****************************************************************************/
{
	SANE_Byte bData;

	DBG(DBG_SCAN, "Hp44x0_SetLamp....\n");
	if ( !(Hp_rts_RegRead(iHandle, LAMPREG, &bData))) return SANE_FALSE;
	if (fLampOn) {
		if ( !(Hp_rts_RegWrite(iHandle, LAMPREG, bData | 0x80))) return SANE_FALSE;
	}
	else {
		if ( !(Hp_rts_RegWrite(iHandle, LAMPREG, bData & ~0x80))) return SANE_FALSE;
	}
	return SANE_TRUE;
}


/****************************************************************************/
SANE_Int
Hp44x0c_nvram_enable_controller(SANE_Int iHandle,SANE_Int enable)
/****************************************************************************/
{
	SANE_Byte r;

	DBG(DBG_SCAN, "Hp44x0c_nvram_enable_controller....\n");
	if ( !(Hp_rts_RegRead(iHandle, 0x1d, &r))) return SANE_FALSE;
	if (enable)
		r |= 1;
	else
		r &= ~1;
	return Hp_rts_RegWrite(iHandle,0x1d, r);

}

