# Samizdat resource representation
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'
require 'samizdat/helpers/resource_helper'
require 'samizdat/helpers/message_helper'

class Resource

  # _id_ should be translatable to an Integer above zero
  #
  # returns nil if _id_ is invalid
  #
  def Resource.validate_id(id)
    return nil if
      id.to_i.to_s != id.to_s or
      not id.to_i > 0

    id.to_i
  end

  # get reference for _request_, validate _id_, determine
  # _type_
  #
  def initialize(request, id)
    @request = request

    @id = Resource.validate_id(id)
    @id or raise ResourceNotFoundError, id.to_s

    @type = cache.fetch_or_add(%{resource_type/#{@id}}) do
      external, literal, label =
        db.select_one("SELECT uriref, literal, label FROM Resource WHERE literal = 'false' AND id = ?", @id)

      if external
        'Uriref'
      elsif literal
        'Literal'
      else   # internal resource
        case label
        when 'Member', 'Message', 'Statement', 'Vote', 'Item'
          label
        when nil
          raise ResourceNotFoundError, @id.to_s
        else
          raise RuntimeError,
            sprintf(_("Unknown resource type '%s'"), CGI.escapeHTML(label))
        end
      end
    end

    @type.untaint
    @component = instance_eval(@type + 'Component.new(@request, @id)')
  end

  attr_reader :request, :id, :type

  def to_i
    @id
  end

  def _rgettext_hack   # :nodoc:
    [ _('Uriref'), _('Literal'), _('Member'), _('Message'), _('Statement'), _('Vote') ]
  end
  private :_rgettext_hack

  # delegate known actions to the component and cache results when appropriate
  #
  def method_missing(action, *args)
    key = cache_key(action)
    case key
    when :uncacheable
      @component.send(action, *args)
    when String
      cache.fetch_or_add(key) do
        @component.send(action, *args)
      end
    else
      super
    end
  end

  private

  # compose a unique cache key for resource component action
  #
  # when necessary, include request parameters in the key
  #
  # returns +:uncacheable+ if action cannot be cached
  #
  def cache_key(action)
    case action
    when :links
      %{resource/#{@id}/#{action}}
    when :title, :list_item, :short, :full
      %{resource/#{@id}/#{action}/#{@request['page'].to_i}/#{@request.accept_language.join(':')}}
    when :page, :messages, :buttons, :moderation_log
      :uncacheable
    else
      nil
    end
  end
end

class ResourceComponent
  include ResourceHelper

  def initialize(request, id)
    @request = request
    @session = @request.session
    @id = Resource.validate_id(id)
    @id or raise ResourceNotFoundError, id.to_s

    @title = nil
    @info = nil

    @links = {}
  end

  attr_reader :links

  # resource title (HTML-escaped)
  #
  def title
    CGI.escapeHTML(limit_string(@title.to_s))
  end

  # render resource as a list item
  #
  def list_item
    resource(@id, @title, info)
  end

  # short rendering of the resource
  #
  def short
    info
  end

  # full rendering of the resource
  #
  def full
    short
  end

  def focuses
    dataset = RdfDataSet.new(%{
SELECT ?focus
WHERE (rdf::subject ?stmt #{@id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt ?focus)
      (s::rating ?stmt ?rating FILTER ?rating > 0)
ORDER BY ?rating DESC})

    focuses = {}
    dataset[0].each {|focus,|
      f = Focus.new(@request, focus, @id)
      focuses[f.id] = f
    }
    if @request.access('vote') and not rdf.get_property(@id, 's::inReplyTo').nil?
      # publish translations as replies (todo: dct:isPartOf)
      translation = Focus.new(@request, 'focus::Translation', @id)
      focuses.update({translation.id => translation})
    end
    focus_box(@id, focuses.values)
  end

  def messages
    nil
  end

  def buttons
    nil
  end

  # render resource itself along with focuses, messages, and buttons
  #
  # only one resource per page may be rendered in this mode
  #
  def page
    full << focuses << messages.to_s << buttons.to_s
  end

  def moderation_log(page = 1)
    Moderation.find(@id)
  end

  private

  attr_reader :info
end

class UrirefComponent < ResourceComponent
  def initialize(request, id)
    super

    label, = db.select_one("SELECT label FROM Resource WHERE literal = 'false' AND id = ?", @id)

    @title = Samizdat::SquishQuery.ns_shrink(label, rdf.ns)
    @title.gsub!(/\Afocus::/, '') and @title = _(@title)   # focus is special

    @info = '<div>' + sprintf(_('refers to <a href="%s">external uriref</a>'), label) + '</div>'
    # todo: select all statements with this subject
  end
end

class LiteralComponent < ResourceComponent
  def initialize(request, id)
    super

    label, = db.select_one("SELECT label FROM Resource WHERE literal = 'false' AND id = ?", @id)
    @uriref = @request.base + @id.to_s

    @title = label
    @info = %{<a href="#{@uriref}">#{Samizdat::SquishQuery.ns_shrink(@uriref, rdf.ns)}</a> = #{label}}
  end
end

class MessageComponent < ResourceComponent
  include MessageHelper

  def initialize(request, id)
    super

    @message = Message.cached(@id)

    # use translation to preferred language if available
    @translation = @message.select_translation(@request.accept_language)
    @title = @translation.content.title

    # navigation links
    @links['made'] = @message.creator.id
    @links['up'] = @message.parent if @message.parent
  end

  def list_item
    if @message.nrelated > 0
      resource(@id, Focus.focus_title(@title), info)
    else
      super
    end
  end

  def short
    # rely on ApplicationHelper#message to take care of translation
    message(@message, :short)
  end

  def full
    message(@message, :full)
  end

  def messages
    if @message.nreplies > 0   # add replies
      dataset = RdfDataSet.new(%{
SELECT ?msg
WHERE (s::inReplyTo ?msg #{@id})
      (s::hidden ?msg ?hidden #{filter_hidden})
OPTIONAL (rdf::predicate ?stmt dc::relation)
         (rdf::subject ?stmt ?msg)
         (rdf::object ?stmt focus::Translation)
         (s::rating ?stmt ?rating FILTER ?rating > 0)
LITERAL ?stmt IS NULL
ORDER BY ?msg})

      page = (@request['page'] or 1).to_i
      replies = dataset[page - 1].collect {|reply,|
        Resource.new(@request, reply).short
      }

      box(
        _('Replies') + page_number(page),
        list(replies, nav(dataset)),
        'replies'
      ) if replies.size > 0
    end
  end

  def buttons
    message_buttons(@message)
  end

  def moderation_log(page = 1)
    Moderation.find(@id, { :message => true })
  end

  private

  def info
    message_info(@message, :list)
  end
end

class MemberComponent < ResourceComponent
  def initialize(request, id)
    super

    @login, @title = rdf.select_one %{
SELECT ?login, ?full_name
WHERE (s::login #{@id} ?login)
      (s::fullName #{@id} ?full_name)}
    @info = _('Login') + ": #{@login}"

    # used to check if account is blocked
    password, @prefs = db.select_one(
      'SELECT password, prefs FROM Member WHERE id = ?', @id)
    @blocked = password.nil?
  end

  def full
    body = '<p>' + @info + '</p>'

    if @blocked
      blocked_by = yaml_hash(@prefs)['blocked_by'].to_i
      if blocked_by > 0
        b_name, = db.select_one(
          'SELECT full_name FROM Member WHERE id = ?', blocked_by)
        body << '<p>' <<
          sprintf(
            _('Account blocked by moderator: %s.'),
            resource_href(blocked_by, CGI.escapeHTML(b_name))
          ) << '</p>'
      end
    end

    box(nil, body)
  end

  def messages
    dataset = RdfDataSet.new(%{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (dc::creator ?msg #{@id})
      (dct::isVersionOf ?msg ?version_of)
LITERAL ?version_of IS NULL
ORDER BY ?date DESC})

    page = (@request['page'] or 1).to_i
    messages = dataset[page - 1].collect {|msg,|
      Resource.new(@request, msg).short
    }

    if messages.size > 0
      box(
        _('Latest Messages') + page_number(page),
        list(messages, nav(dataset))
      )
    end
  end

  def buttons
    if @request.moderate? and not config['access']['moderators'].include? @login
      # show block/unblock button
      %{<div class="foot"><a class="moderator_action" href="member/#{@id}/} <<
        (@blocked ? 'unblock' : 'block') << '">' <<
        (@blocked ? _('UNBLOCK') : _('BLOCK')) <<
        %{</a></div>\n}
    end
  end
end

class StatementComponent < ResourceComponent
  def initialize(request, id)
    super

    @title = _('Statement') + ' ' + @id.to_s

    @predicate, @subject, @object = rdf.select_one %{
SELECT ?p, ?s, ?o
WHERE (rdf::predicate #{@id} ?p)
      (rdf::subject #{@id} ?s)
      (rdf::object #{@id} ?o)}

    @info = %{
(<a href="#{@predicate}">} + _('Predicate') + %{ #{@predicate}</a>,
<a href="#{@subject}">} + _('Subject') + %{ #{@subject}</a>,
<a href="#{@object}">} + _('Object') + %{ #{@object}</a>)}
  end

  def short
    n = [_('Predicate'), _('Subject'), _('Object')]
    [@predicate, @subject, @object].collect {|resource|
      box(n.shift, Resource.new(@request, resource).list_item)
    }.join
  end

  def full
    short << box(nil,
      '<p><a href="query/run?q=' <<
      CGI.escape("SELECT ?vote WHERE (s::voteProposition ?vote #{@id})") <<
      '">' << _('Votes') << '</a></p>')
  end
end

class VoteComponent < ResourceComponent
  def initialize(request, id)
    super

    @title = _('Vote') + ' ' + @id.to_s

    date, @stmt, @member, name, rating = rdf.select_one %{
SELECT ?date, ?stmt, ?member, ?name, ?rating
WHERE (dc::date #{@id} ?date)
      (s::voteProposition #{@id} ?stmt)
      (s::voteMember #{@id} ?member)
      (s::fullName ?member ?name)
      (s::voteRating #{@id} ?rating)}
    @info = sprintf(_('<a href="%s">%s</a> gave rating %4.2f to the <a href="%s">Statement %s</a> on %s.'),
      @member, name, rating, @stmt, @stmt, format_date(date).to_s)

    @links['made'] = @member
  end

  def short
    box(nil, @info) <<
      box(_('Vote Proposition'), Resource.new(@request, @stmt).short)
  end
end

class ItemComponent < ResourceComponent
end
