/*
 * ipl_tools - Linux for System z reipl and shutdown tools
 *
 * Shared system functions
 *
 * Copyright IBM Corp. 2008, 2010
 * Author(s): Hans-Joachim Picht <hans@linux.vnet.ibm.com>
 *            Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#include "ipl_tools.h"

/*
 * Check if we are running in an LPAR environment
 */
int is_lpar(void)
{
	size_t bytes_read;
	char buf[2048];
	int rc = 0;
	FILE *fh;

	fh = fopen("/proc/cpuinfo", "r");
	if (fh == NULL)
		ERR_EXIT_ERRNO("Could not open \"/proc/cpuinfo\"");
	bytes_read = fread(buf, 1, sizeof(buf), fh);
	if (bytes_read == 0)
		ERR_EXIT("Could not read \"/proc/cpuinfo\"");
	buf[bytes_read] = '\0';
	if (strstr(buf, "version = FF") == NULL)
		rc = 1;
	fclose(fh);
	return rc;
}

/*
 * Check whether we are started as root
 */
int is_root(void)
{
	if (geteuid() == 0)
		return 1;
	else
		return 0;
}

/*
 * Convert a string to lower case
 */
void strlow(char *s)
{
	while (*s) {
		*s = tolower(*s);
		s++;
	}
}

/*
 * Read a string from a particular file
 */
void read_str(char *string, const char *file, size_t len)
{
	char path[PATH_MAX];
	size_t rc;
	FILE *fh;

	snprintf(path, sizeof(path), "/sys/firmware/%s", file);
	fh = fopen(path, "rb");
	if (fh == NULL)
		ERR_EXIT_ERRNO("Could not open \"%s\"", file);
	rc = fread(string, 1, len - 1, fh);
	if (rc == 0 && ferror(fh))
		ERR_EXIT_ERRNO("Could not read \"%s\"", file);
	fclose(fh);
	string[rc] = 0;
	if (string[strlen(string) - 1] == '\n')
		string[strlen(string) - 1] = 0;
}

/*
 * Print content of a file (path = dir/file)
 */
void print_str(const char *fmt, const char *dir, const char *file)
{
	char path[PATH_MAX], str[4096];

	snprintf(path, sizeof(path), "%s/%s", dir, file);
	read_str(str, path, sizeof(str));
	printf(fmt, str);
}

/*
 * Write a string to a file
 */
void write_str(char *string, char *file)
{
	char path[PATH_MAX], value[4096];
	int fh;

	snprintf(value, sizeof(value), "%s\n", string);
	snprintf(path, sizeof(path), "/sys/firmware/%s", file);
	fh = open(path, O_WRONLY);
	if (fh < 0)
		ERR_EXIT_ERRNO("Could not open \"%s\"", file);
	if (write(fh, value, strlen(value)) < 0)
		ERR_EXIT_ERRNO("Could not set \"%s\"", file);
	close(fh);
}
