/*
 * ipl_tools - Linux for System z reipl and shutdown tools
 *
 * Command: lsshut
 *
 * Copyright IBM Corp. 2008, 2010
 * Author(s): Hans-Joachim Picht <hans@linux.vnet.ibm.com>
 *            Michael Holzheu <holzheu@linux.vnet.ibm.com>
 */

#include "ipl_tools.h"

static const char *const usage_lsshut =
"Usage: %s [OPTIONS]\n"
"\n"
"Show actions to be taken after the kernel has shut down.\n"
"\n"
"OPTIONS:\n"
"  -h, --help           Print this help, then exit\n"
"  -v, --version        Print version information, then exit\n";

static void print_usage_lsshut_exit(void)
{
	printf(usage_lsshut, g.prog_name);
	exit(0);
}

static void parse_lsshut_options(int argc, char *argv[])
{

	int opt, idx;
	const struct option long_opts[] = {
		{ "help",	 no_argument,		NULL, 'h' },
		{ "version",	 no_argument,		NULL, 'v' },
		{ NULL,		 0,			NULL,  0  }
	};

	while ((opt = getopt_long(argc, argv, "hv", long_opts, &idx)) != -1) {
		switch (opt) {
		case 'h':
			print_usage_lsshut_exit();
		case 'v':
			print_version_exit();
		default:
			print_help_hint_exit();
		}
	}
	/* don't run with too many arguments */
	if (optind != argc)
		ERR_EXIT("Invalid positional parameter \"%s\" specified",
			 argv[optind]);
}

/*
 * VMCMDs can have up to 128 characters. Newlines mark the end of a CP command.
 * Therefore we can have up to 64 single CP commands (with one character).
 * With quotes (2) and commas (1) we can have at most 4 * 64 = 256 characters
 * for the output string.
 */
static void read_vmcmd(char *str, const char *path)
{
	char buf[128], tmp[512];
	char *ptr_old, *ptr;

	*str = 0;
	ptr_old = ptr = buf;
	read_str(buf, path, sizeof(buf));
	while ((ptr = strchr(ptr_old, '\n'))) {
		*ptr = 0;
		sprintf(tmp, "\"%s\",", ptr_old);
		strcat(str, tmp);
		ptr_old = ptr + 1;
	}
	sprintf(tmp, "\"%s\"", ptr_old);
	strcat(str, tmp);
}

void cmd_lsshut(int argc, char *argv[])
{
	char tmp[1024], cmd[1024];

	parse_lsshut_options(argc, argv);

	printf("Trigger          Action\n");
	printf("========================\n");

	read_str(tmp, "shutdown_actions/on_halt", sizeof(tmp));
	if (strncmp(tmp, "vmcmd", strlen("vmcmd")) == 0) {
		read_vmcmd(cmd, "vmcmd/on_halt");
		printf("Halt             %s (%s)\n", tmp, cmd);
	} else {
		printf("Halt             %s\n", tmp);
	}

	read_str(tmp, "shutdown_actions/on_panic", sizeof(tmp));
	if (strncmp(tmp, "vmcmd", strlen("vmcmd")) == 0) {
		read_vmcmd(cmd, "vmcmd/on_panic");
		printf("Panic            %s (%s)\n", tmp, cmd);
	} else {
		printf("Panic            %s\n", tmp);
	}

	read_str(tmp, "shutdown_actions/on_poff", sizeof(tmp));
	if (strncmp(tmp, "vmcmd", strlen("vmcmd")) == 0) {
		read_vmcmd(cmd, "vmcmd/on_poff");
		printf("Power off        %s (%s)\n", tmp, cmd);
	} else {
		printf("Power off        %s\n", tmp);
	}

	read_str(tmp, "shutdown_actions/on_reboot", sizeof(tmp));
	if (strncmp(tmp, "vmcmd", strlen("vmcmd")) == 0) {
		read_vmcmd(cmd, "vmcmd/on_reboot");
		printf("Reboot           %s (%s)\n", tmp, cmd);
	} else {
		printf("Reboot           %s\n", tmp);
	}
}
