#include <algorithm>
#include <cstddef>
#include <functional>
#include <numeric>
#include <iostream>
#include <string>
#include <stdexcept>
#include <assert.h>
#include <cmath>

#include <rumba/manifold.h>
#include <rumba/arghandler.h>

#include <gauss.h>

#define PRECISION 1e-5




using namespace Gauss;



// get subsample offset corresponding to i_{th} point out of array of size s,
// subsampled down to size N
double getOffset(int i, int N, int s)
{
	int offset = static_cast<int>((i+0.5) * double(s) / double(N));
	assert (offset >=0 && offset < s);

	return offset;
}

std::string help_message()
{
	return std::string ( "usage: qqplot -i filename [--xlower|-l <lower> ] [--xupper|-u  <upper> ] [--table|-t   <table> ]" );
}


RUMBA::Argument myArgs[] = 
{
	RUMBA::Argument("table", RUMBA::ALPHA, 't' ),
	RUMBA::Argument("xlower",RUMBA::NUMERIC, 'l' ),
	RUMBA::Argument("xupper",RUMBA::NUMERIC, 'u' ),
	RUMBA::Argument()
};

void get_table ( const std::string& s, std::map<double,double> & M)
{
	RUMBA::Manifold<double> x ( s.c_str());
	double lower = x.headerData()["validMin"].asDouble();
	double upper = x.headerData()["validMax"].asDouble();

	for ( int i = 0; i < x.size(); ++i )
	{
		M[x[i]+0.5] = (upper - lower)*i / x.size() + lower;
	}
}

double p_to_z ( const std::map<double, double>& M, double search)
{
	double tmp;
	std::map<double,double>::const_iterator i,j;
	i = M.lower_bound(search);

	if (i == M.end()) 
	{
		return M.rbegin()->second;
	}
	if (i == M.begin())
	{
		return M.begin()->second;
	}
	else
	{
		j = i;
		j--;
		return (search - j->first) * (i->second-j->second)/(i->first-j->first)
		+ j->second;
	}
}

int main(int argc, char** argv)
{
	RUMBA::Manifold<double> x;
	std::map<double,double> lookup_table;
	std::string infile;
	std::string table_arg;
	double xlower = -10;
	double xupper = 10;
	double mean=0.0,sumSquares=0.0,sd=0.0,sum=0.0;
	double div_sd = 0, mean_div_sd = 0;
	const int N = 100000;

	double tmp = 0;

	const std::string TABLE_PATH = 
		std::string (RUMBA_PREFIX)  + "/share/rumba/tables/";

	std::vector<double> result( (size_t) N);
	try
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			std::cerr << help_message() << std::endl;
			return 0;
		}
		argh.arg("infile",infile);
		x.load( infile.c_str());


		if (argh.arg("xupper")) argh.arg("xupper", xupper);
		if (argh.arg("xlower")) argh.arg("xlower", xlower);
		

		if (!argh.arg ("table"))
			table_arg = TABLE_PATH + "normal.hdr";
		else
			argh.arg("table", table_arg);

		get_table (table_arg, lookup_table );	

		std::sort(x.begin(), x.end());

		sum=std::accumulate(x.begin(),x.end(),0.0);

		mean = sum/x.size();
		sumSquares=std::inner_product(x.begin(),x.end(),x.begin(),0.0);
		std::transform(x.begin(),x.end(),x.begin(),
			std::bind2nd(std::minus<double>(), mean ));
		sd = std::sqrt( (sumSquares - (sum*sum)/x.size()) / x.size());

		if ( sd < 1e-9 )
			throw RUMBA::Exception ("Standard deviation is too small");

		std::transform(x.begin(),x.end(),x.begin(),
			std::bind2nd(std::divides<double>(), sd ));

		cerr << "sd: " << sd << endl;
		cerr << "mean: " << mean << endl;
		cerr << "size: " << x.size() << endl;

		div_sd = 1/sd;
		mean_div_sd = mean / sd;


		for (int i = 0; i < N; ++i)
		{
			result[i] = x.getElement(getOffset(i,N,x.size()));
			 
		}


		bool flag = false;
		int first_index;	
		for ( int i = 0; i < N; ++i )
		{
			tmp = p_to_z(lookup_table, i/(double)N );
			if (tmp >= xlower && tmp <= xupper )
			{
				std::cout << tmp << " " << result[i]*sd + mean << std::endl;
				// remember index of first plotted point 
				if (!flag){ first_index = i; flag = true; }
			}
		}
		// tack coords of first point to end of file
		std::cout << p_to_z(lookup_table, first_index/(double)N )
			<< " " << result[first_index] * sd +mean << std::endl;
		


	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl;
	}
	catch (RUMBA::Exception& e)
	{
		std::cerr << "Exception: " << e.error() << std::endl;
	}
}
