#include <string>
#include <rumba/manifold.h>
#include <rumba/numerics.h>
#include <rumba/arghandler.h>
#include <cmath>

using std::sqrt;

RUMBA::Argument myArgs [] = 
{
	RUMBA::Argument("phase", RUMBA::ALPHA, 'p', RUMBA::Splodge(), false ),
	RUMBA::Argument()
};

void usage()
{
	std::cerr << "fftmap (--infile|-i) infile (--outfile|o) powermap-file \n"
		" [ (--phase|-p) phasemap-file ]"  << std::endl;

}

template<class TYPE>
void interpolate (std::vector<TYPE> & v, int start_index, int end_index )
{
	double ratio = (end_index - start_index) / (double) v.size();
	double tmp;
	for (int i = v.size()-1; i >= 0; --i )
	{
		tmp = ratio * i;
		v[i] = (tmp - (int)tmp) * (v[(int)(tmp+1)] - v[(int)tmp] )+v[(int)tmp];
	}

}



inline double normsquared(const std::complex<double> & x)
{
	return x.real()*x.real()+x.imag()*x.imag();
}

int next_power_of_2(int x)
{
	int res = 1;
	while (res <x) res *=2;
	return res;
}

int main(int argc,char** argv)
{
	RUMBA::ManifoldFile* fin=0;
	RUMBA::ManifoldFile* power=0;
	RUMBA::ManifoldFile* phase=0;
	RUMBA::intPoint dims;
	int px,tp;
	bool have_phase;
	std::vector<std::complex<double> > v;

	std::string infile,outfile, phasefile;
	try 
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			usage(); 
			return 0;
		}
		argh.arg("infile",infile);
		argh.arg("outfile",outfile);
		have_phase = argh.arg("phase");
		if(have_phase) argh.arg("phase",phasefile);
		fin = RUMBA::ManifoldFile::construct (infile.c_str());
		fin->setCacheStrategy(RUMBA::CACHE_TIME);
		dims = fin->extent();
		dims.t() = (dims.t()+1)/2;
	
		power = RUMBA::ManifoldFile::construct ( 
				outfile.c_str(), "float64", dims 
				);
		power->setCacheStrategy(RUMBA::CACHE_TIME);
	
		if (have_phase)
		{
			phase = RUMBA::ManifoldFile::construct (
					phasefile.c_str(), "float64", dims
					);
			phase->setCacheStrategy(RUMBA::CACHE_TIME);
		}
	
		px = fin->pixels();
		tp = fin->timepoints();

	
		v.resize(next_power_of_2(tp));
	
		for ( int i = 0; i < px; ++i )
		{
			fill(v.begin(),v.end(),complex<double>(0));
			for ( int j = 0; j < tp; ++j )
				v[j] = complex<double>(fin->getElementDouble(j*px+i),0);
			interpolate(v,0,tp);	
			RUMBA::fft(v);
			for (int j =0; j < v.size(); ++j )
				if (norm(v[j]) > 1e18 )
				{
					cerr << "overflow at " << i << endl;
					exit(1);
				}
	
	
			for ( int j = 0; j < power->timepoints(); ++j )
				(*power)[j*px+i] = std::sqrt(norm(v[j]));
	
			if (phase)
				for ( int j = 0; j < power->timepoints(); ++j )
					(*phase)[j*px+i] = arg(v[j]);
		}
	
		if (power) delete power;
		if (phase) delete phase;
	}
	catch ( RUMBA::Exception & e )
	{
		std::cerr << "Fatal: " << e.error() << std::endl;
		exit(1);
	}


	return 0;
}
