#ifndef __RTHREAD_H__
#define __RTHREAD_H__


int rthread_enabled;

/*
 * Win32 thread implementation 
 */ 

#if RTHREAD_USING_WIN32

#include <windows.h>
#include <process.h>

/* Main. */

typedef struct
{
    HANDLE handle;
    int retval;
    char *errormessage;
} RThread;

#define rthread_err(thread) (thread->retval)

char *
rthread_strerror (RThread *thread);

#define rthread_yield_if_coop()


/* Mutexes. */

typedef HANDLE RThreadMutex;

#define RTHREAD_MUTEX_ENTER(mutex) \
    rthread_enabled ? WaitForSingleObject (mutex, INFINITE) : 0; \
    do

#define RTHREAD_MUTEX_EXIT(mutex) \
    while (0); \
    rthread_enabled ? ReleaseMutex (mutex) : 0;
    

#define rthread_mutex_lock(mutex) rthread_enabled ? WaitForSingleObject(mutex, INFINITE) : 0
#define rthread_mutex_unlock(mutex) rthread_enabled ? ReleaseMutex(mutex) : 0

int
rthread_mutex_trylock (RThreadMutex mutex);


/* Events. */

typedef HANDLE RThreadEvent;

#define rthread_event_wait(event) \
    WaitForSingleObject (event, INFINITE)

#define rthread_event_set(event) \
    SetEvent (event)

#define rthread_event_reset(event) \
    ResetEvent (event)

#elif RTHREAD_USING_PTHREADS

/*
 * pthread thread implementation.
 */


/* Main. */

typedef struct
{
    pthread_t pthread;
    int retval;
} RThread;

#define rthread_err(thread) (thread->retval)
#define rthread_strerror(thread) (strerror (thread->retval))

#ifdef HAVE_PTH_YIELD
#define rthread_yield_if_coop pthread_yield_np
#else
#define rthread_yield_if_coop()
#endif /* HAVE_PTH_YIELD */


/* Mutexes. */
typedef pthread_mutex_t * RThreadMutex;

#define RTHREAD_MUTEX_ENTER(mutex) \
    rthread_enabled ? pthread_mutex_lock(mutex) : 0; \
    do

#define RTHREAD_MUTEX_EXIT(mutex) \
    while (0); \
    rthread_enabled ? pthread_mutex_unlock(mutex) : 0;
    
#define rthread_mutex_trylock(mutex) rthread_enabled ? pthread_mutex_trylock(mutex) : 0
#define rthread_mutex_lock(mutex) rthread_enabled ? pthread_mutex_lock(mutex) : 0
#define rthread_mutex_unlock(mutex) rthread_enabled ? pthread_mutex_unlock(mutex) : 0

/* Events. */
struct _RThreadEvent
{
    RThreadMutex mutex;
    pthread_cond_t cond;
    int isset;
    int auto_reset;
};

typedef struct _RThreadEvent * RThreadEvent;

void
rthread_event_wait (RThreadEvent event);

void
rthread_event_set (RThreadEvent event);

void
rthread_event_reset (RThreadEvent event);

#elif RTHREAD_USING_PTH

/*
 * GNU PTH thread implementation.
 * Used as a fallback or alternative
 * to pthreads on UNIX.
 */

/* Main. */

typedef struct
{
    pth_t pthread;
    int retval;
} RThread;

#define rthread_err(thread) (thread->retval)
#define rthread_strerror(thread) (strerror (thread->retval))

#define rthread_yield_if_coop() pth_yield(NULL)


/* Mutexes. */

typedef pth_mutex_t * RThreadMutex;

#define RTHREAD_MUTEX_ENTER(mutex) \
    rthread_enabled ? pth_mutex_acquire(mutex,FALSE,NULL) : 0; \
    do

#define RTHREAD_MUTEX_EXIT(mutex) \
    while (0); \
    rthread_enabled ? pth_mutex_release(mutex) : 0;

#define rthread_mutex_trylock(mutex) rthread_enabled ? pth_mutex_acquire(mutex,TRUE,NULL) : 0
#define rthread_mutex_lock(mutex) rthread_enabled ? pth_mutex_acquire(mutex,FALSE,NULL) : 0
#define rthread_mutex_unlock(mutex) rthread_enabled ? pth_mutex_release(mutex) : 0


/* Events. */

typedef struct _RThreadEvent * RThreadEvent;

struct _RThreadEvent
{
    RThreadMutex mutex;
    pth_cond_t cond;
    int isset;
    int auto_reset;
};

#define RTHREAD_EVENT_AUTO_RESET 0x1
#define RTHREAD_EVENT_SIGNALED 0x2

void
rthread_event_wait (RThreadEvent event);

void
rthread_event_set (RThreadEvent event);

void
rthread_event_reset (RThreadEvent event);

#endif /* RTHREAD_USING_PTH */


/* These are all common interfaces for all threading types. */


typedef void * (RThreadFunction) (void *user_data);

RThread *
rthread_create (RThreadFunction *function, void *user_data);

int
rthread_waitfor (RThread *thread, void **user_data);

RThreadMutex
rthread_mutex_new (void);

void
rthread_mutex_destroy (RThreadMutex mutex);

RThreadEvent
rthread_event_new (int auto_reset);

void
rthread_event_destroy (RThreadEvent event);


void
rthread_exit (void *user_data);

void
rthread_free (RThread *thread);

/*** PRIVATE ***/
void
rthread_init__P (void);

#endif /* __RTHREAD_H__ */

