#include <roy.h>

#if RTHREAD_USING_WIN32

void 
rthread_init__P (void)
{
}


RThread *
rthread_create (RThreadFunction *function, void *user_data)
{
    RThread *thread;
    long tid;
    unsigned long arg = (unsigned long) user_data;

    thread = rchunk_alloc (sizeof (RThread));
    
    if (!rthread_enabled) {
        thread->retval = 1;
        thread->handle = NULL;
        fprintf (stderr, "** If you wish to use threads, you should initilize Roy with rinit_threaded().\n");
        return (thread);
    }

    /* We use _beginthreadex() instead of CreateThread because it will
     * always cleanup behind itself. */

    thread->handle = (HANDLE) _beginthreadex (NULL, 0, (void *) function,
                                              user_data, 0, &tid);
    if (thread->handle == NULL) {
        thread->retval = GetLastError ();
    } else {
        thread->retval = 0;
    }

    thread->errormessage = NULL;

    return (thread);
}

void
rthread_free (RThread *thread)
{
    if (thread->errormessage) {
        LocalFree (thread->errormessage);
    }

    rchunk_free (thread, sizeof (RThread));
}

int
rthread_waitfor (RThread *thread, void **user_data)
{
    long exit_code;
    int retval;

    if (!rthread_enabled)
        return (-1);

    retval = WaitForSingleObject (thread->handle, INFINITE);
    if (user_data) {
        GetExitCodeThread (thread->handle, &exit_code);
        *user_data = (void *) exit_code;
    }
    
    thread->retval = retval;
    
    return (retval);
}

void
rthread_exit (void *user_data)
{
    _endthreadex ((unsigned int) user_data);
}

char *
rthread_strerror (RThread *thread)
{
    LPVOID lpMsgBuf;

    FormatMessage (FORMAT_MESSAGE_ALLOCATE_BUFFER | 
                   FORMAT_MESSAGE_FROM_SYSTEM | 
                   FORMAT_MESSAGE_IGNORE_INSERTS,
                   NULL,
                   thread->retval,
                   MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                   (LPTSTR) &lpMsgBuf,
                   0,
                   NULL );

    thread->errormessage = (char *) lpMsgBuf;
    return ((char *) lpMsgBuf);
}

RThreadMutex
rthread_mutex_new (void)
{
    RThreadMutex mutex = NULL;
    
    if (rthread_enabled)
        mutex = CreateMutex (NULL, FALSE, NULL);

    return (mutex);
}

int
rthread_mutex_trylock (RThreadMutex mutex)
{
    int retval;
    
    if (!rthread_enabled)
        return (1);

    retval = WaitForSingleObject (mutex, 0);
    if (retval == WAIT_FAILED ||
        retval == WAIT_TIMEOUT) {
        return (1);
    } else {
        return (0);
    }
}

void
rthread_mutex_destroy (RThreadMutex mutex)
{
    if (rthread_enabled)
        CloseHandle (mutex);
}


RThreadEvent
rthread_event_new (int auto_reset)
{
    RThreadEvent event = NULL;

    /* Second argument to CreateEvent() is 'manual' event,
     * so we have to negate the argument passed by the user. */
    if (rthread_enabled)
        event = CreateEvent (NULL, !auto_reset, FALSE, NULL);

    return (event);
}

void
rthread_event_destroy (RThreadEvent event)
{
    if (rthread_enabled)
        CloseHandle (event);
}



#elif RTHREAD_USING_PTHREADS 

void 
rthread_init__P (void)
{
}

RThread *
rthread_create (RThreadFunction *function, void *user_data)
{
    RThread *thread;

    thread = rchunk_alloc (sizeof (RThread));

    if (rthread_enabled) {
        thread->retval = pthread_create (&thread->pthread, NULL, function, user_data);
    } else {
        thread->retval = EAGAIN;
        fprintf (stderr, "** If you wish to use threads, you should initilize Roy with rinit_threaded().\n");
    }

    return (thread);
}

void
rthread_free (RThread *thread)
{
    rchunk_free (thread, sizeof (RThread));
}

int
rthread_waitfor (RThread *thread, void **user_data)
{
    int retval;

    if (!rthread_enabled)
        return (-1);
    
    retval = pthread_join (thread->pthread, user_data);
    thread->retval = retval;

    return (retval);
}

void
rthread_exit (void *user_data)
{
    pthread_exit (user_data);
}

RThreadMutex
rthread_mutex_new (void)
{
    RThreadMutex mutex;

    if (!rthread_enabled)
        return (NULL);

    /* This must be done with rmem_alloc() because rchunk
     * requires a mutex to work. */
    mutex = rmem_alloc (sizeof (pthread_mutex_t));
    pthread_mutex_init (mutex, NULL);

    return (mutex);
}

void
rthread_mutex_destroy (RThreadMutex mutex)
{
    if (!rthread_enabled)
        return;

    pthread_mutex_destroy (mutex);
    rmem_free (mutex);
}


RThreadEvent
rthread_event_new (int auto_reset)
{
    RThreadEvent event;
    
    if (!rthread_enabled)
        return (NULL);

    event = rchunk_alloc (sizeof (struct _RThreadEvent));
    
    event->auto_reset = auto_reset;
    event->isset = FALSE;
    pthread_cond_init (&event->cond, NULL);
    event->mutex = rthread_mutex_new ();

    return (event);
}

void
rthread_event_destroy (RThreadEvent event)
{
    if (!rthread_enabled)
        return;

    rthread_mutex_destroy (event->mutex);
    pthread_cond_destroy (&event->cond);
    rchunk_free (event, sizeof (struct _RThreadEvent));
}

void
rthread_event_wait (RThreadEvent event)
{
    if (!rthread_enabled)
        return;

    /* First thing we have to do is grab the 
     * mutex lock. */
   
    RTHREAD_MUTEX_ENTER (event->mutex) {
        if (event->isset == FALSE) {
            pthread_cond_wait (&event->cond, event->mutex);
        }
        if (event->auto_reset) {
            event->isset = FALSE;
        }
    } RTHREAD_MUTEX_EXIT (event->mutex);
}

void
rthread_event_set (RThreadEvent event)
{
    if (!rthread_enabled)
        return;
    
    event->isset = TRUE;
   
    if (event->auto_reset) {
        pthread_cond_signal (&event->cond);
    } else {
        pthread_cond_broadcast (&event->cond);
    }
}

void
rthread_event_reset (RThreadEvent event)
{
    if (!rthread_enabled)
        return;

    RTHREAD_MUTEX_ENTER (event->mutex) {
        event->isset = FALSE;
    } RTHREAD_MUTEX_EXIT (event->mutex);
}

#elif RTHREAD_USING_PTH

void 
rthread_init__P (void)
{
    if (rthread_enabled) {
        pth_init ();
    }
}

RThread *
rthread_create (RThreadFunction *function, void *user_data)
{
    RThread *thread;

    thread = rchunk_alloc (sizeof (RThread));
        
    if (!rthread_enabled) {
        thread->pthread = NULL;
        thread->retval = EAGAIN;
        fprintf (stderr, "** If you wish to use threads, you should initilize Roy with rinit_threaded().\n");
        return (thread);
    }

    thread->pthread = pth_spawn (NULL, function, user_data);
    if (thread->pthread == NULL)
        thread->retval = errno;

    return (thread);
}

void
rthread_free (RThread *thread)
{
    rchunk_free (thread, sizeof (RThread));
}

int
rthread_waitfor (RThread *thread, void **user_data)
{
    if (!rthread_enabled)
        return (-1);

    if (!pth_join (thread->pthread, user_data)) {
        thread->retval = errno;
        return (errno);
    }
    return (0);
}

void
rthread_exit (void *user_data)
{
    if (rthread_enabled) {
        pth_exit (user_data);
    }
}


RThreadMutex
rthread_mutex_new (void)
{
    RThreadMutex mutex;

    if (!rthread_enabled)
        return (NULL);

    /* This must be done with rmem_alloc() because rchunk
     * requires a mutex to work. */
    mutex = rmem_alloc (sizeof (pth_mutex_t));
    if (!pth_mutex_init (mutex))
        return (NULL);

    return (mutex);
}

void
rthread_mutex_destroy (RThreadMutex mutex)
{
    if (!rthread_enabled)
        return;

    rmem_free (mutex);
    mutex = NULL;
}

RThreadEvent
rthread_event_new (int auto_reset)
{
    RThreadEvent event;
    
    if (!rthread_enabled)
        return (NULL);

    event = rchunk_alloc (sizeof (struct _RThreadEvent));
    
    event->auto_reset = auto_reset;
    event->isset = FALSE;
    pth_cond_init (&event->cond);
    event->mutex = rthread_mutex_new ();

    return (event);
}

void
rthread_event_destroy (RThreadEvent event)
{
    if (!rthread_enabled)
        return;

    rthread_mutex_destroy (event->mutex);
    rchunk_free (event, sizeof (struct _RThreadEvent));
}

void
rthread_event_wait (RThreadEvent event)
{
    if (!rthread_enabled)
        return;

    RTHREAD_MUTEX_ENTER (event->mutex) {
        if (event->isset == FALSE) {
            pth_cond_await (&event->cond, event->mutex, NULL);
        }
        if (event->auto_reset) {
            event->isset = FALSE;
        }
    } RTHREAD_MUTEX_EXIT (event->mutex);
}

void
rthread_event_set (RThreadEvent event)
{
    if (!rthread_enabled)
        return;
    
    event->isset = TRUE;
    
    /* Second arg is set to TRUE to do a broadcast style
     * wakeup, so we just use negated auto_reset variable. */
    pth_cond_notify (&event->cond, !event->auto_reset);
}

void
rthread_event_reset (RThreadEvent event)
{
    if (!rthread_enabled)
        return;

    RTHREAD_MUTEX_ENTER (event->mutex) {
        event->isset = FALSE;
    } RTHREAD_MUTEX_EXIT (event->mutex);
}


#endif /* PTH */


