
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#if !RCHUNK_ENABLED

void
rchunk_free (void *chunk, unsigned int size)
{
    free (chunk);
}


void *
rchunk_alloc (unsigned int size)
{
    return (malloc (size));
}

void *
rchunk_alloc0 (unsigned int size)
{
    return (calloc (size, 1));
}

void
rchunk_cleanup__P (void)
{
    return;
}

void
rchunk_init__P (void)
{
    return;
}


#else


/* This is just used as a 'template' over the raw
 * chunks so we can setup the linked list easily
 * using the pointer within it. */
typedef struct _RChunk RChunk;

struct _RChunk
{
    RChunk *next;   
};


/* Maximum allocation size by rchunk */

#define ROY_CHUNK_MAX_SIZE sizeof (void *) * 8192

/* divide by the size of a pointer because we only allocate on
 * multiples of pointer size */
static RChunk *rchunk_lists[ROY_CHUNK_MAX_SIZE / sizeof (void *)];

static RThreadMutex rchunk_lists_mutex;


void
rchunk_cleanup__P (void)
{
    rthread_mutex_destroy (rchunk_lists_mutex);
}


void
rchunk_init__P (void)
{
    rchunk_lists_mutex = rthread_mutex_new ();
}


void
rchunk_free (void *ptr, unsigned int size)
{
    int listnum = 0;
    RChunk *chunk = ptr;

    if (chunk == NULL)
        return;

    /* Size must be a multiple of the size of a pointer on some
     * systems. The common case is that the size is already divisable by
     * the size of a pointer. If it's not however, we must adjust
     * it.  We let integer rounding take care of it. */
    if (size % sizeof (void *)) {
        size = ((size / sizeof (void *)) + 1) * sizeof (void *);
    }

    listnum = size / sizeof (void *);

    RTHREAD_MUTEX_ENTER (rchunk_lists_mutex) {
        /* Make the new chunk point to the previous head of the list. */
        chunk->next = rchunk_lists[listnum];

        /* Make the new chunk the new head. */
        rchunk_lists[listnum] = chunk;
    } RTHREAD_MUTEX_EXIT (rchunk_lists_mutex);
}


void *
rchunk_alloc0 (unsigned int size)
{
    void *mem;
    mem = rchunk_alloc (size);
    memset (mem, 0, size);
    return (mem);
}

#define NUMCHUNKS 18
/* #define NUMCHUNKS 128 */

void *
rchunk_alloc (unsigned int size)
{
    int listnum = 0;
    RChunk *chunk;

    /* Size must be a multiple of the size of a pointer on some
     * systems. The common case is that the size is already divisable by
     * the size of a pointer. If it's not however, we must adjust
     * it.  We let integer rounding take care of it. */
    if (size % sizeof (void *)) {
        size = ((size / sizeof (void *)) + 1) * sizeof (void *);
    }

    /* We find our list by looking straight into the lists array, after we
     * divide by the size of a pointer, which is how we round off the
     * sizes anyway.  This gives us a lot higher density in our chunk
     * lists. */
    listnum = size / sizeof (void *);

    RTHREAD_MUTEX_ENTER (rchunk_lists_mutex) {
       
        /* If this chunk list is empty, we have to allocate more for it. */
        if (!rchunk_lists[listnum]) {
            char *newmem;
            RChunk *curchunk;
            RChunk *nextchunk;
            int i;

            /* Allocate chunks.  We found the point of diminishing return
             * was at about 12 chunks or so so we went to allocating 18
             * at a time. */
            newmem = malloc (NUMCHUNKS * size);

            curchunk = (RChunk *) newmem;

            rchunk_lists[listnum] = curchunk;

            for (i = 1; i < NUMCHUNKS; i++) {
                newmem += size;
                nextchunk = (RChunk *) newmem;

                curchunk->next = nextchunk;

                curchunk = nextchunk;
            }

            /* ensure that the last block has a NULL pointer */
            curchunk->next = NULL;
        }
        
        /* Get our memory from the first available block */
        chunk = rchunk_lists[listnum];

        /* Update the head of our chunk list to point to the next chunk. */
        rchunk_lists[listnum] = chunk->next;

    } RTHREAD_MUTEX_EXIT (rchunk_lists_mutex);

    return ((void *)chunk);
}

#endif /* RCHUNK_ENABLED */


