/* main.c: main(), initialisation and cleanup*/

/*  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License , or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  You may contact the author by e-mail:  hlub@knoware.nl
 */

#include "rlwrap.h"

/* global vars */
int master_pty_fd;		     /* master pty (rlwrap uses this to communicate with client) */
int slave_pty_fd;		     /* slave pty (client uses this to communicate with rlwrap,
				      * we keep it open after forking in order to keep track of
				      * client's terminal settings */
FILE *debug_fp;			     /* filehandle of debugging log (!= stderr) */
char *program_name, *command_name;   /* "rlwrap" and (base-)name of command */
int within_line_edit = FALSE;	     /* TRUE while user is editing input */
int always_readline = FALSE;	     /* -a option */
char *password_prompt_search_string  /* argument of -a option (cf manpage) */
   = NULL;
char *history_format = NULL;         /* format for history entries */
int complete_filenames = FALSE;	     /* -c option */
int child_pid = 0;		     /* pid of child (client) */
int i_am_child = FALSE;		     /* after forking, child will set this to TRUE */
int nowarn = FALSE;		     /* argument of -n option (suppress warnings) */
int debug = 0;			     /* debugging mask (0 = no debugging) */
int ignore_queued_input = FALSE;     /* read and then ignore all characters in input queue until it is empty (i.e. read would block) */
int history_duplicate_avoidance_policy =
   ELIMINATE_SUCCESIVE_DOUBLES;      /* whether and how to avoid duplicate history entries */
int one_shot_rlwrap = FALSE;

/* private vars */

static char *history_filename = NULL;
static int  histsize = 300;
static int  write_histfile = TRUE;
static char *completion_filename, *default_completion_filename;
static char *full_program_name;
static void init_rlwrap(void);
static void fork_child(char *command_name, char **argv);
static char *read_options_and_command_name(int argc, char **argv);
static void main_loop(void);
static void flush_output_queue(void);
static int  output_queue_is_nonempty(void);
static int  last_option_didnt_have_optional_argument = FALSE;
static int  last_opt = -1;
static int  bleach_input = FALSE;


int
main(int argc, char **argv)
{
  char *command_name;
  init_completer();
  command_name = read_options_and_command_name(argc, argv);
  /* by now, optind points to <command>, and &argv[optind] is <command>'s argv */
  if (!isatty(STDIN_FILENO) && execvp(argv[optind], &argv[optind]) < 0)
       /* if stdin is not a tty, just execute <command> */ 
       myerror("Cannot execute %s", argv[optind]);	
  init_rlwrap();
  fork_child(command_name, argv);
  main_loop();
  return 42;			/* not reached, but some compilers are unhappy without this ... */
}


/*
 * create pty pair and fork using my_pty_fork; parent returns immediately; child
 * executes the part of rlwrap's command line that remains after
 * read_options_and_command_name() has harvested rlwrap's own options
 */  
void
fork_child(char *command_name, char **argv)
{
  int pid =
    my_pty_fork(&master_pty_fd, &saved_terminal_settings, &window_size);
  if (pid > 0)			/* parent: */
    child_pid = pid;
  else {			/* child: */
    DPRINTF1(DEBUG_TERMIO, "preparing to execute %s", argv[optind]);
    close_open_files_without_writing_buffers();
   
    if (execvp(argv[optind], &argv[optind]) < 0) {
      if (last_opt > 0 && last_option_didnt_have_optional_argument) /* e.g. 'rlwrap -a Password: sqlpus' will try to exec 'Password:' */
	fprintf(stderr, "Did you mean '%s' to be an option argument?\nThen you should write '-%c%s', without the space(s)\n",
		      argv[optind], last_opt, argv[optind]); 
      myerror("Cannot execute %s", argv[optind]);   	/* stillborn child, parent will live on and display child's last gasps */
    }
  }
}


/*
 * main loop: listen on stdin (for user input) and master pty (for command output),
 * and try to write output_queue to master_pty (if it is not empty)
 * This function never returns.
 */
void
main_loop()
{				
  int nfds;			
  fd_set readfds;	
  fd_set writefds;
  int nread;		
  struct timeval immediately = { 0, 0 }; /* zero timeout when child is dead */
  struct timeval *forever = NULL;
  char buf[BUFFSIZE], *last_nl;
  int messysigs[] = { SIGWINCH, SIGTSTP, 0 };	/* we don't want to catch these
						   while readline is processing */
  int termwidth;		/* width of terminal window */
  int promptlen = 0, removed;	/* used when prompt is wider than terminal */


  init_readline("");
  set_echo(FALSE);		/* This will also put the terminal in CBREAK mode */

  /* ------------------------------  main loop  -------------------------------*/
  while (TRUE) {
    /* listen on both stdin and pty_fd */
    FD_ZERO(&readfds);
    FD_SET(STDIN_FILENO, &readfds);
    FD_SET(master_pty_fd, &readfds);

    /* try to write output_queue to master_pty (if it is nonempty) */
    FD_ZERO(&writefds);
    if (output_queue_is_nonempty())
      FD_SET(master_pty_fd, &writefds);


    unblock_signals(messysigs);	/* Handle SIGWINCH and SIGTSTP now (and only now) */
    nfds = select(1 + master_pty_fd,
		  &readfds,
		  &writefds,
		  (fd_set *) NULL, (child_is_dead || ignore_queued_input ? &immediately : forever));
    block_signals(messysigs);	/* Dont't disturb now */

    DPRINTF3(DEBUG_TERMIO, "select(..) = %d (%s), within_line_edit=%d", nfds,
	     (FD_ISSET(master_pty_fd, &readfds) ? "pty" : "stdin"),
	     within_line_edit);


    
    if (nfds < 0) {		/* exception  */
      if (errno == EINTR)	/* interrupted by signal, don't worry */
	continue;
      else
	myerror("select received exception");
    } else if (nfds == 0) {	/* timeout, which can only happen when .. */
      if (ignore_queued_input) {       /* we have read all the input keystrokes that should be ignored */
	ignore_queued_input = FALSE;
	continue;
      }                         /* or else, ic child is dead, ... */
      if (promptlen > 0)	/* ... its last words were not terminated by \n ... */
	my_putchar('\n');	/* print the \n ourselves */
      DPRINTF2(DEBUG_SIGNALS,
	       "select returned 0, child_is_dead=%d, childs_exit_status=%d",
	       child_is_dead, childs_exit_status);
      cleanup_rlwrap_and_exit(EXIT_SUCCESS);
    } else if (nfds > 0) {	/* something to read or write */

      /* --------------------- read pty ------------------------------*/
      if (FD_ISSET(master_pty_fd, &readfds)) {
 
	if ((nread = read(master_pty_fd, buf, BUFFSIZE - 1)) <= 0) {
	  if (nread == EINTR)	/* interrupted by signal, don't worry */
	    continue;
	  else if (child_is_dead || nread < 0) {
	    if (promptlen > 0)	/* commands dying words not terminated by \n ... */
	      my_putchar('\n');	/* provide the missing \n */
	    cleanup_rlwrap_and_exit(EXIT_SUCCESS);
	  } else
	    myerror("read error on master pty");
	}
	buf[nread] = '\0';

        if (bleach_input)
	  bleach(buf);
	
	if (within_line_edit)	/* client output arrives while we're editing keyboard input  */
	  save_rl_state(&saved_rl_state);

	if (remember_for_completion)
	  feed_line_into_completion_list(buf);	/* here we assume that buf doesn't end with an incomplete word
						   user *in*put is remembered this way too if we assume that <command>
						   echoes user input. Both assumptions are reasonable for line-oriented
						   interactive programs */

	write(STDOUT_FILENO, buf, strlen(buf));
	DPRINTF2(DEBUG_READLINE, "wrote %d bytes: %s", strlen(buf),
		 mangle_string_for_debug_log(buf, 40));

	write_logfile(buf);

	/* now determine the text *after* the last newline. This wil be the
	   "prompt" for the readline input routine; we'll update
	   saved_rl_state.prompt accordingly   */
	last_nl = strrchr(buf, '\n');
	if (last_nl != NULL) {
	  /* newline seen, will get new prompt: */
	  mystrlcpy(saved_rl_state.prompt, last_nl + 1,
		    sizeof(saved_rl_state.prompt));
	} else {
	  /* no newline, extend old prompt: */
	  /* prompts longer than BUFFSIZE will be truncated */
	  mystrlcat(saved_rl_state.prompt, buf,
		    sizeof(saved_rl_state.prompt));
	}


	/* If the prompt is wider than the current terminal width,
	   assume that it wrapped around and take only the last line of it to be the
	   new prompt. This may go wrong if the prompt contains BS or control
	   sequences (which is unusual for programs needing rlwrap)  @@@FIXME? */

	termwidth = window_size.ws_col;
	promptlen = strlen(saved_rl_state.prompt);
	if (termwidth != 0 &&	/* this might sometimes be 0 on some weird systems */
	    promptlen > termwidth) {
	  removed = (promptlen / termwidth) * termwidth;	/* integer arithmetic */
	  memmove(saved_rl_state.prompt, saved_rl_state.prompt + removed, strlen(saved_rl_state.prompt) + 1 - removed);
	      /* +1, as we want to move the final '\0' as well */
	}

	DPRINTF2(DEBUG_READLINE, "Prompt now: '%s' (%d bytes)",
		 mangle_string_for_debug_log(saved_rl_state.prompt, 40),
		 strlen(saved_rl_state.prompt));
	saved_rl_state.prompt[BUFFSIZE - 1] = '\0';	/* just to make sure.... */
        
	if (within_line_edit)
	  restore_rl_state(&saved_rl_state);
	
      }

      /* ----------------------------- key pressed: read stdin -------------------------*/
      if (FD_ISSET(STDIN_FILENO, &readfds)) {	/* key pressed */
	nread = read(STDIN_FILENO, buf, 1);	/* read next byte of input   */
	if (nread < 0)
	  myerror("Unexpected error");
	else if (ignore_queued_input)
	  continue;             /* do nothing with it*/
	else if (nread == 0)	/* EOF on stdin */
	  cleanup_rlwrap_and_exit(EXIT_SUCCESS);

	if (slave_is_in_raw_mode() && !always_readline) {	/* just pass it on */
	  buf[nread] = '\0';	/* only necessary for DPRINTF, 
				   the '\0' won't be written to master_pty */
	  DPRINTF2(DEBUG_READLINE,
		   "passed on %d bytes in transparent mode: %s", strlen(buf),
		   mangle_string_for_debug_log(buf, 40));
	  write(master_pty_fd, buf, nread);
	} else {		/* hand it over to readline */
	  if (!within_line_edit) {	/* start a new line edit    */
	    DPRINTF0(DEBUG_READLINE, "Starting line edit");
	    within_line_edit = TRUE;
	    restore_rl_state(&saved_rl_state);
	  }

	  rl_pending_input = buf[0];	/* stuff it back in readlines input queue */

	  DPRINTF2(DEBUG_READLINE, "Character %d (%c)", rl_pending_input,
		   (isalpha(rl_pending_input) ? rl_pending_input : '.'));

	  if (buf[0] == term_eof && strlen(rl_line_buffer) == 0)	/* kludge: readline on Solaris does not interpret CTRL-D as EOF */
	    line_handler(NULL);
	  else
	    rl_callback_read_char();
	}
      }
      /* -------------------------- write pty --------------------------------- */
      if (FD_ISSET(master_pty_fd, &writefds))
	flush_output_queue();


    }				/* if (ndfs > 0)         */
  }				/* while (1)             */
}				/* void main_loop()      */


/* Read history and completion word lists */
void
init_rlwrap()
{

  char *homedir, *histdir, *homedir_prefix;

  /* open debug file if necessary */

  if (debug) {    
    debug_fp = fopen(DEBUG_FILENAME, "w");
    if (!debug_fp)
      myerror("Couldn't open debug file %s", DEBUG_FILENAME);
    setbuf(debug_fp, NULL);
  }

  DPRINTF0(DEBUG_TERMIO, "Initialising");
  init_terminal();


  /* Determine rlwrap home dir and prefix for default history and completion filenames */
  homedir = (getenv("RLWRAP_HOME") ? getenv("RLWRAP_HOME") : getenv("HOME"));
  homedir_prefix = (getenv("RLWRAP_HOME") ?                    /* is RLWRAP_HOME set?                */
		    add2strings(getenv("RLWRAP_HOME"), "/") :  /* use $RLWRAP_HOME/<command>_history */
		    add2strings(getenv("HOME"), "/."));	       /* if not, use ~/.<command>_history   */

  /* Determine history file name and check its existence and permissions */

  if (history_filename) {
    histdir = mydirname(history_filename);
  } else {
    histdir = homedir;
    history_filename = add3strings(homedir_prefix, command_name, "_history");
  }
  if (write_histfile) {
    if (access(history_filename, F_OK) == 0) {	/* already exists, can we read/write it? */
      if (access(history_filename, R_OK | W_OK) != 0) {
	myerror("cannot read and write %s", history_filename);
      }
    } else {			/* doesn't exist, can we create it? */
      if (access(histdir, W_OK) != 0) {
	myerror("cannot create history file in %s", histdir);
      }
    }
  } else {			/* ! write_histfile */
    if (access(history_filename, R_OK) != 0) {
      myerror("cannot read %s", history_filename);
    }
  }

  /* Initialize history */
  using_history();
  stifle_history(histsize);
  read_history(history_filename);	/* ignore errors here: history file may not yet exist, but will be created on exit */


  /* Determine completion file name (completion files are never written to,
     and ignored when unreadable or non-existent) */

  completion_filename =
    add3strings(homedir_prefix, command_name, "_completions");
  default_completion_filename =
    add3strings(DATADIR, "/rlwrap/", command_name);

  rl_readline_name = command_name;

  /* Initialize completion list (if <completion_filename> is readable) */
  if (access(completion_filename, R_OK) == 0) {
    feed_file_into_completion_list(completion_filename);
  } else if (access(default_completion_filename, R_OK) == 0) {
    feed_file_into_completion_list(default_completion_filename);
  }


}

/*
 * On systems where getopt doens't handle optional argments, warn the user whenever an
 * argument of the form -<letter> is seen, or whenever the argument is the last item on the command line
 * (e.g. 'rlwrap -a command', which will be parsed as 'rlwrap --always-readline=command')
 */

static char *
check_optarg(char opt, int remaining)
{
  if (!optarg)
    last_option_didnt_have_optional_argument = TRUE; /* if command is not found, suggest that it may have been meant
						        as optional argument (e.g. 'rlwrap -a password sqlplus' will try to
						        execute 'password' ) */
#ifndef GETOPT_GROKS_OPTIONAL_ARGS
  if (optarg &&    /* is there an optional arg? have a look at it: */
      ((optarg[0] == '-' && isalpha(optarg[1])) || /* looks like next option */
       remaining == 0)) /* or is last item on command line */

  mywarn
      ("on this system, the getopt() library function doesn't\n"
       "grok optional arguments, so '%s' is taken as an argument to the -%c option\n"
       "Is this what you meant? If not, please provide an argument"
#ifdef HAVE_GETOPT_LONG
       ", or use a --long-option.\n",
#else
       ".\n",
#endif
       optarg, opt);
#endif
  
  return optarg;
}


#ifdef GETOPT_GROKS_OPTIONAL_ARGS
static char optstring[] = "+:a::b:cC:d::D:f:F:hH:il:nm::P:rs:tv";	/* +: is not really documented. configure checks wheteher it works as expected
					        			   if not, GETOPT_GROKS_OPTIONAL_ARGS is undefined. @@@ */
#else
static char optstring[] = "+:a:b:cC:d:D:f:F:hH:il:nm:P:rs:tv";	/* it is possible that on some systems :: works but not +:. If so, please tell me! */
#endif

#ifdef HAVE_GETOPT_LONG
static struct option longopts[] = {
  {"always-readline", 		optional_argument, 	NULL, 'a'},
  {"break-chars", 		required_argument, 	NULL, 'b'},
  /*  {"bleach",                    no_argument,		NULL, 'B'}, still not usable */
  {"complete-filenames", 	no_argument, 		NULL, 'c'},
  {"command-name", 		required_argument, 	NULL, 'C'},
  {"debug", 			optional_argument, 	NULL, 'd'},
  {"history-no-dupes", 		required_argument, 	NULL, 'D'},
  {"file", 			required_argument, 	NULL, 'f'},
  {"history-format", 		required_argument, 	NULL, 'F'},
  {"help", 			no_argument, 		NULL, 'h'},
  {"history-filename", 		required_argument, 	NULL, 'H'},
  {"case-insensitive", 		no_argument, 		NULL, 'i'},
  {"logfile", 			required_argument, 	NULL, 'l'},
  {"no-warnings", 		no_argument, 		NULL, 'n'},
  {"multi-line", 		optional_argument, 	NULL, 'm'},
  {"pre-given", 		required_argument, 	NULL, 'P'},
  {"remember", 			no_argument, 		NULL, 'r'},
  {"version", 			no_argument, 		NULL, 'v'},
  {"histsize", 			required_argument, 	NULL, 's'},
  {"test-terminal",  		no_argument, 		NULL, 't'},
  {0, 0, 0, 0}
};
#endif


static const char *current_option(int opt, int longindex) {
  static char buf[BUFFSIZE];
#ifdef HAVE_GETOPT_LONG
  if (longindex >=0) {
    sprintf(buf, "--%s", longopts[longindex].name);
    return buf;
  }	
#endif
  sprintf(buf, "-%c", opt);
  return buf;
}

char *
read_options_and_command_name(int argc, char **argv)
{
  int pid, c;
  char *opt_C = NULL;
  int opt_b = FALSE;
  int opt_f = FALSE;
  int remaining = -1; /* remaining number of argumants on command line */
  int longindex = -1; /* index of current option in longopts[], set by getopt_long */
  
  full_program_name = mysavestring(argv[0]);
  program_name = mysavestring(mybasename(full_program_name));	/* normally "rlwrap"; needed by myerror() */
  rl_basic_word_break_characters = " \t\n\r(){}[],+-=&^%$#@\";|\\";

  opterr = 0;			/* we do our own error reporting */

  while (1) {
#ifdef HAVE_GETOPT_LONG
    c = getopt_long(argc, argv, optstring, longopts, &longindex);
#else
    c = getopt(argc, argv, optstring);
#endif

    if (c == EOF)
      break;
    last_option_didnt_have_optional_argument = FALSE;
    remaining = argc - optind;
    last_opt = c;    
    switch (c) {
      case 'a':
	always_readline = TRUE;
	if (check_optarg('a', remaining))
	  password_prompt_search_string = mysavestring(optarg);
	break;
      case 'b':
	rl_basic_word_break_characters = add3strings("\r\n \t", optarg, "");
	opt_b = TRUE;
	break;
      case 'B':
	bleach_input = TRUE;
	break;
      case 'c':
	complete_filenames = TRUE;
	break;
      case 'C':
	opt_C = mysavestring(optarg);	
	break;
      case 'd':
#ifdef DEBUG
	if (check_optarg('d', remaining))
	  debug = atoi(optarg);
	else
	  debug = DEBUG_ALL;
#else
	mywarn
	  ("To use debugging, configure %s with --enable-debug and rebuild",
	   program_name);
#endif
	break;

      case 'D': 
        history_duplicate_avoidance_policy=atoi(optarg);
	if (history_duplicate_avoidance_policy < 0 || history_duplicate_avoidance_policy > 2)
	  myerror("%s option with illegal value %d, should be 0, 1 or 2",
		  current_option('D', longindex), history_duplicate_avoidance_policy);
	break;
      case 'f':
	feed_file_into_completion_list(optarg);
	opt_f = TRUE;
	break;
      case 'F':
        history_format = mysavestring(optarg);
	if (isspace(history_format[0]))
	  myerror("%s option argument should start with non-space", current_option('F', longindex));
	if(history_format[0] == '%' && history_format[1] != ' ')
	  myerror("if %s option argument starts with '%%', is should start with '%% ' i.e. '%%<space>'", current_option('F', longindex));
	break;
      case 'h':
	usage(EXIT_SUCCESS);		/* will call exit() */
      case 'H':
	history_filename = mysavestring(optarg);
	break;
      case 'i':
	if (opt_f)
	  myerror("-i option has to precede -f options");
	completion_is_case_sensitive = FALSE;
	break;
      case 'l':
	open_logfile(optarg);
	break;
      case 'n':
	nowarn = TRUE;
	break;
      case 'm':
#ifndef HAVE_SYSTEM
	mywarn("the -m option doesn't work on this system");
#endif
	multiline_separator =
	  (check_optarg('m', remaining) ? mysavestring(optarg) : " \\ ");
	break;
      case 'P':
        pre_given = mysavestring(optarg);
	always_readline = TRUE; /* pre_given does not work well with transparent mode */
	one_shot_rlwrap = TRUE;
	break;
      case 'r':
	remember_for_completion = TRUE;
	break;
      case 's':
	histsize = atoi(optarg);
	if (histsize < 0) {
	  write_histfile = 0;
	  histsize = -histsize;
	}
	break;
#ifdef DEBUG
      case 't':
	test_terminal();
        exit(EXIT_SUCCESS);
#endif
      case 'v':
	printf("rlwrap %s\n",  VERSION);
	exit(EXIT_SUCCESS);
      case '?':
	assert(optind > 0);
	myerror("unrecognised option %s\ntry '%s --help' for more information",
		argv[optind-1], full_program_name);
      case ':':
	assert(optind > 0);
	myerror
	  ("option %s requires an argument \ntry '%s --help' for more information",
	   argv[optind-1], full_program_name);
      default:
	usage(EXIT_FAILURE);
    }
  }

  if (!complete_filenames && !opt_b) {	/* use / and . as default breaking characters whenever we don't complete filenames */
    rl_basic_word_break_characters =
      add3strings(rl_basic_word_break_characters, "/.", "");
  }

  
  if (optind >= argc) /* rlwrap -a -b -c with no command specified */
    usage(EXIT_FAILURE);

  if (opt_C) {
    int countback = atoi(opt_C);	/* try whether -C option is numeric */

    if (countback > 0) {	/* e.g -C 1 or -C 12 */
      if (argc - countback < optind)	/* -C 66 */
	myerror("when using -C %d you need at least %d non-option arguments",
		countback, countback);
      else if (argv[argc - countback][0] == '-')	/* -C 2 perl -d blah.pl */
	myerror("the last argument minus %d appears to be an option!",
		countback);
      else {			/* -C 1 perl test.cgi */
	command_name = mysavestring(mybasename(argv[argc - countback]));

      }
    } else if (countback == 0) {	/* -C name1 name2 or -C 0 */
      if (opt_C[0] == '0' && opt_C[1] == '\0')	/* -C 0 */
	myerror("-C 0 makes no sense");
      else if (strlen(mybasename(opt_C)) != strlen(opt_C))	/* -C dir/name */
	myerror("-C option argument should not contain directory components");
      else if (opt_C[0] == '-')	/* -C -d  (?) */
	myerror("-C option needs argument");
      else			/* -C name */
	command_name = opt_C;
    } else {			/* -C -2 */
      myerror
	("-C option needs string or positive number as argument, perhaps you meant -C %d?",
	 -countback);
    }
  } else {			/* no -C option given, use command name */
    command_name = mysavestring(mybasename(argv[optind]));
  }
  assert(command_name != NULL);
  return command_name;
}


/*
 * Since version 0.24, rlwrap only writes to master_pty
 * asynchronously, keeping a queue of pending output. The readline
 * line handler calls put_in_output_queue(usder_input) , while
 * main_loop calls flush_output_queue() as long as there is something
 * in the queue.
 */

static char *output_queue; /* NULL when empty */

int
output_queue_is_nonempty()
{
  return (output_queue ? TRUE : FALSE);
}

void
put_in_output_queue(char *stuff)
{
  if (!output_queue)
    output_queue = mysavestring(stuff);           /* allocate queue (no need to free anything) */
  else {
    char *old_queue = output_queue;

    output_queue = add2strings(old_queue, stuff); /* allocate new queue */
    free(old_queue);                              /* free the old one   */
  }
}

void
flush_output_queue()
{
  int nwritten, queuelen, how_much;
  char *old_queue = output_queue;

  if (!output_queue)
    return;
  queuelen = strlen(output_queue);
  how_much = min(BUFFSIZE, queuelen); /* never write more than BUFFSIZE in one go @@@ is this never too much? */
  nwritten = write(master_pty_fd, output_queue, how_much);
  if (nwritten < 0) {
    struct timeval ten_millisecs = { 0, 10000 };
    switch (nwritten) {
      case EINTR:
	return;
      case EAGAIN:
	select(0, NULL, NULL, NULL, &ten_millisecs);	/* wait a little before re-trying */
	return;
      default:
	myerror("write to master pty failed");
    }
  }

  if (!output_queue[nwritten]) /* nothing left in queue */
    output_queue = NULL;
  else
    output_queue = mysavestring(output_queue + nwritten);	/* this much is left to be written */

  free(old_queue);
}





void
cleanup_rlwrap_and_exit(int status)
{
  DPRINTF0(DEBUG_TERMIO, "Cleaning up");
  if (write_histfile && (histsize==0 ||  history_total_bytes() > 0))
    /* avoid creating .speling_error_history after typo */
    write_history(history_filename);	/* ignore errors */
  close_logfile();
  if (terminal_settings_saved && (tcsetattr(STDIN_FILENO, TCSAFLUSH, &saved_terminal_settings) < 0))	/* reset terminal */
    /* ignore (almost dead anyway) */ ;	 /* mywarn ("tcsetattr error on stdin"); */
                                         /* NOT myerror, as this would call cleanup_rlwrap_and_exit again*/

#ifdef WTERMSIG                 /* is there any system without this macro? */
  if (status != EXIT_SUCCESS)
    exit(status);               /* rlwrap itself has failed, rather than the wrapped command */
  else if (WIFSIGNALED(childs_exit_status)) 
    suicide_by(WTERMSIG(childs_exit_status)); /* child terminated by signal, make rlwrap's
						 parent believe rlwrap was killed by it */ 
  else
    exit(WEXITSTATUS(childs_exit_status)); /* propagate child's exit status */
#else
  exit(status != EXIT_SUCCESS ? status : WEXITSTATUS(childs_exit_status)); /* WEXITSTATUS has a default definition in rlwrap.h */
#endif
  
}
