///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// geo: vtk input
//
// author: Pierre.Saramito@imag.fr
//
// 5 march 2012
//
#include "rheolef/geo.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"
using namespace std;
namespace rheolef { 

template <class T>
idiststream&
geo_get_vtk (idiststream& ips, geo_basic<T,sequential>& omega)
{
  using namespace std;
  typedef typename geo_basic<T,sequential>::size_type size_type;
  typedef typename geo_basic<T,sequential>::node_type node_type;

  check_macro (ips.good(), "bad input stream for vtk");
  istream& is = ips.is();
  geo_header hdr;
  hdr.dimension = 3;
  hdr.order     = 1;
  // ------------------------------------------------------------------------
  // 1) load the coordinates
  // 		POINTS <np> float
  //          {xi yi zi} i=0..np-1
  // ------------------------------------------------------------------------
  check_macro (scatch(is,"POINTS"), "unexpected vtk input file");
  size_type nnod;
  is >> nnod;
  if (nnod == 0) {
    warning_macro("empty vtk mesh file");
    return ips;
  }
  disarray<node_type, sequential> node (nnod);
  scatch (is,"float");
  node.get_values (ips);
  check_macro (ips.good(), "bad input stream for vtk");
  // ------------------------------------------------------------------------
  // 2) load the domain connectivity
  //	LINES <ne> <ntot>
  //    2  {s1 s2} j=0..ne-1
  // or
  //    POLYGONS <ne> <ntot>
  //    p  {s1 .. sp} j=0..ne-1
  // ------------------------------------------------------------------------
  string mark;
  is >> ws >> mark;
  if (mark == "VERTICES") { // skip unused paragraph in polydata v3.0
    size_type n, n2, d, idx;
    is >> n >> n2;
    for (size_type i = 0; i < n; i++) {
      is  >> d >> idx;
    }
  }
  do {
    if (mark == "POLYGONS" || mark == "LINES") break;
  } while (is >> ws >> mark);
  if (mark == "POLYGONS")      hdr.map_dimension = 2;
  else if (mark == "LINES")    hdr.map_dimension = 1;
  else error_macro ("geo: unexpected `" << mark << "' in vtk polydata input.");
  // load the connectivity
  size_type ne, ntot;
  is >> ne >> ntot;
  typedef heap_allocator<size_type>           alloc_t;
  typedef geo_element_auto<alloc_t>           element_t;
  typedef disarray<element_t,sequential,alloc_t> disarray_t;
  disarray_t elt (ne);
  for (size_type ie = 0; ie < ne; ie++) {
    size_type nloc;
    is >> ws >> nloc;
    size_type variant = reference_element::variant (nloc, hdr.map_dimension);
    elt[ie].reset (variant, hdr.order);
    for (size_type iloc = 0 ; iloc < nloc; iloc++) {
      is >> elt[ie][iloc];
    }
    hdr.dis_size_by_dimension [elt[ie].dimension()]++;
    hdr.dis_size_by_variant   [elt[ie].variant()]++;
  }
  hdr.dis_size_by_dimension [0] = nnod;
  hdr.dis_size_by_variant   [0] = nnod;
  // ------------------------------------------------------------------------
  // 2) split elements by variants
  // ------------------------------------------------------------------------
  alloc_t alloc;
  std::array<disarray_t, reference_element::max_variant>  tmp_geo_element;
  if (hdr.map_dimension > 0) {
    for (size_type variant = reference_element::first_variant_by_dimension(hdr.map_dimension);
                   variant < reference_element:: last_variant_by_dimension(hdr.map_dimension); variant++) {
      geo_element_auto<> init_val (variant, hdr.order);
      tmp_geo_element [variant] = disarray_t (hdr.dis_size_by_variant [variant], init_val, alloc);
    }
    std::array<size_type, 4> counter;
    counter.fill(0);
    for (size_type ie = 0; ie < ne; ie++) {
      size_type variant = elt[ie].variant();
      size_type ige = counter[variant];
      tmp_geo_element [variant] [ige] = elt[ie];
      counter[variant]++;
    }
  }
  // ------------------------------------------------------------------------
  // 3) build & upgrade
  // ------------------------------------------------------------------------
  omega.build_from_data (hdr, node, tmp_geo_element, true);
  return ips;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template idiststream& geo_get_vtk<Float> (idiststream&, geo_basic<Float,sequential>&);

}// namespace rheolef
