#ifndef _RHEO_BASIS_ON_POINTSET_H
#define _RHEO_BASIS_ON_POINTSET_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

//! Pre-compute a piola transformation polynomial basis
//! on a lattice definied on the reference element.
//! The computation is performed one time for all
//! the first time the reference element appears.
//! Thus, when looping on the mesh for the Lagrange interpolation
//! on a Lagrange nodal basis : the lattice is the set of nodes
//! the evaluation of the fem polynomial basis.
//! The evaluation is performed only one time.

#include "rheolef/basis.h"
#include "rheolef/quadrature.h"
namespace rheolef {

// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
template<class T>
class basis_on_pointset {
public:

    typedef typename std::vector<T>::size_type                     size_type;
    typedef typename std::vector<T>::const_iterator                const_iterator;
    typedef typename std::vector<point_basic<T> >::const_iterator  const_iterator_grad;

// allocators:

    basis_on_pointset ();
    basis_on_pointset (const quadrature<T>& quad, const basis_basic<T>& b);
    basis_on_pointset (const basis_basic<T>& nb,  const basis_basic<T>& b);

// modifiers:

    void set (const quadrature<T>& quad, const basis_basic<T>& b);
    void set (const basis_basic<T>& nb,  const basis_basic<T>& b);

// accessors:

    const basis_basic<T>& get_basis() const { return _b; }
    size_type size (reference_element hat_K) const;

    // old interface:
    const_iterator begin (reference_element hat_K, size_type q) const;
    const_iterator end   (reference_element hat_K, size_type q) const;

    const_iterator_grad begin_grad (reference_element hat_K, size_type q) const;
    const_iterator_grad end_grad   (reference_element hat_K, size_type q) const;

    // new interface:
    void evaluate      (reference_element hat_K, size_type q) const;
    void evaluate_grad (reference_element hat_K, size_type q) const;
    void evaluate      (reference_element hat_K, const point_basic<T>& hat_xq) const;
    void evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const;
    const_iterator      begin() const;
    const_iterator      end() const;
    const_iterator_grad begin_grad() const;
    const_iterator_grad end_grad() const;

// data:
    typedef enum {
      quad_mode     = 0,
      nodal_mode    = 1,
      specific_mode = 2,
      max_mode      = 3
    } mode_type;
protected:
    basis_basic<T>                                      _b;
    mutable mode_type                                   _mode;
    const quadrature<T>*                                _p_quad; // when mode: on quadrature pointset
    basis_basic<T>                                      _nb;     // when mode: on nodal basis pointset
    mutable std::vector<std::vector<T> >                _val         [reference_element::max_variant];
    mutable std::vector<std::vector<point_basic<T> > >  _grad_val    [reference_element::max_variant];
    mutable std::vector<bool>                           _initialized;
    mutable std::vector<bool>                           _grad_initialized;
    mutable std::vector<T>                              _val_specific;
    mutable std::vector<point_basic<T> >                _grad_val_specific;
    mutable size_type                                   _curr_K_variant;
    mutable size_type                                   _curr_q;
    void _initialize          (reference_element hat_K) const;
    void _grad_initialize     (reference_element hat_K) const;
};
// ----------------------------------------------------------
// inlined
// ----------------------------------------------------------
template<class T>
inline
basis_on_pointset<T>::basis_on_pointset ()
 : _b(),
   _mode(max_mode),
   _p_quad(0),
   _nb(),
   _val(),
   _grad_val(),
   _initialized(reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _val_specific(),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q (std::numeric_limits<size_type>::max())
{
}
template<class T>
inline
basis_on_pointset<T>::basis_on_pointset (const quadrature<T>& quad, const basis_basic<T>& b)
 : _b(b),
   _mode(quad_mode),
   _p_quad(&quad),
   _nb(),
   _val(),
   _grad_val(),
   _initialized(reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _val_specific(),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q (std::numeric_limits<size_type>::max())
{
}
template<class T>
inline
basis_on_pointset<T>::basis_on_pointset (const basis_basic<T>& nb, const basis_basic<T>& b)
 : _b(b),
   _mode(nodal_mode),
   _p_quad(0),
   _nb(nb),
   _val(),
   _grad_val(),
   _initialized(reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _val_specific(),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q (std::numeric_limits<size_type>::max())
{
}
template<class T>
inline
void
basis_on_pointset<T>::set (const quadrature<T>& quad, const basis_basic<T>& b)
{
    // data was not defined on this quadrature formulae : reset
    _b  = b;
    _mode = quad_mode;
    _p_quad = &quad;
    std::fill (_initialized.begin(), _initialized.end(), false);
    std::fill (_grad_initialized.begin(), _grad_initialized.end(), false);
}
template<class T>
inline
void
basis_on_pointset<T>::set (const basis_basic<T>& nb, const basis_basic<T>& b)
{
    // data was not defined on this quadrature formulae : reset
    _b  = b;
    _mode = nodal_mode;
    _nb = nb;
    std::fill (_initialized.begin(), _initialized.end(), false);
    std::fill (_grad_initialized.begin(), _grad_initialized.end(), false);
}
// ==================================================================================
// old interface
// ==================================================================================
template<class T>
inline
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::begin (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_variant = hat_K.variant();
    if (!_initialized [K_variant]) _initialize (hat_K);
    return _val[K_variant][q].begin();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::end (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_variant = hat_K.variant();
    if (!_initialized [K_variant]) _initialize (hat_K);
    return _val[K_variant][q].end();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::begin_grad (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_variant = hat_K.variant();
    if (!_grad_initialized [K_variant]) _grad_initialize (hat_K);
    return _grad_val[K_variant][q].begin();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::end_grad (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_variant = hat_K.variant();
    if (!_grad_initialized [K_variant]) _grad_initialize (hat_K);
    return _grad_val[K_variant][q].end();
}
// ==================================================================================
// new interface
// ==================================================================================
// -------------------
// scalar
// -------------------
template<class T>
inline
void
basis_on_pointset<T>::evaluate (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_variant = hat_K.variant();
    if (!_initialized [K_variant]) _initialize (hat_K);
    _curr_K_variant = K_variant;
    _curr_q     = q;
}
template<class T>
inline
void
basis_on_pointset<T>::evaluate (reference_element hat_K, const point_basic<T>& hat_xq) const
{
    size_type nb = _b.size(hat_K);
    if (nb != _val_specific.size ()) { _val_specific.resize (nb); }
    _b.eval (hat_K, hat_xq, _val_specific);
    _mode = specific_mode;
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::begin() const
{
    if (_mode != specific_mode) return _val[_curr_K_variant][_curr_q].begin();
    else                        return _val_specific.begin();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::end() const
{
    if (_mode != specific_mode) return _val[_curr_K_variant][_curr_q].end();
    else                        return _val_specific.end();
}
// -------------------
// grad
// -------------------
template<class T>
inline
void
basis_on_pointset<T>::evaluate_grad (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_variant = hat_K.variant();
    if (!_grad_initialized [K_variant]) _grad_initialize (hat_K);
    _curr_K_variant = K_variant;
    _curr_q     = q;
}
template<class T>
inline
void
basis_on_pointset<T>::evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const
{
    size_type nb = _b.size(hat_K);
    if (nb != _grad_val_specific.size ()) { _grad_val_specific.resize (nb); }
    _b.grad_eval (hat_K, hat_xq, _grad_val_specific);
    _mode = specific_mode;
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::begin_grad() const
{
    if (_mode != specific_mode) return _grad_val[_curr_K_variant][_curr_q].begin();
    else                        return _grad_val_specific.begin();
}
template<class T>
inline
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::end_grad() const
{
    if (_mode != specific_mode) return _grad_val[_curr_K_variant][_curr_q].end();
    else                        return _grad_val_specific.end();
}

}// namespace rheolef
#endif // _RHEO_BASIS_ON_POINTSET_H
