;;;-*-Emacs-Lisp-*-
(defconst mdb-version "$Id: mdb.el.in,v 1.5 2006/03/19 20:38:14 rockyb Exp $"
  "`mdb-mode' version number.")
;;; $Id: mdb.el.in,v 1.5 2006/03/19 20:38:14 rockyb Exp $
;;; GNU Make debugger mode via GUD and makedb

;; Copyright (C) 2004, 2005, 2006 Rocky Bernstein (rocky@panix.com) 

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 51 Franklin Street, Fifth
;; Floor, Boston, MA 02110-1301 USA.

;; Commentary:
;; 1. Add
;;
;; (autoload 'mdb "mdb" "GNU make Debugger mode via GUD and mdb" t)
;;
;; to your .emacs file.
;; 2. Do M-x mdb

;; Codes:
(require 'gud)
;; ======================================================================
;; mdb functions

;; History of argument lists passed to mdb.
(defvar gud-mdb-history nil)

;; Convert a command line as would be typed normally to run a script
;; into one that invokes an Emacs-enabled debugging session.
;; "--debugger" in inserted as the first switch.

(defun gud-mdb-massage-args (file args)
  (let* ((new-args (list "--debugger"))
	 (seen-e nil)
	 (shift (lambda ()
		  (setq new-args (cons (car args) new-args))
		  (setq args (cdr args)))))
    
    ;; Pass all switches and -e scripts through.
    (while (and args
		(string-match "^-" (car args))
		(not (equal "-" (car args)))
		(not (equal "--" (car args))))
      (funcall shift))
    
    (if (or (not args)
	    (string-match "^-" (car args)))
	(error "Can't use stdin as the script to debug"))
    ;; This is the program name.
    (funcall shift)

  (while args
    (funcall shift))
  
  (nreverse new-args)))

;; There's no guarantee that Emacs will hand the filter the entire
;; marker at once; it could be broken up across several strings.  We
;; might even receive a big chunk with several markers in it.  If we
;; receive a chunk of text which looks like it might contain the
;; beginning of a marker, we save it here between calls to the
;; filter.
(defun gud-mdb-marker-filter (string)
  (setq gud-marker-acc (concat gud-marker-acc string))
  (let ((output ""))

    ;; Process all the complete markers in this chunk.
    ;; Format of line looks like this:
    ;;   (/src/remake/Makefile:16):
    ;; but we also allow DOS drive letters
    ;;   (d:/src/remake/Makefile:16):
    (while (string-match "\\(^\\|\n\\)(\\(\\([a-zA-Z]:\\)?[^:\n]*\\):\\([0-9]*\\)).*\n"
			 gud-marker-acc)
      (setq

       ;; Extract the frame position from the marker.
       gud-last-frame
       (cons (match-string 2 gud-marker-acc)
	     (string-to-int (match-string 4 gud-marker-acc)))

       ;; Append any text before the marker to the output we're going
       ;; to return - we don't include the marker in this text.
       output (concat output
		      (substring gud-marker-acc 0 (match-beginning 0)))

       ;; Set the accumulator to the remaining text.
       gud-marker-acc (substring gud-marker-acc (match-end 0))))

    ;; Does the remaining text look like it might end with the
    ;; beginning of another marker?  If it does, then keep it in
    ;; gud-marker-acc until we receive the rest of it.  Since we
    ;; know the full marker regexp above failed, it's pretty simple to
    ;; test for marker starts.
    (if (string-match "\032.*\\'" gud-marker-acc)
	(progn
	  ;; Everything before the potential marker start can be output.
	  (setq output (concat output (substring gud-marker-acc
						 0 (match-beginning 0))))

	  ;; Everything after, we save, to combine with later input.
	  (setq gud-marker-acc
		(substring gud-marker-acc (match-beginning 0))))

      (setq output (concat output gud-marker-acc)
	    gud-marker-acc ""))

    output))

(defun gud-mdb-find-file (f)
  (save-excursion
    (let ((buf (find-file-noselect f 'nowarn)))
      (set-buffer buf)
      buf)))


(defcustom gud-mdb-command-name "remake -f Makefile"
  "File name for executing make debugger."
  :type 'string
  :group 'gud)

;;;###autoload
(defun mdb (command-line)
  "Run mdb on program FILE in buffer *gud-FILE*.
The directory containing FILE becomes the initial working directory
and source-file directory for your debugger."
  (interactive
   (list (read-from-minibuffer "Run mdb (like this): "
			       (if (consp gud-mdb-history)
				   (car gud-mdb-history)
				 (concat gud-mdb-command-name
					 " "))
			       gud-minibuffer-local-map nil
			       '(gud-mdb-history . 1))))

  (gud-common-init command-line 'gud-mdb-massage-args
		   'gud-mdb-marker-filter 'gud-mdb-find-file)

  (set (make-local-variable 'gud-minor-mode) 'mdb)

  ; (gud-def gud-break  "break %l"   "\C-b" "Set breakpoint at current line.")
  ; (gud-def gud-tbreak "tbreak %l"  "\C-t" "Set temporary breakpoint at current line.")
  ; (gud-def gud-remove "clear %l"   "\C-d" "Remove breakpoint at current line")
  (gud-def gud-step   "step"       "\C-s" "Step one source line with display.")
  (gud-def gud-next   "next"       "\C-n" "Step one line (skip functions).")
  (gud-def gud-cont   "continue"   "\C-r" "Continue with display.")
  (gud-def gud-finish "finish"     "\C-f" "Finish executing current function.")
  (gud-def gud-up     "up %p"      
	   "<" "Up N stack frames (numeric arg).")
  (gud-def gud-down   "down %p"    
	   ">" "Down N stack frames (numeric arg).")
  (gud-def gud-where   "where"
	   "T" "Show stack trace.")

  (gud-def gud-print  "p %e"
	   "\C-p" "Evaluate make expression at point.")

  (gud-def gud-statement "x %e" 
	   "\C-e" "Expand make statement at point.")

  (define-key gud-menu-map [args]      '("Show arguments of current stack" . 
					 gud-args))
  (define-key gud-menu-map [break]     nil)
  (define-key gud-menu-map [down]      '("Down Stack" . gud-down))
  (define-key gud-menu-map [eval]      '("Expand GNU Make expression at point"
					 . gud-statement))
  (define-key gud-menu-map [finish]    '("Finish Function" . gud-finish))
  (define-key gud-menu-map [remove]    nil)
  (define-key gud-menu-map [run]       '("Restart GNU make" . gud-run))
  (define-key gud-menu-map [stepi]     nil)
  (define-key gud-menu-map [tbreak]    nil)
  (define-key gud-menu-map [up]        '("Up Stack" . gud-up))
  (define-key gud-menu-map [where]     '("Show stack trace" . gud-where))

  (local-set-key [menu-bar debug tbreak] nil)
  (local-set-key [menu-bar debug finish] '("Finish Function" . gud-finish))
  (local-set-key [menu-bar debug up] '("Up Stack" . gud-up))
  (local-set-key [menu-bar debug down] '("Down Stack" . gud-down))

  (setq comint-prompt-regexp "^mdb<.*> ")
  (setq paragraph-start comint-prompt-regexp)
  (run-hooks 'mdb-mode-hook)
  )

(provide 'mdb)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mdbtrack --- tracking mdb debugger in an Emacs shell window
;;; Modified from  python-mode in particular the part:
;; pdbtrack support contributed by Ken Manheimer, April 2001.

;;; Code:

(require 'comint)
(require 'custom)
(require 'cl)
(require 'compile)
(require 'shell)

(defgroup mdbtrack nil
  "mdb file tracking by watching the prompt."
  :prefix "mdb-mdbtrack-"
  :group 'shell)


;; user definable variables
;; vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv

(defcustom mdb-mdbtrack-do-tracking-p t
  "*Controls whether the mdbtrack feature is enabled or not.
When non-nil, mdbtrack is enabled in all comint-based buffers,
e.g. shell buffers and the *Python* buffer.  When using mdb to debug a
Python program, mdbtrack notices the mdb prompt and displays the
source file and line that the program is stopped at, much the same way
as gud-mode does for debugging C programs with gdb."
  :type 'boolean
  :group 'make)
(make-variable-buffer-local 'mdb-mdbtrack-do-tracking-p)

(defcustom mdbtrack-minor-mode-string " MDB"
  "*String to use in the minor mode list when mdbtrack is enabled."
  :type 'string
  :group 'make)

(defcustom mdb-temp-directory
  (let ((ok '(lambda (x)
	       (and x
		    (setq x (expand-file-name x)) ; always true
		    (file-directory-p x)
		    (file-writable-p x)
		    x))))
    (or (funcall ok (getenv "TMPDIR"))
	(funcall ok "/usr/tmp")
	(funcall ok "/tmp")
	(funcall ok "/var/tmp")
	(funcall ok  ".")
	(error
	 "Couldn't find a usable temp directory -- set `mdb-temp-directory'")))
  "*Directory used for temporary files created by a *Python* process.
By default, the first directory from this list that exists and that you
can write into: the value (if any) of the environment variable TMPDIR,
/usr/tmp, /tmp, /var/tmp, or the current directory."
  :type 'string
  :group 'make)


;; ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;; NO USER DEFINABLE VARIABLES BEYOND THIS POINT

;; have to bind mdb-file-queue before installing the kill-emacs-hook
(defvar mdb-file-queue nil
  "Queue of Makefile temp files awaiting execution.
Currently-active file is at the head of the list.")

(defvar mdb-mdbtrack-tracking-p t)


;; Constants

(defconst mdb-position-re 
  "\\(^\\|\n\\)(\\([^:]+\\):\\([0-9]*\\)).*\n"
  "Regular expression for a mdb position")

(defconst mdb-marker-regexp-file-group 2
  "Group position in mdb-postiion-re that matches the file name.")

(defconst mdb-marker-regexp-line-group 3
  "Group position in mdb-position-re that matches the line number.")



(defconst mdb-traceback-line-re
  "^#[0-9]+[ \t]+\\((\\([a-zA-Z-.]+\\) at (\\(\\([a-zA-Z]:\\)?[^:\n]*\\):\\([0-9]*\\)).*\n"
  "Regular expression that describes tracebacks.")

;; mdbtrack contants
(defconst mdb-mdbtrack-stack-entry-regexp
  "^=>#[0-9]+[ \t]+\\((\\([a-zA-Z-.]+\\) at (\\(\\([a-zA-Z]:\\)?[^:\n]*\\):\\([0-9]*\\)).*\n"
  "Regular expression mdbtrack uses to find a stack trace entry.")

(defconst mdb-mdbtrack-input-prompt "\nmdb<+.*>+ "
  "Regular expression mdbtrack uses to recognize a mdb prompt.")

(defconst mdb-mdbtrack-track-range 10000
  "Max number of characters from end of buffer to search for stack entry.")


;; Utilities
(defmacro mdb-safe (&rest body)
  "Safely execute BODY, return nil if an error occurred."
  (` (condition-case nil
	 (progn (,@ body))
       (error nil))))


;;;###autoload

(defun mdb-mdbtrack-overlay-arrow (activation)
  "Activate or de arrow at beginning-of-line in current buffer."
  ;; This was derived/simplified from edebug-overlay-arrow
  (cond (activation
	 (setq overlay-arrow-position (make-marker))
	 (setq pos (point))
	 (setq overlay-arrow-string "=>")
	 (set-marker overlay-arrow-position (point) (current-buffer))
	 (setq mdb-mdbtrack-tracking-p t))
	(mdb-mdbtrack-tracking-p
	 (setq overlay-arrow-position nil)
	 (setq mdb-mdbtrack-tracking-p nil))
	))

(defun mdb-mdbtrack-track-stack-file (text)
  "Show the file indicated by the mdb stack entry line, in a separate window.
Activity is disabled if the buffer-local variable
`mdb-mdbtrack-do-tracking-p' is nil.

We depend on the mdb input prompt matching `mdb-mdbtrack-input-prompt'
at the beginning of the line.
" 
  ;; Instead of trying to piece things together from partial text
  ;; (which can be almost useless depending on Emacs version), we
  ;; monitor to the point where we have the next mdb prompt, and then
  ;; check all text from comint-last-input-end to process-mark.
  ;;
  ;; Also, we're very conservative about clearing the overlay arrow,
  ;; to minimize residue.  This means, for instance, that executing
  ;; other mdb commands wipe out the highlight.  You can always do a
  ;; 'where' (aka 'w') command to reveal the overlay arrow.
  (let* ((origbuf (current-buffer))
	 (currproc (get-buffer-process origbuf)))

    (if (not (and currproc mdb-mdbtrack-do-tracking-p))
        (mdb-mdbtrack-overlay-arrow nil)

      (let* ((procmark (process-mark currproc))
             (block (buffer-substring (max comint-last-input-end
                                           (- procmark
                                              mdb-mdbtrack-track-range))
                                      procmark))
             target target_fname target_lineno)

        (if (not (string-match (concat mdb-mdbtrack-input-prompt "$") block))
            (mdb-mdbtrack-overlay-arrow nil)

          (setq target (mdb-mdbtrack-get-source-buffer block))

          (if (stringp target)
              (message "mdbtrack: %s" target)

            (setq target_lineno (car target))
            (setq target_buffer (cadr target))
            (setq target_fname (buffer-file-name target_buffer))
            (switch-to-buffer-other-window target_buffer)
            (goto-line target_lineno)
            (message "mdbtrack: line %s, file %s" target_lineno target_fname)
            (mdb-mdbtrack-overlay-arrow t)
            (pop-to-buffer origbuf t)

            )))))
  )

(defun mdb-mdbtrack-get-source-buffer (block)
  "Return line number and buffer of code indicated by block's traceback text.

We look first to visit the file indicated in the trace.

Failing that, we look for the most recently visited python-mode buffer
with the same name or having 
having the named function.

If we're unable find the source code we return a string describing the
problem as best as we can determine."

  (if (not (string-match mdb-position-re block))

      "line number cue not found"

    (let* ((filename (match-string mdb-marker-regexp-file-group block))
           (lineno (string-to-int 
		    (match-string mdb-marker-regexp-line-group block)))
           funcbuffer)

      (cond ((file-exists-p filename)
             (list lineno (find-file-noselect filename)))

            ((= (elt filename 0) ?\<)
             (format "(Non-file source: '%s')" filename))

            (t (format "Not found: %s" filename)))
      )
    )
  )


;;; Subprocess commands



;; mdbtrack functions
(defun mdb-mdbtrack-toggle-stack-tracking (arg)
  (interactive "P")
  (if (not (get-buffer-process (current-buffer)))
      (error "No process associated with buffer '%s'" (current-buffer)))
  ;; missing or 0 is toggle, >0 turn on, <0 turn off
  (if (or (not arg)
	  (zerop (setq arg (prefix-numeric-value arg))))
      (setq mdb-mdbtrack-do-tracking-p (not mdb-mdbtrack-do-tracking-p))
    (setq mdb-mdbtrack-do-tracking-p (> arg 0)))
  (message "%sabled Make's mdbtrack"
           (if mdb-mdbtrack-do-tracking-p "En" "Dis")))

(defun turn-on-mdbtrack ()
  (interactive)
  (mdb-mdbtrack-toggle-stack-tracking 1))

(defun turn-off-mdbtrack ()
  (interactive)
  (mdb-mdbtrack-toggle-stack-tracking 0))

;; mdbtrack
(add-hook 'comint-output-filter-functions 'mdb-mdbtrack-track-stack-file)

;; Add a designator to the minor mode strings
(or (assq 'mdb-mdbtrack-minor-mode-string minor-mode-alist)
    (push '(mdb-mdbtrack-is-tracking-p
	    mdb-mdbtrack-minor-mode-string)
	  minor-mode-alist))


;;; mdbtrack.el ends here
;;; mdb.el ends here
